#ifndef _STATUS_INFO_H_
#define _STATUS_INFO_H_

#include <string>
#include <map>

using namespace std;

namespace GraphGraphics
{

class GraphViewContext;
    
class StatusInfo
{
protected:
  string            m_status_info_name;
  GraphViewContext  *m_context;        
public:
  StatusInfo(const string& name, GraphViewContext *context) : m_status_info_name(name), m_context(context) {}
  virtual ~StatusInfo() {}
  
  virtual string get_status_str() = 0;
      
  string get_status_name() { return m_status_info_name; }    
};

typedef map<string, StatusInfo*> StatusMap;
typedef map<string, StatusInfo*>::iterator StatusMapIter;

class StatusManager
{
private:
  StatusMap   m_statuses;
  StatusInfo  *m_current_status;
  bool        m_is_blocked;
public:
  StatusManager() : m_is_blocked(false) {}
  
  ~StatusManager() 
  {
    //StatusInfo    *status;  
    StatusMapIter iter = m_statuses.begin();
    StatusMapIter iter_end = m_statuses.end();      
      
    while (iter != iter_end)
    {
      delete iter->second;
      ++iter;        
    }
  }    

  void block(bool is_blocked = true)
  {
    m_is_blocked = is_blocked;
  }
  
  void status_registration(StatusInfo *s)
  {
    m_statuses[s->get_status_name()] = s;
  }
    
  StatusInfo* get_status(const string& status_name)
  {
    StatusMapIter  iter;
    
    iter = m_statuses.find(status_name);
    
    if (iter != m_statuses.end())
      return (*iter).second;
    else
      return 0;
  }

  StatusInfo* get_current_status()
  {
    return m_current_status;
  }

  bool        set_current_status(string& status_name)
  {
    StatusMapIter  iter;
    
    if (m_is_blocked)
      return false;
    
    iter = m_statuses.find(status_name);
      
    if (iter != m_statuses.end())
    {
      m_current_status = (*iter).second;
      return true;  
    }
    else
      return false;
  }
};

class DefaultStatusInfo : public StatusInfo
{
private:
  unsigned int m_nodes_number, m_edges_number, m_sel_nodes_number, m_sel_edges_number;  
public:
  DefaultStatusInfo(GraphViewContext *context) : StatusInfo("default", context) {};
  ~DefaultStatusInfo() {}    

  void set_default_info(unsigned int nodes_number, unsigned int edges_number, unsigned int sel_nodes_number, unsigned int sel_edges_number)    
  {
    m_nodes_number = nodes_number; 
    m_edges_number = edges_number;
    m_sel_nodes_number = sel_nodes_number;
    m_sel_edges_number = sel_edges_number;        
  }

  string get_status_str();
};

class HelpStatusInfo : public StatusInfo
{
private:
  string  m_help_str;
public:
  HelpStatusInfo(GraphViewContext *context) : StatusInfo("help", context), m_help_str("") {};
  ~HelpStatusInfo() {}    

  string get_status_str()
  {
    return m_help_str;
  }    
    
  void  set_help_string(string& help_str)
  {    
    m_help_str = help_str;
  }      
};

}

#endif
