//Implementation of graph laout method.

#ifndef _LAYOUT_H_
#define _LAYOUT_H_

#include <list>

using namespace std;

namespace GraphGraphics
{

typedef vector<struct VRank*> VRankVector;
typedef int (*qsort_cmpf)(const void*, const void*);

enum {
	SPLINE,
	SPLINE4,
	LINE,
	PIXEL
};

struct VRank
{
	int					n;			/* number of nodes in this rank			*/
	VNodeVector	v;			/* ordered list of nodes in rank		*/
	int					an;			/* globally allocated number of nodes	*/
	VNodeVector	av;			/* allocated list of nodes in rank		*/
	int 				ht1,ht2;	/* height below/above centerline		*/
	int 				pht1,pht2;	/* as above, but only primitive nodes 	*/
	bool			candidate;	/* for transpose ()						*/
	bool			valid;
	int				cache_nc;	/* caches number of crossings			*/
/*	adjmatrix_t		*flat;*/
};

/**
* This class incapsulates differents algorithms for graph layout.
* There are 4 stages of layout building:
* 1. Ranking of the nodes. An efficient network simplex algorithm is used for nodes ranking. This algorithm find optimal ordering
*    of graph nodes in vertical direction. 
* 2. Reducing of edge crossings. Some heuristics are implemented to reduce edge crossing. These heuristics work good 
*    for many graphs.
* 3. Node x coordinates assigning. Here we again use simplex network algorithm. But now this algorithm finds optimal x 
*    coordinate of the nodes. This is most slow part of the layout building. For large graph this step can be disabled and 
*    in this case distances between nodes in a rank will be the same for all rank nodes. 
* 4. Edge drawing. For edge drawing an efficient algortihm is used that finds base points of an edge to be used for spline drawing.
* 
* Algorithms of the stages 1-3 are taken from the paper 
* Emden R. Gansner, Eleftherios Koutsofios, Stephen C. North, Kiem-Phong Vo A Technique for Drawing Directed Graphs, AT&T Bell Laboratories
* Murray Hill, New Jersey 07974
*
*/

class GraphLayout
{
	private:
		
		enum {
			SEARCHSIZE = 30,
			MC_SCALE = 256
		};
	
		OneDValue	m_radius; // Radius of the nodes
		OneDValue	m_x_separation; // Horizontal separation between nodes 		
		OneDValue	m_y_separation;	// Vertical separation between nodes			
		
		int	max_ranks_nodes; // Max numbers of vertex inside a rank of the current graph
		int	*Count;	//Numbers of edges in each order position
		
		VNodeVector& m_nodes;
		VEdgeVector& m_edges;
		
		VNodeVector  m_nodes_tmp;
		VEdgeVector	 m_edges_tmp;		

		VNodeVector	m_nodes_comp;

      ViewGroupMap&	m_view_groups;		
		
//Following vectors are used to store spanning tree of the input graph which are used in ranking algorithm	
		VNodeVector Tree_node;
		VEdgeVector Tree_edge;

		VEdgeVector tree_in;
		VEdgeVector tree_out;	
	
		int	N_nodes, N_edges;	
		int	S_i;		/* search index for enter_edge */	
		int	Search_size;
		VEdge	*Enter;	
		int		Low,Lim,Slack;	
		
		int		MinQuit;
		int		MaxIter;
		double	Convergence;

		int		*TI_list;	/*List which will be used to store median values of the vertexes*/
		
		inline int LENGTH(VEdge *e)	{ return m_nodes[e->head_indx]->rank - m_nodes[e->tail_indx]->rank; }	
		inline int SLACK(VEdge *e)	{ return LENGTH(e) - e->minlen; }
		inline bool TREE_EDGE(VEdge *e)	{ return e->tree_index >= 0; }
		inline bool SEQ(int a, int b, int c) { return (a <= b) && (b <= c); }
	
// Layout_Rank functions
		void dfs(VNode *v_node);
  	void comp_dfs(VNode *v, VNodeList &node_list);
/*		void comp_dfs_down(VNode *v, VNodeList &node_list);		*/
    
		void reverse_edge(VEdge *v_edge);	
		bool init_graph();
    
/**
* The function computes initial feasible ranking
*/
		void init_rank(void);	

/**
* The function finds a maximal tree of tight edges containing some fixed node and returns the number of nodes in the tree
*/
		int tight_tree(void);	
		bool treesearch(VNode* v);		
		void add_tree_edge(VEdge* e);
		NodeIndex incident(VEdge* e);
    
/**
* This function constructs an initial feasible spanning tree. The simplex method starts with a feasible solution 
* and maintains this invariant.    
* @see GraphLayout::init_rank()
* @see GraphLayout::tight_tree()
* @see GraphLayout::init_cutvalues()    
*/    
		void feasible_tree(void);
		void x_cutval(VEdge* f);
		int x_val(VEdge* e, VNode* v, int dir);
		void dfs_cutval(VNode* v, VEdge* par);
		int dfs_range(VNode* v, VEdge* par, int low);
    
/**
* The function computes the cut values of the tree edges. For each tree edge, this is computed by marking the nodes as
* belonging to the head or tail component, and then prforming the sum of the signed weights of all edges whose head and 
* tail are in different components, the sign being negative for those adges going from the head to the tail component.
*/
		void init_cutvalues(void);
    
/**
* The function returns a tree edge with a negative cut value, or nil if there is none, meaning the solution is optimal. Any 
* edge with a negative cut value may be selected as the edge to remove.
*/
		VEdge *leave_edge (void);
    
/**
* This function finds a non-tree edge to replace e. This is done by breaking the edge e, which divides the tree into 
* a head and tail component. All edges going from the head component to the tail are considered, with an edge of 
* minimum slack being chosen. This is necessary to maintain feasibility.
*/
		VEdge *enter_edge(VEdge* e);
		void dfs_enter_outedge(VNode* v);
		void dfs_enter_inedge(VNode* v);			
		VNode* treeupdate(VNode *v, VNode *w, int cutvalue, bool dir);
    
/**
* The function exchanges edges, updates the tree and its cut values
*/
		void exchange_tree_edges(VEdge *e, VEdge *f);
		void rerank(VNode* v, int delta);
		void update(VEdge *e, VEdge *f);		
    
/** 
* The function normalize rank values by setting least rank to zero
*/
		void scan_and_normalize(void);	
    
		void LR_balance(void);		
    
		void debug_info(char *info);
    
		void cut_rank_info(char *info);		
		
// Layout_Mincross functions
		bool medians(int r0, int r1);
		void mincross_step(int pass);		
		void reorder(int r, bool reverse, bool hasfixed);
		void exchange(VNode *v, VNode *w);		
		int rcross(int r);
		int ncross();		
		void install_in_rank(VNode* n);
		void build_ranks(int pass);
		void enqueue_neighbors(VNodeQueue& q, VNode* n0, int pass);
		void restore_best();
		void save_best();		
		NodeIndex new_virtual_node();
		VEdge* new_virtual_edge(NodeIndex u_indx, NodeIndex v_indx);
		void make_chain(NodeIndex from_indx, NodeIndex to_indx, bool inverse_edge);		
		
// Layout_Position functions
		void allocate_aux_edges();
		void	make_aux_edge(OneDValue u_indx, OneDValue v_indx, int len, int wt);
		void make_LR_constraints();
		void make_edge_pairs();
		void set_ycoords();		
		void set_xcoords();		
		void remove_aux_edges();	

// Layout_Edges functions


	public:
		GraphLayout(VNodeVector& nodes, VEdgeVector& edges, ViewGroupMap& view_groups) : m_nodes(nodes), m_edges(edges), m_view_groups(view_groups) 
		{
      Count = 0;
		}
		
		~GraphLayout() 
		{
			NodeIndex	i;
			
			for (i = 0; i < (long)m_ranks.size(); i++)
				delete m_ranks[i];
      
 			if (Count != 0)
        delete[] Count;
		}
			
   bool  less_order(VNode* n, VNode* v)
   {
      if (n->order < v->order)
         return true;
      else
         return false;
   }
	
		void	SetRadius(OneDValue radius) { m_radius = radius; }
		OneDValue	GetRadius() { return m_radius; }

		void	SetSeparation(OneDValue x_separation, OneDValue y_separation) { m_x_separation = x_separation; m_y_separation = y_separation; }
/*		OneDValue	GetSeparation() { return m_separation; }*/

		int	minrank, maxrank;	// min and max numbers of ranks;	
		VRankVector	 m_ranks;			
    
/**
* The function make the graph acyclic. It is based on depth-first search 
*/ 		
		void acyclic();
    
    void rank_connected_comps(int balance, int maxiter);
    
/**
* This function finds an optimal ranking of the graph nodes. Finding an optimal ranking can be formulated as the following integer 
* problem:
*   min(sum(e.weight)*(w.rank-v.rank), for all e from E), subject to (w.rank-v.rank) >= e.minlen for all e from E.
* Here: e = (v,w) - edge of the graph, v,w - nodes of the graph.    
* We use network simplex method to solve this problem. A few definitions. A feasible ranking is one satisfying the length constraints
* w.rank - v.rank >= e.minlen for all e = (v,w), value of w.rank - v.rank will be called as a length of the edge. Given any ranking, 
* not necessary feasible, the slack of an edge is the difference of its length and its minimum length. Thus, a ranking is feasible if 
* the slack of every edge is non-negative. An edge is tight if its slack is zero.
* A spanning tree of a graph induces a ranking, or rather, a family of equivalent rankings. (Note that the
* spanning tree is on the underlying unrooted undirected graph, and is not necessarily a directed tree.)
* This ranking is generated by picking an initial node and assigning it a rank. Then, for each node
* adjacent in the spanning tree to a ranked node, assign it the rank of the adjacent node, incremented or
* decremented by the minimum length of the connecting edge, depending on whether it is the head or tail
* of the connecting edge. This process is continued until all nodes are ranked. A spanning tree is feasible
* if it induces a feasible ranking. By construction, all edges in the feasible tree are tight.
* Given a feasible spanning tree, we can associate an integer cut value with each tree edge as follows. If
* the tree edge is deleted, the tree breaks into two connected components, the tail component containing
* the tail node of the edge, and the head component containing the head node. The cut value is defined as
* the sum of the weights of all edges from the tail component to the head component, including the tree
* edge, minus the sum of the weights of all edges from the head component to the tail component.
* Typically (but not always because of degeneracy) a negative cut value indicates that the weighted edge
* length sum could be reduced by lengthening the tree edge as much as possible, until one of the head
* component-to-tail component edges becomes tight. This corresponds to replacing the tree edge in the
* spanning tree with the newly tight edge, obtaining a new feasible spanning tree. It is also simple to see
* that an optimal ranking can be used to generate another optimal ranking induced by a feasible spanning
* tree. These observations are the key to solving the ranking problem in a graphical rather than algebraic
* context. Tree edges with negative cut values are replaced by appropriate non-tree edges, until all tree
* edges have non-negative cut values. The resulting spanning tree corresponds to an optimal ranking.
*/    
    void rank(int balance, int maxiter);	
		
    void init_mincross();		

/**
* The function implements heuristics for reducing edge crossings. It works as follows. First, an initial ordering 
* within each rank is computed. Then a sequence of iteration is performed to try to improve the orderings. Each 
* iteration traverses from the first rank to the last one, or vice versa. When visiting a rank, each of its 
* vertices is assigned a weight based on the relative positions of its incident vertices on the preceding rank.
* Then the vertices in the rank are re-ordered by sorting on these weights. We use here median weighting method. Let 
* v be a vertex and P the list of positions of its incedent vertices on the appropriate adjacent rank. Position 
* of an adjacent node is its ordinal number in the current ordering. The median method defines the weight of v as 
* the median of elements in P. When the number of elements in P is even then right median is used.    
*/    
		int mincross(int startpass, int endpass);			
    
/**
* The function sets (x,y) coordinates of the graph nodes. Setting of y coordinate is a straightforward - all nodes in a rank have the same
* y coordinate. Distance between two adjacent rank is m_y_separation. Calculating of x coordinates uses following algorithm. 
* We should solve following integer problem:
*   min(sum(e.weight)*|w.x-v.x|, for all e = (v,w) from E), subject to (w.x-v.x) >= 2*m_radius+m_x_separation for all e from E.
* We again use network simplex algorithm but on another graph. This auxiliary graph, G1, is built from the original graph. 
* Nodes of the G1 are the nodes of the original graph plus, for every edge e in G , there is a new node n(e). 
* There are two kinds of edges in G1. One edge class encodes the cost of the original edges. Every edge e = (u,v) in G is replaced by 
* two edges (n(e) ,u) and (n(e), v) with minlen = 0 and weight = e.weight. The other class of edges separates nodes in the same rank. If
* v is the left neighbor of w, then G1 has an edge f = e(v,w) with f.minlen = 2*m_radius+m_x_separation and f.weight = 0. This edge
* forces the nodes to be sufficiently separated but does not affect the cost of the layout. Then we find optimal layout of the G1 using 
* function rank() and obtained v.rank values will be x coordinates of the corresponded nodes.   
*/
		void position();				
		
		void set_edges_coords();	
    
/**
* The function finds coordinates of the points that are used for edge spline drawing.
*/
		void set_edges_coords_new( );		
		void set_selfedges( );
		void set_selfedges_coords( );		
};

}

#endif //_LAYOUT_H_
