#ifndef _INFO_STORAGE_CPP_
#define _INFO_STORAGE_CPP_

#include  <list>
#include  <vector>
#include  <ext/hash_map>
#include  "drawing.h" 

namespace GraphGraphics
{
  
class InfoValue;
  
/**
* Base class of the click handler class. Function object of the inherited class can be set for the value object.
* For example, for the value that is an URL the function can run browser and load the site. 
*/
class ClickHandler
{
public:
  virtual ~ClickHandler() {}
  virtual void operator() ( InfoValue &value ){} 
};

typedef list<InfoValue*> InfoValueList;
typedef list<InfoValue*>::iterator InfoValueListIter;

/**
* This class incapsulates different information about item value.
*/
class InfoValue
{
private:
  string        m_name;
  ClickHandler  *m_click_handler;
  Color         m_color;
  StyleFlags    m_font_style;

  InfoValue& operator = ( const InfoValue &value ) 
  { 
/*    m_name = value.m_name; 
    m_color = value.m_color;
    m_font_style = value.m_font_style;
    memcpy( m_click_handler, value.m_click_handler, sizeof(ClickHandler));*/
    
    return *this; 
  }

  InfoValue( const InfoValue &value )
  {
/*    m_name = value.m_name; 
    m_click_handler = value.m_click_handler;
    m_color = value.m_color;
    m_font_style = value.m_font_style;*/
  }


public:
 
/**
* Constructor of the class.
*/
  InfoValue() : m_name(""), m_color(Color::BLACK), m_font_style( Font::NONE )
  {
    m_click_handler = 0;
  };
  
/**
* Constructor of the class.
* @param name Name of the value
*/
  InfoValue(const string& name) : m_name(name), m_color(Color::BLACK), m_font_style( Font::NONE )
  {
    m_click_handler = 0;    
  };
  
/**
* Constructor of the class.
* @param name Name of the value
* @param color Value text color  
* @param color Text font style
* @see Color
*/
  InfoValue(const string& name, const Color& color, const StyleFlags& font_style) : m_name(name)
  {
    m_click_handler = 0;    
    m_color = color;
    m_font_style = font_style;
  };

/**
* Destructor of the class.
*/
  ~InfoValue()
  {
    if (m_click_handler != 0)
      delete m_click_handler;
  }
  
/**
* Sets name of the value.
*/
  void set_name(const string& name)
  {
    m_name = name;
  }

/**
* Gets name of the value.
*/
  string get_name( )
  {
    return m_name;
  }

/**
* Sets click handler object.
* @param click_handler Handler object.
* @see ClickHandler
*/
  void set_click_handler(ClickHandler *click_handler)
  {
    m_click_handler = click_handler;
  }
  
/**
* Gets click handler object.
* @return Handler object.
* @see ClickHandler
*/
  ClickHandler* get_click_handler( )
  {
    return m_click_handler;
  }

/**
* Sets color of the value text.
* @param color Color to be set.
* @see Color
*/
  void set_color(const Color& color)
  {
    m_color = color;
  }
  
/**
* Gets color of the value text.
* @return Current color.
* @see Color
*/
  Color get_color( )
  {
    return m_color;
  }

/**
* Sets font style of the value text.
* @param font_style Font style to be set.
*/
  void set_font_style(const StyleFlags& font_style)
  {
    m_font_style = font_style;    
  }
  
/**
* Gets font style of the value text.
* @return Current font style.
*/
  StyleFlags get_font_style( )
  {
    return m_font_style;    
  }
  
};

/**
* Base class for the information level.This is abstract class and object of this class can't be directly created.
* @see InfoStorageSubLevel
* @see InfoStorageItemLevel
*/
class InfoStorageLevel
{
private:  
  string            m_name;
  
public:

/**
* Constructor of the class.
* @param name Name of the level
*/
  InfoStorageLevel(const string& name) : m_name(name) { };
  
/**
* Destructor of the class.
*/
  virtual ~InfoStorageLevel() { };
  
/**
* Gets name of the level.
* @return Name of the level.
*/
  string  get_name() { return m_name; };
  void    set_name( const string& name ) { m_name = name; }
  
/**
* Virtual function that should returns status of the level. 
* @return <b>true</b> if the level is a sub-level.
*/
  virtual bool  is_sub_level() = 0;
};

/**
* This class represents sub-level of the information tree. This level can contain other sub-levels or leafs (objects of type InfoStorageItemLevel) 
* @see InfoStorageLevel
* @see InfoStorageItemLevel
*/
class InfoStorageSubLevel : public InfoStorageLevel
{
private:
  vector<InfoStorageLevel*> m_child_levels;  
public:  
/**
* Constructor of the class.
* @param name Name of the level
*/
  InfoStorageSubLevel(const string& name) : InfoStorageLevel(name) { };
  
/**
* Destructor of the class.
*/
  virtual ~InfoStorageSubLevel();

//Methods for working with sublevels

/**
* Adds new sub-level to the current one. 
* @param name_sublevel Pointer to the new sub-level
*/
  void  add_sublevel(InfoStorageLevel *new_sublevel);

/**
* Gets number of the sub-levels in the current one. 
* @return Number of sub-levels.
*/
  int get_level_count() { return m_child_levels.size(); };
  
/**
* Gets sub-level with the specified index. 
* @param Index of the sub-level.
* @return Sub-level with the index <code>i</code>  
*/
  InfoStorageLevel *get_sublevel(int i);
  
/**
* Overriden method of the base class. Returns always <code>true</code>. 
* @return <code>true</code>  
*/
  bool  is_sub_level() { return true; }

/**
* Clear storage and set it to initial state.
*/
  void  clear();
};

/**
* This class represents item level (or leaf) of the information tree. This level can contain none, one or more associated values.  
* @see InfoStorageSubLevel
* @see InfoStorageLevel
*/
class InfoStorageItemLevel : public InfoStorageLevel
{
private:  
  list<InfoValue*> m_values;   
public:  
//Constructors and destructors
/**
* Constructor of the class.
* @param name Name of the level
*/
  InfoStorageItemLevel(const string& name) : InfoStorageLevel(name) { };
  
/**
* Destructor of the class.
*/
  virtual ~InfoStorageItemLevel();
  
//Method for working with item values
  
/**
* Replaces current value list with the given one. 
* @param values Values list.
* @see InfoValue
*/
  void  set_values(const InfoValueList& values);
  
/**
* Adds new value to the current level. 
* @param value New value to be added.
* @see InfoValue
*/
  void  add_value(InfoValue *value);

/**
* Gets current values list. 
* @return Values list.
* @see InfoValue
*/
  InfoValueList* get_values() { return &m_values; }

/**
* Overriden method of the base class. Returns always <code>false</code>. 
* @return <code>false</code>  
*/
  bool  is_sub_level() { return false; }
};

/**
* This is base class for storing information in tree form. This class is used in some GUI classes for storing semantic information
* related to the graph.
*/
class InfoStorage
{
private:
/*  InfoHashMap       info_map;*/
  InfoStorageSubLevel  *root_level;
  
public:
  InfoStorage();
  virtual ~InfoStorage();
  
/**
* This function returns root level of the information tree stored in this class. 
* @return Root level of the tree. 
* #see InfoStorageSubLevel
*/
  InfoStorageSubLevel *get_root_level() { return root_level; };
  virtual void set_root_level( InfoStorageSubLevel *level ) 
  {
    if ( level != 0 )
    {
      delete root_level;
      root_level = level;
      root_level->set_name( "root" );  
    }        
  }
  
/**
* Clear storage and set it to initial state.
*/
  void clear() { root_level->clear(); }
};

}

#endif
