
#ifndef  _GUI_WINDOWS_H_
#define  _GUI_WINDOWS_H_

#include <list>

#include "storable.h"
#include "InfoStorage.hpp"
#include "nodeattr.h"

using namespace std;

/** 
* @file GUIWindows.h
*  The file includes all OS-independed interfaces for OS-depended GUI window classes.
*/ 

namespace GraphGraphics
{
/**
* <b>GUICloseHandler</b> is base class-function interface for "window closing" event handling.
* Classes of non-modal GUI windows allow to register one or more <b>GUICloseHandler</b> objects to handle close event.
* @see GraphGUIWindow
*/  
class GUICloseHandler
{
public:
  /**
  * Default virtual destructor.
  */
  virtual ~GUICloseHandler(){};
  
  /**
  * Virtual parenthesis operator.
  * If a <b>GUICloseHandler</b> object is registered in a GUI window class the operator will be called before the window is closed/destroyed.  
  */
  virtual void operator () ( ) = 0;
};

typedef list<GUICloseHandler*>::iterator CloseHandlerIter;

/**
* <b>GraphGUIWindow</b> represents base abstract class for all OS-dependent non-modal GUI window classes.
* The class provides functionality to register "window closing" event handlers to allow extra action on the event from application level.
* @see GUICloseHandler
* @see GraphGUIModalWindow
* @see GraphGenerationGUI
* @see GraphInformationGUI
*/
class GraphGUIWindow
{
protected:
  /**
  * The list of registered "window closing" event handlers
  * @see GUICloseHandler
  * @see ::register_close_handler(...)
  */
  list<GUICloseHandler*> m_close_handler_list;    
public:
  /**
  * Default virtual destructor.
  */
  virtual ~GraphGUIWindow()
  {
    CloseHandlerIter  iter = m_close_handler_list.begin();
    CloseHandlerIter  end_iter = m_close_handler_list.end();
    
    for (; iter != end_iter; ++iter)
      delete *iter;
    
    m_close_handler_list.clear();
  }
  
  /**
  * The virtual method is used to show or hide GUI window.
  * Any class based on the <b>GraphGUIWindow</b> have to implement the method.
  * @param show_window If <b>true</b> the GUI window will be shown otherwise it will be hiden.
  */
  virtual void show( bool show_window ) = 0;
  
  /**
  * The method registers "window closing" event handler.
  * Once a handler object is registered it will be automatically destroyed when the <b>GraphGUIWindow</b> is destroyed. 
  * Any class based on the <b>GraphGUIWindow</b> should call parenthesis operator of registered objects on closing of the window.
  * @warning To prevent crashing do not destroy registered objects and do not register the same object twice.
  * @param handler The pointer to <b>GUICloseHandler</b> object that should be registered.
  * @see GUICloseHandler
  */
  void register_close_handler( GUICloseHandler *handler )
  {
    if ( handler != 0 )
      m_close_handler_list.push_back(handler);        
  }
};

/**
* <b>GraphGUIModalWindow</b> represents base interface for all OS-dependent modal GUI window classes.
* @see GraphGUIWindow
* @see GraphNodeSearchGUI
* @see GraphViewPropertiesGUI
* @see GraphAttrFilterGUI
* @see ComponentManagerGUI
* @see GraphSaveGUI
*/
class GraphGUIModalWindow
{
public:
  /** Returned values of <b>run()</b> method.
  * @see ::run()
  */
  enum RunResults
  {
    CANCELED = 0, /**< User canceled any action pressing <b>Cancel</b> or <b>Close</b> dialog button.*/
    APPLIED       /**< User allowed all actions pressing <b>OK</b> or <b>Apply</b> dialog button.*/
  };
  
public:
  /**
  * Default virtual destructor.
  */
  virtual ~GraphGUIModalWindow(){};
   
  /**
  * The virtual method is used to run modal window.
  * Any class based on the <b>GraphGUIModalWindow</b> have to implement the method.
  * @return The result of modal window running (one of the <b>RunResults</b> constants)
  * @see ::RunResults
  */
  virtual int run( ) = 0;
};

class GUIDockMenuHandler
{
public:
  /**
  * Default virtual destructor.
  */
  virtual ~GUIDockMenuHandler(){};
  
  virtual void operator () ( bool dock ) = 0;
};

class GraphGUIDockableWindow: public GraphGUIWindow
{
protected:
  list<GUIDockMenuHandler*> m_dock_menu_handler_list;    
  StyleFlags                m_type;
public:
  virtual ~GraphGUIDockableWindow()
  {
    list<GUIDockMenuHandler*>::iterator  iter = m_dock_menu_handler_list.begin();
        
    for (; iter != m_dock_menu_handler_list.end(); ++iter)
      delete *iter;
    
    m_dock_menu_handler_list.clear();
  }
  
  void set_dockable_type( StyleFlags type ) { m_type = type; }
  StyleFlags get_dockable_type( ) { return m_type; }
  
  virtual string get_label() = 0;
  virtual void make_docked( bool docked ) = 0;
  virtual bool is_docked() = 0;
  virtual void* get_dockable() = 0;
    
  void register_dock_menu_handler( GUIDockMenuHandler *handler )
  {
    if ( handler != 0 )
      m_dock_menu_handler_list.push_back(handler);        
  }
};

/**
* <b>GUIGraphLoadHandler</b> is base class-function interface for "graph loaded" event handling.
* The event is generated by Graph Generation GUI window after graph is loaded.
* The main porpose of the class is provide provide the event extra handling from application level (for example repainting of drawing area of main window). 
* @see GraphGenerationGUI
*/
class GUIGraphLoadHandler
{
public:
  /**
  * Default virtual destructor.
  */
  virtual ~GUIGraphLoadHandler(){};
    
  /**
  * Virtual parenthesis operator.
  * If a <b>GUIGraphLoadHandler</b> object is registered in GraphGenerationGUI class the operator will be called after graph is loaded and constructed.  
  */
  virtual void operator () ( ) = 0;
};

typedef list<GUIGraphLoadHandler*>::iterator LoadHandlerIter;

/**
* <b>GraphGenerationGUI</b> is base abstract class for OS-dependent graph generation/loading GUI dialog class.
* This dialog is assigned to a <code>ScenarioApp</code> instance, which <code>GraphGUI</code> object is used to create <b>GraphGenerationGUI</b>.
* The GUI dialog allows to generate a graph file using a XML model/Promela model and/or load a graph file.
* As far as graph file generation is very slow operation it is processed in another thread. 
* It allows to do any other operations with existing graph during a new one is generated. 
* This class provide functionality to get last used graph file name, model (promela program) 
* and XML network model file name that are used for the graph file generation.
* It also allow to register "graph loaded" event handling to provide some additional actions from application level.
* A <b>GraphGenerationGUI</b> object is not be deleted after its window is closed (destroyed) so it should be deleted at application level.
* @warning To avoid any graph loading problem it is strongly recomended not to create/run more than one <b>GraphGenerationGUI</b> 
* object for one <b>ScenarioApp</b> instance.
* @see GraphGUI::create_graph_generation_window(...)
* @see ScenarioEventHandler::handle_load()
* @see GraphGUIWindow
*/
class GraphGenerationGUI: public GraphGUIWindow
{
protected:
  
  /**
  * The list of registered "graph loaded" event handlers
  * @see GUIGraphLoadHandler
  * @see ::register_load_handler(...)
  */
  list<GUIGraphLoadHandler*> m_load_handler_list;
public:
 
  /**
  * Default virtual destructor.
  */
  virtual ~GraphGenerationGUI()
  {
    LoadHandlerIter  iter = m_load_handler_list.begin();
    LoadHandlerIter  end_iter = m_load_handler_list.end();
    
    for (; iter != end_iter; ++iter)
      delete *iter;
    
    m_load_handler_list.clear();
  }
  
  /**
  * The method registers "graph loaded" event handler.
  * Once a handler object is registered it will be automatically destroyed when a <b>GraphGenerationGUI</b> object is destroyed. 
  * Any class based on the <b>GraphGenerationGUI</b> should call parenthesis operator of registered objects after a graph is loaded.
  * @warning To prevent crashing do not destroy registered objects and do not register the same object twice.
  * @param handler The pointer to <b>GUIGraphLoadHandler</b> object that should be registered.
  * @see GUIGraphLoadHandler
  */
  void   register_load_handler( GUIGraphLoadHandler *handler )
  {
    m_load_handler_list.insert(m_load_handler_list.begin(), handler);  
  }
  
  /**
  * The method returns last-used model file name (Promela or NuSMV program file).
  */
  virtual string get_model_file_name() = 0;
  
  /**
  * The method returns last-used graph file name.
  */
  virtual string get_graph_file_name() = 0;
  
  /**
  * The method returns last-used network model file name (XML file).
  */
  virtual string get_XML_file_name() = 0;
  
  /**
  * The method tries to close GUI dialog.   
  * @param force_close If <b>true</b> any promt dialog will not be shown;   
  * graph generation process will be stopped and dialog window will be closed in any case.
  * Otherwise if graph generation is in process a user promt dialog is shown.   
  * If user choose do not interupt process the dialog window will be canceled.
  * @return Whether the dialog window is closed.  
  */
  virtual bool   destroy( bool force_close ) = 0;
};

/**
* <b>GraphNodeSearchGUI</b> is base interface for OS-dependent node search patern creation GUI dialog class .
* The GUI dialog allows visual creation of graph attributes pattern string to search graph nodes by graph attributes. 
* It also allows to check whether the search result should be shown in a new window.
* @see  GraphGUI::create_node_search_dialog(...)
* @see GraphGUIModalWindow
*/
class GraphNodeSearchGUI: public GraphGUIModalWindow
{
public:
  
  /**
  * Default virtual destructor.
  */
  virtual ~GraphNodeSearchGUI() {}
    
  /**
  * The method sets graph attributes template string.
  * This attribute template is used for visual creation graph apttributes search pattern.
  * The method should be called before calling <b>run()</b> method.
  * @param attr_string Graph attributes in string format.
  */
  virtual void set_attr_template_string( const string &attr_string ) = 0;
    
  /**
  * The method returns resulting graph attributes pattern.
  * @return Resulting graph attributes in string format.
  */
  virtual string get_result_attr_string( ) = 0;
    
  /**
  * The method returns whether a user selects to show search result in new window.
  */
  virtual bool get_in_new_window_flag() = 0;
};

class AttrStringTemplateStorage;
  
/**
* <b>GraphAttrFilterGUI</b> is base interface for OS-dependent graph attributes filter creation/selection GUI dialog class.
* @see GraphGUI::create_node_attr_filter_dialog(...)
* @see AttrStringTemplateStorage
* @see GraphGUIModalWindow
*/
class GraphAttrFilterGUI: public GraphGUIModalWindow
{
public:
  /**
  * Default virtual destructor.
  */
  virtual ~GraphAttrFilterGUI() {}
    
  /**
  * The method sets graph attributes template string.
  * This attribute template is used for visual creation graph apttributes filter pattern.
  * The method should be called before calling <b>run()</b> method.
  * @param attr_template Graph attributes in string format.
  */
  virtual void set_attr_template( const string &attr_template ) = 0;
  
  /**
  * The method sets attribute filters storage.
  * @param filter_storage The <code>AttrStringTemplateStorage</code> object to be copied.
  * @see GraphView::get_node_attr_filters()
  */
  virtual void set_filters( AttrStringTemplateStorage* filter_storage ) = 0;
    
  /**
  * The method returns resulting attribute filters storage.
  * @see GraphView::set_node_attr_filters(...)
  */
  virtual AttrStringTemplateStorage* get_filetrs() = 0;
  
};

/**
* <b>GraphViewPropertiesGUI</b> is base interface for OS-dependent graph view properties GUI dialog class and node view properties group GUI dialog class.
* The view properties GUI dialog allows user to change general view properties such as default edge colors and graph layout properties (vertical/horisontal distance) etc.
* The node view properties group dialog allows to change/create/remove properties group (foregraund color, pen, font etc.) for drawing nodes and assign the group to nodes using graph attributes.
* @see GraphViewPropertiesGroup
* @see GraphViewPropertiesStorage
* @see GraphView::get_view_properties_storage()
* @see GraphGUI::create_view_preferences_dialog(...)
* @see GraphGUI::create_group_properties_dialog(...)
*/
class GraphViewPropertiesGUI: public GraphGUIModalWindow
{
public:
  
  /** 
  * Possible levels of changes. 
  * @see ::get_changes_level()
  */
  enum ChangesLevel
  {
    PROPERTIES_UNCHANGED = 0, /**< Nothing was changed.*/
    NEED_REDRAW_ONLY,         /**< Changes do not change graph layout.*/
    NEED_EDGES_RECONSTRUCTION /**< Graph layout changed */
  };
  
public:
  
  /**
  * Default virtual destructor.
  */
  virtual ~GraphViewPropertiesGUI() {}
    
  /**
  * The method returns level of changes.
  * Returned value is one of <code>ChangesLevel</code> constans and indicate what should be done to correctly process changes.
  * If it is <b>PROPERTIES_UNCHANGED</b> nothing need to do.
  * If it is <b>NEED_REDRAW_ONLY</b> invalidating draw area rectangle is enough.
  * If it is <b>NEED_EDGES_RECONSTRUCTION</b> the method <code>GraphView::redraw_edges(...)</code> need to be called.
  */
  virtual int get_changes_level() = 0;
};

/**
* <b>ComponentManagerGUI</b> is base inerfce for OS-dependent component activity manager GUI dialog class.
* The GUI dialog allows enabling/desabling registered components functionality.
* @see GraphGUI::create_comp_manager_dialog(...)
* @see ComponentManager
* @see CompInterface
*/
class ComponentManagerGUI: public GraphGUIModalWindow
{
public:
  
  /**
  * Default virtual destructor.
  */
  virtual ~ComponentManagerGUI(){}
};

/**
* <b>GraphSaveGUI</b> is base interface for OS-dependent graph saving GUI dialog class.
* The GUI dialog allows a user to set options and save graph to file.
* @see GraphGUI::create_graph_save_dialog( )
* @see ScenarioEventHandler::handle_save(...)
*/
class GraphSaveGUI: public GraphGUIModalWindow
{
public:
  virtual ~GraphSaveGUI() {}
  virtual void set_file_info( const GraphFileInfo &info ) {}
};

/**
* <b>GraphInformationGUI</b> is base interface for OS-dependent graph information GUI dialog.
* The GUI dialog allows to show any information that consists of pairs ("name", "value") in tree-like or list-like form.
* It also allows to set visual attributes and "mouse button click" event handler for each pair.
* The <b>GraphInformationGUI</b> object is automatically deleted when the GUI dialog is closed.
* @see GraphGUI::create_information_window(...)
* @see GraphGUIWindow
* @see InfoStorage
*/
class GraphInformationGUI: public GraphGUIDockableWindow, public InfoStorage
{
public:
  
  /**
  * Default virtual destructor.
  */
  virtual ~GraphInformationGUI() {}
};

/**
* This is base interface for OS-dependent graph node attributes window
*/
class GraphNodeAttributesGUI: public GraphGUIDockableWindow
{
public:
  /**
  * Default virtual destructor.
  */
  virtual ~GraphNodeAttributesGUI() {}
  
  virtual void set_attributes( const vector<string> &attr_lists ) = 0;
  virtual void set_attributes( const vector<NodeAttributeList> &attr_lists ) = 0;
  virtual void set_attribute_string( const string& attr_string ) = 0;
};

class GraphScenarioGroupRestrictionGUI: public GraphGUIModalWindow
{
public:
  virtual ~GraphScenarioGroupRestrictionGUI() {}
  virtual string get_include_nodes_restriction() = 0;
  virtual string get_exclude_nodes_restriction() = 0;
};

};

#endif
