#ifndef EVENTHANDLE_H_HEADER_INCLUDED_C1038CEC
#define EVENTHANDLE_H_HEADER_INCLUDED_C1038CEC

#include "graphics.h"
#include "Events.h"
#include "CompInterface.h"

/********************************************************************/

namespace GraphGraphics
{
  
  /**
  * This class contains methods for default handling of the scenario events.Invoking of these methods also leads to invoking of 
  * the corresponded methods of all active registered components. It is impossible to create an independent object of the class.
  * The object is automatically created and stored in <code>ScenarioApp</code> object and can be retrieved using
  * <code>ScenarioApp::get_event_handler()</code> method.
  */  
class ScenarioEventHandler
{
  friend class ScenarioApp;
private:
  GraphViewContext *m_graph_view_context;
  GraphGUI         *m_graph_gui;
  ComponentsVector &m_components;
  
  /** @name GUI fields. */
  //@{
  TwoDPoint    m_gui_start_point; /**< Left-top corner of GUI selecting rectangle.*/
  TwoDPoint    m_gui_end_point;   /**< Right-bottom corner of GUI selecting rectangle.*/
  bool         m_gui_view_rect;   /**< Selecting rectangle drawing mode (<b>true</b> means the mode is switched on).*/
  /** 
  * If it is not equal to <code>Graph::INVALID_NODE_ID</code>
  * the edge/path selecting mode is switched on.
  * Index of start node of edge/path that are going to be selected. 
  */
  NodeIndex    m_gui_start_edge_node_id; 
  /**
  * Edge mark id used to select edge/path (specifies which mode is switched on).
  * If it is <code>GraphViewPropertiesStorage::EDGE_SELECTED</code> - edge selecting mode is switched on.
  * If it is <code>GraphViewPropertiesStorage::EDGE_IN_SCENARIO</code> - scenario path selecting mode is switched on.
  * If it is <code>GraphViewPropertiesStorage::EDGE_UNMARKED</code> - both modes are switched off.
  */
  StyleFlags   m_gui_edge_mark; 
  StyleFlags   m_gui_last_edge_mark; /**< Edge mark id of last selected edge.*/
  NodeIndex    m_node_mouse_over; /**< Node index of node the mouse pointer is over or Graph::INVALID_NODE_ID.*/
  bool         m_ctrl_pressed;    /**< Is CTRL key is pressed and held*/
  //@}  

  unsigned int m_modes; /**< Behaviour Modes. @see ScenarioApp::ScenarioModes*/
  
  GraphPath    m_selected_pathes; /**< Found path groups.*/
  int          m_curr_selected_path; /**< Currently selected path (GraphPath::INVALID_PATH_INDEX - none of the pathes is selected) */

private:
  /** 
  * The method is invoked in mouse event hendlers.
  * @param event An event object.
  */
  void mouse_interaction( const Event &event );

  /**
  * This constructor creates an scenario event handler object.
  * It is private to make object creation posible only in <code>ScenarioApp</code>. 
  */
  ScenarioEventHandler( GraphViewContext *scenario_app_context, GraphGUI *graph_gui, ComponentsVector& components ): m_components(components)
  {
    m_graph_view_context = scenario_app_context;
    m_graph_gui          = graph_gui;
    //m_components         = components;
    
    m_gui_start_edge_node_id = Graph::INVALID_NODE_ID;
    m_gui_edge_mark = GraphViewPropertiesStorage::EDGE_UNMARKED;
    m_gui_last_edge_mark = GraphViewPropertiesStorage::EDGE_UNMARKED;
    m_gui_view_rect = false;
    m_node_mouse_over = Graph::INVALID_NODE_ID;
    m_modes = 0;
    m_ctrl_pressed = false;
    m_curr_selected_path = GraphPath::INVALID_PATH_INDEX;
  }
  
  /**
  * The method draws selecting rectangle.
  * @param g A pointer to <code>GraphicArea</code> object to draw.
  */
  void draw_gui_rect( GraphicArea *g )
  {
    if ( !(m_gui_start_point == m_gui_end_point) && g != 0 )
    {
      g->set_paint_mode( GraphicArea::XOR_MODE );
      g->set_foreground_color( RGB( 128, 128, 128 ) );
      g->set_drawing_pen( DrawingPen( DrawingPen::ON_OFF_DASH, 1 ) );
        
      Rectangle rect( GGMIN( m_gui_start_point.get_x(), m_gui_end_point.get_x()),
                      GGMIN( m_gui_start_point.get_y(), m_gui_end_point.get_y()),
                      GGMAX( m_gui_start_point.get_x(), m_gui_end_point.get_x()),
                      GGMAX( m_gui_start_point.get_y(), m_gui_end_point.get_y()) );
      rect.draw( g );
      g->set_paint_mode( GraphicArea::SOLID_COLOR );
    }
  }
  
  /**
  * This method enables or disables <code>ScenarioApp::EDGE_SELECTING</code> mode.
  */
  void enable_edge_selecting_mode( bool enable );
  
  /**
  * This method selects a path with identifier <code>path_id</code> and make it "currently selected".
  * @param path_id An identifier of path to select.
  * @param g A pointer to <code>GraphicArea</code> object to draw.
  * @see m_selected_pathes
  * @see m_curr_selected_path
  */
  bool select_path( int path_id, GraphicArea *g );
  
  /**
  * This method unselects a path with identifier <code>path_id</code>.
  * @param path_id An identifier of path to select.
  * @param g A pointer to <code>GraphicArea</code> object to draw.
  * @see m_selected_pathes
  */
  bool unselect_path( int path_id, GraphicArea *g );
  
  void select_all_pathes( GraphicArea *g );
  
public:
  /**
  * Destructor.
  */
  ~ScenarioEventHandler() {};
  
  bool find_path_group( GraphicArea *g, NodeIndex start_node, const NodeAttributeList &end_nodes, 
                        const NodeAttributeList &include_nodes, 
                        const NodeAttributeList &exclude_nodes );
    
  bool find_path_group( GraphicArea *g, NodeIndex start_node, const NodeAttributeList &end_nodes );  
    
  bool find_path_sub_group( GraphicArea *g, const NodeAttributeList &include_nodes, const NodeAttributeList &exclude_nodes );
    
  int  get_sub_group_finding_number() { return m_selected_pathes.sub_group_finding_count(); }
  bool undo_find_path_sub_group( GraphicArea *g );
  /**
  * The method returns reference on <code>GraphPath</code> object that contains last found scenario path.
  */
  GraphPath& get_found_pathes() { return m_selected_pathes; }
  
  /**
  * The method returns currently selected path identifier.
  */
  int get_current_selected_path_id() { return m_curr_selected_path; }
    
  /**
  *  Invoke this method to open graph file.
  *  @param file_name Name of the file to be opened.
  */
  bool handle_open( const string &file_name );

  /**
  *  Invoke this method to save the graph to the file.
  *  @param save_file_info <code>GraphFileInfo</code> structure.
  *  @see GraphFileInfo
  */
  bool handle_save( const GraphFileInfo &save_file_info );

  /**
  *  Invoke this method to close the main window.
  */
  void handle_close_window();

  /**
  *  Handling of mouse events.Invoke this function for all mouse events, like mouse press, mouse release and mouse motion.
  *  @param event Description of the mouse event.
  *  @see Event
  */
  void handle_mouse_events( const Event &event );

  /**
  *  Zoom in of the graph view.
  */
  void handle_zoom_in();
  
  /**
  *  Zoom out of the graph view.
  */
  void handle_zoom_out();

  /**
  *  Fit graph view to the window bounds.
  */
  void handle_fit_to_window();
    
  /**
  *  Fit graph view to the window width.
  */
  void handle_fit_to_width();

  /**
  *  Fit graph view to the window height.
  */
  void handle_fit_to_height();

  /**
  *  Fit selected sub-graph to the window bounds.
  */
  void handle_fit_selection_to_window();

  /**
  *  Set arbitrary size of the graph view.
  */
  void handle_arbitrary_size();

  /**
  *  Centering of the selected sub-graph on the window rectangle.If selected sub-graph doesn't fit to the rectangle than the graph
  *  will be properly zoomed and scrolled.
  *  @param display_rect Rectangle for centering  
  */
  bool handle_selection_centering( const Rectangle &display_rect );
  
  /**
  *  Initialization of edge selecting.
  *  @param g Graphic area object of the window.
  *  @param start_node_id First node index of the edge.    
  *  @see GrpahicArea 
  */
  void handle_init_edge_selecting( GraphicArea *g, NodeIndex start_node_id );
  
  /**
  *  Initialization of scenario sub-graph finding.
  */
  void handle_init_path_finding(  );
  
  /**
  * Invoke the method to select particular path of found scenario sub-graph.
  * @param path_id An identifier of path to be selected. It have to be in range from 0 to  
  * <code>get_found_pathes().get_pathes_count()</code>.
  * @param g A pointer to <code>GraphicArea</code> object to draw.
  */
  bool handle_path_selecting( int path_id, GraphicArea *g );
};

};
#endif /* EVENTHANDLE_H_HEADER_INCLUDED_C1038CEC */
