#ifndef COMPINTERFACE_H_HEADER_INCLUDED_C103902C
#define COMPINTERFACE_H_HEADER_INCLUDED_C103902C

#include <list>
#include <string>
#include "graphics.h"
#include "Events.h"

using namespace std;

namespace GraphGraphics
{
 
  /**
  * This is base interface for handlers of "Activity Changed" event that is generated by a component
  * if its activity status is changed.
  * @see CompInterface::register_comp_activity_change_handler(...)
  */
class CompActivityChangesEventHandler
{
public:
  virtual ~CompActivityChangesEventHandler() {}
  virtual void operator () ( bool active ) = 0;
};

  /**
  * <code>CompInterface</code> is a base class for all components. This class consists of the methods that will be invoked 
  * when corresponded events occured in the system. To make this work a component object need to be registered in  
  * <code>ComponentManager</code> instance. Each component has a name and generates "Activity Changed" event on activity status changing.
  */  
class CompInterface
{
protected:
  
typedef list<CompActivityChangesEventHandler*>::iterator CompActivityHandlerIter;

  /*
  * This is list of registered "Activity Changed" event handlers. 
  */
  list<CompActivityChangesEventHandler*> m_activity_change_handler_list;

  /*
  * The component name.
  */
  string m_component_name;

  /*
  * Component activity status.
  */
  bool   m_is_component_active;
  
public:
  /**
  *  This constructor create <code>CompInterface</code>.
  */
  CompInterface(): m_activity_change_handler_list(), m_component_name(""), m_is_component_active(false) {}
    
  /**
  *  Destructor of <code>CompInterface</code>.
  */
  virtual ~CompInterface() 
  {
    CompActivityHandlerIter  iter = m_activity_change_handler_list.begin();
    CompActivityHandlerIter  end_iter = m_activity_change_handler_list.end();
    
    for (; iter != end_iter; ++iter)
      delete *iter;
    
    m_activity_change_handler_list.clear();
  }
  
  /**
  *  This function register "Activity Changed" event handler class <code>CompActivityChangesEventHandler</code>.
  *  The event is generated every time the set_component_active(...) is invoked.
  *  @param handler pointer to the object of class <code>CompActivityChangesEventHandler</code>
  */
  void register_comp_activity_change_handler( CompActivityChangesEventHandler *handler )
  {
    m_activity_change_handler_list.push_back(handler);        
  }
  
  /**
  *  Returns name of the component.
  *  @return Name of the component.
  */
  string  get_component_name() { return m_component_name; }
  
  /**
  *  Sets name of the component.
  *  @param comp_name Name of the component.
  */
  void    set_component_name( const string &comp_name ) { m_component_name = comp_name; }
  
  /**
  *  Sets activity status of the component.
  *  @param active Activity flag to be set.
  */
  void    set_component_active( bool active ) 
  { 
    bool act_changed = (m_is_component_active && !active || !m_is_component_active && active);
    m_is_component_active = active; 
    
    if ( act_changed )
    {
      CompActivityHandlerIter  iter = m_activity_change_handler_list.begin();
      CompActivityHandlerIter  end_iter = m_activity_change_handler_list.end();
    
      for (; iter != end_iter; ++iter)
        (**iter)( active );
    }
  }
  
  /**
  *  Returns <code>true</code> if the component is active.
  *  @return Activity flag of the component.
  */
  bool    is_component_active() { return m_is_component_active; }    
  
  /**
  *  This method will be invoked when the system registers the component. This function must be overriden in the inherited component class.
  *  @param main_app_context Pointer to the GraphViewContext object of the main scenario component.
  *  @param gui Pointer to the GraphGUI object.
  *  @return <code>true</code> if the registration ended with success.
  *  @see ComponentManager
  */
  virtual bool on_register( GraphViewContext *main_app_context, GraphGUI *gui ) = 0;
  
  /**
  *  This method will be invoked when the system destroys the component.
  */
  virtual void on_destroy() {};
      
  /**
  *  This method will be invoked when the system opens graph file.
  *  @param file_name Name of the file to be opened.
  */
  virtual void on_open(const string &file_name) {};

  /**
  *  This method will be invoked when the system saves graph file.
  *  @param save_file_info <code>GraphFileInfo</code> structure.
  *  @see GraphFileInfo  
  */
  virtual void on_save( GraphFileInfo &save_file_info ) {};

  /**
  *  This method will be invoked when the system closes main window.
  */
  virtual void on_close_window() {};

  /**
  *  This method will be invoked when mouse button press event occures.
  *  @param event Event structure with the description of the event.
  *  @see Event    
  */

  virtual void on_mouse_button_press( const Event &event ) {};

  /**
  *  This method will be invoked when mouse button release event occures.
  *  @param event Event structure with the description of the event.
  *  @see Event    
  */
  virtual void on_mouse_button_release( const Event &event ) {};
  
  /**
  *  This method will be invoked when mouse motion event occures.
  *  @param event Event structure with the description of the event.
  *  @see Event    
  */
  virtual void on_mouse_motion_notify( const Event &event ) {};

  /**
  *  This method will be invoked when the system zooms in the graph.
  */
  virtual void on_zoom_in() {};

  /**
  *  This method will be invoked when the system zooms out the graph.
  */
  virtual void on_zoom_out() {};

  /**
  *  This method will be invoked when the system fit the graph to the window.
  */
  virtual void on_fit_to_window() {};

  /**
  *  This method will be invoked when the system fit the graph to the window width. 
  */
  virtual void on_fit_to_width() {};
    
  /**
  *  This method will be invoked when the system fit the graph to the window height. 
  */
  virtual void on_fit_to_height() {};

  /**
  *  This method will be invoked when the system fit selected sub-graph to the window. 
  */
  virtual void on_fit_selection_to_window() {};

  /**
  *  This method will be invoked when the system sets arbitrary size of the graph . 
  */
  virtual void on_arbitrary_size() {};

  /**
  *  This method will be invoked when the system sets selected sub-graph to the center of the window. 
  */
  virtual void on_selection_centering() {};

  /**
  *  This method will be invoked after initilization of the scenario sub-graph finding. 
  */
  virtual void on_init_path_finding() {};
      
  /**
  *  This method will be invoked after starting of scenario sub-graph finding.
  *  @param first_selected_node_id First node index of the scenario.    
  */
  virtual void on_begin_path_finding( NodeIndex first_selected_node_id, GraphicArea *g ) {};
    
  /**
  *  This method will be invoked after finishing of scenario sub-graph finding.
  *  @param path Found scenario sub-graph.            
  */
  virtual void on_end_path_finding( const GraphPath &path, GraphicArea *g ) {};
  
  /**
  * This method will be invoked after a path in the <code>path</code> scenario sub-graph is selected.
  * @param path_ind An index of selected path.
  * @param path A scenario sub-graph, which particular path is selected.
  * @param g A pointer to <code>GraphicArea</code> object to draw.
  */    
  virtual void on_path_selecting( int path_ind, const GraphPath &path, GraphicArea *g) {};
  
  /**
  *  This method will be invoked after initilization of the edge. 
  */
  virtual void on_init_edge_selecting( NodeIndex node ) {};
    
  /**
  *  This method will be invoked after finishing of edge selecting.
  *  @param edge Selected edge.    
  *  @param edge_exist <code>true</code> if such edge exists.            
  */
  virtual void on_end_edge_selecting( const Edge& edge, bool edge_exist ) {};
};

/**
* Components vector-container type.
*/
typedef vector<CompInterface*> ComponentsVector;

};

#endif /* COMPINTERFACE_H_HEADER_INCLUDED_C103902C */
