#ifndef _NETWORK_COMP_H_
#define _NETWORK_COMP_H_

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#include "graphics.h"
#include "ScenarioApp.h"
#include "NetGUI.h"
#include "NetAttacks.h"
#include "StatusInfo.h"

namespace GraphGraphics
{

typedef vector<struct Node*>      NodeVector;
typedef vector<HostID>            HostVector; 
typedef map<string, HostVector*>  NetworkGroups;  
typedef list<string>              StringList;

class CVEClickHandler : public ClickHandler
{
public:
  virtual ~CVEClickHandler() {}
  virtual void operator()( InfoValue &value )
  {
    string cvename = value.get_name();
    unsigned int ind;    
      
    for (unsigned int i = 0; i < cvename.length(); i++ )
      if ( cvename[i] == '_' )
        cvename[i] = '-';
    
    if ((ind = cvename.find(':')) != cvename.npos)
      cvename = cvename.substr(ind+2);      

//    cout << cvename;    
      
    string url = "http://icat.nist.gov/icat.cfm?cvename=" + cvename;
    char**  argv = new char*[5];
    int     program_pid;

    argv[0] = "xterm";
    argv[1] = "-e";
    argv[2] = "lynx";   
    argv[3] = (char*)url.c_str();
    argv[4] = 0;  
     
    program_pid = vfork();
		
    if (program_pid == 0)
	  {
/*    cout << "enter to child process\n";*/
      execvp((char*)argv[0], (char**)argv);    
      
      perror((char*)argv[0]);
      exit(errno);
	  } 
    else
    {
      delete argv;
    }
    
  }
};

class NetworkComp : public CompInterface
{
  friend class NetPathCloseWindowHandler;
  friend class NodeClickHandler;
private:
  Network           *m_network;
  GraphViewContext  *network_context;
  GraphicArea       *network_g;
  GraphViewContext  *m_app_context;
  GraphGUI          *m_gui;
  NetworkGUI        *m_net_gui;
 
  TwoDPoint    m_gui_start_point;
  TwoDPoint    m_gui_end_point;
  bool         m_gui_view_rect;
  NodeIndex    m_gui_start_edge_node_id;
  StyleFlags   m_gui_edge_mark;
  StyleFlags   m_gui_last_edge_mark;
  bool         m_is_file_loaded;
  StyleFlags   m_path_mark;
  bool         m_init_path_selecting;

  GraphInformationGUI *m_path_info_window;
  StyleFlags   m_curr_sel_node;

  static bool m_xml_parser_initialized;

private:
  void calc_node_measure_mark( Graph *graph, NodeIndex node_id, const AttackMeasuresSet &measures, GraphMeasureMarks &graph_measure_marks );
  AttackMeasureID get_most_frequent_measure_id( Graph *graph, const AttackMeasuresSet &measures );

  void remove_path_node_info_selecting( GraphicArea *g );

public:
  static void initialize_globals();
  NetworkComp();
  virtual ~NetworkComp();

  void set_selnode_group(StyleFlags selnode_group) { m_curr_sel_node = selnode_group; }
  StyleFlags get_selnode_group() { return m_curr_sel_node; }  

//Implementation of CompInterface functions
  virtual bool on_register( GraphViewContext *main_app_context, GraphGUI *gui );
  virtual void on_open(const string &file_name);
  virtual void on_mouse_button_press( const Event &event );
  virtual void on_path_selecting( int path_id, const GraphPath &path, GraphicArea *g);
  virtual void on_init_edge_selecting( NodeIndex node );
  virtual void on_end_edge_selecting( const Edge& edge, bool edge_exist );
  virtual void on_mouse_motion_notify( const Event &event );
  virtual void on_init_path_finding();
  virtual void on_begin_path_finding( NodeIndex first_selected_node_id, GraphicArea *g );
  virtual void on_end_path_finding( const GraphPath &path, GraphicArea *g );
//
  
//Utility functions
  void select_subgraph(NodeIndex host_id, GraphicArea *g);  
  void mark_network_path( NodeIndex host_id1, NodeIndex host_id2 );  
  bool get_network_info( NodeIndex node_ind, NetworkInfo &net_info );  
  Graph* generate_network_graph();     
  int handle_mouse_events( const Event &event, GraphicArea  *main_g );  
  void set_graphic_area(GraphicArea* g);
  GraphicArea* get_graphic_area();
  void draw_gui_rect( GraphicArea *g );
  void set_attack_information( Attack* attack, InfoStorageSubLevel *level );
  void set_edge_information( InfoStorageSubLevel *root_level, NodeIndex node_id );
  void set_host_information(  Host* host, InfoStorageSubLevel *level );
  void set_firewall_information( Firewall* firewall, InfoStorageSubLevel *level );
  void set_scenario_information( InfoStorageSubLevel *root_level, const EdgeList &edges, NodeIndex start_node_id );  
  void set_attack_triples_information( InfoStorageSubLevel *root_level, const AttackTriplesSet &attacks );
  void set_attack_measures_information( InfoStorageSubLevel *root_level, const AttackMeasuresSet &measures );
  void set_scenario_nodes_information( InfoStorageSubLevel *root_level, const NodeLinkList &nodes, GraphicArea *g );

  Graph* remove_attacks_from_scenario( const AttackTriplesSet &attacks );
  
  bool   gready_measure_set( const AttackMeasuresSet &measure_set, AttackMeasuresSet &res_subset, StyleFlags edge_mark = GraphViewPropertiesStorage::EDGE_UNMARKED, GraphicArea  *main_g = 0 );
  bool   gready_attack_set( const AttackTriplesSet &attacks, AttackTriplesSet &res_subset, StyleFlags edge_mark = GraphViewPropertiesStorage::EDGE_UNMARKED, GraphicArea  *main_g = 0 );

  GraphViewContext*  get_network_context();
  GraphViewContext*  get_app_context();
  GraphGUI*  get_GUI();
  
  NetworkGUI*    get_net_GUI();
  Network*       get_network_model_info() { return m_network; }
  bool           is_network_model_info_enabled() { return (m_network != 0); }
  void           clear_network_context();
//

//Implementation of network specific event handling
  void handle_zoom_in();
  void handle_zoom_out();
  void handle_fit_to_window();
  void handle_fit_to_width();
  void handle_fit_to_height();
  void handle_fit_selection_to_window();
  void handle_arbitrary_size();
  bool handle_selection_centering( const Rectangle &display_rect );
//

};

class NodeClickHandler : public ClickHandler
{
private:
  NodeIndex m_node_num;     
  GraphViewContext  *m_app_context;
  NetworkComp  *m_net;
  GraphicArea *m_g;
public:
  virtual ~NodeClickHandler() { delete m_g; }
  virtual void operator()( InfoValue &value )
  {
    GraphViewPropertyGroup scen_node_group;         
    StyleFlags selnode_group;// = m_net->get_selnode_group();
      
    //if (selnode_group != GraphViewPropertyGroup::DEFAULT)  
    //{
      //m_app_context->get_view()->get_view_properties_storage().remove_view_group(selnode_group);    
    //if (m_g != 0)
    m_net->remove_path_node_info_selecting( m_g );
    //}
    
    ostringstream   s_stream( "" );
    s_stream << m_node_num << ":Node{}";
      
    scen_node_group.set_shape_type( GraphViewPropertyGroup::CIRCLE );
    scen_node_group.set_name( "Scenario node" );
    scen_node_group.set_shape_filled( true );            
    scen_node_group.set_color( Color( RGB( 0, 0, 255) ) );            
      
    selnode_group = m_app_context->get_view()->get_view_properties_storage().add_view_group( s_stream.str(), scen_node_group );
    
    m_net->set_selnode_group(selnode_group);

    if (m_g != 0)
    {
      NodeIndexSet nodes;
      nodes.insert( m_node_num );
      m_app_context->get_view()->redraw_nodes( m_g, nodes, true );      
    }
  }
  
  void set_node_num(NodeIndex node_num)
  {
    m_node_num = node_num;    
  }
  
  void set_app_context(GraphViewContext *app_context)
  {
    m_app_context = app_context;    
  }
  
  void set_net(NetworkComp *net)
  {
    m_net = net;    
  }

  void set_graphic_area(GraphicArea *g)
  {
    m_g = g->clone();    
  }

};

class NetInfoStatusInfo : public StatusInfo
{
private:
  Node    *m_node;    
  NetworkComp *m_netcomp;
public:
  NetInfoStatusInfo(GraphViewContext *context, NetworkComp *netcomp) : StatusInfo("net_node_info", context), m_node(0) 
  {
    m_netcomp = netcomp;
  };
  ~NetInfoStatusInfo() {}    

  void set_node(Node *node) { m_node = node; }
      
  string get_status_str( )
  {
    int         src_host_id = 0, dest_host_id = -1, dest_firewall_id = -1;
    int         attack_id = 0;
    string      src_host_str, dest_host_str, attack_str;  
//  NetworkInfo info;
    GraphAttributes *attr = m_node->get_attributes();
    AttributeValue  *value;
    string      type;
    ostringstream ost_res;        
    
    if (m_node == 0 || m_netcomp == 0)
      return "";
    
    Network *network = m_netcomp->get_network_model_info();
    
    if ( network == 0 || !m_netcomp->is_component_active() )
      return "";
    
    bool attr_exist = attr->query_value( "::GlobalVariables::net.attack_id", type, &value );
  
    if ( !attr_exist )
      return "";
  
    attack_id = (int)value->get_integer();
    delete value;
  
    attr_exist = attr->query_value( "::GlobalVariables::net.source_ip", type, &value );
  
    if ( !attr_exist )
      return "";    
  
    src_host_id = (int)value->get_integer();
    delete value;
  
    attr_exist = attr->query_value( "::GlobalVariables::net.target_ip", type, &value );
  
    if ( !attr_exist )
      return "";
  
    dest_host_id = (int)value->get_integer();
    delete value;
    
    attr_exist = attr->query_value( "::GlobalVariables::net.target_fwi", type, &value );
  
    if ( attr_exist )
    {
      dest_firewall_id = (int)value->get_integer();
      delete value;
    }

    if (src_host_id < network->GetNumHosts())
    {
      ostringstream ost;        
      ost << network->GetHost(src_host_id)->NetworkName() << "." << network->GetHost(src_host_id)->GetName();
      src_host_str = ost.str();  
    }
    else
      return "";
  
    if (dest_host_id < network->GetNumHosts() && dest_host_id >= 0)
    {
      ostringstream ost;          
      ost << network->GetHost(dest_host_id)->NetworkName() << "." << network->GetHost(dest_host_id)->GetName();
      dest_host_str = ost.str();  
    }
    else if ( dest_host_id < 0 && dest_firewall_id >= 0 )
    {
      if ( dest_firewall_id < network->GetFirewallNumber() )
      {
        ostringstream ost;          
        ost << "fw: " << network->GetFirewallName(dest_firewall_id);
        dest_host_str = ost.str();  
      }
      else
        return "";
    }
    else
      return "";
  
    if (attack_id < network->GetNumAttacks())
    {
      ostringstream ost;          
      ost << network->GetAttack(attack_id)->GetName();
      attack_str = ost.str();  
    }
    else
      return "";

    ost_res <<  "Attack info - Src:" << src_host_str << " " << "Tgt:" << dest_host_str << " " <<
          "Name:" << attack_str;
  
    string status_str = ost_res.str();
    return status_str;
  }
};

class SelNodesStatusInfo : public StatusInfo
{
private:
  StringList  tgt_list;    
  unsigned int m_sel_nodes_number, m_sel_edges_number;  

public:
  SelNodesStatusInfo(GraphViewContext *context) : StatusInfo("sel_nodes_info", context) {};
  ~SelNodesStatusInfo() {}    

  void add_target(const string& tgt) { tgt_list.push_back(tgt); }
  
  void remove_all_targets() {tgt_list.clear();}
      
  void set_select_info(unsigned int sel_nodes_number, unsigned int sel_edges_number)    
  {
    m_sel_nodes_number = sel_nodes_number;
    m_sel_edges_number = sel_edges_number;        
  }
  
  string get_status_str()
  {
    StringList::iterator  iter;    
    StringList::iterator  end_iter = tgt_list.end();    
    ostringstream ost_res;        
          
    if (tgt_list.size() == 1)
      ost_res << "Nodes with tgt ";
    else
      ost_res << "Nodes with tgts: ";
    
    for (iter = tgt_list.begin(); iter != end_iter; ++iter)
    {
      ost_res << *iter;
      if (iter != --end_iter)
        ost_res << ", ";
      ++end_iter;
    }
    
    
    if ( m_context->get_graph()->count_nodes() > 0 )
    {
      //GraphView *view = m_context->get_view();
      ost_res << "\t" << "Sel nodes num:" << m_sel_nodes_number << "\t" << "Sel edges num:" << m_sel_edges_number;
    }
    
    string status_str = ost_res.str();    
    return status_str;
  }
};

}

#endif
