#include <unistd.h>
#include <fcntl.h>
#include <dirent.h>

#include <iostream>
#include <fstream>
#include <sstream>

#include <xercesc/util/PlatformUtils.hpp>
#include <xercesc/util/TranscodingException.hpp>

#include "NetworkComp.h"

Network *xml_parser(string xmlname);
bool init_xml_parser();

namespace GraphGraphics
{
class NetPathCloseWindowHandler: public GUICloseHandler
{
private:
  NetworkComp *m_comp;
  GraphInformationGUI *m_window;

public:
  NetPathCloseWindowHandler( NetworkComp *comp, GraphInformationGUI *window ) 
  {
    m_comp = comp;
    m_window = window;
  }
  
  virtual ~NetPathCloseWindowHandler() {}
  
  void operator () ( )
  {
    if ( m_comp->m_path_info_window == m_window )
      m_comp->m_path_info_window = 0;
    
    m_window = 0;
  }
};
};

using namespace GraphGraphics;

bool NetworkComp::m_xml_parser_initialized = false;

void NetworkComp::initialize_globals()
{
  m_xml_parser_initialized = init_xml_parser();
}

const string AttackTriple::TARGET_FW_TYPE = "[fw]:";

bool AttackTriple::make_attack_triple( GraphAttributes *attr )
{
  AttributeValue *attack_id = 0;
  AttributeValue *src_host_id = 0;
  AttributeValue *dest_host_id = 0;
  AttributeValue *dest_firewall_id = 0;
  string         type;
  bool           fw_exists = attr->query_value( "::GlobalVariables::net.target_fwi", type, &dest_firewall_id );
    
  bool res;
      
  if ( (res = (attr->query_value( "::GlobalVariables::net.attack_id", type, &attack_id ) &&
       attr->query_value( "::GlobalVariables::net.source_ip", type, &src_host_id ) &&
       (attr->query_value( "::GlobalVariables::net.target_ip", type, &dest_host_id ) || fw_exists))) )
  {
     m_attack_id = attack_id->get_integer();
     m_src_host_id = src_host_id->get_integer();
    
     if ( dest_firewall_id != 0 )
     {
       if ( dest_firewall_id->get_integer() >= 0 )
       {
         m_is_dest_firewall = true;
         m_dest_id = dest_firewall_id->get_integer();
       }
       else if ( dest_host_id != 0 )
       {
         m_is_dest_firewall = false;
         m_dest_id = dest_host_id->get_integer();
       }
       else
         res = false;
     }
     else
     {
       m_is_dest_firewall = false;
       m_dest_id = dest_host_id->get_integer();
     }
  }
  
  if ( attack_id != 0 )
    delete attack_id;
  
  if ( src_host_id != 0 )
    delete src_host_id;
  
  if ( dest_host_id != 0 )
    delete dest_host_id;
  
  if ( dest_firewall_id != 0 )
    delete dest_firewall_id;
      
  return res;
}
    
bool AttackTriple::make_attack_triple( const string &attack_name, const string &source_host, const string &dest, Network* network, bool is_dest_firewall )
{
  bool res = false;
      
  m_src_host_id = -1;
  m_dest_id = -1;
  m_attack_id = -1;
  m_is_dest_firewall = is_dest_firewall;
    
  if ( network != 0 )
  {
    string name;
    for ( int host_ind = 0; host_ind < network->GetNumHosts() && 
                            (m_src_host_id == -1 || m_dest_id == -1 && !m_is_dest_firewall); host_ind++ )
    {
      name = string( ((network->GetHost(host_ind))->GetName()).transcode() );
          
      if ( name == source_host )
        m_src_host_id = host_ind;
          
      if ( name == dest && !m_is_dest_firewall)
        m_dest_id = host_ind;
    }
    
    if ( m_is_dest_firewall )
      for ( FirewallID fw_ind = 0; fw_ind < network->GetFirewallNumber() && m_dest_id == -1; fw_ind++ )
      {
        name = string( (network->GetFirewallName(fw_ind)).transcode() );
        
        if ( name == dest )
          m_dest_id = fw_ind;
      }
        
    for ( int attack_ind = 0; attack_ind < network->GetNumAttacks() && m_attack_id == -1; attack_ind++ )
    {
      name = string( ((network->GetAttack( attack_ind ))->GetName()).transcode() );
          
      if ( name == attack_name )
        m_attack_id = attack_ind;
    }
        
    res = (m_attack_id >= 0 && m_src_host_id >= 0 && m_dest_id >= 0);
  }
      
  return res;
}

string AttackTriple::make_attack_triple_string( Network *net ) const
{
  NetworkInfo *info = get_network_info( net );
  stringstream  str_in("");       
  
  if ( info != 0 )
  {
    str_in << info->attack->GetName() << " ";
    str_in << info->src_host->GetName() << " ";
    
    if ( info->dest_host != 0 )
      str_in << info->dest_host->GetName();
    else
      str_in << TARGET_FW_TYPE << " " << info->dest_firewall->GetName();
    
    delete info;
  }    
  
  return str_in.str();
}

bool AttackTriple::make_attack_triple( const string &str_attack_triple, Network *net )
{
  bool res = true;
  string data = str_attack_triple;
  unsigned int pos = data.find( ' ' );
  string attack_name;
  string src_host;
  string dst;
  string dst_type;
  bool is_dst_firewall = false;
  
         
  if ( pos == string::npos )
    res = false;
  else
  {
    attack_name = data.substr( 0, pos );
            
    while ( data[pos] == ' ' || data[pos] == '\t' ) pos++;
    data = data.substr( pos );
  }            
          
  if ( res )
  {
    pos = data.find( ' ' );
          
    if ( pos == string::npos )
      res = false;
    else
    {
      src_host = data.substr( 0, pos );
            
      while ( data[pos] == ' ' || data[pos] == '\t' ) pos++;
      data = data.substr( pos );
    }
  }
        
  if ( res )
  {
    pos = data.find( ' ' );
    if ( pos == string::npos )
      dst = data;
    else
    {
      dst_type = data.substr( 0, pos );
      
      if ( dst_type != TARGET_FW_TYPE )
        dst = dst_type;
      else
      {
        is_dst_firewall = true;
        while ( data[pos] == ' ' || data[pos] == '\t' ) pos++;
        data = data.substr( pos );
        
        pos = data.find( ' ' );
        if ( pos == string::npos )
          dst = data;
        else
          dst = data.substr( 0, pos );
      }
    }
  }
        
  if ( res )
    res = make_attack_triple( attack_name, src_host, dst, net, is_dst_firewall );
  
  return res;
}

void AttackMeasure::write_to( ostream& s_out, Network *net, bool save_name ) const
{
  string data;
  
  if ( save_name )
  {
    if ( !m_name.empty() )
      s_out << "=" << m_name << "\n";
    s_out << "{" << "\n";
  }
  
  for ( AttackMeasure::iterator p = begin(); p != end(); p++ )
  {
    data = p->make_attack_triple_string( net );
    
    if ( !data.empty() )
      s_out << data << "\n"; 
  }
  
  if ( save_name )
    s_out << "}" << "\n";
}

bool AttackMeasure::read_from( istream& s_in, Network *net )
{
  bool res = false;
  string data;
  AttackTriple attack;
  
  res = Storable::read_min_ASCII_block( s_in, data );
  
  while ( !s_in.eof() && res && data.empty() )
    res = Storable::read_min_ASCII_block( s_in, data );
  
  if ( res && data[0] == '=' )
  {
    m_name = data.substr( 1 );
    res = Storable::read_min_ASCII_block( s_in, data );
  }
  
  while ( !s_in.eof() && res && data.empty() )
    res = Storable::read_min_ASCII_block( s_in, data );
  
  if ( res && data[0] == '{')
    res = Storable::read_min_ASCII_block( s_in, data );
    
  while ( !s_in.eof() && res )
  {
    if ( data.empty() )
    {
      if ( !s_in.eof() )
        res = Storable::read_min_ASCII_block( s_in, data );
      continue;
    }
    
    if ( data[0] == '}' )
      break;
        
    if ( data[0] == '#' )
    {
      if ( !s_in.eof() )
        res = Storable::read_min_ASCII_block( s_in, data );
      continue;
    }
    
    res = attack.make_attack_triple( data, net );
    
    if ( res )
    {
      insert( attack );
      
      if ( !s_in.eof() )
        res = Storable::read_min_ASCII_block( s_in, data );
    }
  }
  
  return res;
}

bool AttackMeasure::load_from_file( const string& file_name, Network *net )
{
  bool res = false;
  ifstream s_in( file_name.c_str() );
    
  if ( !s_in.fail() && !s_in.eof() )
    res = read_from( s_in, net );
  
  return res;
}

bool AttackMeasure::save_into_file( const string& file_name, Network *net ) const
{
  bool res = false;
  ofstream s_out( file_name.c_str() );
    
  if ( (res = (!s_out.fail())) )
    write_to( s_out, net );
  
  return res;
}

bool AttackMeasuresSet::load_from_file( const string& file_name, Network *net )
{
  bool res = false;
  ifstream s_in( file_name.c_str() );
  
  res = !s_in.fail() && !s_in.eof();
  
  while ( !s_in.eof() && res )
  {
    AttackMeasure measure;
    
    res = measure.read_from( s_in, net );
    
    if ( res && (measure.size() > 0 || !measure.get_name().empty()) )
      add_measure( measure );
  }
  
  return res;
}

bool AttackMeasuresSet::save_into_file( const string& file_name, Network *net ) const
{
  bool res = false;
  ofstream s_out( file_name.c_str() );
  
  if ( (res = (!s_out.fail())) )
  {
    for ( AttackMeasuresMap::const_iterator p = m_measures.begin(); p != m_measures.end(); p++ )
    {
      p->second.write_to( s_out, net, true );
      s_out << "\n";
    }
  }
  
  return res;
}

AttackMeasureID AttackMeasuresSet::add_measure( const AttackMeasure& measure )
{
  AttackMeasureID id = -1;
  AttackTripleMeasuresMap::iterator ap; 
  for (AttackMeasuresMap::const_iterator mp = m_measures.begin(); mp != m_measures.end(); mp++ )
    if ( mp->second == measure )
    {
      id = mp->first;
      break;
    }
      
  if ( id < 0 )
  {
    m_measures[(id = (m_last_measure_id++))] = measure;
      
    for ( AttackMeasure::const_iterator p = measure.begin(); p != measure.end(); p++ )
    {
      if ( (ap = m_attack_measure.find( *p )) == m_attack_measure.end() )
      {
        AttackMeasureIDSet id_set;
        id_set.insert( id );
        m_attack_measure[*p] = id_set;
      }
      else
        ap->second.insert( id );
    }
  }    
    
  return id;
}

void AttackMeasuresSet::set_measure( AttackMeasureID measure_id, const AttackMeasure &measure )
{
  if ( measure_id < 0 )
    return;
    
  AttackTripleMeasuresMap::iterator ap;
  AttackMeasure::iterator p;
    
  if ( measure_id >= m_last_measure_id )
  {
    m_last_measure_id = measure_id + 1;
    m_measures[ measure_id ] = measure;
  }
  else
  {
    AttackMeasuresMap::const_iterator mp; 
      
    if ( (mp = m_measures.find( measure_id )) != m_measures.end() )
    {
      for ( p = mp->second.begin(); p != mp->second.end(); p++ )
      {
        if ( (ap = m_attack_measure.find( *p )) != m_attack_measure.end() )
           ap->second.erase( measure_id );            
      }
    }
      
    m_measures[ measure_id ] = measure;
  }
    
  for ( p = measure.begin(); p != measure.end(); p++ )
  {
    if ( (ap = m_attack_measure.find( *p )) == m_attack_measure.end() )
    {
      AttackMeasureIDSet id_set;
      id_set.insert( measure_id );
      m_attack_measure[*p] = id_set;
    }
    else
      ap->second.insert( measure_id );
  }
}

/*****************************************************************************/

NetworkComp::NetworkComp()
{
  char  *home_path = getenv("HOME");
  char  pref_dir_path[255];
  char  pref_path[255];
  int   fd;
  DIR   *dd;
  
  m_network = 0;
  m_curr_sel_node = GraphViewPropertyGroup::DEFAULT;  
  m_gui_start_edge_node_id = Graph::INVALID_NODE_ID;  
  m_is_file_loaded = false;
  network_context = 0;
  m_path_mark = GraphViewPropertiesStorage::EDGE_UNMARKED;
  m_path_info_window = 0;
      
  //create preferences
  if (home_path != 0)
  {
    printf("%s\n", home_path);  
    sprintf(pref_dir_path, "%s/.sgraph", home_path);    
        
    if ((dd = opendir(pref_dir_path)) == 0)
    {
      if (mkdir(pref_dir_path, S_IRWXU|S_IRWXG|S_IRWXO) == -1)
        perror("opendir");
    }
    else
      closedir(dd);
      
    sprintf(pref_dir_path, "%s/.sgraph/netgraph", home_path);
    
    if ((dd = opendir(pref_dir_path)) == 0)
    {
      if (mkdir(pref_dir_path, S_IRWXU|S_IRWXG|S_IRWXO) == -1)
        perror("opendir");
    }
    else
      closedir(dd);
    
    sprintf(pref_dir_path, "%s/.sgraph/mdlgen", home_path);
    
    if ((dd = opendir(pref_dir_path)) == 0)
    {
      if (mkdir(pref_dir_path, S_IRWXU|S_IRWXG|S_IRWXO) == -1)
        perror("opendir");
    }
    else
      closedir(dd);
    
    sprintf(pref_dir_path, "%s/.sgraph/mdlgen/nessus", home_path);
    
    if ((dd = opendir(pref_dir_path)) == 0)
    {
      if (mkdir(pref_dir_path, S_IRWXU|S_IRWXG|S_IRWXO) == -1)
        perror("opendir");
    }
    else
      closedir(dd);
    
    sprintf(pref_path, "%s/.sgraph/netgraph/preferences", home_path);
    
    if ((fd = open(pref_path, O_RDONLY, 0)) == -1)
    {
      printf("%s\n", pref_path);  
      if ((fd = open(pref_path, O_WRONLY|O_CREAT, S_IRUSR|S_IWUSR)) == -1)
        perror("openfile");
      else
      {
        unsigned char layout = GraphView::ARBITRARY_SIZE;
        write(fd, &layout, 1);
        close(fd);
      }
    }
    else
      close(fd);
  }
}

NetworkComp::~NetworkComp()
{
  if (network_context != 0)
    delete network_context;
}

bool NetworkComp::on_register( GraphViewContext *main_app_context, GraphGUI *gui )
{
  GraphView              *view;
  GraphViewPropertyGroup connection_group;
  GraphViewPropertyGroup host_group;
  GraphViewPropertyGroup hor_fire_group;     
  GraphViewPropertyGroup vert_fire_group;     
  GraphViewPropertyGroup scen_node_group;         

  NetInfoStatusInfo *status_info = new NetInfoStatusInfo(main_app_context, this);
  main_app_context->get_status_mngr()->status_registration(status_info); 
    
  SelNodesStatusInfo *status_sel_info = new SelNodesStatusInfo(main_app_context);
  main_app_context->get_status_mngr()->status_registration(status_sel_info); 
  m_init_path_selecting = false;

  network_context = new GraphViewContext();  
  
  if (network_context == 0)
    return false;
  
  network_context->set_graph( new Graph() );
  view = new GraphView( network_context );
  view->set_arrows_drawn( false );
  view->add_node_label_template( "{ }", "{::name}" );
  network_context->set_view( view );
  
  //read preferences
  char      *home_path = getenv("HOME");
  char       pref_path[255];
    
  memset( pref_path, 0, sizeof(pref_path) );
  sprintf(pref_path, "%s/.sgraph/netgraph/preferences", home_path);
    
  ifstream   in_stream( (const char*)pref_path );
    
  if ( in_stream.is_open() )
  {
    unsigned char layout;
    
    layout = in_stream.get( );
    
    view->set_layout( layout );
    in_stream.close();
  }
  //////////////////////////
  
  m_app_context = main_app_context;
  m_gui = gui;
  
  m_net_gui = new NetworkGUI();

  connection_group.set_shape_type( GraphViewPropertyGroup::HIDEN_POINT );
  connection_group.set_name( "Connection point" );
  network_context->get_view()->get_view_properties_storage().add_view_group( "{ node_type=1;}", connection_group );

  host_group.set_shape_type( GraphViewPropertyGroup::SQUARE );
  host_group.set_shape_xy_ratio( 2.0, 1.0 );
  host_group.set_name( "Hosts" );
  host_group.set_shape_filled( true );
  host_group.set_color( Color( RGB( 128, 128, 128) ) );
  host_group.set_font( host_group.get_font(), Color( RGB( 255, 255, 255) ) );
  network_context->get_view()->get_view_properties_storage().add_view_group( "{ node_type=0;}", host_group );
          
  hor_fire_group.set_shape_type( GraphViewPropertyGroup::SQUARE );
  hor_fire_group.set_shape_xy_ratio( 4.0, 1.0 );
  hor_fire_group.set_name( "Horisontal firewalls" );
  hor_fire_group.set_shape_filled( true );      
  hor_fire_group.set_color( Color( RGB( 239, 186, 34) ) );      
  network_context->get_view()->get_view_properties_storage().add_view_group( "{ node_type=2;}", hor_fire_group );

  vert_fire_group.set_shape_type( GraphViewPropertyGroup::SQUARE );
  vert_fire_group.set_shape_xy_ratio( 1.0, 4.0 );
  vert_fire_group.set_name( "Vertical firewalls" );
  vert_fire_group.set_shape_filled( true );            
  vert_fire_group.set_color( Color( RGB( 239, 186, 34) ) );            
  network_context->get_view()->get_view_properties_storage().add_view_group( "{ node_type=3;}", vert_fire_group );

  network_context->get_view()->set_radius(40);
  network_context->get_view()->set_x_separation(40);
  network_context->get_view()->set_y_separation(40);	   

  network_context->get_view()->get_view_properties_storage().set_edge_pen( DrawingPen( DrawingPen::SOLID, 2) );
  
  Color c = network_context->get_view()->get_view_properties_storage().get_edge_color();  
  network_context->get_view()->get_view_properties_storage().set_edge_mark_color( GraphViewPropertiesStorage::EDGE_IN_SCENARIO, c );
  network_context->get_view()->get_view_properties_storage().set_outgoing_edge_color( c );
  network_context->get_view()->get_view_properties_storage().set_incoming_edge_color( c );
  
  m_path_mark = network_context->get_view()->get_view_properties_storage().add_edge_mark( "Net path", Color::RED );
  //network_context->get_view()->get_view_properties_storage().set_reserved_edge_color(GraphView::EDGE_RESERVED1, Color::RED);
        
  return true;
}

GraphViewContext*  NetworkComp::get_network_context()
{
  return network_context;
}

GraphViewContext*  NetworkComp::get_app_context()
{
  return m_app_context;
}

GraphGUI*  NetworkComp::get_GUI()
{
  return m_gui;
}

NetworkGUI*  NetworkComp::get_net_GUI()
{
  return m_net_gui;
}

void NetworkComp::set_graphic_area(GraphicArea* g)
{
  network_g = g;
}

void NetworkComp::clear_network_context()
{
  network_context->set_graph( new Graph() );
  network_context->get_view()->make_graph_view();
  
  if ( m_network != 0 )
  {
    delete m_network;
    m_network = 0;
  }
}
  
void NetworkComp::on_open(const string &file_name)
{
  Graph *net_graph;  
  Graph *app_graph;
  string xml_name;
  GraphFileInfo load_info;  
  
  m_is_file_loaded = false;
  
  if ( m_network != 0 )
  {
    delete m_network;
    m_network = 0;
  }
  
  load_info = m_app_context->get_load_info();
  
  remove_path_node_info_selecting( 0 );
  set_selnode_group(GraphViewPropertyGroup::DEFAULT);      
    
  if ( m_path_info_window != 0 )
  {
    delete m_path_info_window;
    m_path_info_window = 0;
  }
  
  xml_name = load_info.model_file;
  
  if ( !xml_name.empty() )
  {
    std::ifstream in_xml(xml_name.c_str());
    
    if (in_xml.is_open()) 
      in_xml.close();
    else //there is no xml file
      xml_name = "";
  }
  
  if (xml_name.empty())
  {
    unsigned int pos = load_info.file_name.rfind( '.' );
  
    if ( pos == string::npos )
      xml_name = file_name;
    else
      xml_name = file_name.substr( 0, pos );
  
    if ( !xml_name.empty() )
      xml_name += ".xml";
  }
  
  if ( !xml_name.empty() )
  {
    string converted_xml_name = xml_name + "_converted";
    std::ifstream in_xml(xml_name.c_str());
    
    if (in_xml.is_open()) 
      in_xml.close();
    else //there is no xml file
      xml_name = "";
    
    std::ifstream in_converated(converted_xml_name.c_str());
    
    if (in_converated.is_open()) 
    {
      xml_name = converted_xml_name;
      in_converated.close();
    }
  }

  if ( !xml_name.empty() && m_xml_parser_initialized )
  {
    m_gui->show_wait_dialog( "Network scanning. Please wait..." );
    
    if ((m_network = xml_parser( xml_name )) != 0)
    {
      app_graph = m_app_context->get_graph();
      
      if ( app_graph->count_nodes() > 0 )
        if ( app_graph->get_node(0)->get_links_count() <= 1 )
          app_graph->remove_node(0);
      
      net_graph = generate_network_graph();
  
      network_context->set_graph( net_graph );
        
      network_context->get_view()->make_graph_view();
      
      m_is_file_loaded = true;
    } 
    else
    {
      net_graph = new Graph();
      network_context->set_graph( net_graph );
  
      network_context->get_view()->make_graph_view();
    }
    
    m_gui->hide_wait_dialog();
    
  } 
  else
  {
    net_graph = new Graph();
    network_context->set_graph( net_graph );
  
    network_context->get_view()->make_graph_view();
  }
  
  // set attribute aliases
  if ( m_network != 0 )
  {
    GraphAttrNameStorage *name_storage = app_graph->get_graph_attr_name_storage();
    ServiceID sid;
    
    int id = name_storage->add_new_alias_set( );
    name_storage->set_curr_alias_set( id );
    
    StringContainer full_attr_name;
    StringContainer attr_alias;
    string attr_name;
    string name;
    stringstream s_str;
    
    set<string> host_obj_named_values;
    set<int>    host_obj_int_values;
    int         ivalue;
    int         ind;
    int         host_ind;
    
    DOMStringSet *attr_values = m_network->GetHostObjectValues();
    
    for ( DOMStringSet::iterator val_it = attr_values->begin(); val_it != attr_values->end(); val_it++ )
    {
      name = val_it->transcode();
      
      if ( isdigit( name[0] ) ) {
        sscanf( name.c_str(), "%d", &ivalue );
        host_obj_int_values.insert( ivalue );
      }
      else      
        host_obj_named_values.insert( name );
    }
    
    full_attr_name.push_back( "Node" );
    attr_alias.push_back( "Node" );
    
    //set alias for highest attr. level 
    name_storage->set_meta_attr_name_alias( full_attr_name, attr_alias );
    
    //
    full_attr_name.push_back( "accept" );
    attr_alias.push_back( "Accepted state" );
    name_storage->set_attr_name_alias( full_attr_name, "bool", attr_alias );
    attr_alias.pop_back();
    full_attr_name.pop_back();
    
    full_attr_name.push_back( "final" );
    attr_alias.push_back( "Final state" );
    name_storage->set_attr_name_alias( full_attr_name, "bool", attr_alias );
    attr_alias.pop_back();
    full_attr_name.pop_back();
    
    //
    full_attr_name.push_back( "GlobalVariables" );
    
    //set type values aliases
    name_storage->set_attr_type_value_alias( "bool", 0, "No" );
    name_storage->set_attr_type_value_alias( "bool", 1, "Yes" );
    
    if ( m_network->DoesModelHaveFirewalls() )
    {
      //Firewalls
      attr_alias.push_back( "Firewalls" );
      
      for ( FirewallID f_id = 0; f_id < m_network->GetFirewallNumber(); f_id++ )
      {
        name = m_network->GetFirewallName( f_id ).transcode();
        attr_alias.push_back( name );
        
        attr_alias.push_back( "Status" );
        
        s_str.str("");
        s_str << "net.firewalls";
        if ( m_network->GetFirewallNumber() > 1 )
          s_str << "[" << f_id << "]";
        s_str << ".on";
        full_attr_name.push_back( s_str.str() );
        
        name_storage->set_attr_name_alias( full_attr_name, "bool", attr_alias );
        name_storage->set_attr_value_alias( full_attr_name, 0, "Off" );
        name_storage->set_attr_value_alias( full_attr_name, 1, "On" );
        
        full_attr_name.pop_back();
        attr_alias.pop_back();
        
        attr_alias.push_back( "Rules" );
        for ( host_ind = 0; host_ind < m_network->GetNumHosts(); host_ind++ )
        {
          name = "Source host: ";
          name += (m_network->GetHost( host_ind )->GetName().transcode());
          attr_alias.push_back( name );
          for ( PortID port_ind1 = 0; port_ind1 < m_network->GetFirewallPortNumber(); port_ind1++ )
          {
            name = "Source port: ";
            name += (m_network->GetFirewallPortName( port_ind1 ).transcode());
            attr_alias.push_back( name );
            for ( ind = 0; ind < m_network->GetNumHosts(); ind++ )
            {
              name = "Destination host: ";
              name += (m_network->GetHost( ind )->GetName().transcode());
              attr_alias.push_back( name );
              
              if ( m_network->GetFirewallPortNumber() == 1 )
              {
                name = "Destination port: ";
                name += (m_network->GetFirewallPortName(0).transcode());
                attr_alias.push_back( name );
                
                s_str.str("");
                s_str << "net.firewalls";
                if ( m_network->GetFirewallNumber() > 1 )
                  s_str << "[" << f_id << "]";
                s_str << ".rule.src[" << host_ind << "].srcport.dst[" << ind << "].dstport";
                
                full_attr_name.push_back( s_str.str() );
                name_storage->set_attr_name_alias( full_attr_name, "bool", attr_alias );
                name_storage->set_attr_value_alias( full_attr_name, 0, "Deny" );
                name_storage->set_attr_value_alias( full_attr_name, 1, "Allow" );
                full_attr_name.pop_back();
                
                attr_alias.pop_back();
              }
              else
                for ( PortID port_ind2 = 0; port_ind2 < m_network->GetFirewallPortNumber(); port_ind2++ )
                {
                  name = "Destination port: "; 
                  name += (m_network->GetFirewallPortName( port_ind2 ).transcode());
                  attr_alias.push_back( name );
                  
                  s_str.str("");
                  s_str << "net.firewalls";
                  if ( m_network->GetFirewallNumber() > 1 )
                    s_str << "[" << f_id << "]";
                  s_str << ".rule.src[" << host_ind << "].srcport[" << port_ind1 
                        << "].dst[" << ind << "].dstport[" << port_ind2 << "]";
                  
                  full_attr_name.push_back( s_str.str() );
                  name_storage->set_attr_name_alias( full_attr_name, "bool", attr_alias );
                  name_storage->set_attr_value_alias( full_attr_name, 0, "Deny" );
                  name_storage->set_attr_value_alias( full_attr_name, 1, "Allow" );
                  full_attr_name.pop_back();
                  
                  attr_alias.pop_back();
                }
                
              attr_alias.pop_back();
            }              
            attr_alias.pop_back();
          }
          attr_alias.pop_back();
        }          
        attr_alias.pop_back();
        
        attr_alias.pop_back();
      }
      attr_alias.pop_back();
      //-- End firewalls
    }
    
    //set host attributes aliases
    attr_alias.push_back( "Hosts" );
    for ( host_ind = 0; host_ind < m_network->GetNumHosts(); host_ind++ )
    {
      name = string( ((m_network->GetHost(host_ind))->GetName()).transcode() );
      
      attr_alias.push_back( name );
      
      //services
      attr_alias.push_back( "Services" );
      for (sid = 1; sid <= (ServiceID)m_network->GetNumServices(); sid++)
      {
        s_str.str("");
        attr_name = string((m_network->GetServiceName( sid )).transcode());
        s_str << "net.hosts[" << host_ind << "].running_" << attr_name;
        full_attr_name.push_back( s_str.str() );
        attr_alias.push_back( attr_name );
        
        name_storage->set_attr_name_alias( full_attr_name, "bool", attr_alias );
        name_storage->set_attr_value_alias( full_attr_name, 0, "Not Run");
        name_storage->set_attr_value_alias( full_attr_name, 1, "Run");
        
        attr_alias.pop_back();
        full_attr_name.pop_back();
      }
      attr_alias.pop_back();
      
      //vulnerabilities
      DOMStringSet *vuln = m_network->GetVulnerabilities();
      attr_alias.push_back( "Vulnerabilities" );
      for ( DOMStringSet::iterator vit = vuln->begin(); vit != vuln->end(); vit++ )
      {
        s_str.str("");
        attr_name = string((*vit).transcode());
        s_str << "net.hosts[" << host_ind << "]." << attr_name;
        full_attr_name.push_back( s_str.str() );
        attr_alias.push_back( attr_name );
        
        name_storage->set_attr_name_alias( full_attr_name, "bool", attr_alias );
        name_storage->set_attr_value_alias( full_attr_name, 0, "Not Detected");
        name_storage->set_attr_value_alias( full_attr_name, 1, "Detected");
        
        attr_alias.pop_back();
        full_attr_name.pop_back();
      }
      attr_alias.pop_back();
      
      //objects
      HostObjectAttrNameMap *obj = m_network->GetHostObjectNames();
      if ( !obj->empty() )
      {
        attr_alias.push_back( "Objects" );
        
        HostObjectAttrNameMap::const_iterator obj_it;
        for (obj_it = obj->begin(); obj_it != obj->end(); obj_it++) 
        {
          s_str.str("");
          name = string((obj_it->first).transcode()); 
          s_str << "net.hosts[" << host_ind << "].objects." << name;
          
          attr_alias.push_back( name );
          
          name = s_str.str();
          
          //exists attr.
          attr_name = name + ".exists";
          full_attr_name.push_back( attr_name );
          attr_alias.push_back( "Exists" );
          name_storage->set_attr_name_alias( full_attr_name, "bool", attr_alias );
          attr_alias.pop_back();
          full_attr_name.pop_back();
          
          //source host attr.
          attr_name = name + ".source_host";
          full_attr_name.push_back( attr_name );
          attr_alias.push_back( "Source" );
          
          name_storage->set_attr_name_alias( full_attr_name, "int", attr_alias );
          
          for ( ind = 0; ind < m_network->GetNumHosts(); ind++ )
          {
            attr_name = string( ((m_network->GetHost(ind))->GetName()).transcode() );
            name_storage->set_attr_value_alias( full_attr_name, ind, attr_name );
          }
          attr_alias.pop_back();
          full_attr_name.pop_back();
          
          //services
          attr_name = name + ".service";
          full_attr_name.push_back( attr_name );
          attr_alias.push_back( "Service" );
          
          name_storage->set_attr_name_alias( full_attr_name, "int", attr_alias );
          name_storage->set_attr_value_alias( full_attr_name, 0, "None" );
          
          for (sid = 1; sid <= (ServiceID)m_network->GetNumServices(); sid++) 
          {
            s_str.str("");
            attr_name = string((m_network->GetServiceName( sid )).transcode());
            name_storage->set_attr_value_alias( full_attr_name, sid, attr_name );
          }
          attr_alias.pop_back();
          full_attr_name.pop_back();
          
          //the other obj. attributes
          for (DOMStringSet::iterator attr = obj_it->second.begin(); attr != obj_it->second.end(); attr++ ) 
          {
            s_str.str("");
            attr_name = string((*attr).transcode());
            
            full_attr_name.push_back( name + "." + attr_name );
            attr_alias.push_back( attr_name );
            
            name_storage->set_attr_name_alias( full_attr_name, "int", attr_alias );
            
            for ( DOMStringSet::iterator val_it = attr_values->begin(); val_it != attr_values->end(); val_it++ )
            {
              s_str.str("");
              attr_name = string((*val_it).transcode());
                            
              ivalue = 512;
              for ( set<string>::const_iterator p = host_obj_named_values.begin(); p != host_obj_named_values.end(); p++, ivalue++ )
              {
                while ( host_obj_int_values.find(ivalue) != host_obj_int_values.end() ) 
                  ivalue++;
      
                name_storage->set_attr_value_alias( full_attr_name, ivalue, *p );
              }
            }
            attr_alias.pop_back();
            full_attr_name.pop_back();
          }
          attr_alias.pop_back();
        }
        attr_alias.pop_back();
        //-- End Host objects
      }
      attr_alias.pop_back();
    }
    attr_alias.pop_back();
    //--End hosts
    
    
    if ( !m_network->DoesModelHaveFirewalls() ) 
    {
      // Connectivity
      attr_alias.push_back( "Connectivity" );
      for ( host_ind = 0; host_ind < m_network->GetNumHosts(); host_ind++ )
        for ( ind = 0; ind < m_network->GetNumHosts(); ind++ )
        {
          s_str.str("");
          name = string( ((m_network->GetHost(host_ind))->GetName()).transcode() ) +
                 " -> " + string( ((m_network->GetHost(ind))->GetName()).transcode() );
          
          attr_alias.push_back( name );
          
          s_str << "net.hosts[" << host_ind << "].conn[" << ind << "].";
          name = s_str.str();
          
          attr_alias.push_back( "Using path" );
          full_attr_name.push_back( name + "path" );
          name_storage->set_attr_name_alias( full_attr_name, "bool", attr_alias );
          full_attr_name.pop_back();
          attr_alias.pop_back();
          
          attr_alias.push_back( "Using services" );
          
          for (sid = 1; sid <= (ServiceID)m_network->GetNumServices(); sid++) 
          {
            attr_name = string( (m_network->GetServiceName(sid)).transcode() );
            attr_alias.push_back( attr_name );
            
            attr_name = name + attr_name;
            full_attr_name.push_back( attr_name );
            
            name_storage->set_attr_name_alias( full_attr_name, "bool", attr_alias );
            //name_storage->set_attr_value_alias( full_attr_name, 0, "Not Allowed");
            //name_storage->set_attr_value_alias( full_attr_name, 1, "Allowed");
            full_attr_name.pop_back();
            attr_alias.pop_back();
          }
          
          attr_alias.pop_back();
          attr_alias.pop_back();
        }
      attr_alias.pop_back();
      //-- End connectivity
    }
    else
    {
      // Firewalls between hosts
      attr_alias.push_back( "Firewalls between hosts" );
      for (host_ind = 0; host_ind < m_network->GetNumHosts(); host_ind++)
      {
        name = "Source host: "; 
        name += (m_network->GetHost(host_ind)->GetName().transcode());
        attr_alias.push_back( name );
        for ( ind = 0; ind < m_network->GetNumHosts(); ind++ )
        {
          name = "Destination host: "; 
          name += (m_network->GetHost(ind)->GetName().transcode());
          attr_alias.push_back( name );
          BoolVectorContainer* paths = m_network->GetFWPathsBetweenHosts( host_ind, ind );
          for ( int i = 0; i < m_network->GetMaxFWHostPaths(); i++ )
          {
            if ( i >= paths->size() )
            {
              attr_alias.push_back( "Non-existent path" );
            }
            else
            {
              s_str.str("");
              s_str << "Path " << i;
              name = s_str.str();
              attr_alias.push_back( name );
            }
            
            if ( m_network->GetFirewallNumber() == 1 )
            {
              s_str.str("");
              name = m_network->GetFirewallName( 0 ).transcode();
              attr_alias.push_back( name );
              
              s_str << "net.hosts[" << host_ind << "].path[" << ind << "]._";
              if ( m_network->GetMaxFWHostPaths() > 1 )
                s_str << "[" << i << "]";
              s_str << ".firewalls";
              
              attr_name = s_str.str();
              
              full_attr_name.push_back( attr_name );
              name_storage->set_attr_name_alias( full_attr_name, "bool", attr_alias );
              name_storage->set_attr_value_alias( full_attr_name, 0, "Does Not Exist");
              name_storage->set_attr_value_alias( full_attr_name, 1, "Exists");
              full_attr_name.pop_back();
              
              attr_alias.pop_back();
            }
            else
              for ( FirewallID f_id = 0; f_id < m_network->GetFirewallNumber(); f_id++ )
              {
                s_str.str("");
                name = m_network->GetFirewallName( f_id ).transcode();
                attr_alias.push_back( name );
                
                s_str << "net.hosts[" << host_ind << "].path[" << ind << "]._";
                if ( m_network->GetMaxFWHostPaths() > 1 )
                  s_str << "[" << i << "]";
                s_str << ".firewalls[" << f_id << "]";
                
                attr_name = s_str.str();
                
                full_attr_name.push_back( attr_name );
                name_storage->set_attr_name_alias( full_attr_name, "bool", attr_alias );
                name_storage->set_attr_value_alias( full_attr_name, 0, "Does Not Exist");
                name_storage->set_attr_value_alias( full_attr_name, 1, "Exists");
                full_attr_name.pop_back();
                
                attr_alias.pop_back();
              }
            
            attr_alias.pop_back();
          }
          attr_alias.pop_back();
        }
        attr_alias.pop_back();
      }
        
      attr_alias.pop_back();
      //-- End firewalls between hosts
      
      // Firewall reachability
      attr_alias.push_back( "Hosts reachability" );
      for ( host_ind = 0; host_ind < m_network->GetNumHosts(); host_ind++ )
      {
        name = "From host: "; 
        name += (m_network->GetHost(host_ind)->GetName().transcode());
        attr_alias.push_back( name );
        for ( ind = 0; ind < m_network->GetNumHosts(); ind++ )
        {
          name = "To host: "; 
          name += (m_network->GetHost(ind)->GetName().transcode());
          attr_alias.push_back( name );
          if ( m_network->GetFirewallPortNumber() == 1 )
          {
            s_str.str("");
            name = "Source port: "; 
            name += (m_network->GetFirewallPortName(0).transcode());
            name += ", destination port: "; 
            name += (m_network->GetFirewallPortName(0).transcode());
            attr_alias.push_back( name );
            
            s_str << "net.hosts[" << host_ind << "].fw_reachable[" << ind << "].srcport.dstport";
            attr_name = s_str.str();
            
            full_attr_name.push_back( attr_name );
            name_storage->set_attr_name_alias( full_attr_name, "bool", attr_alias );
            full_attr_name.pop_back();
            
            attr_alias.pop_back();
          }
          else 
            for ( PortID port_ind1 = 0; port_ind1 < m_network->GetFirewallPortNumber(); port_ind1++ )
            {
              name = "Source port: "; 
              name += (m_network->GetFirewallPortName(port_ind1).transcode());
              attr_alias.push_back( name );
              for ( PortID port_ind2 = 0; port_ind2 < m_network->GetFirewallPortNumber(); port_ind2++ )
              {
                s_str.str("");
                name = "Destination port: "; 
                name += (m_network->GetFirewallPortName(port_ind2).transcode());
                attr_alias.push_back( name );
                
                s_str << "net.hosts[" << host_ind << "].fw_reachable[" << ind << "].srcport["
                      << port_ind1 << "].dstport[" << port_ind2 << "]";
                attr_name = s_str.str();
            
                full_attr_name.push_back( attr_name );
                name_storage->set_attr_name_alias( full_attr_name, "bool", attr_alias );
                full_attr_name.pop_back();
                
                attr_alias.pop_back();
              }
              
              attr_alias.pop_back();
            }
          attr_alias.pop_back();
        }
        attr_alias.pop_back();
      }
      attr_alias.pop_back();
      //-- End Firewall reachability
    }
    
    //Trust
    attr_alias.push_back( "Trust" );
    for ( host_ind = 0; host_ind < m_network->GetNumHosts(); host_ind++ )
      for ( ind = 0; ind < m_network->GetNumHosts(); ind++ )
      {
        s_str.str("");
        name = string( ((m_network->GetHost(host_ind))->GetName()).transcode() ) +
               " trusts " + string( ((m_network->GetHost(ind))->GetName()).transcode() ); 
        
        s_str << "net.hosts[" << host_ind << "].trust[" << ind << "]";
        attr_alias.push_back( name );
        full_attr_name.push_back( s_str.str() );
        
        name_storage->set_attr_name_alias( full_attr_name, "bool", attr_alias );
                
        full_attr_name.pop_back();
        attr_alias.pop_back();
      }        
    attr_alias.pop_back();
      
    //intruder
    attr_alias.push_back( "Intruder" );  
    //intruder privileges
    attr_alias.push_back( "Privileges" );
    for ( host_ind = 0; host_ind < m_network->GetNumHosts(); host_ind++ )
    {
      s_str.str("");
      name = string( ((m_network->GetHost(host_ind))->GetName()).transcode() );
      
      s_str << "net.intruder.privilege[" << host_ind << "]";
      
      attr_alias.push_back( name );
      full_attr_name.push_back( s_str.str() );
        
      name_storage->set_attr_name_alias( full_attr_name, "byte", attr_alias );
      name_storage->set_attr_value_alias( full_attr_name, 0, "none" );
      name_storage->set_attr_value_alias( full_attr_name, 1, "user" );
      name_storage->set_attr_value_alias( full_attr_name, 2, "root" );
      
      full_attr_name.pop_back();
      attr_alias.pop_back();
    }
    attr_alias.pop_back();
    
    // Knowledge
    attr_alias.push_back( "Knowledge" );
    DOMStringSet know_names = m_network->adversary.GetKnowledgeNames();
    for (DOMStringSet::const_iterator kit = know_names.begin(); kit != know_names.end(); kit++)
    {
      name = string( kit->transcode() );
      
      attr_name = "net.intruder." + name;
      attr_alias.push_back( name );
      full_attr_name.push_back( attr_name );
      name_storage->set_attr_name_alias( full_attr_name, "bool", attr_alias );
      full_attr_name.pop_back();
      attr_alias.pop_back();
    }
    attr_alias.pop_back();
    
    attr_alias.pop_back();
    //--End Intruder
    
    
    //Intrusion Detection System
    attr_alias.push_back( "Intrusion Detection System" );
    for ( host_ind = 0; host_ind < m_network->GetNumHosts(); host_ind++ )
      for ( ind = 0; ind < m_network->GetNumHosts(); ind++ )
      {
        s_str.str("");
        name = string( ((m_network->GetHost(host_ind))->GetName()).transcode() ) +
               " detects " + string( ((m_network->GetHost(ind))->GetName()).transcode() ); 
        
        s_str << "net.ids[" << host_ind << "].detected[" << ind << "]";
        attr_alias.push_back( name );
        full_attr_name.push_back( s_str.str() );
        
        name_storage->set_attr_name_alias( full_attr_name, "bool", attr_alias );
                
        full_attr_name.pop_back();
        attr_alias.pop_back();
      }
   attr_alias.pop_back();
      
   //Attack
   attr_alias.push_back( "Attack Info" );
   
   full_attr_name.push_back( "net.attack_id" );
   attr_alias.push_back( "Attack name" );
   name_storage->set_attr_name_alias( full_attr_name, "byte", attr_alias );
      
   for ( ind = 0; ind < m_network->GetNumAttacks(); ind++ )
   {
     name = string( ((m_network->GetAttack(ind))->GetName()).transcode() );
     name_storage->set_attr_value_alias( full_attr_name, ind, name );
   }
   full_attr_name.pop_back();
   attr_alias.pop_back();
   
   full_attr_name.push_back( "net.source_ip" );
   attr_alias.push_back( "Source host" );
   name_storage->set_attr_name_alias( full_attr_name, "byte", attr_alias );
   
   for ( ind = 0; ind < m_network->GetNumHosts(); ind++ )
   {
     name = string( ((m_network->GetHost(ind))->GetName()).transcode() );
     name_storage->set_attr_value_alias( full_attr_name, ind, name );
   }
   full_attr_name.pop_back();
   attr_alias.pop_back();
   
   full_attr_name.push_back( "net.target_ip" );
   attr_alias.push_back( "Target host" );
   if ( m_network->DoesModelHaveFirewalls() )
   {
     name_storage->set_attr_name_alias( full_attr_name, "int", attr_alias );
     name_storage->set_attr_value_alias( full_attr_name, -1, "NONE" );
   }
   else
     name_storage->set_attr_name_alias( full_attr_name, "byte", attr_alias );
   
   for ( ind = 0; ind < m_network->GetNumHosts(); ind++ )
   {
     name = string( ((m_network->GetHost(ind))->GetName()).transcode() );
     name_storage->set_attr_value_alias( full_attr_name, ind, name );
   }
   full_attr_name.pop_back();
   attr_alias.pop_back();
   
   if ( m_network->DoesModelHaveFirewalls() )
   {
     full_attr_name.push_back( "net.target_fwi" );
     attr_alias.push_back( "Target firewall" );
     name_storage->set_attr_name_alias( full_attr_name, "int", attr_alias );
     name_storage->set_attr_value_alias( full_attr_name, -1, "NONE" );
     for ( FirewallID fw_ind = 0; fw_ind < m_network->GetFirewallNumber(); fw_ind++ )
     {
       name = string( (m_network->GetFirewallName(fw_ind)).transcode() );
       name_storage->set_attr_value_alias( full_attr_name, fw_ind, name );
     }
     
     full_attr_name.pop_back();
     attr_alias.pop_back();
   }     
   
   full_attr_name.push_back( "net.attack_flavor" );
   attr_alias.push_back( "Attack flavor" );
   name_storage->set_attr_name_alias( full_attr_name, "bool", attr_alias );
   name_storage->set_attr_value_alias( full_attr_name, 0, "Stealthy" );
   name_storage->set_attr_value_alias( full_attr_name, 1, "Detectable" );
   full_attr_name.pop_back();
   attr_alias.pop_back();
   
   full_attr_name.push_back( "net.alarm" );
   attr_alias.push_back( "Alarm" );
   name_storage->set_attr_name_alias( full_attr_name, "bool", attr_alias );
   full_attr_name.pop_back();
   attr_alias.pop_back();
      
   attr_alias.pop_back();
   //--End attack
   
   if ( (m_gui->get_scenario_app()->get_extra_behavior_modes( ) & ScenarioApp::USE_ATTR_ALIASES) == 0 )
     name_storage->set_curr_alias_set( -1 );
  }
} 

void NetworkComp::on_init_path_finding()
{
  m_init_path_selecting = true;
}

void NetworkComp::on_begin_path_finding( NodeIndex first_selected_node_id, GraphicArea *g )
{
  remove_path_node_info_selecting( g );
  set_selnode_group(GraphViewPropertyGroup::DEFAULT);
}

void NetworkComp::on_end_path_finding( const GraphPath &path, GraphicArea *g )
{
  m_init_path_selecting = false;
}

void NetworkComp::remove_path_node_info_selecting( GraphicArea *g )
{
  StyleFlags selnode_group = get_selnode_group();
      
  if (selnode_group != GraphViewPropertyGroup::DEFAULT)  
  {
    Graph *graph = m_app_context->get_graph();
    NodeIndexSet nodes;
    NodeAttributeList attr_list;
    string attr_string = m_app_context->get_view()->get_view_properties_storage().get_view_group( selnode_group ).get_assigned_attributes( );
    
    if ( attr_string != "" )
    {
      bool is_attr_str = GraphAttributes::parse_string( attr_string, attr_list );
  
      if ( is_attr_str )
      {
        for ( NodeIndex id = graph->find_first_node( attr_list ); id != Graph::INVALID_NODE_ID; id = graph->find_next_node( attr_list, id ) )
          nodes.insert( id );
      }
    }
    
    m_app_context->get_view()->get_view_properties_storage().remove_view_group( selnode_group );
    
    m_app_context->get_view()->redraw_nodes( g, nodes, true );
  }
}

void NetworkComp::on_path_selecting( int path_id, const GraphPath &path, GraphicArea *g )
{
  if ( !path.empty() && path_id != GraphPath::INVALID_PATH_INDEX )
  {
    InfoStorageSubLevel *root_level = new InfoStorageSubLevel( "root" );
    NodeLinkList nodes;
      
    path.get_path_nodes( path_id, nodes );
      
    remove_path_node_info_selecting( g );
    set_selnode_group(GraphViewPropertyGroup::DEFAULT);
      
    set_scenario_nodes_information( root_level, nodes, g );
    //set_scenario_information( window->get_root_level(), edges, path.get_init_node() );
              
    if ( m_path_info_window == 0 )
    {
      m_path_info_window = m_gui->create_information_window( GraphGUI::DOCKABLE_PATH_INFO, root_level, true );
      m_path_info_window->register_close_handler( new NetPathCloseWindowHandler( this, m_path_info_window ) );
    }
    else
    {
      m_path_info_window->set_root_level( root_level );
      m_gui->generate_create_dockable_event( m_path_info_window, false );
    }
      
    m_path_info_window->show( true );
  }
}

void NetworkComp::on_init_edge_selecting( NodeIndex node )
{
  m_gui_start_edge_node_id = node;  
}
    
void NetworkComp::on_end_edge_selecting( const Edge& edge, bool edge_exist )
{
  Node      *node;
  GraphAttributes *attr;
  AttributeValue  *value;
  string    type;
  int       src_host_id = 0, dest_host_id = 0;
  bool      src_attr_exist;
  bool      dest_attr_exist;
    
  m_gui_start_edge_node_id = Graph::INVALID_NODE_ID;
  
  if (!m_is_file_loaded || !edge_exist)
    return;
  
  node = m_app_context->get_graph()->get_node(edge.end_node);
  
  if (node != 0)
  {
    GraphInformationGUI* window;
    InfoStorageSubLevel *root_level = new InfoStorageSubLevel( "root" );
  
    attr = node->get_attributes();
  
    src_attr_exist = attr->query_value( "::GlobalVariables::net.source_ip", type, &value );
  
    if ( value != 0 )
    {
      src_host_id = (int)value->get_integer();
      delete value;
    }
  
    dest_attr_exist = attr->query_value( "::GlobalVariables::net.target_ip", type, &value );
     
    if ( value != 0 )    
    {
      dest_host_id = (int)value->get_integer();
      delete value;
    }
  
/*  network_context->get_view()->unselect_all_nodes();
  network_context->get_view()->mark_all_adges( event.get_graphic_area() );      */
              
    if (src_attr_exist && dest_attr_exist && dest_host_id >= 0)
      mark_network_path( src_host_id, dest_host_id );
    
    set_edge_information( root_level, edge.end_node);
    
    window = m_gui->create_information_window( GraphGUI::DOCKABLE_EDGE_INFO, root_level );
      
    window->show( true );
  }
}
    
void NetworkComp::on_mouse_button_press( const Event &event )
{
  if (!m_is_file_loaded)
    return;

  if (m_app_context->get_status_mngr()->get_current_status()->get_status_name() == "sel_nodes_info")
      m_app_context->get_status_mngr()->block(false);              
  
  network_context->get_view()->unselect_all_nodes( network_g );
  network_context->get_view()->mark_all_edges( network_g );      
  
  if ( !m_init_path_selecting )
  {
    remove_path_node_info_selecting( event.get_graphic_area() );
    set_selnode_group(GraphViewPropertyGroup::DEFAULT);      
        
    if ( m_path_info_window != 0 )
    {
      delete m_path_info_window;
      m_path_info_window = 0;
    }
  }
}

void NetworkComp::on_mouse_motion_notify( const Event &event )
{
  TwoDPoint   point = event.get_point();    
  NodeIndex   node_id;
  Node        *node;
  NetInfoStatusInfo *status;  
  string status_name;  
    
  node_id = m_app_context->get_view()->get_node_id( point );
    
  if (node_id != Graph::INVALID_NODE_ID)
  {
    status_name = "net_node_info";  
    node = m_app_context->get_graph()->get_node(node_id);
    status = (NetInfoStatusInfo*)m_app_context->get_status_mngr()->get_status(status_name);
    status->set_node(node);
    m_app_context->get_status_mngr()->set_current_status(status_name);
  }      
  else  
  {
    status_name = "default";      
    m_app_context->get_status_mngr()->set_current_status(status_name);      
  }
}

void NetworkComp::mark_network_path( NodeIndex host_id1, NodeIndex host_id2 )
{
	Host      *host1, *host2;
  Node      *node1 = network_context->get_graph()->get_node(host_id1);     
  Node      *node2 = network_context->get_graph()->get_node(host_id2);       
  NodeLinkListIter  node_iter;
  NodeLinkListIter  end_node_iter;

  if (!is_network_model_info_enabled())
    return;

  //test if network structure is set
  if (!(host_id1 < m_network->GetNumHosts() && host_id2 < m_network->GetNumHosts()))
    return;
  
  host1 = m_network->GetHost(host_id1);
  host2 = m_network->GetHost(host_id2);
  
  if (host1 == 0 || host2 == 0)
    return;
  
  network_context->get_view()->select_node(host_id2, network_g);
  
  if (host_id1 != host_id2)
/*    select_node(host_id1);
  else*/
  //hosts in one network
  if (strcmp(host1->NetworkName(), host2->NetworkName()) == 0)
  {
    bool  found_edge;
    
    node_iter = node1->get_links()->begin();
    end_node_iter = node1->get_links()->end();
        
    found_edge = false;
    while (node_iter != end_node_iter)
    {
      if (*node_iter == host_id2)
      {
        found_edge = true;
        break;
      }
      node_iter++;
    }
    
    if (!found_edge)
    {
      node_iter = node2->get_links()->begin();
      end_node_iter = node2->get_links()->end();
      
      while (node_iter != end_node_iter)
      {
        if (*node_iter == host_id1)
        {
          found_edge = true;
          break;
        }
        node_iter++;
      }
    }
    
    if (found_edge)
    {
      Edge  e(host_id1, host_id2);
      network_context->get_view()->mark_edge( e, network_g, m_path_mark );
    }
  }
  else
  { // hosts in different networks
    NodeAttributeList attr_list;
    string str_conn("{ node_type=1;}");
    bool      is_attr_str;
    Graph     *conn_graph = 0;
    Graph     *fire_graph = 0;
    NodeIndex conn_node_index1 = 0;
    NodeIndex conn_node_index2 = 0;    
    bool      found_conn1 = false;
    bool      found_conn2 = false;    
    Node      *conn_node1 = 0;
    Node      *conn_node2 = 0;    
    
    is_attr_str = GraphAttributes::parse_string( str_conn, attr_list );
    
    //drawing of the edges between hosts and connection points
    if (is_attr_str)
    {
      conn_graph = network_context->get_graph()->get_subgraph(attr_list);
      
      if (conn_graph == 0)
        return;
            
      for (int ind = 0; ind < conn_graph->count_nodes(); ind++)
      {
        conn_node_index1 = conn_graph->get_node(ind)->get_sub_id();
                
        node_iter = node1->get_links()->begin();
        end_node_iter = node1->get_links()->end();
        while (node_iter != end_node_iter)
        {
          if (*node_iter == conn_node_index1)
            break;
          node_iter++;
        }
        
        if (node_iter == end_node_iter)
        {
          conn_node1 = network_context->get_graph()->get_node(conn_node_index1);                         
          node_iter = conn_node1->get_links()->begin();
          end_node_iter = conn_node1->get_links()->end();
          
/*          v_iter = m_nodes[conn_node_index1]->out_edges.begin();*/
          while (node_iter != end_node_iter)
          {
            if (*node_iter == host_id1)
              break;
            node_iter++;
          }
          
          if (node_iter != end_node_iter)
          {
            Edge  e(conn_node_index1, host_id1);
            network_context->get_view()->mark_edge( e, network_g, m_path_mark );            
            found_conn1 = true;
            break;
          }
        }
        else
        {
          Edge  e(host_id1, conn_node_index1);          
          network_context->get_view()->mark_edge( e, network_g, m_path_mark );
          found_conn1 = true;          
          break;
        }
      }

      ////////
      for (int ind = 0; ind < conn_graph->count_nodes(); ind++)
      {
        conn_node_index2 = conn_graph->get_node(ind)->get_sub_id();
     
        node_iter = node2->get_links()->begin();
        end_node_iter = node2->get_links()->end();
        while (node_iter != end_node_iter)
        {
          if (*node_iter == conn_node_index2)
            break;
          node_iter++;
        }

        if (node_iter == end_node_iter)
        {
          conn_node2 = network_context->get_graph()->get_node(conn_node_index2);               
          node_iter = conn_node2->get_links()->begin();
          end_node_iter = conn_node2->get_links()->end();
          
/*          v_iter = m_nodes[conn_node_index1]->out_edges.begin();*/
          while (node_iter != end_node_iter)
          {
            if (*node_iter == host_id2)
              break;
            node_iter++;
          }

        if (node_iter != end_node_iter)
          {
            Edge  e(conn_node_index2, host_id2);
            network_context->get_view()->mark_edge( e, network_g, m_path_mark );            
            found_conn2 = true;
            break;
          }            
        }
        else
        {
          Edge  e(host_id2, conn_node_index2);          
          network_context->get_view()->mark_edge( e, network_g, m_path_mark );
          found_conn2 = true;
          break;
        }          
      }
        
    //drawing of the edges between connection points and firewalls     
        if (found_conn1 && found_conn2)
        {
          string str_fire_vert("{ node_type=2;}");
          string str_fire_hor("{ node_type=3;}");
          bool      is_attr_str = GraphAttributes::parse_string( str_fire_vert, attr_list );
          NodeIndex fire_node_index;
          bool      fire_found = false;
          bool      hor_attr_loop = false;
          
          NodeLinkListIter  conn_node1_iter;
          NodeLinkListIter  end_conn_node1_iter;
          NodeLinkListIter  conn_node2_iter;
          NodeLinkListIter  end_conn_node2_iter;
          NodeLinkListIter  fire_node_iter;
          NodeLinkListIter  end_fire_node_iter;
          Node              *fire_node;
          
          if (conn_node1 == 0)
            conn_node1 = network_context->get_graph()->get_node(conn_node_index1);                         
          
          if (conn_node2 == 0)
            conn_node2 = network_context->get_graph()->get_node(conn_node_index2);                         
          
          conn_node1_iter = conn_node1->get_links()->begin();
          end_conn_node1_iter = conn_node1->get_links()->end();
          conn_node2_iter = conn_node2->get_links()->begin();
          end_conn_node2_iter = conn_node2->get_links()->end();
          

          while (is_attr_str)
          {
            fire_graph = network_context->get_graph()->get_subgraph(attr_list);
        
            for (int ind = 0; ind < fire_graph->count_nodes(); ind++)
            {
              bool  found_conn1 = false, 
                    found_conn2 = false;
              
              fire_node_index = fire_graph->get_node(ind)->get_sub_id();            
              fire_node = network_context->get_graph()->get_node(fire_node_index);               
              fire_node_iter = fire_node->get_links()->begin();
              end_fire_node_iter = fire_node->get_links()->end();
              
              for (; fire_node_iter < end_fire_node_iter; ++fire_node_iter)
                if (*fire_node_iter == conn_node_index1)
                  found_conn1 = true;
                else if (*fire_node_iter == conn_node_index2)                
                  found_conn2 = true;
                
              if (found_conn1 && found_conn2)
              {
                Edge  e1(fire_node_index, conn_node_index1);          
                Edge  e2(fire_node_index, conn_node_index2);          
                network_context->get_view()->mark_edge( e1, network_g, m_path_mark );
                network_context->get_view()->mark_edge( e2, network_g, m_path_mark );                
                fire_found = true;
                break;
              } else if (found_conn1)
              {
                for (; conn_node2_iter < end_conn_node2_iter; ++conn_node2_iter)
                  if (*conn_node2_iter == fire_node_index)
                  {
                    found_conn2 = true;
                    break;
                  }
                  
                if (found_conn2)
                {
                  Edge  e1(fire_node_index, conn_node_index1);          
                  Edge  e2(conn_node_index2, fire_node_index);          
                  network_context->get_view()->mark_edge( e1, network_g, m_path_mark );
                  network_context->get_view()->mark_edge( e2, network_g, m_path_mark );                
                  fire_found = true;
                  break;
                }
              } else if (found_conn2)
              {
                for (; conn_node1_iter < end_conn_node1_iter; ++conn_node1_iter)
                  if (*conn_node1_iter == fire_node_index)
                  {
                    found_conn1 = true;
                    break;
                  }
                  
                if (found_conn1)
                {
                  Edge  e2(fire_node_index, conn_node_index2);          
                  Edge  e1(conn_node_index1, fire_node_index);          
                  network_context->get_view()->mark_edge( e1, network_g, m_path_mark );
                  network_context->get_view()->mark_edge( e2, network_g, m_path_mark );                
                  fire_found = true;
                  break;
                }
              } else
              {
                for (; conn_node1_iter < end_conn_node1_iter; ++conn_node1_iter)
                  if (*conn_node1_iter == fire_node_index)
                  {
                    found_conn1 = true;
                    break;
                  }
                  
                for (; conn_node2_iter < end_conn_node2_iter; ++conn_node2_iter)
                  if (*conn_node2_iter == fire_node_index)
                  {
                    found_conn2 = true;
                    break;
                  }

                if (found_conn1 && found_conn2)
                {
                  Edge  e1(conn_node_index1, fire_node_index);          
                  Edge  e2(conn_node_index2, fire_node_index);          
                  network_context->get_view()->mark_edge( e1, network_g, m_path_mark );
                  network_context->get_view()->mark_edge( e2, network_g, m_path_mark );                
                  fire_found = true;
                  break;
                }
              }
              
              if (fire_found)
                break;
            }
            
            delete fire_graph;
            
            if (!fire_found && !hor_attr_loop)
              is_attr_str = GraphAttributes::parse_string( str_fire_hor, attr_list );            
            else if (!hor_attr_loop)
              hor_attr_loop = true;
            else
              break;
          }
        }
        
       delete conn_graph;  
    }      
  }

}

bool NetworkComp::get_network_info( NodeIndex node_ind, NetworkInfo &net_info )
{
  if (!is_network_model_info_enabled())
    return false;
  
  int         src_host_id = 0, dest_host_id = -1, dest_firewall_id = -1;
  int         attack_id = 0;

  Graph      *graph = m_app_context->get_graph();
  Node       *node  = graph->get_node( node_ind );
  GraphAttributes *attr = node->get_attributes();
  AttributeValue *value;
  string     type;
  
  bool attr_exist = attr->query_value( "::GlobalVariables::net.attack_id", type, &value );
  
  if ( !attr_exist )
    return false;
  
  attack_id = (int)value->get_integer();
  
  delete value;
  
  attr_exist = attr->query_value( "::GlobalVariables::net.source_ip", type, &value );
  
  if ( !attr_exist )
    return false;    
  
  src_host_id = (int)value->get_integer();
  
  delete value;
  
  attr_exist = attr->query_value( "::GlobalVariables::net.target_ip", type, &value );
  
  if ( !attr_exist )
    return false;
  
  dest_host_id = (int)value->get_integer();
  
  delete value;
  
  
  
  if ( attr->query_value( "::GlobalVariables::net.target_fwi", type, &value ) )
    dest_firewall_id = (int)value->get_integer();
  
  delete value;

  
  if (src_host_id < m_network->GetNumHosts())
    net_info.src_host = m_network->GetHost(src_host_id);
  else
  {
    net_info.src_host = 0;
    cerr << "Incorrect host id";
    return false;
  } 
  
  if (dest_host_id < m_network->GetNumHosts() && dest_host_id >= 0 )
    net_info.dest_host = m_network->GetHost(dest_host_id);
  else
  {
    net_info.dest_host = 0;
    
    if ( dest_firewall_id < 0 ) 
    {
      cerr << "Incorrect host id";
      return false;
    }
    
    if ( dest_firewall_id < (int)m_network->GetFirewallNumber() )
      net_info.dest_firewall = m_network->GetFirewall( dest_firewall_id );
    else
    {
      cerr << "Incorrect firewall id";
      return false;
    }
  }
  
  if (attack_id < m_network->GetNumAttacks())
    net_info.attack = m_network->GetAttack(attack_id);
  else
  {
    net_info.attack = 0;
    cerr << "Incorrect attack id";
    return false;
  }
  
  return true;
  
} 

Graph* NetworkComp::generate_network_graph()
{
  Graph*              sub_graph = new Graph( );
  StringSet           net_names;
  StringSet::iterator net_iter;
  Host*               host;
  NetworkGroups       groups;
  HostVector*         vec;
  char                buf[255];
  char*               attr_templ = "NetObject{ byte: node_type=%i; text: name=\"%s\";}";  
                                                      //node_type==0 - host nodes
                                                      //node_type==1 - connection nodes
                                                      //node_type==2 - hor. firewall nodes  
                                                      //node_type==3 - vert. firewall nodes  
  GraphAttrNameStorage *attr_storage = sub_graph->get_graph_attr_name_storage();
  attr_storage->init_name_set( "NetObject{ byte: node_type=0; text: name=\"\";}" );
  
  if ( is_network_model_info_enabled() )
  {
    Firewall *firewall;
    GraphAttributes*    attr = 0;
    bool      exists;
    NodeIndex ind = 0;
    map< int, NodeIndex > host_nodes;
    map< FirewallID, NodeIndex > firewall_nodes;
    if ( m_network->DoesModelHaveFirewalls() )
    {
      //Creating of host nodes
      for ( int host_ind = 0; host_ind < m_network->GetNumHosts(); host_ind++ )
      {
        host = m_network->GetHost(host_ind);
        memset(buf, 0, sizeof(buf));
        sprintf(buf, attr_templ, 0, host->GetName().transcode());
        attr = new GraphAttributes( attr_storage );
    
        if (attr->set_attribute_string( string( buf ) ))
        {  
          ind = sub_graph->add_node(attr);
          host_nodes[host_ind] = ind;
        }
        else
        {
          cout << "Incorrect attribute string";
          delete attr;
        }
      }
      
      //Creating of firewall nodes
      for ( FirewallID f_ind = 0; f_ind < m_network->GetFirewallNumber(); f_ind++ )
      {
        firewall = m_network->GetFirewall( f_ind );
        memset(buf, 0, sizeof(buf));
        sprintf(buf, attr_templ, 2, firewall->GetName().transcode());
        
        attr = new GraphAttributes( attr_storage );
    
        if (attr->set_attribute_string( string( buf ) ))
        {
          ind = sub_graph->add_node(attr);
          firewall_nodes[f_ind] = ind; 
        }
        else
        {
          cout << "Incorrect attribute string";
          delete attr;
        }
      }
      
      // Creating connectivity edges
      for ( int host_ind1 = 0; host_ind1 < m_network->GetNumHosts(); host_ind1++ )
        for ( int host_ind2 = host_ind1; host_ind2 < m_network->GetNumHosts(); host_ind2++ )
        {
          if ( m_network->AreFWHostsConnected( host_ind1, host_ind2 ) )
          {
            exists = false;
            FirewallID f_ind;
            for ( f_ind = 0; f_ind < m_network->GetFirewallNumber() && !exists; f_ind++ )
            {
              firewall = m_network->GetFirewall( f_ind );
              
              exists = firewall->IsBetweenHosts( host_ind1, host_ind2 );
            }
            
            if ( exists )
            {
              sub_graph->add_edge( host_nodes[host_ind1], firewall_nodes[f_ind - 1]);
              sub_graph->add_edge( firewall_nodes[f_ind - 1], host_nodes[host_ind2] );
            }
            else
              sub_graph->add_edge( host_nodes[host_ind1], host_nodes[host_ind2]);
          }
        }
      
      
    }
    else
    {
      GraphAttributes*    attr = 0;
      bool                is_attr_correct = false;
      NodeIndexVector     conn_vec;
      
      for (int host_ind = 0; host_ind < m_network->GetNumHosts(); host_ind++)
      {
        host = m_network->GetHost(host_ind);
        net_names.insert(string(host->NetworkName()));
      }
      
      for (net_iter = net_names.begin(); net_iter != net_names.end(); net_iter++)
      {
        vec = new HostVector();
        groups[*net_iter] = vec;
      }    
      
      //Creating of host nodes
      for (int host_ind = 0; host_ind < m_network->GetNumHosts(); host_ind++)
      {
        host = m_network->GetHost(host_ind);
        vec = groups[string(host->NetworkName())];
        vec->push_back(host->GetId());
        
        memset(buf, 0, sizeof(buf));
        sprintf(buf, attr_templ, 0, host->GetName().transcode() );
        
        attr = new GraphAttributes( attr_storage );
        is_attr_correct = attr->set_attribute_string( string( buf ) );
    
        if (is_attr_correct)
          sub_graph->add_node(attr);
        else
          cout << "Incorrect attribute string";
      }
    
    //Creating of connection nodes  
      
        memset(buf, 0, sizeof(buf));
        sprintf(buf, attr_templ, 1, "" );
      
        for (net_iter = net_names.begin(); net_iter != net_names.end(); net_iter++)
        {
          int conn_ind = 0;
        
          attr = new GraphAttributes( attr_storage );
          is_attr_correct = attr->set_attribute_string( string( buf ) );
        
          if (is_attr_correct)
            conn_ind = sub_graph->add_node(attr);
          else
            cout << "Incorrect attribute string";
        
          conn_vec.push_back(conn_ind);
        
          vec = groups[*net_iter];
        
          for (int ind1 = 0; ind1 < (int)vec->size(); ind1++)
          {
            if (net_names.size() > 1)
            {
              if (net_iter == net_names.begin())
                sub_graph->add_edge((NodeIndex)(*vec)[ind1], conn_ind);
              else
                sub_graph->add_edge(conn_ind, (NodeIndex)(*vec)[ind1]);        
            }
            
            for (int ind2 = ind1+1; ind2 < (int)vec->size(); ind2++ )
              sub_graph->add_edge((NodeIndex)(*vec)[ind1], (NodeIndex)(*vec)[ind2]);
          }
        }
    
    //Creating of firewall nodes    
      memset(buf, 0, sizeof(buf));
    
      for (int i = 0; i < (int)conn_vec.size(); i++)
        for (int j = i+1; j < (int)conn_vec.size(); j++)
        {
          int fire_ind = 0;
          
          if (i == 0)
            sprintf(buf, attr_templ, 2, "");
          else
            sprintf(buf, attr_templ, 3, "");
          
          attr = new GraphAttributes( attr_storage );
          is_attr_correct = attr->set_attribute_string( string( buf ) );
        
          if (is_attr_correct)
            fire_ind = sub_graph->add_node(attr);
          else
            cout << "Incorrect attribute string";
    
          if (i > 0)
          {
            LinkAttr attr(0, 0);
            sub_graph->add_edge(conn_vec[i], fire_ind, attr);
            sub_graph->add_edge(fire_ind, conn_vec[j], attr);        
          }
          else
          {
            sub_graph->add_edge(conn_vec[i], fire_ind);
            sub_graph->add_edge(fire_ind, conn_vec[j]);        
          }
        }
      
      for (net_iter = net_names.begin(); net_iter != net_names.end(); net_iter++)
      {
        groups[*net_iter]->clear();
        delete groups[*net_iter];
      }    
      groups.clear();
      net_names.clear();
      conn_vec.clear();
    }
  }
  
  return sub_graph;
}

void NetworkComp::select_subgraph(NodeIndex host_id, GraphicArea *g)
{
  Graph     *graph = m_app_context->get_graph();
  AttributeValue *value1;  
  AttributeValue *value2;  
  //long      head_value;    
  //long      tail_value;      
  string    type;
  
  for (NodeIndex i = 0; i < (NodeIndex)graph->count_nodes(); i++)
  {
    GraphAttributes *attr = graph->get_node(i)->get_attributes();
  
    bool src_attr_exist = attr->query_value( "::GlobalVariables::net.source_ip", type, &value1 );
    
    if (src_attr_exist )
    {
      if ( value1->get_integer() == host_id )
      {
        Node  *node = graph->get_node(i);
        NodeLinkListIter  node_iter = node->get_links()->begin();
        NodeLinkListIter  end_node_iter = node->get_links()->end();
        
        m_app_context->get_view()->select_node(i, g);
        
        while (node_iter != end_node_iter)
        {
          attr = graph->get_node(*node_iter)->get_attributes();
          src_attr_exist = attr->query_value( "::GlobalVariables::net.source_ip", type, &value2 );
          
          if (src_attr_exist )
          {  
            if ( value2->get_integer() == host_id )
            {
              Edge  e(i, *node_iter);
              m_app_context->get_view()->mark_edge( e, g );          
            }
            delete value2;
          }
            
          node_iter++;
        }
        
      }
      delete value1;
    }
  }
}

int NetworkComp::handle_mouse_events( const Event &event, GraphicArea  *main_g )
{
  GraphicArea  *g  = event.get_graphic_area();  
  TwoDPoint    point = event.get_point();
  NodeIndex    node_id;
/*  GraphView   *main_view = net_context->get_dependent_context()->get_view(0);*/
  
  if (!m_is_file_loaded)
    return 1;
  
  switch ( event.get_type() )
  {
  case Event::MOUSE_PRESSED:
  
    if (m_app_context->get_status_mngr()->get_current_status()->get_status_name() == "sel_nodes_info")
      m_app_context->get_status_mngr()->block(false);              
  
    if ( m_gui_start_edge_node_id != Graph::INVALID_NODE_ID )
      break;
    
    m_gui_start_point = point;
    m_gui_end_point = point;
  
    if ( (event.get_modifiers() & Event::SHIFT_MASK) == 0 )
    {
      network_context->get_view()->unselect_all_nodes( g );
      network_context->get_view()->mark_all_edges( g );
      m_app_context->get_view()->unselect_all_nodes( main_g );
      m_app_context->get_view()->mark_all_edges( main_g );
        
      remove_path_node_info_selecting( event.get_graphic_area() );
      set_selnode_group(GraphViewPropertyGroup::DEFAULT);      
            
      if ( m_path_info_window != 0 )
      {
        delete m_path_info_window;
        m_path_info_window = 0;
      }
    }
    
    m_gui_view_rect = false;    
    
    break;

  case Event::MOUSE_RELEASED:
    
    node_id = network_context->get_view()->get_node_id( point );  
  
/*    if (event.get_modifiers() & Event::SHIFT_MASK == 0)
      unselect_all( g );*/
  
    if ( m_gui_view_rect )
    {
      NodeIndexVector nodes;
      Rectangle       rect( GGMIN( m_gui_start_point.get_x(), m_gui_end_point.get_x()),
                            GGMIN( m_gui_start_point.get_y(), m_gui_end_point.get_y()),
                            GGMAX( m_gui_start_point.get_x(), m_gui_end_point.get_x()),
                            GGMAX( m_gui_start_point.get_y(), m_gui_end_point.get_y()) );
            
      draw_gui_rect( g );
      
      if ( rect.get_width() < 4 && rect.get_height() < 4 && 
            event.get_button_code() == Event::MOUSE_LEFT_BUTTON )
      {
        node_id = network_context->get_view()->get_node_id( point );
      
        if (node_id != Graph::INVALID_NODE_ID)
        {
/*          if ((event.get_modifiers() & Event::SHIFT_MASK) == 0)
            main_view->unselect_all(main_g);*/
          
          network_context->get_view()->select_node(node_id, g);      
          select_subgraph(node_id, main_g);
          
          if ( is_network_model_info_enabled() )
            if (node_id < m_network->GetNumHosts())
            {                                  
              string status_str = "sel_nodes_info";  
              SelNodesStatusInfo *status;
              if (m_app_context->get_status_mngr()->set_current_status(status_str))
              { 
                ostringstream ost;        
                ost << m_network->GetHost(node_id)->NetworkName() << "." << m_network->GetHost(node_id)->GetName();
              
                status = (SelNodesStatusInfo*)m_app_context->get_status_mngr()->get_current_status();
                
                if ( (event.get_modifiers() & Event::SHIFT_MASK) == 0 )                
                  status->remove_all_targets();  
                
                status->add_target(ost.str());            
                
                unsigned int nodes_number, edges_number, sel_nodes_number, sel_edges_number;              
                m_app_context->get_view()->get_number_info(nodes_number, edges_number, sel_nodes_number, sel_edges_number);              
                status->set_select_info(sel_nodes_number, sel_edges_number);    
                
                m_app_context->get_status_mngr()->block(true);
              }
            }
        }

      }
      else
      {        
        network_context->get_view()->get_node_id_from_rect( rect, nodes );
        
/*        if (nodes.size() > 0 && (event.get_modifiers() & Event::SHIFT_MASK) == 0)
          main_view->unselect_all( main_g );*/
        string status_str = "sel_nodes_info";  
        SelNodesStatusInfo *status;
        bool  is_status_set = false;  
          
        if ( is_network_model_info_enabled() )
          for ( unsigned int i = 0; i < nodes.size(); i++ )
            if (nodes[i] < m_network->GetNumHosts())
              is_status_set = true;


        if (is_status_set && (is_status_set = m_app_context->get_status_mngr()->set_current_status(status_str)))
        { 
          status = (SelNodesStatusInfo*)m_app_context->get_status_mngr()->get_current_status();
          
          if ( (event.get_modifiers() & Event::SHIFT_MASK) == 0 )
            status->remove_all_targets();  
        }

        for ( unsigned int i = 0; i < nodes.size(); i++ )
        {
          network_context->get_view()->select_node(nodes[i], g);      
          select_subgraph(nodes[i], main_g);
            
          if ( is_network_model_info_enabled() )
            if (is_status_set && (nodes[i] < m_network->GetNumHosts()))
            {              
              ostringstream ost;        
              ost << m_network->GetHost(nodes[i])->NetworkName() << "." << m_network->GetHost(nodes[i])->GetName();
              status->add_target(ost.str());                            
            }
        }
        
        if (is_status_set)
        {
          unsigned int nodes_number, edges_number, sel_nodes_number, sel_edges_number;              
          m_app_context->get_view()->get_number_info(nodes_number, edges_number, sel_nodes_number, sel_edges_number);              
          status->set_select_info(sel_nodes_number, sel_edges_number);    
            
          m_app_context->get_status_mngr()->block(true);
        }
        
      }
    }
    else
    {
      node_id = network_context->get_view()->get_node_id( point );
      
      if (node_id != Graph::INVALID_NODE_ID)
      {
/*        if ((event.get_modifiers() & Event::SHIFT_MASK) == 0)
          main_view->unselect_all(main_g);*/
        
        network_context->get_view()->select_node(node_id, g);      
        select_subgraph(node_id, main_g);
          
        if ( is_network_model_info_enabled() )
          if (node_id < m_network->GetNumHosts())
          {            
            string status_str = "sel_nodes_info";  
            SelNodesStatusInfo *status;
            if (m_app_context->get_status_mngr()->set_current_status(status_str))
            { 
              ostringstream ost;        
              ost << m_network->GetHost(node_id)->NetworkName() << "." << m_network->GetHost(node_id)->GetName();
              
              status = (SelNodesStatusInfo*)m_app_context->get_status_mngr()->get_current_status();
                
              if ( (event.get_modifiers() & Event::SHIFT_MASK) == 0 )              
                status->remove_all_targets();  
              
              status->add_target(ost.str());                
              
              unsigned int nodes_number, edges_number, sel_nodes_number, sel_edges_number;              
              m_app_context->get_view()->get_number_info(nodes_number, edges_number, sel_nodes_number, sel_edges_number);              
              status->set_select_info(sel_nodes_number, sel_edges_number);    
              
              m_app_context->get_status_mngr()->block(true);
            }
          }
      }
    }
    
    break;

  case Event::MOUSE_MOVE:
    
    if ( event.get_button_code() == Event::MOUSE_LEFT_BUTTON && 
         m_gui_start_edge_node_id == Graph::INVALID_NODE_ID)
    {
      draw_gui_rect( g );
      
      m_gui_end_point = point;
      draw_gui_rect( g );
      m_gui_view_rect = true;
    }
    
    break; 
 
  default:;
  }
  
  return 0;
}

void NetworkComp::handle_zoom_in()
{
  if (!m_is_file_loaded)
    return;

  GraphView *view = network_context->get_view();
  view->zoom_in();
}

void NetworkComp::handle_zoom_out()
{
  if (!m_is_file_loaded)
    return;

  GraphView *view = network_context->get_view();
  view->zoom_out();
}

void NetworkComp::handle_fit_to_window()
{
  if (!m_is_file_loaded)
    return;

  GraphView *view = network_context->get_view();
  view->set_layout( GraphView::FIT_TO_WINDOW );
}

void NetworkComp::handle_fit_to_height()
{
  if (!m_is_file_loaded)
    return;

  GraphView *view = network_context->get_view();
  view->set_layout( GraphView::FIT_TO_HEIGHT );
}

void NetworkComp::handle_fit_to_width()
{
  if (!m_is_file_loaded)
    return;

  GraphView *view = network_context->get_view();
  view->set_layout( GraphView::FIT_TO_WIDTH );
}

void NetworkComp::handle_fit_selection_to_window()
{
  if (!m_is_file_loaded)
    return;

  GraphView *view = network_context->get_view();
  view->set_layout( GraphView::FIT_SELECTION_TO_WINDOW );
}

void NetworkComp::handle_arbitrary_size()
{
  if (!m_is_file_loaded)
    return;

  GraphView *view = network_context->get_view();
  view->set_layout( GraphView::ARBITRARY_SIZE );
}

bool NetworkComp::handle_selection_centering( const Rectangle &display_rect )
{
  if (!m_is_file_loaded)
    return false;

  GraphView *view = network_context->get_view();
  GraphGUI  *gui  = m_gui;
  bool  res = false;
  NodeIndexVector   selected_nodes;
  
  view->get_selected_nodes( selected_nodes );
  
  if ( selected_nodes.size() == 0 )
    gui->show_message_dialog( "Please select nodes" );
  else
  {
    view->centering( display_rect );
    res = true;
  }
    
  return res;
}

void NetworkComp::draw_gui_rect( GraphicArea *g )
{
  if (!m_is_file_loaded)
    return;

  if ( !(m_gui_start_point == m_gui_end_point) && g != 0 )
  {
    g->set_paint_mode( GraphicArea::XOR_MODE );
    g->set_foreground_color( RGB( 128, 128, 128 ) );
    g->set_drawing_pen( DrawingPen( DrawingPen::ON_OFF_DASH, 1 ) );
      
    Rectangle rect( GGMIN( m_gui_start_point.get_x(), m_gui_end_point.get_x()),
                    GGMIN( m_gui_start_point.get_y(), m_gui_end_point.get_y()),
                    GGMAX( m_gui_start_point.get_x(), m_gui_end_point.get_x()),
                    GGMAX( m_gui_start_point.get_y(), m_gui_end_point.get_y()) );
    rect.draw( g );
    g->set_paint_mode( GraphicArea::SOLID_COLOR );
  }
}

Graph* NetworkComp::remove_attacks_from_scenario( const AttackTriplesSet &attacks )
{
  vector<NodeAttributeList> attr_lists;
  NodeAttributeList attr_list;
  Graph  *graph_curr = m_app_context->get_graph();
  Graph  *graph = 0;
  
  for ( AttackTriplesSet::const_iterator p = attacks.begin(); p != attacks.end(); p++ )
  {
    p->make_graph_attr_list( attr_list );
    
    if ( attr_list.size() > 0 )
      attr_lists.push_back( attr_list );
  }
  
  if ( attr_lists.size() <= 0 )
    m_gui->show_message_dialog( "Attack list is not correct" );
  else
  {
    graph = graph_curr->get_reduced_graph( attr_lists );
        
    if ( graph == 0 )
      m_gui->show_message_dialog( "There are no more scenarios" );
  }
  
  return graph;
} 
 
void NetworkComp::set_host_information(  Host* host, InfoStorageSubLevel *level )
{
  if ( host == 0 )
    return;
  
  InfoStorageItemLevel *item_level;
  ostringstream   s_stream( "" );
    
    //cout << "\tname:" << info.src_host->GetName() << "\n";
  item_level = new InfoStorageItemLevel("Host name");
  s_stream << host->GetName();
  item_level->add_value(new InfoValue(s_stream.str().c_str()));
  level->add_sublevel(item_level);
  s_stream.str("");
        
    //cout << "\tid:" << info.src_host->GetId() << "\n";
  item_level = new InfoStorageItemLevel("Host id");
  s_stream << host->GetId();
  item_level->add_value(new InfoValue(s_stream.str().c_str()));
  level->add_sublevel(item_level);
  s_stream.str("");
    
    //cout << "\tIP:\n\t\t";
  item_level = new InfoStorageItemLevel("Host IP");
  StringSet* ips = host->IPs();
  for (StringSet::iterator ip = ips->begin(); ip != ips->end(); ip++)
  {
    s_stream << *ip;
    item_level->add_value(new InfoValue(s_stream.str().c_str()));      
    s_stream.str("");
  }
  level->add_sublevel(item_level);
    
    //cout << "\tNetwork name:" << info.src_host->NetworkName();
  item_level = new InfoStorageItemLevel("Network name");
  s_stream << host->NetworkName();
  item_level->add_value(new InfoValue(s_stream.str().c_str()));
  level->add_sublevel(item_level);
  s_stream.str("");

    //cout << "\n\tVulnerabilities:\n\t\t";
  item_level = new InfoStorageItemLevel("Vulnerabilities");    
  DOMStringSet *vuls = host->GetVulnerabilities();
  for (DOMStringSet::iterator vul = vuls->begin(); vul != vuls->end(); vul++)
  { 
    CVEClickHandler *cve_handler = new CVEClickHandler();          
    s_stream << *vul;
    InfoValue *val = new InfoValue(s_stream.str().c_str(), Color::BLUE, Font::UNDERLINED);
    val->set_click_handler(cve_handler);      
    item_level->add_value( val );            
    s_stream.str("");
  }
  level->add_sublevel(item_level);
}

void NetworkComp::set_firewall_information( Firewall* firewall, InfoStorageSubLevel *level )
{
  InfoStorageItemLevel *item_level;
    
  item_level = new InfoStorageItemLevel("Name");
  item_level->add_value( new InfoValue( firewall->GetName().transcode() ) );
  level->add_sublevel(item_level);
}

void NetworkComp::set_attack_information( Attack* attack, InfoStorageSubLevel *level )
{
  InfoStorageItemLevel *item_level;
    
  item_level = new InfoStorageItemLevel("Name");
  item_level->add_value( new InfoValue( attack->GetName().transcode() ) );
  level->add_sublevel(item_level);
    
  item_level = new InfoStorageItemLevel("Description");
  item_level->add_value( new InfoValue( attack->GetDescription().transcode() ) );
  level->add_sublevel(item_level);
     
  DOMStringSet* veids = attack->GetCveids();
  item_level = new InfoStorageItemLevel("CVE");    

  for (DOMStringSet::iterator vul = veids->begin(); vul != veids->end(); vul++)
  {
    CVEClickHandler *cve_handler = new CVEClickHandler();          
    InfoValue *val = new InfoValue( vul->transcode(), Color::BLUE, Font::UNDERLINED);
    val->set_click_handler(cve_handler);
        
    item_level->add_value( val );        
   }
   
   level->add_sublevel(item_level);
}
 
void NetworkComp::set_attack_triples_information( InfoStorageSubLevel *root_level, const AttackTriplesSet &attacks )
{
  NetworkInfo          *info;
  InfoStorageSubLevel *item_level;  
  InfoStorageSubLevel *host_level;
  InfoStorageSubLevel *attack_level;
  string               level_name;

  if ( root_level != 0 && is_network_model_info_enabled() )
    for ( AttackTriplesSet::const_iterator p = attacks.begin(); p != attacks.end(); p++ )
    {
      info = p->get_network_info( m_network );
      
      if ( info != 0 )
      {
        level_name = string(info->attack->GetName().transcode()) + ": " +
                     string(info->src_host->GetName().transcode());
        if ( info->dest_host != 0 )
          level_name += " -> " + string(info->dest_host->GetName().transcode());
        else if ( info->dest_firewall != 0 )
          level_name += " -> " + string(info->dest_firewall->GetName().transcode());
        
        item_level = new InfoStorageSubLevel(level_name);
        
        host_level = new InfoStorageSubLevel("Source host");
        set_host_information( info->src_host, host_level );
        item_level->add_sublevel( host_level );
        
        if ( info->dest_host != 0 )
        {
          host_level = new InfoStorageSubLevel("Destination host");
          set_host_information( info->dest_host, host_level );
          item_level->add_sublevel( host_level );
        }
        else if ( info->dest_firewall != 0 )
        {
          host_level = new InfoStorageSubLevel("Destination firewall");
          set_firewall_information( info->dest_firewall, host_level );
          item_level->add_sublevel( host_level );
        }          
        
        attack_level = new InfoStorageSubLevel("Attack");
        set_attack_information( info->attack, attack_level );
        item_level->add_sublevel( attack_level );
        
        root_level->add_sublevel( item_level );
        
        delete info;
      }
    }
}

void NetworkComp::set_attack_measures_information( InfoStorageSubLevel *root_level, const AttackMeasuresSet &measures )
{
  AttackMeasure        measure;
  InfoStorageSubLevel *item_level;  
    
  if ( root_level != 0 && is_network_model_info_enabled())
    for ( AttackMeasureID id = measures.get_first_measure_id(); id >= 0; id = measures.get_next_measure_id( id ) )
    {
      measure = measures.get_measure( id );
      
      item_level = new InfoStorageSubLevel( measure.get_name() );
      set_attack_triples_information( item_level, measure );
      
      root_level->add_sublevel( item_level );
    }
}

void NetworkComp::set_edge_information( InfoStorageSubLevel *root_level, NodeIndex node_id)
{
  vector<string> value_list;
  InfoStorageSubLevel *src_level = new InfoStorageSubLevel("Source host");  
  InfoStorageSubLevel *dest_level;    
  InfoStorageSubLevel *attack_level = new InfoStorageSubLevel("Attack");      
  NetworkInfo     info;
  bool            is_attack_graph = get_network_info( node_id, info );
  //InfoStorageItemLevel *item_level;  
  ostringstream   s_stream( "" );  

  if ( is_attack_graph )
  {
    set_host_information( info.src_host, src_level );
    root_level->add_sublevel(src_level);
    
    if ( info.dest_host != 0 )
    {
      dest_level = new InfoStorageSubLevel("Destination host");
      set_host_information( info.dest_host, dest_level );   
      root_level->add_sublevel(dest_level);      
    }
    else if ( info.dest_firewall != 0 )
    {
      dest_level = new InfoStorageSubLevel("Destination firewall");
      set_firewall_information( info.dest_firewall, dest_level );   
      root_level->add_sublevel(dest_level);
    }
    set_attack_information( info.attack, attack_level );    

/*    
    item_level = new InfoStorageItemLevel("Name");
    s_stream << info.attack->GetName();
    item_level->add_value(new InfoValue(s_stream.str().c_str()));
    attack_level->add_sublevel(item_level);
    s_stream.str("");
    
    item_level = new InfoStorageItemLevel("Description");
    s_stream << info.attack->GetDescription();
    item_level->add_value(new InfoValue(s_stream.str().c_str()));
    attack_level->add_sublevel(item_level);
    s_stream.str("");
    
    DOMStringSet* veids = info.attack->GetCveids();
    item_level = new InfoStorageItemLevel("CVE");    
      for (DOMStringSet::iterator vul = veids->begin(); vul != veids->end(); vul++)
      {
        CVEClickHandler *cve_handler = new CVEClickHandler();          
        s_stream << *vul;
        InfoValue *val = new InfoValue(s_stream.str().c_str(), Color::BLUE, Font::UNDERLINED);
        val->set_click_handler(cve_handler);
        
        item_level->add_value( val );        
        s_stream.str("");
      } 
    attack_level->add_sublevel(item_level);
*/      
    
    
    root_level->add_sublevel(attack_level);      
  } 
}

typedef set<NodeIndex>  NodeIndexSet;

void NetworkComp::set_scenario_information( InfoStorageSubLevel *root_level, const EdgeList &edges, NodeIndex start_node_id )
{
  if ( !is_network_model_info_enabled() )
    return;
  
  bool res = true;
  NetworkInfo  info;
  NodeIndexSet node_set;
  set<string>  hosts_set;
  set<string>  firewalls_set;
  set<string>  CVEs_set;
  Edge edge;
  vector<string> hosts;
  vector<string> CVEs;
  ostringstream   s_stream( "" );
  InfoStorageItemLevel *firewalls_level = 0;
  InfoStorageItemLevel *hosts_level = new InfoStorageItemLevel("Host names");  
  InfoStorageItemLevel *cves_level = new InfoStorageItemLevel("CVEs");    
  //InfoStorageItemLevel *item_level;  
   
  for ( EdgeList::const_iterator iter = edges.begin(); iter != edges.end(); iter++ )
  {
    edge = *iter;
    
    if ( edge.start_node != start_node_id )
      node_set.insert( edge.start_node );
    
    if ( edge.end_node != start_node_id )
      node_set.insert( edge.end_node );
  }
  
  for ( NodeIndexSet::iterator p = node_set.begin(); p != node_set.end(); p++ )
  {
    res = get_network_info( *p, info );
    
    if ( res )
    {
      s_stream << info.src_host->NetworkName() << "." << info.src_host->GetName();
      hosts_set.insert( s_stream.str() );
      s_stream.str( "" );
      
      if ( info.dest_host != 0 )
      {
        s_stream << info.dest_host->NetworkName() << "." << info.dest_host->GetName();
        hosts_set.insert( s_stream.str() );
        s_stream.str( "" );
      }
      else if ( info.dest_firewall != 0 )
      {
        s_stream << info.dest_firewall->GetName();
        firewalls_set.insert( s_stream.str() );
        s_stream.str( "" );
      }        
      DOMStringSet* veids = info.attack->GetCveids();
    
      for (DOMStringSet::iterator vul = veids->begin(); vul != veids->end(); vul++)
      {
        s_stream << *vul;
        CVEs_set.insert( s_stream.str() );
        s_stream.str("");
      }
    }
  }
  
  if ( !hosts_set.empty() )
  {
    set<string>::iterator it = hosts_set.begin();
    for ( it++; it != hosts_set.end(); it++ )
    {      
      hosts_level->add_value(new InfoValue(it->c_str()));
    }
  } 
  
  if ( !firewalls_set.empty() )
  {
    firewalls_level = new InfoStorageItemLevel("Firewall names");  
    set<string>::iterator it = firewalls_set.begin();
    for ( it++; it != firewalls_set.end(); it++ )
    {      
      firewalls_level->add_value(new InfoValue(it->c_str()));
    }   
  }
  
  if ( !CVEs_set.empty() )
  {
    set<string>::iterator it = CVEs_set.begin();
    for ( it++; it != CVEs_set.end(); it++ )
    {
      CVEClickHandler *cve_handler = new CVEClickHandler();        
      InfoValue *val = new InfoValue(it->c_str(), Color::BLUE, Font::UNDERLINED);
      val->set_click_handler(cve_handler);
      
      cves_level->add_value(val);      
    }
  }
  
  root_level->add_sublevel(hosts_level);
  if ( firewalls_level != 0 ) 
    root_level->add_sublevel(firewalls_level);
  root_level->add_sublevel(cves_level);      
  
}

void NetworkComp::set_scenario_nodes_information( InfoStorageSubLevel *root_level, const NodeLinkList &nodes, GraphicArea *g )
{
  bool res = true;
  NetworkInfo  info;
  ostringstream   s_stream( "" );
    
  NodeLinkList::const_iterator p = nodes.begin();
  
  if ( p != nodes.end() && is_network_model_info_enabled() )
  {
    InfoStorageItemLevel *node_level;
      
    node_level = new InfoStorageItemLevel("Node ");          
    
    s_stream << "Start node";  
    node_level = new InfoStorageItemLevel(s_stream.str());  
    s_stream.str( "" );
   
    s_stream << "Number : " << *(p);
    NodeClickHandler *node_handler = new NodeClickHandler();        
    node_handler->set_node_num(*p);  
    node_handler->set_app_context(m_app_context);          
    node_handler->set_net(this);                  
    node_handler->set_graphic_area(g);            
    InfoValue *val = new InfoValue(s_stream.str(), Color::RED, Font::UNDERLINED);
    val->set_click_handler(node_handler);
    node_level->add_value(val);
    root_level->add_sublevel(node_level);
    s_stream.str( "" );
        
    for ( p++; p != nodes.end(); p++ )
    {
      res = get_network_info( *p, info );
        
      if ( res )
      {
        node_level = new InfoStorageItemLevel("Node ");          
    
        s_stream << "Node";  
        node_level = new InfoStorageItemLevel(s_stream.str());  
        s_stream.str( "" );
   
        s_stream << "Number : " << *(p);
        NodeClickHandler *node_handler = new NodeClickHandler();        
        node_handler->set_node_num(*p);  
        node_handler->set_app_context(m_app_context);          
        node_handler->set_net(this);                  
        node_handler->set_graphic_area(g);            
        InfoValue *val = new InfoValue(s_stream.str(), Color::RED, Font::UNDERLINED);
        val->set_click_handler(node_handler);
        node_level->add_value(val);
        s_stream.str( "" );
            
        s_stream << "Attack name : " << info.attack->GetName();
        node_level->add_value(new InfoValue(s_stream.str()));
        s_stream.str( "" );
     
        s_stream << "Src host : " << info.src_host->NetworkName() << "." << info.src_host->GetName();
        node_level->add_value(new InfoValue(s_stream.str()));
        s_stream.str( "" );
    
        if ( info.dest_host != 0 )
        {
          s_stream << "Tgt host : " << info.dest_host->NetworkName() << "." << info.dest_host->GetName();
          node_level->add_value(new InfoValue(s_stream.str()));
          s_stream.str( "" );
        }
        else if ( info.dest_firewall != 0 )
        {
          s_stream << "Tgt firewall : " << info.dest_firewall->GetName();
          node_level->add_value(new InfoValue(s_stream.str()));
          s_stream.str( "" );
        } 
        
        DOMStringSet* veids = info.attack->GetCveids();
        
        if (!veids->empty())     
        {
          CVEClickHandler *cve_handler = new CVEClickHandler();        
          s_stream << "CVE : " << *(veids->begin());          
          InfoValue *val = new InfoValue(s_stream.str(), Color::BLUE, Font::UNDERLINED);
          val->set_click_handler(cve_handler);
          node_level->add_value(val);      
          s_stream.str( "" );
        }
    
        root_level->add_sublevel(node_level);      
      }
    }
  }  
}

void NetworkComp::calc_node_measure_mark( Graph *graph, NodeIndex node_id, const AttackMeasuresSet &measures, GraphMeasureMarks &graph_measure_marks )
{
  Node* node = graph->get_node( node_id );
  
  NodeLinkList *node_links = node->get_links();
  AttackTriple  attack;
  Node         *link;
  AttackMeasureIDSet measure_covering;
  string        type;
  AttributeValue  *value;
  bool          successful_node;
    
  node->set_node_mark( Node::NODE_NOT_IN_PATH );
  
  for ( NodeLinkList::iterator iter = node_links->begin(); iter != node_links->end(); iter++ )
  {
    successful_node = false;
    link   = graph->get_node( *iter );
    attack.make_attack_triple( link->get_attributes() );
    measure_covering = measures.get_measures_covering( attack );
    
    if ( link->get_attributes()->query_value( "::accept", type, &value ) )
    {
      successful_node = (value->get_integer() != 0);
      delete value;
    }
    
    if ( !successful_node && link->get_attributes()->query_value( "::final", type, &value ) )
    {
      successful_node = (value->get_integer() != 0);
      delete value;
    }
    
    if ( successful_node )
      link->set_node_mark( Node::NODE_IN_PATH );
          
    if ( link->get_node_mark() == Node::NODE_UNMARKED )
      calc_node_measure_mark( graph, *iter, measures, graph_measure_marks );
      
    if ( link->get_node_mark() != Node::NODE_NOT_IN_PATH )
    {
      for ( AttackMeasuresCount::iterator p = graph_measure_marks[*iter].begin(); p != graph_measure_marks[*iter].end(); p++ )
        if ( p->first >= 0)
        {
          if ( measure_covering.find( p->first ) == measure_covering.end() ) 
            graph_measure_marks[node_id][p->first] += p->second;
        }
        else
          graph_measure_marks[node_id][-1] += ((link->get_links_count() > 0 && !successful_node)? graph_measure_marks[*iter][-1]: 1);
     }
     else
       graph_measure_marks[node_id][-1] += 1;
      
    for ( AttackMeasureIDSet::const_iterator cp = measure_covering.begin(); cp != measure_covering.end(); cp++ )
      graph_measure_marks[node_id][*cp] += ((graph_measure_marks[*iter][-1] > 0)? graph_measure_marks[*iter][-1]: 1);

  }
  
  node->set_node_mark( Node::NODE_IN_PATH );
}

AttackMeasureID NetworkComp::get_most_frequent_measure_id( Graph *graph, const AttackMeasuresSet &measures )
{
  AttackMeasureID   res = -1;
  unsigned int      res_count = 0;
  GraphMeasureMarks node_measure_marks( graph->count_nodes() );
  NodeIndex         init_node = graph->get_init_node();
  
  for ( unsigned int i = 0; i < node_measure_marks.size(); i++ )
  {
    for ( AttackMeasureID ms_id = measures.get_first_measure_id(); ms_id >= 0; ms_id = measures.get_next_measure_id( ms_id ) )
      node_measure_marks[i][ms_id] = 0;
    node_measure_marks[i][-1] = 0;
  }    
  calc_node_measure_mark( graph, init_node, measures, node_measure_marks );
  
  AttackMeasuresCount measure_count = node_measure_marks[init_node];
  
  for ( AttackMeasuresCount::const_iterator p = measure_count.begin(); p != measure_count.end(); p++ )
  {
    if ( p->first >= 0 && res_count < p->second )
    {
      res_count = p->second;
      res = p->first;
    }
  }
  
  return res;
}

bool NetworkComp::gready_measure_set( const AttackMeasuresSet &measure_set, AttackMeasuresSet &res_subset, StyleFlags edge_mark, GraphicArea  *main_g )
{
  bool res = false;
  
  res_subset.clear();
  
  if ( m_app_context == 0 || measure_set.size() == 0 )
    return res;
  
  Graph *scenario_graph = m_app_context->get_graph();
  GraphView *view = m_app_context->get_view();
  
  if ( scenario_graph == 0 || view == 0 )
    return res;
  
  AttackMeasuresSet curr_measures = measure_set;
  Graph           *curr_graph   = new Graph();
  Graph           *reduced_graph = 0;
  GraphAttributes	*attr;
  Node*            node = 0;
  Node*            new_node = 0;
  NodeLinkList	  *links;
  vector<NodeAttributeList> attr_lists;
  NodeIndex        ind, node_id;
  AttackMeasureID measure_id;
  AttackMeasure   measure;
  NodeIndexSet    nodes;
  unsigned int    i;
  bool            build_measures = (curr_measures.size() == 0);
  AttackTriple    attack;
  
  view->unselect_all_nodes( main_g );
  view->remark_edges( edge_mark, main_g, GraphViewPropertiesStorage::EDGE_UNMARKED );
  
  //copy nodes from the graph to work_graph (without copying attributes)
  for ( ind = 0; ind < scenario_graph->count_nodes(); ind++ )
  {
    node = scenario_graph->get_node( ind );
    attr = node->get_attributes();
    new_node = new Node( );
    new_node->set_attributes( attr );
    new_node->set_sub_id( node->get_sub_id() ); 
           
    links = node->get_links();  
      
    for (NodeLinkList::iterator lit = links->begin(); lit != links->end(); lit++)
    {
      node_id = *lit;
      new_node->add_link( node_id );
    }
    curr_graph->add_node( new_node );
    
    if ( build_measures && ind != scenario_graph->get_init_node() )
    {
      if ( attack.make_attack_triple( attr ) )
      {
        measure.clear();
        measure.insert( attack );
        curr_measures.add_measure( measure );
      }
    }
  }
  
  while ( curr_graph != 0 && curr_measures.size() > 0 )
  {
    //Step 1
    measure_id = get_most_frequent_measure_id( curr_graph, curr_measures );
    
    if ( measure_id < 0 )
      break;
        
    //Step 2
    measure = curr_measures.get_measure( measure_id );
    res_subset.add_measure( measure );
    
    curr_measures.get_graph_attr_lists( measure_id, attr_lists );
    
    curr_measures.remove_measure( measure_id );
    
    /// find corresponding nodes
    for ( i = 0; i < attr_lists.size(); i++ )
      for ( node_id = scenario_graph->find_first_node( attr_lists[i] ); node_id != Graph::INVALID_NODE_ID; node_id = scenario_graph->find_next_node( attr_lists[i], node_id ) )
        nodes.insert( node_id );
    
    reduced_graph = curr_graph->get_reduced_graph( attr_lists, false );
    
    for ( ind = 0; ind < curr_graph->count_nodes(); ind++ )
    {
      node = curr_graph->get_node( ind );
      node->set_attributes( 0, false );
    }
    delete curr_graph;
    
    curr_graph = reduced_graph;
    
    if ( curr_graph != 0 )
    {
      for ( ind = 0; ind < curr_graph->count_nodes(); ind++ )
      {
        new_node = curr_graph->get_node( ind );
        node = scenario_graph->get_node( new_node->get_sub_id() );
         
        attr = node->get_attributes();
        new_node->set_attributes( attr );
      }        
    }
  }
  
  if ( !(res = (curr_graph == 0)) )
  {
    for ( ind = 0; ind < curr_graph->count_nodes(); ind++ )
    {
      node = curr_graph->get_node( ind );
      node->set_attributes( 0, false );
    }
    delete curr_graph;
  }
  
  //Draw edges
  if ( edge_mark != GraphViewPropertiesStorage::EDGE_UNMARKED )
  {
    for ( NodeIndexSet::const_iterator nit = nodes.begin(); nit != nodes.end(); nit++ )
    {
      view->select_node( *nit, main_g );
      view->mark_incoming_edges( *nit, main_g, edge_mark );
    }
  }
  
  return res;
}

bool NetworkComp::gready_attack_set( const AttackTriplesSet &attacks, AttackTriplesSet &res_subset, StyleFlags edge_mark, GraphicArea  *main_g )
{
  bool              res = !attacks.empty();
  AttackMeasuresSet measure_set;
  AttackMeasuresSet res_measure_set;
  
  res_subset.clear();
  
  if ( res )
  {
    for ( AttackTriplesSet::const_iterator p = attacks.begin(); p != attacks.end(); p++ )
    {
      AttackMeasure measure;
      measure.insert( *p );
      measure_set.add_measure( measure );
    }
    
    res = gready_measure_set( measure_set, res_measure_set, edge_mark, main_g );
    
    if ( res_measure_set.size() > 0 )
    {
      AttackMeasure measure;
      for ( AttackMeasureID id = res_measure_set.get_first_measure_id(); id >= 0; id = res_measure_set.get_next_measure_id( id ) )
      {
        measure = res_measure_set.get_measure( id );
        
        res_subset.insert( measure.begin(), measure.end() );
      }
    }
  }
  
  return res;
}
