#include <ext/hash_set>
#include "graph.hpp"

ofstream graph_file;
ofstream  dfs_graph_file;
bool     ascii_file = false;
bool     bin32_file = false;
bool     do_lefty_output = false;
bool     edge_capping = false;
bool     f_rs_mode = false;
bool     i_rs_mode = false;
int      edge_cap = 0;
bool     supertrace = false;
bool     detect_collisions = false;
long     dfs_edges_count = 0;
long     hash_comparison = 0;
long     state_restoring = 0;
long     state_comparison = 0;
long     collisions_count = 0;
unsigned long     reconstructing_stack_summ = 0;
long     max_depth = 0;
bool     make_dfs_graph = false;
GraphMetrics graph_metrix;
bool     output_graph_metrics = false;

long     scenario_mem_used = 0;

void start_timer(struct timeval *timer)
{
  struct timezone tz;
  gettimeofday(timer, &tz);
}

void report_timer(ostream& out, struct timeval start)
{
  struct timeval stop;
  struct timezone tz;
  double seconds;
  
  gettimeofday(&stop, &tz);
  seconds = (stop.tv_sec-start.tv_sec) + ((double)stop.tv_usec-(double)start.tv_usec)/1000000;
  char snum[30];
  sprintf(snum, "%.3f", seconds);

  out << snum;
}

NodeHashSet Nodes(1048576);
NodePointerList NodeList;

/////////////////////
// A stack of nodes

inline NodeStack::NodeStack() {
  top = NULL;
  length = 0;
  max_length = 0;
}

inline NodeStack::~NodeStack() {
  while (!Empty()) Pop();
}

inline bool NodeStack::Empty() { return (top == NULL); }
inline Node *NodeStack::Top() { if (Empty()) return NULL; else return top->node; }

inline void NodeStack::Push(Node *node) {
  NodeElement *element = new NodeElement;
  element->node = node;
  element->next = top;
  top = element;
  length++;
    
  if ( length > max_length )
    max_length = length;  
}

inline Node *NodeStack::Pop() {
  if (Empty()) return NULL;
  NodeElement *element = top;
  top = top->next;
  Node *node = element->node;
  delete element;
  length--;
  return node;
}

inline Node *NodeStack::OnStack(Node *node) {
  NodeElement *element = top;
  while (element != NULL) {
    if (*(element->node) == *node)
      return element->node;
    element = element->next;
  }
  return NULL;
}

void NodeStack::Print() {
  NodeElement *element = top;
  while (element != NULL) {
    element->node->Print();
    element = element->next;
    if (element != NULL) printf(" ");
  }
}

//////////////////////////
// Main search procedure

Graph *BuildScenarioGraph(Node *init)
{
  Graph *Scenario = new Graph();
  NodeStack ExploredNodes;          // The main DFS stack
  // Hash table for checking quickly if a node is in
  // current play.  Invariant: CurrentDFSStack = ExploredNodes union FinishedNodes
  //NodePointerHashSet CurrentDFSStack(1048576);
  NodeStack FinishedNodes;          // Nodes that have been fully explored and
                                    // await placement in a component
  int number = 1;                   // Node number
  int ComponentNumber = 0;          // Uniq component ID that is set after 
                                    // the component is constructed to mark the 
                                    // component's nodes as already analysed 

  Nodes.insert(init);
  init->SetLowLink(number);
  init->SetNumber(number);

  if ( do_lefty_output ) {
    init->LeftyInsertNode(stdout, "fsm");
    fprintf(stdout, "highlight#");
    init->LeftyHighlight(stdout, "fsm");
    fprintf(stdout, "\ncommand done\n");
  }

  init->GenerateChildren();
  init->ResetLastChild();
  ExploredNodes.Push(init);
  //CurrentDFSStack.insert(init);
  init->SetFixed( true );

  // Main loop; works with the node on top of the stack.
  // An iterative implementation of Tarjan's strongly connected
  // components algorithm.
  
  while (!ExploredNodes.Empty()) {
    
    Node *CurrentNode = ExploredNodes.Top();

    if (edge_capping) {
      if (dfs_edges_count >= edge_cap)
        return Scenario;
    }
    
    // First, check if we have just finished exploring a child of the current
    // node.  If so, check if the child's lowlink is smaller than ours
    Node *LastChild = CurrentNode->GetLastChild();
    
    if (LastChild != NULL) {
      
      if (LastChild->GetComponentNumber() == 0) {
        // If last child is in already constructed and analysed component
        // the low link of the parent node should not be changed      
        int ChildLowLink = LastChild->GetLowLink();
        if ( ChildLowLink < CurrentNode->GetLowLink() )
          CurrentNode->SetLowLink(ChildLowLink);
        
        if ( do_lefty_output ) {
          CurrentNode->LeftyUpdateNode(stdout, "fsm");
          fprintf(stdout, "highlight#");
          CurrentNode->LeftyHighlightLabel(stdout, "fsm");
          fprintf(stdout, "\ncommand done\n");
        }
      }
      else if ( LastChild->InScenario() ) {
        // The (CurrentNode, LastChild) is transition between currently constructing
        // component and an analysed component that is in scenario -
        // mark the node to add whole component to scenario after construction
        Scenario->Add(CurrentNode);
      }
    }

    // Get the next child
    Node *Child = CurrentNode->GetNextChild();

    if (Child != NULL) {
      if (Child->Unexplored()) {
        // There is an unexplored child of the current node. Put the child on
        // the stack and repeat with the child.  This edge will be a tree edge.
        number++;
        Child->SetLowLink(number);
        Child->SetNumber(number);

        if (do_lefty_output) {
          Child->LeftyInsertNode(stdout, "fsm");
          Child->LeftyInsertEdge(stdout, CurrentNode, Child, "fsm");
          fprintf(stdout, "highlight#");
          Child->LeftyHighlight(stdout, "fsm");
          fprintf(stdout, "$");
          CurrentNode->LeftyHighlightEdge(stdout, CurrentNode, Child, "fsm");
          fprintf(stdout, "\ncommand done\n");
        }
  

        Child->GenerateChildren();
        Child->ResetLastChild();
        ExploredNodes.Push(Child);
        //CurrentDFSStack.insert(Child);
        Child->SetFixed( true );
        continue;
      }
      else {
        // This child had been explored previously; check the
        // lowlink against its number.
        if ( do_lefty_output ) {
          Child->LeftyInsertEdge(stdout, CurrentNode, Child, "fsm");
          fprintf(stdout, "highlight#");
          CurrentNode->LeftyHighlightEdge(stdout, CurrentNode, Child, "fsm");
          fprintf(stdout, "\ncommand done\n");
        }

        //if (CurrentDFSStack.find(Child) != CurrentDFSStack.end()) {
        if ( Child->IsFixed() ) {
          int ChildNumber = Child->GetNumber();
          if (ChildNumber < CurrentNode->GetLowLink()) {
            CurrentNode->SetLowLink(ChildNumber);
            CurrentNode->LeftyUpdateNode(stdout, "fsm");
      
            if ( do_lefty_output ) {
              fprintf(stdout, "highlight#");
              CurrentNode->LeftyHighlightLabel(stdout, "fsm");
              fprintf(stdout, "\ncommand done\n");
            }
          }
        }
        //if ( Child->InScenario() )
        //  Scenario->Add(CurrentNode);
      }
    }
    else {
      // All children of the current node have been explored.
      if (CurrentNode->GetLowLink() == CurrentNode->GetNumber()) {
        // Current node is the root of a component.  Look through
        // the nodes comprising the component.  If the component is
        // nontrivial and contains an acceptance state, add the
        // component and the path to it from the original state
        // to the scenario graph.  I.e., add the entire stack

        NodeStack Component;
        Node *node = ExploredNodes.Pop();
        //CurrentDFSStack.erase(node);
        node->SetFixed( false );
        
        bool accepting = node->IsAccepting();
        bool NonTrivial = node->HasSelfLoop();
        // If the component has nodes from scenario then there is transition 
        // between the componet and previously analysed one that is in scenario -
        // add the component to scenario (RSET-EXISTENTIAL)
        bool HasInScenario = node->InScenario(); 
        
        ComponentNumber++;
        
        Component.Push(node);
        node->SetComponentNumber( ComponentNumber );
        
        // Add nodes to component
        int CurrentNumber = CurrentNode->GetNumber();
        if (!FinishedNodes.Empty()) {
          while (FinishedNodes.Top()->GetNumber() >= CurrentNumber) {
            node = FinishedNodes.Pop();
            //CurrentDFSStack.erase(node);
            node->SetFixed( false );
            
            Component.Push(node);
            node->SetComponentNumber( ComponentNumber );
            accepting = accepting || node->IsAccepting();
            HasInScenario = HasInScenario || node->InScenario();
                        
            if (FinishedNodes.Empty()) 
              break;
          }
        }
        
        if ( Component.GetLength() > 1 )
          NonTrivial = true;

        if ( accepting && NonTrivial ||
             HasInScenario ) {
          Scenario->Add(Component);
          //Scenario->Add(ExploredNodes);
        }
        else if (supertrace) {
          while ( !Component.Empty() ) {
            node = Component.Pop();
            node->PackNode();
          }
        }
      }
      else {
        // Current node has been explored, but is not the root of a component.
        // It gets put on the stack of nodes waiting to be placed in a component.
        FinishedNodes.Push(ExploredNodes.Pop());
      }
    }
  }
  
  scenario_mem_used += ExploredNodes.GetUsedMem();
  scenario_mem_used += FinishedNodes.GetUsedMem();

  return Scenario;
}


Graph *BuildRepresentativeScenarioGraph(Node *init)
{
  Graph *Scenario = new Graph();
  NodeStack ExploredNodes;          // The main DFS stack
  // Hash table for checking quickly if a node is in
  // current play.  Invariant: CurrentDFSStack = ExploredNodes union FinishedNodes
  //NodePointerHashSet CurrentDFSStack(1048576);
  NodeStack FinishedNodes;          // Nodes that have been fully explored and
                                    // await placement in a component
  
  int number = 1;                   // Node number
  int ComponentNumber = 0;          // Component ID

  Nodes.insert(init);
  init->SetLowLink(number);
  init->SetNumber(number);

 // init->LeftyInsertNode(stdout, "fsm");

  init->GenerateChildren();
  init->ResetLastChild();
  ExploredNodes.Push(init);
  //CurrentDFSStack.insert(init);
  init->SetFixed( true );

  // Main loop; works with the node on top of the stack.
  // An iterative implementation of Tarjan's strongly connected
  // components algorithm.
  
  while (!ExploredNodes.Empty()) {
    
    Node *CurrentNode = ExploredNodes.Top();

    if (edge_capping) {
      if (dfs_edges_count >= edge_cap)
        return Scenario;
    }
    
    // First, check if we have just finished exploring a child of the current
    // node.  If so, check if the child's lowlink is smaller than ours
    Node *LastChild = CurrentNode->GetLastChild();
    
    if (LastChild != NULL) {
      
      if (LastChild->GetComponentNumber() == 0) {
        
        // If last child is in already constructed and analysed component
        // the low link of the parent node should not be changed      
        int ChildLowLink = LastChild->GetLowLink();
        if (ChildLowLink < CurrentNode->GetLowLink())
          CurrentNode->SetLowLink(ChildLowLink);
      }
      else {
        // Transition to alredy analysed component
        if ( LastChild->InScenarioNotInRS() )
          // LastChild is possible border node
          CurrentNode->SetBorderTransStart( true );
        else if ( LastChild->InScenario() )
          Scenario->Add(CurrentNode);
      }
    }

    // Get the next child
    Node *Child = CurrentNode->GetNextChild();

    if (Child != NULL) {
      if (Child->Unexplored()) {
        // There is an unexplored child of the current node. Put the child on
        // the stack and repeat with the child.  This edge will be a tree edge.
        number++;
        Child->SetLowLink(number);
        Child->SetNumber(number);

        Child->GenerateChildren();
        Child->ResetLastChild();
        ExploredNodes.Push(Child);
        //CurrentDFSStack.insert(Child);
        Child->SetFixed( true );
        continue;
      }
      else {
        // This child had been explored previously; check the
        // lowlink against its number.
        
        //if (CurrentDFSStack.find(Child) != CurrentDFSStack.end()) {
        if ( Child->IsFixed() ) {
          
          int ChildNumber = Child->GetNumber();
          if (ChildNumber < CurrentNode->GetLowLink())
            CurrentNode->SetLowLink(ChildNumber);
        }
        
        //if ( Child->InScenarioNotInRS() )
        // CurrentNode->SetBorderTransStart( true );
        //else if ( Child->InScenario() )
        //  Scenario->Add(CurrentNode);
      }
    }
    else {
      // All children of the current node have been explored.
      if (CurrentNode->GetLowLink() != CurrentNode->GetNumber()) {
        // Current node has been explored, but is not the root of a component.
        // It gets put on the stack of nodes waiting to be placed in a component.
        FinishedNodes.Push(ExploredNodes.Pop());
      }
      else {
        // Current node is the root of a component.  Look through
        // the nodes comprising the component.  If the component is
        // nontrivial and contains an acceptance state, add the
        // component and the path to it from the original state
        // to the scenario graph.  I.e., add the entire stack

        // Add the root of the component
        NodeStack Component;
        NodeStack Border;
        Node *node = ExploredNodes.Pop();
        Node *child = 0;
        //CurrentDFSStack.erase(node);
        node->SetFixed( false );
        
        bool accepting = node->IsAccepting();
        bool NonTrivial = node->HasSelfLoop();
        bool HasBorderTrans = node->HasBorderChild();
        bool HasScenarioTrans = (node->InScenario() && !HasBorderTrans);
        
        
        ComponentNumber++;
        
        Component.Push(node);
        node->SetComponentNumber( ComponentNumber );
        
        if ( HasBorderTrans )
          for ( child = node->GetNextChild(); child != 0; child = node->GetNextChild() )
            if ( child->InScenarioNotInRS() )
              Border.Push( child );
        
        // Add nodes to component and verify Q_s
        int CurrentNumber = CurrentNode->GetNumber();
        if ( !FinishedNodes.Empty() ) {
          while (FinishedNodes.Top()->GetNumber() >= CurrentNumber) {
            node = FinishedNodes.Pop();
            //CurrentDFSStack.erase(node);
            node->SetFixed( false );
            
            Component.Push(node);
            node->SetComponentNumber( ComponentNumber );
             
            if ( node->HasBorderChild() ) {
              // Find border
              for ( child = node->GetNextChild(); child != 0; child = node->GetNextChild() )
                if ( child->InScenarioNotInRS() )
                  Border.Push( child );
              HasBorderTrans = TRUE;
            }
            else
              HasScenarioTrans = HasScenarioTrans || node->InScenario();
                
            accepting = accepting || node->IsAccepting();
                        
            if (FinishedNodes.Empty()) 
              break;
          }
        }
        
        if ( Component.GetLength() > 1 )
          NonTrivial = true;
        

        if (!accepting || !NonTrivial) {
          if ( HasScenarioTrans || HasBorderTrans ) {
            Scenario->Add( Component );
            Scenario->Add( Border);
          }
          else if (supertrace) {
            while ( !Component.Empty() ) {
              node = Component.Pop();
              node->PackNode();
            }
          }
        }
        else {
          // Q_s is true, the component is in scenario
          
          // RSET-UNIVERSAL condition Q_i = (each cycle has an accepting state)
          // and all components that the current one has transition to must also satisfy Q_i
          bool rset_universal_condition;          
          
          if ( HasScenarioTrans )
            rset_universal_condition = false;          
          else {
            //RSET-UNIVERSAL algorithm for i-representative scenario
            NodeStack PathNodesStack;
            Node     *DFS_node;
            
            rset_universal_condition = true;
          
            Component.ResetIterator();
            for (node = Component.Next(); node != NULL && rset_universal_condition; node = Component.Next()) {
              // If the node is accepted it is removed - verify connection to another component and skip
              // If the node is already passed all cycles that can be passed from the node 
              // already verified - skip           
              if ( node->IsPassed() )
                continue;
              
              if ( node->IsAccepting() ) {
                for ( child = node->GetNextChild(); child != NULL; child = node->GetNextChild() )
                  if ( child->GetComponentNumber() != ComponentNumber )
                    if ( !child->InScenarioNotInRS() ) {
                      // Stop search with fail
                      rset_universal_condition = false;
                      break;
                    }
                 node->SetPassed( true );
              }
              else {
                // Begin deep-first-search
                PathNodesStack.Push( node );
                node->SetPassed( true );
                node->SetFixed( true );
                
                while ( !PathNodesStack.Empty() && rset_universal_condition ) {
                  DFS_node = PathNodesStack.Top();
                  
                  child = DFS_node->GetNextChild();
              
                  if ( child == NULL ) {
                    // All children are passed
                    PathNodesStack.Pop();
                    DFS_node->SetFixed( false );
                  }
                  else if ( child->GetComponentNumber() != ComponentNumber ) {
                
                    if ( !child->InScenarioNotInRS() ) {
                      // Stop search with fail
                      rset_universal_condition = false;
                    }         
                  }
                  else if ( child->IsPassed() ) {
                    // All passed nodes are verified
                    // The exeption is
                    if ( child->IsFixed() ) {
                      // An unaccepted cycle is detected
                      rset_universal_condition = false;
                    }
                  }
                  else if ( child->IsAccepting() ) {
                    // Verify connection to another component and skip
                    DFS_node = child;
                    for ( child = node->GetNextChild(); child != NULL; child = node->GetNextChild() )
                      if ( child->GetComponentNumber() != ComponentNumber )
                        if ( !child->InScenarioNotInRS() ) {
                          //Stop search with fail
                          rset_universal_condition = false;
                          break;
                        }
                    DFS_node->SetPassed( true );
                  }
                  else {
                    //Add it to stack
                    PathNodesStack.Push( child );
                    child->SetPassed( true );
                    child->SetFixed( true );
                  }
                }
                
                while ( !PathNodesStack.Empty() ) {
                  DFS_node = PathNodesStack.Pop();
                  DFS_node->SetFixed( false );
                }
              }
            }
            
            Component.ResetIterator();
            for (node = Component.Next(); node != NULL; node = Component.Next())
              node->SetPassed( false );
          }
          
          if ( !rset_universal_condition ) {
            // Add to scenario graph
            Scenario->Add(Component);
            Scenario->Add( Border );
          }
          else {
            
            while ( !Component.Empty() ) {
              node = Component.Pop();
              node->SetInScenarioNotInRS(true);
            }
          }
        }
      }
    }
  }
  
  if ( !init->InScenario() && init->InScenarioNotInRS() )
    Scenario->Add( init );
  
  scenario_mem_used += ExploredNodes.GetUsedMem();
  scenario_mem_used += FinishedNodes.GetUsedMem();

  return Scenario;
}

void construct_scenario_graph(bool output_to_file)
{
  State *InitState = new State();
  Graph *gr = 0;

  Node *Init = new Node(InitState, -1, 0, 0);
  unsigned int memory_size = 0;

  if (do_lefty_output) {
    fprintf(stdout, "fa.fsm = dotty.graphs[1];\n");
    fprintf(stdout, "ready\n");
  }

  // Timing
  struct timeval timer;

  
  start_timer(&timer);
  
  if ( i_rs_mode )
    gr = BuildRepresentativeScenarioGraph(Init);
  else
    gr = BuildScenarioGraph(Init);
  
   
  if (output_to_file) {
    unsigned long format_flags = Tracer::GRAPH_ATTR_FORMAT;
    if ( bin32_file )
      format_flags |= Tracer::BIN32; 
    Tracer::init_tracing( &graph_file, format_flags );
    
    gr->PrintToFile();
  }

  cout << "DFS nodes: " << Nodes.size() << endl;
  cout << "DFS edges: " << dfs_edges_count << endl << endl;
  
  cout << "Scenario nodes: " << gr->GetNodeCount() << endl;
  cout << "Scenario edges: " << gr->edgecount << endl << endl;
  
  if ( detect_collisions ) {
    cout << "Node comparison:  " << hash_comparison << endl;
    cout << "State comparison: " << state_comparison << endl;
    cout << "State recreating: " << state_restoring << endl;
    cout << "Collisions:       " << collisions_count << endl << endl;
    
    cout << "Max. state byte array size: " << StateHash::max_st_arr_size << endl;
    cout << "Max. states in memory:      " << StateBase::max_state_in_memory  << endl;
    cout << "States in memory (leak):    " << StateBase::state_in_memory - 1 << endl << endl;
  }
  
  for ( NodePointerList::iterator p = NodeList.begin(); p != NodeList.end(); p++ ) {
    Node *node = *p;
    memory_size += node->CalcSize() + 2 * sizeof(Node*);
  }

  memory_size += StateBase::max_state_size * StateBase::max_state_in_memory;
  memory_size += gr->CalcSize();
  memory_size += Tracer::calc_names_storage_size();
  
  scenario_mem_used += memory_size;
  
  if ( output_graph_metrics ) {
    graph_metrix = gr->CountGraphMetrics( Init );
    
    cout << "Number of accepting/final states:       " << graph_metrix.acceptance_number << endl;
    cout << "Max. out-degree of any node:            " << graph_metrix.max_node_out_degree << endl;
    cout << "Max. out-degree of any non-final node: " << graph_metrix.max_not_final_out_degree << endl;
    cout << "Depth of graph over all nodes:          " << graph_metrix.graph_depth << endl;
    cout << "Depth of graph over non-final nodes:  " << graph_metrix.longest_scenario << endl;
    cout << "Length of shortest path to final node: " << graph_metrix.shortest_scenario << endl << endl;
  }
  
  if ( do_lefty_output )
    fprintf(stdout, "all done\n");
  
  delete gr;
}

unsigned long make_graph_file_header( bool ascii_file, const string &model_file_name, bool DFS_graph = false )
{
  unsigned long offset = 0;
  string        comment = "The file is generated by scenario graph generator";
  ofstream &out_graph = ((DFS_graph)? dfs_graph_file: graph_file); 
  
  out_graph << "graph";
  out_graph << "001.000\n";
  out_graph << "model\n";
  
  if ( ascii_file ) {    
    out_graph << "ASC\n";
    out_graph << "Comment: " << comment << "\n";
    if ( !model_file_name.empty() )
      out_graph << "Model_file: " << model_file_name << "\n";
    out_graph << "\n";
    out_graph << "Node{ bool: accept=0; bool: final=0; ";
    
    if ( DFS_graph )
      out_graph << "bool: in_scenario=0; ";
    
    make_state_attr_format( Tracer::ASCII_OUTPUT, &out_graph );
    out_graph << "}\n";
    out_graph << ":\n";  
  }
  else
  {
    unsigned long flags = Tracer::get_flags();
    ostream  *dbg_stream = Tracer::get_curr_stream();
    
    unsigned long format_flags = Tracer::GRAPH_ATTR_FORMAT;
    if ( bin32_file )
      format_flags |= Tracer::BIN32; 
    Tracer::init_tracing( &out_graph, format_flags );
    
    if ( bin32_file )
      out_graph << "B32\n";
    else
      out_graph << "BIN\n";
    
    Tracer::write_bin_identifier( 0x0D01, &out_graph );//comment
    //Tracer::write_2bytes( (short)0x0D01, &out_graph ); //comment
    Tracer::write_bin_identifier( comment.length(), &out_graph );
    //Tracer::write_2bytes( (short)(comment.length()), &out_graph ); 
    out_graph << comment;
    if ( !model_file_name.empty() )
    {
      Tracer::write_bin_identifier( 0x0D02, &out_graph ); //model file name
      //Tracer::write_2bytes( (short)0x0D02, &out_graph ); //model file name
      Tracer::write_bin_identifier( model_file_name.length(), &out_graph );
      //Tracer::write_2bytes( (short)(model_file_name.length()), &out_graph ); 
      out_graph << model_file_name;
    }
    Tracer::write_bin_identifier( -1, &out_graph );//separator
    //Tracer::write_2bytes( (short)-1, &out_graph ); //separator
    
    Tracer::write_bin_identifier( -256, &out_graph ); //identifier of offset to names set
    //Tracer::write_2bytes( (short)-256, &out_graph ); //identifier of offset to names set
    offset = out_graph.tellp();
    Tracer::write_4bytes( 0, &out_graph ); //offset to names set
        
    //attr. templates
    Tracer::write_bin_identifier( Tracer::get_name_id( "Node" ), &out_graph );//name - "Node"
    //Tracer::write_2bytes( Tracer::get_name_id( "Node" ), &out_graph ); //name - "Node"
    Tracer::write_1bytes( (char)-3, &out_graph ); //type "NODE_START"
    Tracer::write_4bytes( (long)-1, &out_graph ); //value (none)

    Tracer::write_bin_identifier( Tracer::get_name_id( "accept" ), &out_graph );//name - "accept"
    //Tracer::write_2bytes( Tracer::get_name_id( "accept" ), &out_graph ); //name - "accept"
    Tracer::write_1bytes( Tracer::get_type_id( "bool" ), &out_graph ); //type - "bool"
    Tracer::write_1bytes( (unsigned char)0, &out_graph ); //value - 0
    
    Tracer::write_bin_identifier( Tracer::get_name_id( "final" ), &out_graph );//name - "final"
    //Tracer::write_2bytes( Tracer::get_name_id( "final" ), &out_graph ); //name - "final"
    Tracer::write_1bytes( Tracer::get_type_id( "bool" ), &out_graph ); //type - "bool"
    Tracer::write_1bytes( (unsigned char)0, &out_graph ); //value - 0
    
    if ( DFS_graph ) {
      Tracer::write_bin_identifier( Tracer::get_name_id( "in_scenario" ), &out_graph );//name - "in_scenario"
      //Tracer::write_2bytes( Tracer::get_name_id( "in_scenario" ), &out_graph ); //name - "in_scenario"
      Tracer::write_1bytes( Tracer::get_type_id( "bool" ), &out_graph ); //type - "bool"
      Tracer::write_1bytes( (unsigned char)0, &out_graph ); //value - 0
    }
    
    make_state_attr_format( format_flags, &out_graph );
    
    Tracer::write_bin_identifier( Tracer::get_name_id( "Node" ), &out_graph ); //name - "Node"
    //Tracer::write_2bytes( Tracer::get_name_id( "Node" ), &out_graph ); //name - "Node"
    Tracer::write_1bytes( (char)-2, &out_graph ); //type "ATTR_END"
       
    Tracer::write_bin_identifier( -3, &out_graph );//attr separator
    //Tracer::write_2bytes( (short)-3, &out_graph ); //attr separator
    Tracer::write_bin_identifier( -2, &out_graph ); //separator
    //Tracer::write_2bytes( (short)-2, &out_graph ); //separator
    
    /*Restore tracing flags*/
    Tracer::init_tracing( dbg_stream, flags );
  }
  
  return offset;
}

void finish_file_making( unsigned long nameset_offset_field, bool ascii, bool DFS_graph = false )
{
  ofstream &out_graph = ((DFS_graph)? dfs_graph_file: graph_file);
  
  if ( ascii ) {
    out_graph << "\n"; // end of graph info block 
    
    //filters
    out_graph << ":\n";
    //rendering options
/*    out_graph << "Range: 20\n";
    out_graph << "V_distance: 60\n";
    out_graph << "H_distance: 40\n";
    out_graph << "Spline: true\n";
*/    out_graph << ":\n";
    
    //General drawing options
/*    out_graph << "BG_color: 255 255 255 0\n";
    out_graph << "Sel_color: 255 0 0 0\n";
    out_graph << "Edge_color: 60 0 60 0\n";
    out_graph << "Edge_width: 1\n";
*/    out_graph << ":\n";
    
    //groups
    out_graph << "{:\n";
    out_graph << "Name: default\n";
    out_graph << "Shape: circle\n";
    out_graph << "Filled: false\n";
    out_graph << "Shape_xy_ratio: 1.0 1.0\n";
    out_graph << "FG_color: 0 0 0 0\n";
    out_graph << "Label_font: 255 0 0 0 12 NORMAL \n";
    out_graph << "Drawing_pen: SOLID 1\n";
    out_graph << "}:\n";
    
    out_graph << "{:\n";
    out_graph << "Name: Accepted\n";
    out_graph << "Shape: double circle\n";
    out_graph << "Assigning: { accept=1; }\n";
    out_graph << "}:\n";
    
    out_graph << "{:\n";
    out_graph << "Name: Final\n";
    out_graph << "Filled: true\n";
    out_graph << "FG_color: 0 150 60 0\n";
    out_graph << "Assigning: { final=1; }\n";
    out_graph << "}:\n";
    
    if ( DFS_graph ) {
      out_graph << "{:\n";
      out_graph << "Name: Scenario\n";
      out_graph << "FG_color: 100 100 255 0\n";
      out_graph << "Assigning: { in_scenario=1; }\n";
      out_graph << "}:\n";
    }
    
    out_graph << "\n"; // end of graph vis. info block 
  }
  else {
    unsigned long flags = Tracer::get_flags();
    ostream  *dbg_stream = Tracer::get_curr_stream();
    
    unsigned long format_flags = Tracer::GRAPH_ATTR_FORMAT;
    if ( bin32_file )
      format_flags |= Tracer::BIN32; 
    Tracer::init_tracing( &out_graph, format_flags );
    
    if ( nameset_offset_field != 0 ) {
      //put names set
      long curr_offset = 0;
      Tracer::write_bin_identifier( -2, &out_graph ); //separator
      //Tracer::write_2bytes( (short)-2, &out_graph ); //separator
      curr_offset = (long)out_graph.tellp();
      out_graph.seekp( nameset_offset_field, ios::beg );
      Tracer::write_4bytes( curr_offset, &out_graph );
      out_graph.seekp( curr_offset, ios::beg );
      Tracer::write_names_storage( &out_graph );
    }
    Tracer::write_bin_identifier( -1, &out_graph ); //end of graph info block
    //Tracer::write_2bytes( (short)-1, &out_graph ); //end of graph info block
    
    //filters
    Tracer::write_bin_identifier( -2, &out_graph );
    //Tracer::write_2bytes( (short)-2, &out_graph );
    
    //rendering options
/*  Tracer::write_bin_identifier( 0x0A01, &out_graph );  //node range
    //Tracer::write_2bytes( (short)0x0A01, &out_graph ); //node range
    Tracer::write_2bytes( 4, &out_graph );
    Tracer::write_4bytes( 20, &out_graph );
    
    Tracer::write_bin_identifier( 0x0A02, &out_graph ); // v. distance
    //Tracer::write_2bytes( (short)0x0A02, &out_graph ); // v. distance
    Tracer::write_2bytes( 4, &out_graph );
    Tracer::write_4bytes( 60, &out_graph );
    
    Tracer::write_bin_identifier( 0x0A03, &out_graph ); //h. distance
    //Tracer::write_2bytes( (short)0x0A03, &out_graph ); //h. distance
    Tracer::write_2bytes( 4, &out_graph );
    Tracer::write_4bytes( 40, &out_graph );
    
    Tracer::write_bin_identifier( 0x0A04, &out_graph ); //spline
    //Tracer::write_2bytes( (short)0x0A04, &out_graph ); //spline
    Tracer::write_2bytes( 1, &out_graph );
    Tracer::write_1bytes( 1, &out_graph );
*/    
    Tracer::write_bin_identifier( -2, &out_graph );
    //Tracer::write_2bytes( (short)-2, &out_graph );
    
    //general drawing options
/*  Tracer::write_bin_identifier( 0x0B01, &out_graph );  //background color
    //Tracer::write_2bytes( (short)0x0B01, &out_graph ); //background color
    Tracer::write_2bytes( 4, &out_graph );
    Tracer::write_1bytes( (char)255, &out_graph );
    Tracer::write_1bytes( (char)255, &out_graph );
    Tracer::write_1bytes( (char)255, &out_graph );
    Tracer::write_1bytes( 0, &out_graph );
    
    Tracer::write_bin_identifier( 0x0B02, &out_graph );//selected color
    //Tracer::write_2bytes( (short)0x0B02, &out_graph ); //selected color
    Tracer::write_2bytes( 4, &out_graph );
    Tracer::write_1bytes( (char)255, &out_graph );
    Tracer::write_1bytes( 0, &out_graph );
    Tracer::write_1bytes( 0, &out_graph );
    Tracer::write_1bytes( 0, &out_graph );
    
    Tracer::write_bin_identifier( 0x0B03, &out_graph );//edge color
    //Tracer::write_2bytes( (short)0x0B03, &out_graph ); //edge color
    Tracer::write_2bytes( 4, &out_graph );
    Tracer::write_1bytes( 60, &out_graph );
    Tracer::write_1bytes( 0, &out_graph );
    Tracer::write_1bytes( 60, &out_graph );
    Tracer::write_1bytes( 0, &out_graph );
    
    Tracer::write_bin_identifier( 0x0B04, &out_graph ); //edge width
    //Tracer::write_2bytes( (short)0x0B04, &out_graph ); //edge width
    Tracer::write_2bytes( 1, &out_graph );
    Tracer::write_1bytes( 1, &out_graph );
*/    
    Tracer::write_bin_identifier( -2, &out_graph );
    //Tracer::write_2bytes( (short)-2, &out_graph );
    
    //groups
    Tracer::write_bin_identifier( 0x0C00, &out_graph ); //group id
    //Tracer::write_2bytes( (short)0x0C00, &out_graph ); //group id
    
    Tracer::write_bin_identifier( 0x0C01, &out_graph );//name
    //Tracer::write_2bytes( (short)0x0C01, &out_graph ); //name
    Tracer::write_bin_identifier( 7, &out_graph );
    //Tracer::write_2bytes( 7, &out_graph );
    out_graph << "default";
    
    Tracer::write_bin_identifier( 0x0C02, &out_graph ); //shape
    //Tracer::write_2bytes( (short)0x0C02, &out_graph ); //shape
    Tracer::write_bin_identifier( 2, &out_graph );
    //Tracer::write_2bytes( 2, &out_graph );
    Tracer::write_2bytes( 0, &out_graph );
    
    Tracer::write_bin_identifier( 0x0C03, &out_graph ); //filled
    //Tracer::write_2bytes( (short)0x0C03, &out_graph ); //filled
    Tracer::write_bin_identifier( 1, &out_graph );
    //Tracer::write_2bytes( 1, &out_graph );
    Tracer::write_1bytes( 0, &out_graph );
    
    float rv = 1.0;
    Tracer::write_bin_identifier( 0x0C04, &out_graph ); // x/y ratio
    //Tracer::write_2bytes( (short)0x0C04, &out_graph ); // x/y ratio
    Tracer::write_bin_identifier( 8, &out_graph );
    //Tracer::write_2bytes( 8, &out_graph );
    Tracer::write_4bytes( *((long*)(&rv)), &out_graph );
    Tracer::write_4bytes( *((long*)(&rv)), &out_graph );
    
    Tracer::write_bin_identifier( 0x0C05, &out_graph ); //fg color
    //Tracer::write_2bytes( (short)0x0C05, &out_graph ); //fg color
    Tracer::write_bin_identifier( 4, &out_graph );
    //Tracer::write_2bytes( 4, &out_graph );
    Tracer::write_4bytes( 0, &out_graph );
    
    Tracer::write_bin_identifier( 0x0C06, &out_graph ); //font
    //Tracer::write_2bytes( (short)0x0C06, &out_graph ); //font
    Tracer::write_bin_identifier( 8, &out_graph );
    //Tracer::write_2bytes( 8, &out_graph );
    Tracer::write_1bytes( (char)255, &out_graph );
    Tracer::write_1bytes( 0, &out_graph );
    Tracer::write_1bytes( 0, &out_graph );
    Tracer::write_1bytes( 0, &out_graph );
    Tracer::write_1bytes( 0, &out_graph );
    Tracer::write_1bytes( 12, &out_graph );
    Tracer::write_bin_identifier( 0, &out_graph );
    //Tracer::write_2bytes( 0, &out_graph );
    
    Tracer::write_bin_identifier( 0x0C07, &out_graph ); //pen
    //Tracer::write_2bytes( (short)0x0C07, &out_graph ); //pen
    Tracer::write_bin_identifier( 2, &out_graph );
    //Tracer::write_2bytes( 2, &out_graph );
    Tracer::write_1bytes( 0, &out_graph );
    Tracer::write_1bytes( 1, &out_graph );
    
    Tracer::write_bin_identifier( -3, &out_graph ); //end group id
    //Tracer::write_2bytes( (short)-3, &out_graph ); //end group id
    
    //Accepted group
    Tracer::write_bin_identifier( 0x0C00, &out_graph );
    //Tracer::write_2bytes( (short)0x0C00, &out_graph ); //group id
    
    Tracer::write_bin_identifier( 0x0C01, &out_graph );
    //Tracer::write_2bytes( (short)0x0C01, &out_graph ); //name
    Tracer::write_bin_identifier( 8, &out_graph );
    //Tracer::write_2bytes( 8, &out_graph );
    out_graph << "Accepted";
    
    Tracer::write_bin_identifier( 0x0C02, &out_graph );
    //Tracer::write_2bytes( (short)0x0C02, &out_graph ); //shape
    Tracer::write_bin_identifier( 2, &out_graph );
    //Tracer::write_2bytes( 2, &out_graph );
    Tracer::write_2bytes( 1, &out_graph );
    
    Tracer::write_bin_identifier( 0x0C08, &out_graph );
    //Tracer::write_2bytes( (short)0x0C08, &out_graph ); //Assigning
    Tracer::write_bin_identifier( 12, &out_graph );
    //Tracer::write_2bytes( 12, &out_graph );
    out_graph << "{ accept=1;}";
    
    Tracer::write_bin_identifier( -3, &out_graph );
    //Tracer::write_2bytes( (short)-3, &out_graph ); //end group id
    
    
    //Final group
    Tracer::write_bin_identifier( 0x0C00, &out_graph );
    //Tracer::write_2bytes( (short)0x0C00, &out_graph ); //group id
    
    Tracer::write_bin_identifier( 0x0C01, &out_graph );
    //Tracer::write_2bytes( (short)0x0C01, &out_graph ); //name
    Tracer::write_bin_identifier( 5, &out_graph );
    //Tracer::write_2bytes( 5, &out_graph );
    out_graph << "Final";
    
    Tracer::write_bin_identifier( 0x0C03, &out_graph );
    //Tracer::write_2bytes( (short)0x0C03, &out_graph ); //filled
    Tracer::write_bin_identifier( 1, &out_graph );
    //Tracer::write_2bytes( 1, &out_graph );
    Tracer::write_1bytes( 1, &out_graph );
    
    Tracer::write_bin_identifier( 0x0C05, &out_graph );
    //Tracer::write_2bytes( (short)0x0C05, &out_graph ); //fg color
    Tracer::write_bin_identifier( 4, &out_graph );
    //Tracer::write_2bytes( 4, &out_graph );
    Tracer::write_1bytes( 0, &out_graph );
    Tracer::write_1bytes( 150, &out_graph );
    Tracer::write_1bytes( 60, &out_graph );
    Tracer::write_1bytes( 0, &out_graph );
    
    Tracer::write_bin_identifier( 0x0C08, &out_graph );
    //Tracer::write_2bytes( (short)0x0C08, &out_graph ); //Assigning
    Tracer::write_bin_identifier( 11, &out_graph );
    //Tracer::write_2bytes( 11, &out_graph );
    out_graph << "{ final=1;}";
    
    Tracer::write_bin_identifier( -3, &out_graph );
    //Tracer::write_2bytes( (short)-3, &out_graph ); //end group id
    
    if ( DFS_graph ) {
      Tracer::write_bin_identifier( 0x0C00, &out_graph );
      //Tracer::write_2bytes( (short)0x0C00, &out_graph ); //group id
    
      Tracer::write_bin_identifier( 0x0C01, &out_graph );
      //Tracer::write_2bytes( (short)0x0C01, &out_graph ); //name
      Tracer::write_bin_identifier( 8, &out_graph );
      //Tracer::write_2bytes( 8, &out_graph );
      out_graph << "Scenario";
      
      Tracer::write_bin_identifier( 0x0C05, &out_graph );
      //Tracer::write_2bytes( (short)0x0C05, &out_graph ); //fg color
      Tracer::write_bin_identifier( 4, &out_graph );
      //Tracer::write_2bytes( 4, &out_graph );
      Tracer::write_1bytes( 100, &out_graph );
      Tracer::write_1bytes( 100, &out_graph );
      Tracer::write_1bytes( 255, &out_graph );
      Tracer::write_1bytes( 0, &out_graph );
      
      Tracer::write_bin_identifier( 0x0C08, &out_graph );
      //Tracer::write_2bytes( (short)0x0C08, &out_graph ); //Assigning
      Tracer::write_bin_identifier( 17, &out_graph );
      //Tracer::write_2bytes( 17, &out_graph );
      out_graph << "{ in_scenario=1;}";
      
      Tracer::write_bin_identifier( -3, &out_graph );
      //Tracer::write_2bytes( (short)-3, &out_graph ); //end group id
      
    }
    
    Tracer::write_bin_identifier( -1, &out_graph );
    //Tracer::write_2bytes( (short)-1, &out_graph ); //end of graph vis. info block
    
    /*Restore tracing flags*/
    Tracer::init_tracing( dbg_stream, flags );
  }
}

void InitializeGlobals()
{
  ifstream vars_file("./vars.txt");
  char buff[65536];
  
  Tracer::get_name_id( "Node" );
  if (vars_file.is_open())
    while ( !vars_file.eof() ) {
      vars_file.getline( buff, sizeof(buff) );
      if (buff[0] != '\0')
        Tracer::get_name_id( string(buff) );
    }
    
  bin32_file = (!ascii_file && (Tracer::name_number() > 30000));
}

int main( int argc, char *argv[] )
{
  ofstream dbg_file;
  string   graph_file_name = "./test.graph";
  string   model_file = "";
  unsigned long g_flags = 0;
  unsigned long offset = 0;
  string   performance_file("");
    
  while ( argc > 1 ) {
    if ( argv[1][0] != '-') //file name
      graph_file_name = string( argv[1] );
    else
      switch( argv[1][1] ) {
        case 'g': 
          if ( argv[1][2] == 'm' ) //output graph metrics
            output_graph_metrics = true;
          else { //debug info
            dbg_file.open( "./__" );
            g_flags = Tracer::FULL_TRACE | Tracer::ASCII_OUTPUT;  
          }
          break;
        
        case 'd': //put all DFS graph into file (does not work in suppertrace mode)
          make_dfs_graph = true;
          dfs_graph_file.open( argv[2] );
          argc--; argv++;
          if ( !dfs_graph_file.is_open() ) {
            make_dfs_graph = false;
            cerr << "Scenario: cannot open DFS output " << argv[2] << " for writing";
          }
            
          break;
      
        case 'f': //lefty info
          do_lefty_output = true; 
          break;
        
        case 'a': //ascii graph file format
          ascii_file = true;
          break;
        
        case 'm':
          argc--, argv++;
          model_file = string( argv[1] );
          break;
        
        case 'c':
          edge_capping = true;
          edge_cap = atoi(argv[2]);
          argc--; argv++;
          break;

        case 's':
          supertrace = true;
          detect_collisions = (argv[1][2] == '1');
          break;
        
        case 'p':
          performance_file = string( argv[2] );
          argc--; argv++;
          break;
        
        case 'h':
          {
            if ( argv[1][2] == 'f') {
              long hash_func = atol( &(argv[1][3]) );
              
              StateHash::hash_function = (StateHash::HashFunc)hash_func;
            }
            else {
              long hash_bit_count = atol( &(argv[1][2]) );
            
              if ( hash_bit_count > 0 )
                StateHash::HASH_KEY_BYTE_SIZE = (uchar)((hash_bit_count - 1) / CHAR_BIT + 1);
            }
          }
          break;
          
        case 'r': //rs graph
          
          if ( argv[1][2] == 'i') // i-representative
            i_rs_mode = true;
          else if ( argv[1][2] == 'f' ) // f-representative
            f_rs_mode = true;
          break;
          
        default:;
      }
    
      argc--, argv++;
  }
  
  struct timeval timer;
  
  start_timer(&timer);
  
  InitializeGlobals();
 
  /* Default tracing settings:
     C++ output (to out_s), 
     disabled: make lefty input string (minimized output)
     enabled: tracing State, Automata (with tables), Queue (expanded), 
     all actions (currently implemented only for tables),
     local and global variables*/
   
  if ( ascii_file )
    graph_file.open( graph_file_name.c_str() );
  else
    graph_file.open( graph_file_name.c_str(), (ios::binary | ios::trunc | ios::out | ios::in) );
  
  offset = make_graph_file_header( ascii_file, model_file );
    
  Tracer::init_tracing( &dbg_file, g_flags );
  StateHash::init_hashing();
  
  construct_scenario_graph(true);
  
  /*save default visualisation values*/
  finish_file_making( offset, ascii_file );
  
  dbg_file.close();
  graph_file.close();
  
  if ( !performance_file.empty() ) {
    ofstream pout( performance_file.c_str(), ios::out | ios::app );
    
    if ( pout.is_open() ) {
      
      pout << " ";
      
      if ( output_graph_metrics ) {
        pout << graph_metrix.acceptance_number << ",";
        pout << graph_metrix.max_node_out_degree << ",";
        pout << graph_metrix.max_not_final_out_degree << ",";
        pout << graph_metrix.graph_depth << ",";
        pout << graph_metrix.longest_scenario << ",";
        pout << graph_metrix.shortest_scenario << ",";
      }
      report_timer(pout, timer);
      pout << "," << scenario_mem_used;
      
      pout.close();
    }
  }
  
  cout << "Performance time: ";
  report_timer(cout, timer);
  cout << endl;
  cout << "Memory used:       " << scenario_mem_used << endl << endl;
  
  //make DFS grapf file
  if ( make_dfs_graph ) {
    NodePointerList::iterator it;
    Tracer::init_tracing( &dbg_file, 0 );
    offset = make_graph_file_header( ascii_file, model_file, true );
    
    //print nodes
    for ( it = NodeList.begin(); it != NodeList.end(); it++ )
      (*it)->LeftyInsertNode( stdout, "dfs", true );
    
    unsigned long format_flags = Tracer::GRAPH_ATTR_FORMAT;
    if ( bin32_file )
      format_flags |= Tracer::BIN32; 
    Tracer::init_tracing( &dfs_graph_file, format_flags );
    
    //print edges
    for ( it = NodeList.begin(); it != NodeList.end(); it++ ) {
      Node *node = (*it);
      node->ResetLastChild();
      for ( Node *child = node->GetNextChild(); child != NULL; child = node->GetNextChild() )
        child->LeftyInsertEdge(stdout, node, child, "fsm", true, true );
    }
    
    /*save default visualisation values*/
    finish_file_making( offset, ascii_file, true );
  }
  
  Tracer::clear_names_storage();

  return 0;
}
