
#ifndef _GRAPH_INCLUDED_
#define _GRAPH_INCLUDED_

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <list>
#include <ext/hash_set>
#include <sys/time.h>
#include "gstructs.h"

#define TRUE  1
#define FALSE 0

class Node;
class Graph;

using namespace std;

extern ofstream graph_file;
extern ofstream dfs_graph_file;
extern bool ascii_file;
extern bool bin32_file;
extern bool do_lefty_output;
extern bool supertrace;
extern bool detect_collisions;
extern long dfs_edges_count;
extern long hash_comparison;
extern long state_restoring;
extern long state_comparison;
extern long collisions_count;
extern bool edge_capping;
extern bool abridged_mode;
extern bool make_dfs_graph;

typedef list<State*> LISTSTATE;

typedef struct NodeElementStruct {
  Node *node;
  struct NodeElementStruct *next;
} NodeElement;

typedef struct PackedState_struct
{
  uchar *hash_key;
  int    ltl_trans_id;
  int    autom_trans_id;
  short  autom_number;
} PackedState;

class NodeStack
{
public:

  NodeStack();
  ~NodeStack();
  bool Empty(); 
  Node *Top(); 
  
  void Push(Node *node); 
  Node *Pop(); 
  Node *OnStack(Node *node);

  void ResetIterator() { current = top; }
  Node *Next() {
    if (current == NULL)
      return NULL;
    else {
      Node *node = current->node;
      current = current->next;
      return node;
    }
  }

  void Print();
  
  long GetUsedMem() { return max_length * sizeof(NodeElement); }
  long GetLength() { return length; } 
  
private:

  NodeElement *top;
  NodeElement *current;
  long         length;
  long         max_length;
};

class Node
{
  friend class Graph;
  
public:
  
  enum NodeFlags
  {
     NONE = 0,
     FORCED_ACCEPTED = 1,
     FORCED_FINAL = 2,
     IN_SCENARIO  = 4,
     R_SCENARIO   = 8,
//     I_ACCEPTANCE = 16,
     //
     FIXED        = 32,
     PASSED       = 64,
     //
     TRANS_TO_BORDER = 128
  };

  Node(State *st, short autom_num, int autom_trans_id, int ltl_trans_id) {
    lowlink = 0;
    number = 0;
    LastChild = NULL;
    Children = NULL;
    state = st;
    parent = 0;
    m_flags = NONE;
    component_number = 0;
    packed_state = new PackedState;
    packed_state->autom_trans_id = autom_trans_id;
    packed_state->ltl_trans_id = ltl_trans_id;
    packed_state->autom_number = autom_num;
    StateHash::pack_state_hash_key( st, &(packed_state->hash_key) );
  }

  ~Node() {
    if ( state != 0 )
      delete state;
    
    delete[] packed_state->hash_key;
    delete   packed_state;
  }

  bool operator==(Node &other);

  //////////////
  // Accessors
  
  void SetLowLink(int ll) { lowlink = ll; }
  void SetNumber(int num) { number = num; }
  int  GetLowLink() { return lowlink; }
  int  GetNumber()  { return number; }

  State *GetState() { return state; }
  PackedState *GetPackedState() { return packed_state; }
  
  void ResetLastChild() { LastChild = NULL; }
  Node *GetLastChild() {
    if (LastChild == NULL)
      return NULL;
    else
      return LastChild->node;
  }
  
  Node *GetNextChild() {
    if (LastChild == NULL)
      LastChild = Children;
    else
      LastChild = LastChild->next;

    return GetLastChild();
  }
  
  bool LastChildIsFirst() { return LastChild == Children; }
  
  void SetCurrNodeElement(NodeElement *curr_el) {
    LastChild = curr_el;
  }
  
  NodeElement *GetCurrNodeElement() {
    return LastChild;
  }

  State* GenerateNodeState(); // This function reconstruct packed state of the node
  void GenerateAtomicStates(int procnum, State *state, LISTSTATE &states);  
  void GenerateChildren();  
  void AutomsProcesing(State *state, State *NextState_never, int ltl_trans_id);

  bool HasSelfLoop() {
    NodeElement *Child = Children;
    while (Child != NULL) {
      if (Child->node == this) return TRUE;
      Child = Child->next;
    }
    return FALSE;
  }

  bool Unexplored() { return (number == 0); }
  
  void ForceAccepting( bool accp ) { 
    if ( accp )
      m_flags |= FORCED_ACCEPTED;
    else
      m_flags &= (~FORCED_ACCEPTED);
  }
  
  void ForceFinal( bool final ) { 
    if ( final )
      m_flags |= FORCED_FINAL; 
    else
      m_flags &= (~FORCED_FINAL);
  }

  bool IsAccepting() { 
    if ( (m_flags & FORCED_ACCEPTED) == FORCED_ACCEPTED )
      return true;
    
    if ( state == 0 )
      return false;
    
    CAutomBase *never = state->get_never_claim();

    if ( never == 0 )
      return false;
    /*accpstate[NEVER][state->never]; */
    return never->is_state_accepted( never->get_curr_state() ); 
  }  
  
  bool IsFinal() {
    if ( (m_flags & FORCED_FINAL) == FORCED_FINAL )
      return true;
    
    if ( state == 0 )
      return false;
    
    CAutomBase *never = state->get_never_claim();

    if ( never == 0 )
      return false;
    
    return never->is_state_final( never->get_curr_state() ); 
  }  

  void Print();
  void PrintSubgraphDotFormat(FILE *fd);
  
  void PackNode( bool in_scenario = false ) {
    if ( supertrace )
      if (parent != 0) {
        if ( state != 0 ) {
          CAutomBase *never = state->get_never_claim();
          if ( never != 0 ) {  
            if ( never->is_state_final( never->get_curr_state() ) )
              ForceFinal( true );
            
            if (never->is_state_accepted( never->get_curr_state() ))
              ForceAccepting( true );
          }
          
          delete state;
        }
            
        state = 0;
      }
  }
  
  Node* GetParent() { return parent; }
  
  unsigned int CalcSize( bool involve_states = false ) {
    unsigned int size = sizeof(*this) + sizeof(PackedState) + StateHash::HASH_KEY_BYTE_SIZE + sizeof(NodeElement);
    
    for (NodeElement *ch = Children; ch != 0; ch = ch->next )
      size += sizeof(NodeElement);
    
    if ( involve_states && state != 0 )
      size += state->calc_size();
    
    return size;
  }
private:

  void AddChild(Node *node) {
    NodeElement *element = new NodeElement;
    if (node->parent == 0)
      node->parent = this;
    element->node = node;
    element->next = Children;
    Children = element;
  }

  Node *GenerateChild(int procnum, int pr_trans_id, int trans_id);
  
  void GenerateChild(State *NextState, short procnum, int autom_trans_id, int ltl_trans_id);
  
public:
  ////////////////////////
  // Lefty script output

  void LeftyInsertNode(FILE *fd, char *GraphTableName, bool DFS_output = false); 
  void LeftyInsertEdge(FILE *fd, Node *source, Node *sink, char *GraphTableName, bool insert_to_file = true, bool DFS_output = false );
  void LeftyUpdateNode(FILE *fd, char *GraphTableName);
  void LeftyInsertNodeStep(FILE *fd, char *GraphTableName);
  void LeftyInsertEdgeStep(FILE *fd, Node *source, Node *sink, char *GraphTableName);
  void LeftyUpdateNodeStep(FILE *fd, char *GraphTableName);
  void LeftyHighlight(FILE *fd, char *GraphTableName);	    
  void LeftyHighlightLabel(FILE *fd, char *GraphTableName);	    
  void LeftyHighlightEdge(FILE *fd, Node *source, Node *sink, char *GraphTableName);
  void PrintLeftyProgram(FILE *fd, char *GraphTableName);

public:
  void ProfileHashCollisions();

  bool InScenario() { return ((m_flags & IN_SCENARIO) == IN_SCENARIO); }
  void SetInScenario( bool in_scenario ) {
    if ( in_scenario )
      m_flags |= IN_SCENARIO;
    else
      m_flags &= (~IN_SCENARIO);
  }
  
  bool InScenarioNotInRS() { return ((m_flags & R_SCENARIO) == R_SCENARIO); }
  void SetInScenarioNotInRS( bool r_scenario ) { 
    if ( r_scenario )
      m_flags |= R_SCENARIO;
    else
      m_flags &= (~R_SCENARIO);
  }
  
  bool IsFixed() { return ((m_flags & FIXED) == FIXED); }
  
  void SetFixed( bool fixed ) { 
    if ( fixed )
      m_flags |= FIXED;
    else
      m_flags &= (~FIXED);
  }
  
  bool IsPassed() { return ((m_flags & PASSED) == PASSED); }
  
  void SetPassed( bool passed ) { 
    if ( passed )
      m_flags |= PASSED;
    else
      m_flags &= (~PASSED);
  }
  
  void SetFlags( uchar flags, bool enable ) {
     if ( enable )
      m_flags |= flags;
    else
      m_flags &= (~flags);
  }
  
  int GetComponentNumber() { return component_number; }
  void SetComponentNumber( int number ) { component_number = number; }
  
  void SetBorderTransStart( bool border ) { 
    if ( border )
      m_flags |= TRANS_TO_BORDER;
    else
      m_flags &= (~TRANS_TO_BORDER);
  }
  
  bool HasBorderChild() { return ((m_flags & TRANS_TO_BORDER) == TRANS_TO_BORDER); }
  
  void RemoveDuplicatedChildren() {
    NodeElement *child;
    NodeElement *del;
    NodeElement *prev_child = 0;
    
    LastChild = NULL;
    child = Children;
    while ( child ) {
      if ( child->node->IsPassed() ) {
        del = child;
        child = child->next;
        prev_child->next = child;
        delete del;
      }
      else {
        child->node->SetPassed( true );
        prev_child = child;
        child = child->next;
      }
    }
    
    child = Children;
    for ( ; child; child = child->next )
      child->node->SetPassed( false );
  }

private:

  State *state;

  PackedState *packed_state;
  Node        *parent;
  
  uchar        m_flags;

  int lowlink;
  int number;
  int component_number;

  NodeElement *Children;            // List of children
  NodeElement *LastChild;
};

///////////////////
// Scenario Graph

struct GraphMetrics
{
  int acceptance_number;
  int max_node_out_degree;
  int max_not_final_out_degree;
  int graph_depth;
  int longest_scenario;
  int shortest_scenario;
  
  GraphMetrics(): acceptance_number(0), max_node_out_degree(0), 
                  max_not_final_out_degree(0), graph_depth(0),
                  longest_scenario(0), shortest_scenario(0) {}
};

typedef list<Node*> NodePointerList;

class Graph
{
private:

  NodePointerList GraphNodes;
public:
  
  Graph() { edgecount = 0; }
  
  void Add( Node *node ) {
    if ( !node->InScenario() ) {
      node->LeftyInsertNode(stdout, "scen");
      node->PackNode( true );
      GraphNodes.push_back(node);
      node->SetInScenario( true );
    }
  }
  
  void Graph::Add(NodeStack &nodes) {
    // Add the nodes
    nodes.ResetIterator();
    for (Node *node = nodes.Next(); node != NULL; node = nodes.Next()) {
      if (!node->InScenario() ) {
        node->LeftyInsertNode(stdout, "scen");
        node->PackNode( true );
        GraphNodes.push_back(node);
        node->SetInScenario( true );
      }
    }
  }
  
  void PrintToFile();
  
  void PrintDotFormat(FILE *fd) {
  }

  void PrintLeftyProgram(FILE *fd) {
  }
  
  unsigned int GetNodeCount() { GraphNodes.size(); }
  unsigned int CalcSize( ) {
    return (sizeof(*this) + GraphNodes.size() * sizeof(Node*));
  }
  
  bool ProcessIAcceptance( Node *init_node );
  
  GraphMetrics CountGraphMetrics( Node *init_node );

public:

  unsigned int edgecount;
};

typedef  size_t   ub4;        /* unsigned 4-byte quantities */
typedef  unsigned char ub1;   /* unsigned 1-byte quantities */

#define hashsize(n) ((ub4)1<<(n))
#define hashmask(n) (hashsize(n)-1)

/*
--------------------------------------------------------------------
mix -- mix 3 32-bit values reversibly.
For every delta with one or two bits set, and the deltas of all three
  high bits or all three low bits, whether the original value of a,b,c
  is almost all zero or is uniformly distributed,
* If mix() is run forward or backward, at least 32 bits in a,b,c
  have at least 1/4 probability of changing.
* If mix() is run forward, every bit of c will change between 1/3 and
  2/3 of the time.  (Well, 22/100 and 78/100 for some 2-bit deltas.)
mix() was built out of 36 single-cycle latency instructions in a 
  structure that could supported 2x parallelism, like so:
      a -= b; 
      a -= c; x = (c>>13);
      b -= c; a ^= x;
      b -= a; x = (a<<8);
      c -= a; b ^= x;
      c -= b; x = (b>>13);
      ...
--------------------------------------------------------------------
*/
#define mix(a,b,c) \
{ \
  a -= b; a -= c; a ^= (c>>13); \
  b -= c; b -= a; b ^= (a<<8); \
  c -= a; c -= b; c ^= (b>>13); \
  a -= b; a -= c; a ^= (c>>12);  \
  b -= c; b -= a; b ^= (a<<16); \
  c -= a; c -= b; c ^= (b>>5); \
  a -= b; a -= c; a ^= (c>>3);  \
  b -= c; b -= a; b ^= (a<<10); \
  c -= a; c -= b; c ^= (b>>15); \
}

namespace __gnu_cxx
{
  struct hash<Node *> {
    /*
      --------------------------------------------------------------------
      jenkins_hash() -- hash a variable-length key into a 32-bit value
      k       : the key (the unaligned variable-length array of bytes)
      len     : the length of the key, counting by bytes
      initval : can be any 4-byte value
      Returns a 32-bit value.  Every bit of the key affects every bit of
      the return value.  Every 1-bit and 2-bit delta achieves avalanche.
      About 6*len+35 instructions.
    
      The best hash table sizes are powers of 2.  There is no need to do
      mod a prime (mod is sooo slow!).  If you need less than 32 bits,
      use a bitmask.  For example, if you need only 10 bits, do
      h = (h & hashmask(10));
      In which case, the hash table should have hashsize(10) elements.
    
      If you are hashing n strings (ub1 **)k, do it like this:
      for (i=0, h=0; i<n; ++i) h = hash( k[i], len[i], h);
    
      By Bob Jenkins, 1996.  bob_jenkins@burtleburtle.net.  You may use this
      code any way you wish, private, educational, or commercial.  It's free.
    
      See http://burtleburtle.net/bob/hash/evahash.html
      Use for hash table lookup, or anything where one collision in 2^^32 is
      acceptable.  Do NOT use for cryptographic purposes.
      --------------------------------------------------------------------
    */
  
    size_t jenkins_hash(register ub1* k, register ub4 length, register ub4 initval) const
    {
      register ub4 a,b,c,len;
    
      /* Set up the internal state */
      len = length;
      a = b = 0x9e3779b9;  /* the golden ratio; an arbitrary value */
      c = initval;         /* the previous hash value */
    
      /*---------------------------------------- handle most of the key */
      while (len >= 12) {
        a += (k[0] +((ub4)k[1]<<8) +((ub4)k[2]<<16) +((ub4)k[3]<<24));
        b += (k[4] +((ub4)k[5]<<8) +((ub4)k[6]<<16) +((ub4)k[7]<<24));
        c += (k[8] +((ub4)k[9]<<8) +((ub4)k[10]<<16)+((ub4)k[11]<<24));
        mix(a,b,c);
        k += 12; len -= 12;
      }

      /*------------------------------------- handle the last 11 bytes */
      c += length;
      switch(len)              /* all the case statements fall through */
      {
      case 11: c+=((ub4)k[10]<<24);
      case 10: c+=((ub4)k[9]<<16);
      case 9 : c+=((ub4)k[8]<<8);
        /* the first byte of c is reserved for the length */
      case 8 : b+=((ub4)k[7]<<24);
      case 7 : b+=((ub4)k[6]<<16);
      case 6 : b+=((ub4)k[5]<<8);
      case 5 : b+=k[4];
      case 4 : a+=((ub4)k[3]<<24);
      case 3 : a+=((ub4)k[2]<<16);
      case 2 : a+=((ub4)k[1]<<8);
      case 1 : a+=k[0];
        /* case 0: nothing left to add */
      }
      mix(a,b,c);
      /*-------------------------------------------- report the result */
      return c;
    }

    size_t operator()(Node *nd) const {
      assert(nd != NULL);
      return jenkins_hash((ub1 *) ((nd->GetPackedState())->hash_key), StateHash::HASH_KEY_BYTE_SIZE, 0x9e3779b9);
    }
  };
}

struct eqnode
{
  bool operator()(Node *n1, Node *n2) const {
    assert(n1 != NULL);
    assert(n2 != NULL);
    
    return (*n1 == *n2);
  }
};

struct eqnodepointer
{
  bool operator()(Node *n1, Node *n2) const {
    assert(n1 != NULL);
    assert(n2 != NULL);
    
    return (n1 == n2);
  }
};

typedef __gnu_cxx::hash_set<Node *, __gnu_cxx::hash<Node *>, eqnode> NodeHashSet;
typedef __gnu_cxx::hash_set<Node *, __gnu_cxx::hash<Node *>, eqnodepointer> NodePointerHashSet;


#endif // _GRAPH_INCLUDED_
