
#include <string>
#include <vector>
#include <cstdio>
#include <iostream>
#include <fstream>
#include <sstream>
#include <map>
#include <limits.h>

using namespace std;

typedef unsigned char uchar;
typedef int QueueRef;

class State;

#define jmix(a,b,c) \
{ \
  a -= b; a -= c; a ^= (c>>13); \
  b -= c; b -= a; b ^= (a<<8); \
  c -= a; c -= b; c ^= (b>>13); \
  a -= b; a -= c; a ^= (c>>12);  \
  b -= c; b -= a; b ^= (a<<16); \
  c -= a; c -= b; c ^= (b>>5); \
  a -= b; a -= c; a ^= (c>>3);  \
  b -= c; b -= a; b ^= (a<<10); \
  c -= a; c -= b; c ^= (b>>15); \
}

class StateHash
{
  static uchar PearsonX[256];
  static uchar DWORD_BYTE_PER_KEY;
public:
  enum HashFunc
  {
    JENKINS_FUNC = 0,
    PEARSON_FUNC
  };
  
  static uchar HASH_KEY_BYTE_SIZE;
  static int   max_st_arr_size;
  static HashFunc hash_function;

  static void init_hashing();
  
  static void jenkins_hash( uchar *st_arr, int arr_size, uchar *key )
  {
    register size_t a,b,c;
    int      i, j;
    size_t   rest;
        
    /* Set up the initial state */
    rest = a = b = 0x9e3779b9;  /* the golden ratio; an arbitrary value */
    for ( i = 0, j = 0; i < DWORD_BYTE_PER_KEY; i += 4 )
      *(size_t*)(key + i) = 0x9e3779b9;
    
    for ( i = 0, j = 0; i < arr_size; i += 12 )
    {
      if ( j < DWORD_BYTE_PER_KEY)
        c = *(size_t*)(key + j);
      else
        c = rest;
                
      /*---------------------------------------- handle most of the key */
      if (arr_size - i >= 12) 
      {
        a += *(size_t*)(st_arr + i);
        b += *(size_t*)(st_arr + i + 4);
        c += *(size_t*)(st_arr + i + 8);
        jmix(a,b,c);
      }
      else
      {
        /*------------------------------------- handle the last 11 bytes */
        c += arr_size;
        switch (arr_size - i)              /* all the case statements fall through */
        {
        case 11: c+=((size_t)st_arr[i + 10]<<24);
        case 10: c+=((size_t)st_arr[i + 9]<<16);
        case 9 : c+=((size_t)st_arr[i + 8]<<8);
          /* the first byte of c is reserved for the length */
        case 8 : b+=((size_t)st_arr[i + 7]<<24);
        case 7 : b+=((size_t)st_arr[i + 6]<<16);
        case 6 : b+=((size_t)st_arr[i + 5]<<8);
        case 5 : b+=(size_t)st_arr[i + 4];
        case 4 : a+=((size_t)st_arr[i + 3]<<24);
        case 3 : a+=((size_t)st_arr[i + 2]<<16);
        case 2 : a+=((size_t)st_arr[i + 1]<<8);
        case 1 : a+=(size_t)st_arr[i];
          /* case 0: nothing left to add */
        }
        jmix(a,b,c);
      }
      
      /* store result */
      if ( j < DWORD_BYTE_PER_KEY )
      {
        *(size_t*)(key + j) = c;
        j += sizeof( size_t );
        if ( j >= DWORD_BYTE_PER_KEY )
          j = 0;
      }
      else
      {
        rest = c;
        j = 0;
      }
    }
    
    j = HASH_KEY_BYTE_SIZE & 0x3;
    switch ( j )
    {
    case 3:
      key[HASH_KEY_BYTE_SIZE - 3] = (uchar)((rest >> 16) & 0xFF);
    case 2:
      key[HASH_KEY_BYTE_SIZE - 2] = (uchar)((rest >> 8) & 0xFF);
    case 1:
      key[HASH_KEY_BYTE_SIZE - 1] = (uchar)(rest & 0xFF);
    }
  }

  static void pearson_hash( uchar *st_arr, int arr_size, uchar *key )
  {
    int i, j;
    unsigned short sum = 0;
    //uchar rest = 0;
    key[0] = (arr_size /*+ HASH_KEY_BYTE_SIZE*/) & 0xFF;
    for ( i = 0; i < HASH_KEY_BYTE_SIZE; i++ )
      key[i] = PearsonX[(key[i - 1] + arr_size) & 0xFF];
    
    for ( i = 0; i < arr_size; i++, j++ )
    {
      if ( j >= HASH_KEY_BYTE_SIZE )
        j = 0;
        
      sum = (unsigned short)key[j] + (unsigned short)st_arr[i] /*+ (unsigned short)rest*/;
      key[j] = PearsonX[sum & 0xFF];
        //rest = (uchar)(sum >> 8);
    }
  }
  //returns size of key
  static uchar pack_state_hash_key( State* st, uchar **key );
};


//Class that helps to work with bit arrays for boolean variable reperesentations
class BoolArr
{
friend uchar pack_state_hash_key( State* st, uchar **key );
private:
  uchar *m_arr;
  unsigned int m_bit_count;

public:
  BoolArr() {
    m_bit_count = 0;
    m_arr = 0;
  }
  
  BoolArr(BoolArr *arr) {
    m_bit_count = arr->m_bit_count;
    int byte_size = (m_bit_count == 0) ? 0 : (m_bit_count-1)/CHAR_BIT+1;
    
    if (byte_size > 0)
    {
      m_arr = new uchar[byte_size];    
      memcpy(m_arr, arr->m_arr, byte_size);
    }
    else
      m_arr = 0;
  }

  BoolArr(int size) {
    m_bit_count = size;
    int byte_size = (size == 0) ? 0 : (size-1)/CHAR_BIT+1;
    
    if (size > 0)
    {
      m_arr = new uchar[byte_size];    
      memset(m_arr, 0, byte_size);
    }
    else
      m_arr = 0;
  }
  
  ~BoolArr() {
    if (m_arr != 0)
      delete[] m_arr;
    m_arr = 0;
  }

  void set_size(int size) {
    if (m_arr != 0)
      delete[] m_arr;
    
    m_bit_count = size;
    int byte_size = (size == 0) ? 0 : (size-1)/CHAR_BIT+1;

    if (size > 0)
    {
      m_arr = new uchar[byte_size];    
      memset(m_arr, 0, byte_size);
    }  
    else
      m_arr = 0;
  }
  
  int get_size() {
    return m_bit_count;
  }
  
  void set_var(int ind, bool val) {
    int   byte_ind = ind/CHAR_BIT;
    uchar &target_byte = m_arr[byte_ind];
    uchar mask;
    
    if (ind < m_bit_count)
    {  
      mask = (uchar)0x80 >> (ind % CHAR_BIT);
    
      if (val)
        target_byte |= mask;
      else
        target_byte &= ~mask;
    }
  }
  
  bool is_var_enabled(int ind) {
    int   byte_ind = ind/CHAR_BIT;
    uchar target_byte = m_arr[byte_ind];
    uchar mask;
    
    if (ind < m_bit_count) {
      mask = (uchar)0x80 >> (ind % CHAR_BIT);
      return target_byte&mask;    
    }
    else
      return false;
  }
  
  bool operator!=(BoolArr &other) {
    if (m_bit_count != other.m_bit_count)
      return true;
    
    int byte_size = (m_bit_count == 0) ? 0 : (m_bit_count-1)/CHAR_BIT+1;
    
 //   for (int i = 0; i < byte_size; i++)
 //     if (m_arr[i] != other.m_arr[i])
 //       return true;
      
 //   return false;
    return (memcmp( m_arr, other.m_arr, byte_size ) == 0);
  }
  
  BoolArr& operator=(const BoolArr &other) {
    
    if (this != &other)
    {
      m_bit_count = other.m_bit_count;
      
      if (m_arr != 0)
        delete[] m_arr;
      
      int byte_size = (m_bit_count == 0) ? 0 : (m_bit_count-1)/CHAR_BIT+1;
    
      m_arr = new uchar[byte_size];    
      memcpy( m_arr, other.m_arr, byte_size );
      //for (int i = 0; i < byte_size; i++)
      //  m_arr[i] = other.m_arr[i];
    }
    
    return *this;
  }

  void debug_print()
  {
    int byte_size = (m_bit_count == 0) ? 0 : (m_bit_count-1)/CHAR_BIT+1;
    
    for (int i = 0; i < byte_size; i++)
      cout << (int)m_arr[i] << ",";
    
    cout << "\n";
  }
  
  void copy_bit_arr( uchar *copy_to )
  {
    int byte_size = (m_bit_count == 0) ? 0 : (m_bit_count-1)/CHAR_BIT+1;
    memcpy( copy_to, m_arr, byte_size );
  }
  
};
  
void make_state_attr_format( unsigned long format_flags, ostream *output );

class CAutomBase;
class QueueEl;
class State;
class Queue;
  
typedef map<string, char> TypeStorage;
typedef map<string, long> NameStorage;
typedef TypeStorage::const_iterator TypeCI;
typedef NameStorage::const_iterator NameCI;

class Tracer
{
public:
   static const unsigned long ASCII_OUTPUT;
   static const unsigned long PRINT_TO_STR;
   static const unsigned long LEFTY_INPUT;
   static const unsigned long GRAPH_ATTR_FORMAT;
   static const unsigned long BIN32;
   
   static const unsigned long TRACE_STATE;
   static const unsigned long TRACE_QUEUE;
   static const unsigned long TRACE_QUEUE_EL;
   static const unsigned long TRACE_AUTOM;
   static const unsigned long ATTR_INIT;

   static const unsigned long EXPAND_QUEUE;
   static const unsigned long TRACE_MEM;
   static const unsigned long TRACE_TABLE;
   static const unsigned long TRACE_LOC_VAR;
   static const unsigned long TRACE_GL_VAR;
   static const unsigned long TRACE_VAR;
   static const unsigned long FULL_TRACE;

   static const unsigned long ACTION_DEL;
   static const unsigned long ACTION_ADD;
   static const unsigned long ACTION_CLONE;
   static const unsigned long ACTION_VAR;
   static const unsigned long ACTION_LOOK;
   static const unsigned long ACTION_FLAGS;
   
   static string ATTR_START;
   static string ATTR_END;

private:
   static ostringstream m_out_sstream;
   static ostream       *m_out_stream;

protected:
   static unsigned long m_tracing_flags;

   static const uchar   QUEUE_CLASS;
   static const uchar   QUEUE_EL_CLASS;
   static const uchar   AUTOM_CLASS;
   static const uchar   STATE_CLASS;

   static const int     MAX_ASCII_LINE_LENGTH;

   static string EOL;
   static string TAB;
   
   static NameStorage  m_name_set;
   static vector<string>  m_name_arr;
   static TypeStorage  m_type_set;
   static vector<string>  m_type_arr;
   static int          m_curr_line_len;

public:
  static inline void write_1bytes( char b, ostream *out_stream ) { out_stream->put( b ); }
  static inline void write_2bytes( short s, ostream *out_stream ) { out_stream->write( (char *)&s, 2 ); }
  static inline void write_4bytes( long l, ostream *out_stream ) { out_stream->write( (char *)&l, 4 ); }
  static inline void write_bin_identifier( long id, ostream *out_stream )
  {
    if ( m_tracing_flags & BIN32 )
      write_4bytes( id, out_stream );
    else
      write_2bytes( (short)id, out_stream );
  }

protected:
   virtual uchar get_tracer_class_id() { return 0; }

/* START: working with name/type storages */   
   static char add_type( const string &type )
   {
      char id = 0;

      if ( !type.empty() )
      {
         TypeStorage::iterator it;
         if ( (it = m_type_set.find( type )) != m_type_set.end() )
            id = it->second;
         else
         {
            m_type_arr.push_back( type );
            id = (char)m_type_arr.size();
            m_type_set[ type ] = id; //id starts from 1
         }
      }

      return id;
   }
   
   static inline long add_name( const string &name )
   {
      long id = 0;

      if ( !name.empty() )
      {
         NameStorage::iterator it;
         if ( (it = m_name_set.find( name )) != m_name_set.end() )
            id = it->second;
         else
         {
            m_name_arr.push_back( name );
            id = (long)m_name_arr.size();
            m_name_set[ name ] = id; // id starts from 1
         }
      }

      return id;
   }
   
public:   
   static inline char get_type_id( const string& type )
   {
      char id = 0;
      TypeCI p;

      if ( type == ATTR_START )
        id = -1;
      else if ( type == ATTR_END )
        id = -2;
      else 
        id = add_type( type );
      
      return id;
   }
   
   static inline long get_name_id( const string& name )
   {
      return add_name( name );
   }
   
   static void clear_names_storage()
   {
      m_name_set.clear();
      m_name_arr.clear();
      m_name_arr.resize(0);
      m_type_set.clear();
      m_type_arr.clear();
      m_type_arr.resize(0);
   }
   
   static long calc_names_storage_size()
   {
     long res = 0;
     unsigned int i;
     for ( i = 0; i < m_name_arr.size(); i++ )
       res += 2 * (m_name_arr[i].length() + sizeof(string)) + sizeof(short);
     
     for ( i = 0; i < m_type_arr.size(); i++ )
       res += 2 * (m_type_arr[i].length() + sizeof(string)) + sizeof(short);
     
     return res;
   }
   
   static long name_number() { return m_name_arr.size(); }
   
   static void write_names_storage( ostream *out_stream )
   {
     unsigned int i = 0;
     for ( ; i < m_name_arr.size(); i++ )
     {
       write_bin_identifier( m_name_arr[i].length(), out_stream );
       (*out_stream) << m_name_arr[i];
     }
     write_bin_identifier( -3, out_stream ); //internal separator 2
     //write_2bytes( (short)-3, out_stream ); 
     
     for ( i = 0; i < m_type_arr.size(); i++ )
     {
       write_bin_identifier( m_type_arr[i].length(), out_stream );
       (*out_stream) << m_type_arr[i];
     }
   }
/* END: working with name/type storages */  

protected:
/* START: put ASCII data only */  
   static inline void print( const string &info_prefix, long info,  const string &info_suffix = "" )
   {
      if ( m_tracing_flags & ASCII_OUTPUT )
      {
        if ( (m_tracing_flags & PRINT_TO_STR) )
            m_out_sstream << info_prefix << info << info_suffix; 
/*      else if ( (m_tracing_flags & C_STYLE_FLAG) )
        {
           fprintf( m_out_fd, "%s%ld%s", info_prefix.c_str(), info, info_suffix.c_str() );
        }
*/      else
        {
           if ( m_out_stream == 0 )
              cout << info_prefix << info << info_suffix;
           else
              (*m_out_stream) << info_prefix << info << info_suffix;
        }
      }
   }

   static inline void print( const string &info_prefix, const string &info, const string &info_suffix = "" )
   {
      if ( m_tracing_flags & ASCII_OUTPUT )
      {
        if ( (m_tracing_flags & PRINT_TO_STR) )
            m_out_sstream << info_prefix << info << info_suffix; 
/*      else if ( (m_tracing_flags & C_STYLE_FLAG) )
        {
           fprintf( m_out_fd, "%s%s%s", info_prefix.c_str(), info.c_str(), info_suffix.c_str() );
        }
*/      else
        {
           if ( m_out_stream == 0 )
              cout << info_prefix << info << info_suffix;
           else
              (*m_out_stream) << info_prefix << info << info_suffix;
        }
      }
   }

   static inline void print( long info, const string &info_suffix = "")
   {
      print( "", info, info_suffix );
   }

   static inline void print( const string &info )
   {
      print( "", info, "" );
   }
   
public:   
   static inline void finish_ASCII_block(bool is_end = true )
   {
     if ( !is_end )
        print( "\\" );
     print( "\n" );
     m_curr_line_len = 0;
   }
/* END: put ASCII data only */   
protected:

/* START: put ASCII and binary data (for storing graph file) */   
   static inline void print_attr( const string&  type, const string &name = "", long value = -1 )
   {
     if ( m_tracing_flags & ASCII_OUTPUT )
     {
       if ( type == ATTR_START )
       {
         if ( value >= 0 )
         {
           print( " ", value, ":" );
           m_curr_line_len += 4; 
         }
         print( " ", name, type );
         m_curr_line_len += name.length() + type.length();
       }
       else if ( type == ATTR_END )
       {
         print( type, EOL);
         m_curr_line_len += type.length();
       }
       else
       {
         print( " " + type + ": " + name + "=", value, ";" + EOL );
         m_curr_line_len += name.length() + type.length() + 7;
       }
       
       if ( m_curr_line_len > MAX_ASCII_LINE_LENGTH )
         finish_ASCII_block( false );
     }
     else
     {
       long id = add_name( name );
       write_bin_identifier( id, m_out_stream );
       //write_2bytes( id, m_out_stream );
       
       if ( type == ATTR_START )
         write_1bytes( 0xFF, m_out_stream );
       else if ( type == ATTR_END )
         write_1bytes( 0xFE, m_out_stream );
       else  
       {
         write_1bytes( add_type( type ), m_out_stream );
       
         if ( type == "bool" || type == "byte" || type == "bit" )
            write_1bytes( (unsigned char)value, m_out_stream );
         else if ( type == "short" || type == "chan" )
            write_2bytes( (short)value, m_out_stream );
         else
            write_4bytes( (long)value, m_out_stream );
       }
     }
   }
   
   static inline void print_attr_ind( const string&  type, long name, long value = -1 )
   {
     if ( m_tracing_flags & ASCII_OUTPUT )
     {
       if ( type == ATTR_START )
       {
         if ( value >= 0 )
         {
           print( " ", value, ":" );
           m_curr_line_len += 4; 
         }
         print( " ", m_name_arr[name - 1], type );
         m_curr_line_len += m_name_arr[name - 1].length() + type.length();
       }
       else if ( type == ATTR_END )
       {
         print( type, EOL);
         m_curr_line_len += type.length();
       }
       else
       {
         print( " " + type + ": " + m_name_arr[name - 1] + "=", value, ";" + EOL );
         m_curr_line_len += m_name_arr[name - 1].length() + type.length() + 7;
       }
       
       if ( m_curr_line_len > MAX_ASCII_LINE_LENGTH )
         finish_ASCII_block( false );
     }
     else
     {
       write_bin_identifier( name, m_out_stream );
       //write_2bytes( (short)name, m_out_stream );
       
       if ( type == ATTR_START )
         write_1bytes( 0xFF, m_out_stream );
       else if ( type == ATTR_END )
         write_1bytes( 0xFE, m_out_stream );
       else  
       {
         write_1bytes( add_type( type ), m_out_stream );
       
         if ( type == "bool" || type == "byte" || type == "bit" )
            write_1bytes( (unsigned char)value, m_out_stream );
         else if ( type == "short" || type == "chan" )
            write_2bytes( (short)value, m_out_stream );
         else
            write_4bytes( (long)value, m_out_stream );
       }
     }
   }
/* END: put ASCII and binary data (for storing graph file) */   
   
public:
   /*The method must not call static trace method in any case*/
   virtual void trace( unsigned long trace_action, long extra_info = 0, const string &line_prefix = "" ) {}

   static inline void put_error( const string &s )
   {
      cerr << s;
      exit(0);
   }

   static unsigned long get_flags() { return m_tracing_flags; }
   static void set_flags( unsigned long tracing_flags ) 
   { 
      if ( tracing_flags != 0xFFFFFFFF )
      {
         if ( (tracing_flags & GRAPH_ATTR_FORMAT) )
         {
            tracing_flags |= TRACE_STATE | TRACE_AUTOM | TRACE_VAR | TRACE_QUEUE | EXPAND_QUEUE;
            TAB = "";
         }
         else if ( (tracing_flags & LEFTY_INPUT) )
         {
            tracing_flags |= TRACE_STATE | TRACE_AUTOM | TRACE_VAR | PRINT_TO_STR;
            TAB = " ";
         }
         else
         {
            TAB = "\t";
         }
         
         m_tracing_flags = tracing_flags; 
         
         if ( (tracing_flags & PRINT_TO_STR) )
            m_out_sstream.str("");
         
         if ( (tracing_flags & GRAPH_ATTR_FORMAT) )
            EOL = "";
         else if ( (tracing_flags & PRINT_TO_STR) )
            EOL = "\\n";
         else
            EOL = "\n";
      }
   }
   
   static ostream *get_curr_stream( ) { return m_out_stream; }
   
   static void init_tracing( ostream *out_stream, unsigned long flags = 0xFFFFFFFF )
   {
      set_flags( flags );
            
      m_out_stream = out_stream;
   }

   static string get_output_str() 
   {
      string str = m_out_sstream.str();
      m_out_sstream.str("");

      return str;
   }

   //static void end_tracing( ) { }

   static inline void trace( unsigned long trace_action, Tracer* tracer, long l_info = 0, const string &line_prefix = "", const string &extra_info = "" )
   {
      if ( (trace_action = (trace_action & m_tracing_flags)) != 0 )
      {
         if ( tracer == 0 )
         {
            print( line_prefix, l_info, extra_info );
         }
         else if ( (trace_action & TRACE_STATE) && tracer->get_tracer_class_id() == STATE_CLASS )
         {
            print( line_prefix, extra_info );
            tracer->trace( trace_action, l_info, line_prefix );
         }
         else if ( ((trace_action & TRACE_QUEUE) || 
                     (trace_action & TRACE_QUEUE_EL)) &&
                     tracer->get_tracer_class_id() == QUEUE_CLASS)
         {
            print( line_prefix, extra_info );
            tracer->trace( trace_action, l_info, line_prefix );
         }
         else if ( ((trace_action & TRACE_AUTOM) || 
                    (trace_action & TRACE_TABLE)) && 
                    tracer->get_tracer_class_id() == AUTOM_CLASS )
         {
            print( line_prefix, extra_info );
            tracer->trace( trace_action, l_info, line_prefix );
         }
         else if ( (trace_action & TRACE_QUEUE_EL) &&
                     tracer->get_tracer_class_id() == QUEUE_EL_CLASS)
         {
            print( line_prefix, extra_info );
            tracer->trace( trace_action, l_info, line_prefix );
        }
      }
   }

   static inline void trace( Tracer* tracer, long l_info = 0, const string &line_prefix = "", const string &extra_info = "" )
   {
      trace( m_tracing_flags, tracer, l_info, line_prefix, extra_info );
   }
};
/**********************************************************************************/

class Clonable
{
public:
   virtual Clonable* clone() = 0;
};

class QueueTypeCarrier: public Clonable
{
public:
   virtual QueueEl* new_element() = 0;
   virtual int get_type() = 0;
};

/* Interface for a channel elements */
/* provides easier processing of various channels*/
class QueueEl: public Clonable, public Tracer
{
protected:
   uchar get_tracer_class_id(){ return QUEUE_EL_CLASS; }
   virtual void  trace_fld( unsigned long trace_action, long extra_info = 0, const string &line_prefix = "" ) {};

public:
   void  trace( unsigned long trace_action, long extra_info = 0, const string &line_prefix = "" );

   virtual ~QueueEl() { };
   virtual void set( int count, ... ) = 0;
   virtual bool get( int ind, ... ) = 0;
   virtual bool equal( string fld_mask, QueueEl *el ) = 0;
   virtual bool equal_fld( int ind, ... /*var*/ ) = 0;
   virtual int fld_number() = 0;
   virtual int get_type() = 0;
};

/* Queue is container of QueueEl* type */
/* This class represents a channel*/
class Queue: public Clonable, public Tracer 
{
   friend class State;

private:
   int               m_owner;       //id of process 
   unsigned int      m_max_length; // max length of queue 
   QueueTypeCarrier* m_type_carrier;
   vector<QueueEl*>  m_items;    // each item structure depends on type of queue 
   bool		           m_is_enable;  // is it currently in use
   bool              m_rvz;

protected:
   uchar get_tracer_class_id() { return QUEUE_CLASS; }

public:
   void trace( unsigned long trace_action, long extra_info = 0, const string &line_prefix = "" );

   Queue( unsigned int max_el, QueueTypeCarrier* q_type_el )
   {
      m_type_carrier       = q_type_el;
      m_rvz                = false; 

      if ( max_el == 0)
      {
         m_max_length = 1;
         m_rvz        = true; 
      }
      else
         m_max_length = max_el;

      m_is_enable = true;
      m_owner = -1;

      m_items.reserve( m_max_length );
      m_items.resize( 0, 0 );
   }

   ~Queue()
   {
      for ( int i = 0; i < m_items.size(); i++ )
         delete m_items[i];

      delete m_type_carrier;
   }

   inline bool is_empty() { return m_items.empty();}
   inline bool is_full() { return (m_items.size() == m_max_length); }
   inline int get_len() { return (m_items.size()); } 
   inline void set_enable( bool enable ) { m_is_enable = enable; }
   inline bool get_enable( ) { return m_is_enable; }
   inline bool is_rvz() { return m_rvz; } 
   inline int  get_type() { return m_type_carrier->get_type(); }
       
   /* Push el to end of stack*/
   inline bool send( QueueEl* el )
   {
      bool bRet = true;
      if ( el == 0 )
         bRet = false;
      else if ( el->get_type() != get_type() )
         bRet = false;
      else
         m_items.push_back( el );
      
      return bRet;
   }

   /* Look at el in end of stack*/
   inline bool look( int ind, QueueEl **el )
   {
      bool bRet = true;
      if ( m_items.size() == 0 )
      {
         *el = 0;
         bRet = false;
      }
      else
        *el = m_items[ ind ];
      
      return bRet;
   }

   /* Look at el in end of stack*/
   inline bool look( QueueEl **el )
   {
      return look( m_items.size() - 1, el );
   }

   /* Pop el from stack*/
   inline bool receive( QueueEl **el ) 
   {
      bool bRet = look( el );

      if (bRet)
         m_items.pop_back();
            
      return bRet;
   }

   inline int has( string fld_mask, QueueEl *el ) 
   {
      for ( int i = m_items.size() - 1; i >= 0; i-- )
         if ( m_items[i]->equal( fld_mask, el ) )
            return i;

      return -1;
   }

   void del( int ind )
   {
      int n = m_items.size();

      if ( ind >= 0 && ind < n )
      {
         int pos = n - 1 - ind;

         if ( m_items[pos] != 0 )
            delete m_items[pos];

         for ( int i = pos; i < n - 1; i++ )
         {
             m_items[i] = m_items[i + 1];
         }

         m_items.pop_back();
      }
   }

   long calc_size()
   {
      long size = sizeof(Queue) + sizeof(*m_type_carrier);
      for (int i = 0; i < m_items.size(); i++ )
         if ( m_items[i] != 0 )
            size += sizeof( *m_items[i] );
      size += m_items.size() * sizeof(QueueEl*);
      return size;
   }

   QueueEl* create_el( ) 
   {
      QueueEl* el = m_type_carrier->new_element(); 
      return el;
   }

   Clonable* clone( );
};

/* This structure represents a global state of validation process */
/* It contains list of procedures (automatons), global variables and channels */
class StateBase: public Clonable, public Tracer 
{
public:
   // for debug purposes
   static int state_in_memory;
   static int max_state_in_memory;
   static unsigned int max_state_size;
protected:
  
   vector<CAutomBase*>  m_autom_list;   // the list of all automaton including automata for each Promela proctype
   CAutomBase*          m_ltl;          // never claim (negation LTL formula)	
          
   uchar m_flags; 
//   m_flags&1 (old m_case_found) - if true then correct statement is chosen in if..fi block (equivalent of Trail::o_pm & 1 in pan.c)
//   m_flags&2 (old m_is_timeout) - if true then timeout is enabled (equivalent of Trail::tau & 1 in pan.c)
//   m_flags&4 (old m_is_progress) - if true then this is progress state (Trail::o_pm & 4)
//   m_flags&8 (old m_is_stutter) - if true then current transition is a stutter move (Trail::tau & 128)

protected:
   uchar get_tracer_class_id() { return STATE_CLASS; }
   
   void copy_to( StateBase *st );
   void delete_automs();

public:
   
   StateBase()
   {
      long size = 0;

      m_flags = 0;

      m_ltl = 0;

      m_autom_list.reserve( 1 );
      m_autom_list.resize( 0, 0 );
   }
   
   virtual ~StateBase() { delete_automs(); }

//   bool initialize( );

   CAutomBase * get_never_claim() { return m_ltl; }

/**************************************************************/
/* Start: Automaton routines                                  */

   int add_proc( CAutomBase *autom );

   bool del_proc( int a_num );

   inline CAutomBase* get_proc( int a_num )
   {
      if ( a_num >= m_autom_list.size() || a_num < 0 )
         return 0;
      
      return m_autom_list[ a_num ];
   }

   void clean_proc_list();
   
   inline int proc_count() 
   { 
      clean_proc_list();
      return m_autom_list.size();
   }

   bool get_enabled_flag(int pid);
   bool get_progress_flag();
   int  get_curr_state(int pid);

/* End: Automaton routines                                    */
/**************************************************************/
   inline bool is_stutter() { return m_flags&8; }
   inline void set_stutter( bool stutter ) { if (stutter) m_flags |= 0x08; else m_flags &= 0xF7; }
   inline bool is_progress() { return m_flags&4; }
   inline void set_progress( bool progress ) { if (progress) m_flags |= 0x04; else m_flags &= 0xFB; }
   inline bool is_timeout() { return m_flags&2; }
   inline void set_timeout( bool timeout ) { if (timeout) m_flags |= 0x02; else m_flags &= 0xFD; }
   inline bool is_case_found() { return m_flags&1; }
   inline void set_case_found( bool case_found ) { if (case_found) m_flags |= 0x01; else m_flags &= 0xFE; }
   virtual int  get_boq() { return -1;/*m_boq;*/}
   virtual void set_boq( int boq ) { /*m_boq = boq;*/}

   
   virtual Clonable* clone( ) = 0;
   virtual int       get_state_byte_array( uchar **arr ) = 0;
   unsigned int      calc_size();

   virtual void trace( unsigned long trace_action, long extra_info = 0, const string &line_prefix = "" );
};


/* A transition table used in automaton classes*/
struct Trans
{
   short atom;	         // type of the trans: atomic or local
   int   st;	         // the next state 
   int   this_st;       // the state owner of transition
   string tp;	         // src txt of statement 

   /* <AR> t_id is equal to index of transition in m_sst_table array (it begins from 1) or 0*/
	 int   t_id;	         // transition id, unique within proc 

   int   forw;	         // index forward transition. 

   struct Trans *nxt;   // Next transition in the case if it is non-deterministic state.

   Trans() {};
   Trans( Trans* );
};

/* This is a base class for all generated automaton classes. */
/* The automaton class contains also local variables*/
class CAutomBase: public Clonable, public Tracer
{
   friend class StateBase;
public:
   static const uchar ATOM;
   static const uchar ND_ATOM;
protected:
   static const uchar L_ATOM;
   static const uchar D_ATOM;

   enum StopDirective
   {
     NOT_STOP   = 0,
     FIRST_STOP = 1,
     CONTINUE_STOP = 2
   };
   
   StateBase *m_owner;
   short   _pid;
   short m_curr_state;
   int   m_state_number; 

   uchar  m_flags;
   
   uchar m_stop;
    
//   m_flags&1 == m_test_enabled;
//   m_flags&2 == m_moved; 
//   m_flags&4 == m_first_skip_trans;
//   m_flags&8 == m_inside_atomic; 
//   m_flags&0x10 == m_else_passed;

//   BoolArr autom_bit_arr;
   
private:
   void del_transition( Trans* t )
   {
      if ( t != 0 )
      {
         /*del_transition( t->nxt );*/
         delete t;
      }
   }

   inline void imed( bool *accpstate, bool *progstate, int st1, int st2 )	/* set intermediate state */
   {	
      accpstate[st1] = (accpstate[st1] || accpstate[st2]);
      progstate[st1] = (progstate[st1] || progstate[st2]);
   }

protected:

   /* Tracing */
   uchar get_tracer_class_id() { return AUTOM_CLASS; }

   void trace_autom_base( unsigned long trace_action, long extra_info, const string &name, const string &line_prefix )
   {
      if ( (trace_action & GRAPH_ATTR_FORMAT) )
      {
         if ( extra_info >= 0 )
            print_attr( ATTR_START, name, _pid );
         else
            print_attr( ATTR_START, name, -1 );
      }
      else
      {
        if ( (trace_action & LEFTY_INPUT) == 0 )
        {
           print( line_prefix );
           if ( (trace_action & ACTION_ADD) )
           {
              print( "Add Automaton " );
           }
           else if ( (trace_action & ACTION_DEL) )
           {
              print( "Delete Automaton " );
           }
        }
  
        if ( extra_info >= 0 )
           print( "id: ", _pid, " " );
        print( "(" + name +  ") st:", m_curr_state,  EOL );
      }
   }
   
   void trace_autom_end( unsigned long trace_action, const string &name, const string &line_prefix )
   {
      if ( (trace_action & GRAPH_ATTR_FORMAT) )
      {
         print_attr( ATTR_END, name, -1 );
      }        
      else
        print( line_prefix, ATTR_END, EOL);
   }

   void trace_table( unsigned long trace_action, Trans *** table, int sst_len, bool *accpstate, bool *progstate, long extra_info, const string &line_prefix );
   void rebuild_table( Trans *** sst_table, bool *accpstate, bool *progstate, int sst_len, int &init_st );
   void find_reached( Trans **sst_table, bool *reached, int sst_len, int init_st );
   /**************************************************************************/

   /* <AR> only State or another automaton can create or delete an automaton */
   CAutomBase()
   {
      _pid = 0;
      m_state_number = 0;
      m_curr_state = 0;

      m_flags = 0;
   }

   virtual ~CAutomBase() { }
   /**************************************************************************/

   inline bool is_test_enabled() { return m_flags&1; }
  
   void del_table( Trans ***sst_table, int &table_ref, int &table_length )
   {
      if ( table_ref == 0 )
         return;

      if ( table_ref == 1 )
      {
         if ( sst_table != 0 )
         {
            Tracer::trace( TRACE_TABLE | ACTION_DEL, this, _pid, "", EOL );

            Trans **table = *sst_table;
            if ( table != 0 )
            {
               /* m_sst_table[0] == 0, always */
               for ( int i = 1; i < table_length; i++ )
               {
                  del_transition( table[i] );
                  table[i] = 0;
               }

               table_length = 0;
               delete[] table;
               *sst_table = 0;
            }
         }
      }
      table_ref--;
   }

/***************************************************************/
/*   Transition table initializing routines                    */

   bool start_table_init( Trans ***table, int ref_count, int table_len );
   
   /* <AR> in general t_id is equal to state index of transition table*/
   bool add_transition( 
               Trans ***table,
               int ref_count,
               int table_len,
               int t_id, // state id (index in the transition array)
               short atom, // atom falg
               int st, // next state
               int forw, // forward transition
               int backw, // backward transition
               string tp, // src txt of statement (comment)
               int glob, // global flag
               int tpe0, // [0] class of operation (for reduction)
               int tpe1  // [1] class of operation (for reduction)
            );
   /* <AR> this method used for initializing deadlock states */
   bool add_transition( 
               Trans ***table,
               int ref_count,
               int table_len,
               int ind, // index in the transition array 
               string tp // src txt of statement
            );
            
   bool add_transition( 
               Trans ***table,
               int ref_count,
               int table_len,
               int t_id, // state id (index in the transition array)
               short atom, // atom falg
               int st, // next state
               int forw // forward transition
            );

   void end_table_init( Trans ***table, bool *accptstate, bool *progstate, int &ref_count, int &table_len, int &init_st );
/*                                                             */
/***************************************************************/

   /* Called in constructor of an automaton class*/
   inline bool set_state_number( int len )
   {
      if ( len <= 0 )
         return false;

      m_state_number = len;

      return true;
   }
      
public:
   inline void set_inside_atomic(bool inside_atomic) {if (inside_atomic) m_flags |= 8; else m_flags &= 0xF7; }
   inline bool is_inside_atomic() {return m_flags&8; }
   
   inline void set_first_skip_trans(bool first_skip_trans) {if (first_skip_trans) m_flags |= 4; else m_flags &= 0xFB; }
   inline bool is_first_skip_trans() {return m_flags&4; }

   inline bool is_enable() { return (m_curr_state > 0); }
   inline void set_pid( int pid) { _pid = pid; }

   inline bool is_moved() { return m_flags&2; }
   inline bool set_moved(bool is_moved) { if (is_moved) m_flags |= 2; else m_flags &= 0xFD; } 
   
   inline void set_else_passed(bool else_passed) { if (else_passed) m_flags |= 0x10; else m_flags &= 0xEF; }
   inline bool is_else_passed() {return m_flags&0x10; }
   
   inline bool set_curr_state(int new_state) 
   {
      if ( new_state >= 0 || new_state < m_state_number )
      {
         m_curr_state = new_state; 
         return true;
      }

      return false;
   }
   inline int get_curr_state() { return m_curr_state; }

   inline int get_state_count() { return m_state_number; }
   inline int is_stop_found_first() { return (m_stop == (uchar)FIRST_STOP); }
   
   virtual bool is_state_accepted( int st ) = 0;
   virtual bool is_state_final( int st ) = 0;
   
   virtual bool is_state_in_progress( int st ) = 0;
   virtual int  get_autom_byte_array( uchar *arr ) = 0;

   /* returns trans_id of first transition for automaton state trans_st*/
   /* or 0 in case of error */
   virtual int get_first_trans_id( int trans_st ) = 0;

   /* in case of nondeterministic returns next transition id otherwise it returns 0 */
   /* trans_id - unique transition identifier returned by get_first_trans_id() or get_next_trans_id()*/
   virtual int get_next_trans_id( int trans_id ) = 0;

   /* This function verifies whether transition is enable. */
   /* trans_id - unique transition identifier returned by get_first_trans_id()/get_next_trans_id()*/
   /* trans_id > 0 */
   inline bool is_trans_enable( State *state, int trans_id )
   {
      bool  res = false;
      bool  nd_atom_trans;

      m_flags |= 1; //m_test_enabled = true

      /* trans_id range is verified here*/
      res = new_state( state, trans_id );

      if (!res && is_first_skip_trans())
      {
        int loc_trans_id = get_first_trans_id(get_curr_state());
        while (loc_trans_id != 0)
        {
          res = new_state( state, loc_trans_id );

          if (res)
            break;
          else
            loc_trans_id = get_next_trans_id(loc_trans_id);
        }

      }

      set_first_skip_trans(false);
      m_flags &= 0xFE; //m_test_enabled = false;

      return res;
   }

// This simple function just modifies current state to new one using specifying         
// transition.
   virtual bool new_state( State *state, int trans_id) = 0;		

protected:
/*********************************************************************************/
/* Initialization of the transition table                                        */
/* It should be generating like this                                             */
/* {                                                                             */
/*    start_table_init( len );                                                   */
/*    //...                                                                      */
/*    add_transition( ind_in_table - 1 ... );                                    */
/*    //...                                                                      */
/*    add_transition( ind_in_table - 1 ... );                                    */
/*    //...                                                                      */
/*    end_table_init();                                                          */
/*    //possible another method calling                     */
/* }                                                                             */
/*********************************************************************************/
   /* Called in constructor of an automaton class*/
   virtual bool set_trans_table( ) = 0;
};
