
#include <sstream>
#include <fstream>
#include "graphics.h"
#include "Layout.h"

using namespace GraphGraphics;

const string Storable::ASCII_INTERNAL_SEPARATOR1 = ":";
const string Storable::ASCII_INTERNAL_SEPARATOR2 = ".";
unsigned long Storable::last_offset = 0;
string Storable::m_last_error_description = "";
unsigned int  Storable::m_last_error_level = 0;

const string     Storable::ASCII_START_ID     = "{:";
const string     Storable::ASCII_END_ID       = "}:";
const string     Storable::ASCII_NAME_ID      = "Name:";
const string     Storable::ASCII_SHAPE_ID     = "Shape:";
const string     Storable::ASCII_FILLED_ID    = "Filled:";
const string     Storable::ASCII_XY_RATIO_ID  = "Shape_xy_ratio:";
const string     Storable::ASCII_COLOR_ID     = "FG_color:";
const string     Storable::ASCII_FONT_ID      = "Label_font:";
const string     Storable::ASCII_PEN_ID       = "Drawing_pen:";
const string     Storable::ASCII_ASSIGN_ID    = "Assigning:";
const string     Storable::ASCII_DEFAULT_NAME = "default";
const string     Storable::ASCII_CIRCLE_NAME  = "circle";
const string     Storable::ASCII_DOUBLE_CIRCLE_NAME = "double circle";
const string     Storable::ASCII_RECTANGULAR_NAME = "rectangular"; 
const string     Storable::ASCII_POINT_NAME   = "point";

const string Storable::ASCII_COMMENT_ID     = "Comment:";
const string Storable::ASCII_MODEL_FILE_ID  = "Model_file:";
const string Storable::ASCII_FILTER_ID      = "Filter:";
const string Storable::ASCII_DEF_FILTER_ID  = "Def_filter:";
const string Storable::ASCII_NODE_RANGE_ID  = "Range:";
const string Storable::ASCII_V_DISTANCE_ID  = "V_distance:";
const string Storable::ASCII_H_DISTANCE_ID  = "H_distance:";
const string Storable::ASCII_SPLINE_ID      = "Spline:";
const string Storable::ASCII_BG_COLOR_ID    = "BG_color:";
const string Storable::ASCII_SEL_COLOR_ID   = "Sel_color:";
const string Storable::ASCII_EDGE_COLOR_ID  = "Edge_color:";
const string Storable::ASCII_EDGE_WIDTH_ID  = "Edge_width:";

const char Storable::BIN_START_ATTR_TYPE = (char)0xFF;
const char Storable::BIN_START_NODE_TYPE = (char)0xFD;
const char Storable::BIN_END_ATTR_TYPE = (char)0xFE;

const string Storable::ASCII_INT_TYPE = "int";
const string Storable::ASCII_SHORT_TYPE = "short";
const string Storable::ASCII_BYTE_TYPE = "byte";
const string Storable::ASCII_BOOL_TYPE = "bool";  
const string Storable::ASCII_CHAN_TYPE = "chan";
const string Storable::ASCII_BIT_TYPE = "bit";
const string Storable::ASCII_TEXT_TYPE = "text";
const string Storable::ASCII_STRING_TYPE = "string";



/*************************************************************/
void Color::write_to( ostream &s_out, unsigned long write_flags )
{
  if ( write_flags & ASCII_STORAGE )
  {
    s_out << (unsigned short)m_rgb.red << " " << (unsigned short)m_rgb.green << " ";
    s_out << (unsigned short)m_rgb.blue << " " << (unsigned short)m_alpha; 
  }
  else
  {
    write_1bytes( s_out, m_rgb.red );
    write_1bytes( s_out, m_rgb.green );
    write_1bytes( s_out, m_rgb.blue );
    write_1bytes( s_out, m_alpha );
  }
}
   
bool Color::read_from( istream &s_in, unsigned long read_flags )
{
  bool res = true;
  
  if ( read_flags & ASCII_STORAGE )
  {
    unsigned short v;
    s_in >> v; m_rgb.red = (ColorComponent)v;
    s_in >> v; m_rgb.green = (ColorComponent)v;
    s_in >> v; m_rgb.blue = (ColorComponent)v;
    s_in >> v; m_alpha = (ColorComponent)v;
       
    res = !s_in.fail();
  }
  else
  {
    res = read_1bytes( s_in, *(char*)&m_rgb.red );
    if ( res )
      res = read_1bytes( s_in, *(char*)&m_rgb.green );
    if ( res )
      res = read_1bytes( s_in, *(char*)&m_rgb.blue );
    if ( res )
      res = read_1bytes( s_in, *(char*)&m_alpha );
  }
     
  return res;
}

/*************************************************************/
void DrawingPen::write_to( ostream& s_out, unsigned long write_options )
{
  if ( write_options & ASCII_STORAGE )
  {
    switch( m_line_style )
    {
    case ON_OFF_DASH:
    case DOUBLE_DASH:
      s_out << "DASHED";
      break;
       
    default:
      s_out << "SOLID";
    };
       
    s_out << " " << m_line_width;
  }
  else
  {
    char c;
    switch( m_line_style )
    {
    case ON_OFF_DASH:
    case DOUBLE_DASH:
      write_1bytes( s_out, 1 );
      break;
      
    default:
      write_1bytes( s_out, 0 );
    };
    
    c = (unsigned char)m_line_width;    
    write_1bytes( s_out, c );
  }
}
   
bool DrawingPen::read_from( istream& s_in, unsigned long read_options )
{
  bool res = true;
  if ( read_options & ASCII_STORAGE )
  {
    char buff[255];
    memset( buff, 0, sizeof( buff ) );
     
    s_in.getline( buff, sizeof( buff), ' ' );
       
    while ( buff[0] == 0 && !s_in.fail() )
      s_in.getline( buff, sizeof( buff), ' ' );
       
    if ( s_in.fail() )
      res = false;
    else
    {
      string str( buff );
         
      if ( str == "SOLID" )
        m_line_style = SOLID;
      else if ( str == "DASHED" )
        m_line_style = ON_OFF_DASH;
      else
        res = false;
    }
       
    if ( res )
    {
      s_in >> m_line_width;
      res = !s_in.fail();
    }
  }
  else
  {
    char c = 0;
       
    res = read_1bytes( s_in, c );
      
    if ( res )
    {
      m_line_style = (( c == 1 )? ON_OFF_DASH: SOLID );
      res = read_1bytes( s_in, c );  
         
      if ( res )
        m_line_width = (unsigned char)c;
    }
  }
  
  return res;
}

/*************************************************************/
void Font::write_to( ostream& s_out, unsigned long write_options )
{
  if ( write_options & ASCII_STORAGE )
  {
    bool slash = false;
    s_out << m_size << " ";
    
    if ( m_style & ITALIC )
    {
      s_out << "ITALIC";
      slash = true;
    }
    
    if ( m_style & BOLD )
    {
      if ( slash )
        s_out << "|";
      s_out << "BOLD";
      slash = true;
    }
    
    if ( !slash )
      s_out << "NORMAL";
    
    s_out << " ";
    s_out << m_face_name;
  }
  else
  {
    unsigned char style = (unsigned char)(m_style & (BOLD|ITALIC));
    write_1bytes( s_out, style );
    write_1bytes( s_out, (unsigned char)m_size );
    
    write_bin_identifier( s_out, m_face_name.length(), write_options );

    if ( m_face_name.length() > 0 )
      write_bytes( s_out, (char *)m_face_name.c_str(), m_face_name.length() );
  }
}

bool Font::read_from( istream& s_in, unsigned long read_options )
{
  bool res = true;
  if ( read_options & ASCII_STORAGE )
  {
    s_in >> m_size;
    
    char buff[255];
    memset( buff, 0, sizeof( buff ) );
     
    s_in.getline( buff, sizeof( buff), ' ' );
    while ( buff[0] == 0 && !s_in.fail() )
      s_in.getline( buff, sizeof( buff), ' ' );
       
    if ( s_in.fail() )
      res = false;
    else
    {
      string str( buff );
      string style;
      unsigned int style_pos = str.find( '|' );
      
      m_style = 0;
      
      while ( style_pos != string::npos && res )
      {
        style = str.substr( 0, style_pos );
        str = str.substr( style_pos + 1 );
        
        if ( str == "BOLD" )
          m_style |= BOLD;
        else if ( str == "ITALIC" )
          m_style |= ITALIC;
        else if ( str == "NORMAL" )
          m_style |= NONE;
        else
          res = false;          
        style_pos = str.find( '|' );
      }
      
      if ( res && !str.empty() )
      {
        if ( str == "BOLD" )
          m_style |= BOLD;
        else if ( str == "ITALIC" )
          m_style |= ITALIC;
        else if ( str == "NORMAL" )
          m_style |= NONE;
        else
          res = false;          
      }
      
      if ( res )
      {
        s_in.getline( buff, sizeof( buff) );
        m_face_name = string( buff );
      }
    }
  }
  else
  {
    char v = 0;
    long len = 0;
    res = read_1bytes( s_in, v );
    
    if ( res )
    {
      m_style = (unsigned char)v;
      res = read_1bytes( s_in, v );
    }
    
    if ( res )
    {
      m_size = (unsigned char)v;
      res = read_bin_identifier( s_in, len, read_options );
    }
    
    if ( res )
    {
      if ( len > 0 )
      {
        char *buff = new (char)[len + 1];
        buff[len] = 0;
        res = read_bytes( s_in, buff, len );
        if ( res )
          m_face_name = string( buff );
      
        delete buff;
      }
      else
        m_face_name = "";
    }
  }
  
  return res;
}

/*******************************************************************************/
/*******************************************************************************/

void GraphViewPropertyGroup::write_to( ostream& s_out, unsigned long write_options )
{
  if ( write_options & ASCII_STORAGE )
  {
    s_out << ASCII_START_ID << "\n";
    s_out << ASCII_NAME_ID << " " << (( m_type != DEFAULT )? m_name: ASCII_DEFAULT_NAME) << "\n";
    
    if ( is_shape_set() || m_type == DEFAULT )
    {
      s_out << ASCII_SHAPE_ID << " ";
      switch ( m_shape_type )
      {
      case SQUARE:
        s_out << ASCII_RECTANGULAR_NAME;
        break;
      case DOUBLE_CIRCLE:
        s_out << ASCII_DOUBLE_CIRCLE_NAME;
        break;
      case HIDEN_POINT:
        s_out << ASCII_POINT_NAME;
        break;
      default:
        s_out << ASCII_CIRCLE_NAME;
      }
      s_out << "\n";
    }
    
    if ( is_filled_set() || m_type == DEFAULT )
      s_out << ASCII_FILLED_ID << " " << (m_shape_filled? "true": "false") << "\n";

    if ( is_shape_ratio_set() || m_type == DEFAULT )
      s_out << ASCII_XY_RATIO_ID << " " << m_shape_x_ratio << " " << m_shape_y_ratio << "\n";
    
    if ( is_color_set() || m_type == DEFAULT )
    {
      s_out << ASCII_COLOR_ID << " ";
      m_color.write_to( s_out, write_options );
      s_out << "\n";
    }
    
    if ( is_font_set() || m_type == DEFAULT )
    {
      s_out << ASCII_FONT_ID << " ";
      m_text_color.write_to( s_out, write_options );
      s_out << " ";
      m_font.write_to( s_out, write_options );
      s_out << "\n";
    }
    
    if ( is_pen_set() || m_type == DEFAULT )
    {
      s_out << ASCII_PEN_ID << " ";
      m_pen.write_to( s_out,  write_options );
      s_out << "\n";
    }
    
    s_out << ASCII_ASSIGN_ID << " " << m_attr_string << "\n";
    
    s_out << "}:\n";
  }
  else
  {
    write_bin_identifier( s_out, (long)BIN_START_ID, write_options );
    write_bin_identifier( s_out, (long)BIN_NAME_ID, write_options );
    write_bin_identifier( s_out, m_name.length(), write_options );
    if ( m_name.length() > 0 )
      write_bytes( s_out, (char*)m_name.c_str(), m_name.length() );
    
    if ( is_shape_set() || m_type == DEFAULT )
    {
      write_bin_identifier( s_out, (long)BIN_SHAPE_ID, write_options );
      write_bin_identifier( s_out, 2, write_options );
      write_2bytes( s_out, (short)m_shape_type );
    }
    
    if ( is_filled_set() || m_type == DEFAULT )
    {
      write_bin_identifier( s_out, (long)BIN_FILLED_ID, write_options );
      write_bin_identifier( s_out, 1, write_options );
      write_1bytes( s_out, (char)((m_shape_filled)? 1: 0) );
    }

    if ( is_shape_ratio_set() || m_type == DEFAULT )
    {
      float x = (float)m_shape_x_ratio;
      float y = (float)m_shape_y_ratio;
      write_bin_identifier( s_out, (long)BIN_XY_RATIO_ID, write_options );
      write_bin_identifier( s_out, 8, write_options );
      write_4bytes( s_out, *((long*)&x) );
      write_4bytes( s_out, *((long*)&y) );
    }
    
    if ( is_color_set() || m_type == DEFAULT )
    {
      write_bin_identifier( s_out, BIN_COLOR_ID, write_options );
      write_bin_identifier( s_out, 4, write_options );
      m_color.write_to( s_out, write_options );
    }
    
    if ( is_font_set() || m_type == DEFAULT )
    {
      write_bin_identifier( s_out, BIN_FONT_ID, write_options );
      write_bin_identifier( s_out, 8 + m_font.get_face_name().length(), write_options );
      m_text_color.write_to( s_out, write_options );
      m_font.write_to( s_out, write_options );
    }
    
    if ( is_pen_set() || m_type == DEFAULT )
    {
      write_bin_identifier( s_out, (long)BIN_PEN_ID, write_options );
      write_bin_identifier( s_out, 2, write_options );
      m_pen.write_to( s_out, write_options );
    }
    
    write_bin_identifier( s_out, (long)BIN_ASSIGN_ID, write_options );
    write_bin_identifier( s_out, m_attr_string.length(), write_options );
    if ( m_attr_string.length() > 0 )
      write_bytes( s_out, (char*)m_attr_string.c_str(), m_attr_string.length() );
    
    write_bin_identifier( s_out, (long)BIN_INTERNAL_SEPARATOR2, write_options );//End group   
  }
}

bool GraphViewPropertyGroup::read_from( istream& s_in, unsigned long read_options )
{
  bool res = true;
  m_name = "";
  m_attr_string = "";
  m_is_shape_set = false;
  m_is_filled_set = false;
  m_is_shape_ratio_set = false;
  m_is_font_set = false;
  m_is_pen_set = false;
  m_is_color_set = false;

  if ( read_options & ASCII_STORAGE )
  {
    string str;
    string str_id;
    //the ASCII_START_ID is already read
    res = read_min_ASCII_block( s_in, str, str_id ); // get minimal block
    
    if ( res )
    {
      //res = read_min_ASCII_block( s_in, str, str_id ); // get minimal block
      if ( str_id != ASCII_NAME_ID )
        res = false;
      else
        m_name = str;
    }
    
    while ( res && !s_in.fail() && !s_in.eof() )
    {
      res = read_min_ASCII_block( s_in, str, str_id ); // get minimal block
      
      if ( !res )
        break;
      
      if ( (str.empty() && str_id.empty()) || str_id == ASCII_START_ID )
      {
        res = false;
        break;
      }
      
      if ( str_id == ASCII_END_ID )
        break;
            
      if ( str_id == ASCII_SHAPE_ID )
      {
        if ( str == ASCII_RECTANGULAR_NAME )
          m_shape_type = SQUARE;
        else if ( str == ASCII_DOUBLE_CIRCLE_NAME )
          m_shape_type = DOUBLE_CIRCLE;
        else if ( str == ASCII_POINT_NAME )
          m_shape_type = HIDEN_POINT;
        else
          m_shape_type = CIRCLE;
        
        m_is_shape_set = true;
      }
      else if ( str_id == ASCII_FILLED_ID )
      {
        m_shape_filled = (str == "true");
        m_is_filled_set = true;
      }
      else if ( str_id == ASCII_XY_RATIO_ID ) 
      {
        istringstream str_stream( str );
        str_stream >> m_shape_x_ratio;
        res = !str_stream.fail();
        
        if ( res )
        {
          str_stream >> m_shape_y_ratio;
          res = !str_stream.fail();
        }
        m_is_shape_ratio_set = true;
      }
      else if ( str_id == ASCII_COLOR_ID )
      {
        istringstream str_stream( str );
        res = m_color.read_from( str_stream, read_options );
        m_is_color_set = true;
      }
      else if ( str_id == ASCII_FONT_ID )
      {
        istringstream str_stream( str );
        res = m_text_color.read_from( str_stream, read_options );
        
        if ( res )
          res = m_font.read_from( str_stream, read_options );
        
        m_is_font_set = true;
      }
      else if ( str_id == ASCII_PEN_ID )
      {
        istringstream str_stream( str );
        res = m_pen.read_from( str_stream, read_options );
        m_is_pen_set = true;
      }
      else if ( str_id == ASCII_ASSIGN_ID )
      {
        m_attr_string = str;
      }
      else
        res = false;
    }
  }
  else
  {
    long id, size;
    short svalue;
    char  c;
    char *buff;
    //the BIN_START_ID is already read
    res = read_bin_identifier( s_in, id, read_options );
    
    if ( res )
    {
      //res = read_2bytes( s_in, id );
      //read name  
      if ( id != (long)BIN_NAME_ID )
        res = false;
        
      if ( res )
      {
        res = read_bin_identifier( s_in, id, read_options );
        if ( res )
        {
          buff = new (char)[(unsigned long)id + 1];
          buff[(unsigned long)id] = 0;
          res = read_bytes( s_in, buff, (unsigned long)id );
            
          if ( res )
            m_name = string( buff );
            
          delete buff;
        }
      }
    }
    
    res = read_bin_identifier( s_in, id, read_options );
    
    while ( res && (id != (long)BIN_INTERNAL_SEPARATOR2) && !s_in.fail() && !s_in.eof() )
    {
      res = read_bin_identifier( s_in, size, read_options );
      
      if ( res )
        switch ( id )
        {
        case BIN_SHAPE_ID:
          if ( size != 2 )
            res = false;
          else
          {
            res = read_2bytes( s_in, svalue );
            m_shape_type = svalue;
            m_is_shape_set = true;
          }
          break;
        
        case BIN_FILLED_ID:
          if ( size != 1 )
            res = false;
          else
          {
            res = read_1bytes( s_in, c );
            m_shape_filled = ( c != 0 );
            m_is_filled_set = true;
          }
          break;
        
        case BIN_XY_RATIO_ID:
          if ( size != 8 )
            res = false;
          else
          {
            float x = 1.0;
            float y = 1.0;
            res = read_4bytes( s_in, *((long*)&x) );
            if ( res )
              res = read_4bytes( s_in, *((long*)&y) );
            m_shape_x_ratio = x;
            m_shape_y_ratio = y;
            m_is_shape_ratio_set = true;
          }
          break;
        
        case BIN_COLOR_ID:
          if ( size != 4 )
            res = false;
          {
            res = m_color.read_from( s_in, read_options );
            m_is_color_set = true;
          }
          break;
        
        case BIN_FONT_ID:
          if ( size < 8 )
            res = false;
          else
          {
            res = m_text_color.read_from( s_in, read_options );
            if ( res )
              res = m_font.read_from( s_in, read_options );
            m_is_font_set = true;
          }
          break;
        
        case BIN_PEN_ID:
          if ( size != 2 )
            res = false;
          else
          {
            res = m_pen.read_from( s_in, read_options );
            m_is_pen_set = true;
          }
          break;
        
        case BIN_ASSIGN_ID:
          {
            buff = new (char)[(unsigned long)size + 1];
            buff[(unsigned short)size] = 0;
            res = read_bytes( s_in, buff, (unsigned long)size );
            if ( res )
              m_attr_string = string( buff );
            delete buff;
            
            NodeAttributeList attr_list;
            res = GraphAttributes::parse_string( m_attr_string, attr_list );
            attr_list.clear();
          }
          break;
        
        default:
          { //skip unknown field
            buff = new (char)[(unsigned long)size];
            res = read_bytes( s_in, buff, (unsigned long)size );
                
            delete buff;
          }
        }
        
        if ( res )
          res = read_bin_identifier( s_in, id, read_options );
    }
  }
  
  if ( res )
  {
    if ( m_name != ASCII_DEFAULT_NAME )
      res = !m_attr_string.empty();
    else
    {
      res = m_is_shape_set && m_is_shape_ratio_set && m_is_filled_set &&
            m_is_color_set && m_is_font_set && m_is_pen_set;
      m_attr_string = "{ }";
      m_type = DEFAULT;
    }
  }
  
  return res;
}

/*************************************************************/
void NodeAttributeList::write_bin_to( ostream& s_out, GraphAttrNameStorage *storage, unsigned long write_options )
{
  NodeAttribute *attr;
  NAPvector::iterator p = NAPvector::begin();
  attr = *p;
  long id;
  char t_id;
    
  if ( attr->type == NodeAttribute::ATTR_START )
  {
    id = (storage->get_name_id( attr->name ) + 1);
    Storable::write_bin_identifier( s_out, id, write_options );
    Storable::write_1bytes( s_out, Storable::BIN_START_NODE_TYPE );
    Storable::write_4bytes( s_out, attr->value->get_integer() );
    p++;
  }
  
  for ( ; p != NAPvector::end(); p++ )
  {
    attr = *p;
    id = (storage->get_name_id( attr->name ) + 1);
    Storable::write_bin_identifier( s_out, id, write_options );
    
    if ( attr->type == NodeAttribute::ATTR_START )
      Storable::write_1bytes( s_out, Storable::BIN_START_ATTR_TYPE );
    else if ( attr->type == NodeAttribute::ATTR_END )
      Storable::write_1bytes( s_out, Storable::BIN_END_ATTR_TYPE );
    else
    {
      t_id = (char)(storage->get_type_id( attr->type ) + 1);
      Storable::write_1bytes( s_out, t_id );
      
      if ( attr->type == Storable::ASCII_TEXT_TYPE ||
           attr->type == Storable::ASCII_STRING_TYPE )
      {
        string str_value = attr->value->get_string();
        if ( str_value.empty() )
          Storable::write_bin_identifier( s_out, 0, write_options );
        else
        {
          Storable::write_bin_identifier( s_out, str_value.size(), write_options );
          Storable::write_bytes( s_out, (char *)str_value.c_str(), (unsigned short)str_value.size() );
        }
      }
      else if ( attr->type == Storable::ASCII_BYTE_TYPE || 
           attr->type == Storable::ASCII_BOOL_TYPE || 
           attr->type == Storable::ASCII_BIT_TYPE )
        Storable::write_1bytes( s_out, (unsigned char)attr->value->get_integer() );
      else if ( attr->type == Storable::ASCII_INT_TYPE || attr->type == GraphAttrNameStorage::EMPTY_NAME )
        Storable::write_4bytes( s_out, attr->value->get_integer() );
      else if ( attr->type == Storable::ASCII_SHORT_TYPE || attr->type == Storable::ASCII_CHAN_TYPE )
        Storable::write_2bytes( s_out, (short)attr->value->get_integer() );
      else
      {
        Storable::write_bin_identifier( s_out, 4, write_options ); //bytes count
        Storable::write_4bytes( s_out, attr->value->get_integer() );
      }
    }
  }
  
  Storable::write_bin_identifier( s_out, Storable::BIN_INTERNAL_SEPARATOR2, write_options );
}

bool NodeAttributeList::read_bin_from( istream& s_in, GraphAttrNameStorage *storage, unsigned long read_options )
{
  bool res = true;
  long id;
  char  t_id;
  NodeAttribute *attr;
  bool  add;
  
  res = Storable::read_bin_identifier( s_in, id, read_options );
  
  while ( res && !s_in.eof() && !s_in.fail() )
  {
    if ( id < 0 )
    {
      if ( id != (long)Storable::BIN_INTERNAL_SEPARATOR2 )
        res = false;
      break;
    }
    
    attr = new NodeAttribute();
    attr->name = storage->get_id_name( id - 1 );
    res = Storable::read_1bytes( s_in, t_id );
    add = true;
      
    if ( res )
    {  
      if ( t_id == Storable::BIN_START_NODE_TYPE || t_id == Storable::BIN_START_ATTR_TYPE )
      {
        long value;
        attr->type = NodeAttribute::ATTR_START;
        if ( t_id == Storable::BIN_START_NODE_TYPE )
        {
          res = Storable::read_4bytes( s_in, value );
          attr->value_op = AttributeValue::EQUAL;
          attr->value = new AttributeIntegerValue( value );
        }
        else
        {
          attr->value_op = AttributeValue::NONE;
          attr->value = new AttributeIntegerValue( -1 );
        }
      }
      else if ( t_id == Storable::BIN_END_ATTR_TYPE )
      {
        attr->type = NodeAttribute::ATTR_END;
        attr->value_op = AttributeValue::NONE;
        attr->value = new AttributeIntegerValue( -1 );
      }
      else
      {
        attr->type = storage->get_id_type( t_id - 1 );
        attr->value_op = AttributeValue::EQUAL;
        if ( attr->type == GraphAttrNameStorage::EMPTY_NAME )
        {
          attr->type = Storable::ASCII_INT_TYPE;
          storage->add_type( Storable::ASCII_INT_TYPE );
        }
        
        if ( attr->type == Storable::ASCII_TEXT_TYPE || 
             attr->type == Storable::ASCII_STRING_TYPE )
        {
          res = Storable::read_bin_identifier( s_in, id, read_options );
          if ( res )
          {
            if ( id > 0 )
            {
              char *buff = new char[ (unsigned long)id + 1 ];
              res = Storable::read_bytes( s_in, buff, (unsigned long)id );
              buff[ (unsigned long)id ] = 0;
              
              if ( res )
                attr->value = new AttributeStringValue( string( buff ) );
              delete buff;
            }
            else
              attr->value = new AttributeStringValue( "" );
          }
        }
        else if ( attr->type == Storable::ASCII_INT_TYPE )
        {
          res = Storable::read_4bytes( s_in, id );
          attr->value = new AttributeIntegerValue( id );
        }
        else if ( attr->type == Storable::ASCII_SHORT_TYPE || attr->type == Storable::ASCII_CHAN_TYPE )
        {
          short svalue;
          res = Storable::read_2bytes( s_in, svalue );
          attr->value = new AttributeIntegerValue( svalue );
        }
        else if ( attr->type == Storable::ASCII_BYTE_TYPE || attr->type == Storable::ASCII_BOOL_TYPE ||
                  attr->type == Storable::ASCII_BIT_TYPE )
        {
          res = Storable::read_1bytes( s_in, t_id );
          attr->value = new AttributeIntegerValue( (unsigned char)t_id );
        }
        else
        {
          res = Storable::read_bin_identifier( s_in, id, read_options );
          if ( id < 0 )
            res = false;
          if ( res && id > 0 )
          {
            char *buff = new (char)[id];
            res = Storable::read_bytes( s_in, buff, id );
            delete buff;
          }
          add = false;
        }
      }
    }
    
    if ( res )
    {
      if ( add )
        push_back( attr );
      else
        delete attr;
      
      res = Storable::read_bin_identifier( s_in, id, read_options );
    }
  }
  
  if ( s_in.eof() && s_in.fail() )
    res = false;
  
  
  return res;
}

/*************************************************************/
void GraphAttrNameStorage::write_to( ostream& s_out, unsigned long write_options )
{
  if ( write_options & ASCII_STORAGE )
  {
    s_out << GraphAttributes::make_string( m_search_format ) << "\n";
    s_out << ":\n";
  }
  else
  {
    // store name set
    unsigned int i = 0;
    for ( ; i < m_names.size(); i++ )
    {
      write_bin_identifier( s_out, m_names[i].length(), write_options);
      write_bytes( s_out, (char *)m_names[i].c_str(), m_names[i].length() );
    }
    write_bin_identifier( s_out, (long)BIN_INTERNAL_SEPARATOR2, write_options);
        
    for ( i = 0; i < m_types.size(); i++ )
    {
      write_bin_identifier( s_out, m_types[i].length(), write_options);
      write_bytes( s_out, (char *)m_types[i].c_str(), m_types[i].length() );
    }
    
    write_bin_identifier( s_out, (long)BIN_INTERNAL_SEPARATOR1, write_options);
    //store attributes template
    m_search_format.write_bin_to( s_out, this, write_options );
    
    write_bin_identifier( s_out, (long)BIN_INTERNAL_SEPARATOR1, write_options);
  }
}

bool GraphAttrNameStorage::read_from( istream& s_in, unsigned long read_options )
{
  bool res = true;
  
  clear_name_set( );
      
  if ( read_options & ASCII_STORAGE )
  {
    string data;
    string data_id;
    res = read_min_ASCII_block( s_in, data );
        
    if ( res )
    {
      res = init_name_set( data );
      if ( res )
      {
        res = read_min_ASCII_block( s_in, data, data_id );
      
        while ( data_id != ASCII_INTERNAL_SEPARATOR1 && res )
          res = read_min_ASCII_block( s_in, data, data_id );
        
        if ( !data.empty() )
          res = false;
      }
    }
    
    if ( !res )
      set_error( "Cannot read attributes template" );
  }
  else
  {
    long offset = 0;
    unsigned long curr_offset = 0;
    long    v;
    char     *buff;
    bool     read_name_set = true;
    
    res = read_bin_identifier( s_in, v, read_options );
    
    if ( v < 0 )
    {
      if ( v == (long)BIN_NAME_SET_OFFSET )
      {
        res = read_4bytes( s_in, offset );
        if ( offset <= s_in.tellg() )
          res = false;
      }
      else if ( v == (long)BIN_INTERNAL_SEPARATOR1 )
        read_name_set = false;
      else
        res = false;
    }
    
    if ( res && read_name_set )
    {
      curr_offset = s_in.tellg();
      if ( offset != 0 )
      {
        s_in.seekg( (unsigned long)offset );
        if ( s_in.eof() || s_in.fail() )
          res = false;
        
        if ( res )
          res = read_bin_identifier( s_in, v, read_options );
      }
          
      while( res && v > 0 )
      {
        buff = new char[v + 1];
        res = read_bytes( s_in, buff, v );
        buff[v] = 0;
            
        if ( res )
        {
          add_name( string(buff) );
          res = read_bin_identifier( s_in, v, read_options );
        }
        delete buff;
      }
          
      if ( v != (long)BIN_INTERNAL_SEPARATOR2 )
        res = false;
          
      if ( res )
      {
        res = read_bin_identifier( s_in, v, read_options );
          
        while( res && v > 0 )
        {
          buff = new char[v + 1];
          res = read_bytes( s_in, buff, v );
          buff[v] = 0;
              
          if ( res )
          {
            add_type( string(buff) );
            res = read_bin_identifier( s_in, v, read_options );
          }
          delete buff;
        } 
      } 
    }
    
    if ( res && read_name_set )
    {
      if ( offset != 0 )
      {
        last_offset = s_in.tellg();
        s_in.seekg( curr_offset );
      }
      else
      {
        last_offset = 0;
        if ( v != (long)BIN_INTERNAL_SEPARATOR1 )
          res = false;
      }
    }
    
    if ( !res )
      set_error( "Cannot read graph names set" );
    
    add_type( "int" );
    m_string_types.insert( add_type("string") );
    m_string_types.insert( add_type("text") );
    
    //read tamplates
    if ( res )
    {
      res = m_search_format.read_bin_from( s_in, this, read_options );
      if ( !res )
        set_error( "Cannot read graph attributes templates" );
    }
    
    if ( res )
    {
      NodeAttributeList attr_list;
      unsigned long  attr_offset = s_in.tellg();
      
      res = read_bin_identifier( s_in, v, read_options );
      while ( v != (long)BIN_INTERNAL_SEPARATOR1 && res )
      {
        if ( v < 0 )
          res = false;
        else
        {
          s_in.seekg( attr_offset );
          res = attr_list.read_bin_from( s_in, this, read_options );
        }
        
        if ( res )
        {
          attr_offset = s_in.tellg();
          res = read_bin_identifier( s_in, v, read_options );
        }
      }
      
      if ( !res )
        set_error( "Cannot read graph file" );
    }
  }
  
  return res;
}

/*************************************************************/
void Graph::write_to( ostream& s_out, unsigned long write_options )
{
  NodeLinkList* links = 0;
  NodeAttributeList attr_list;
  unsigned int i;
  
  if ( write_options & ASCII_STORAGE )
  {
    string attr_string;
    m_attr_name_storage->write_to( s_out, write_options );
    
    //save nodes
    for ( i = 0; i < m_nodes.size(); i++ )
    {
      m_nodes[i]->get_attributes()->get_parsed_attributes( attr_list );
      delete attr_list[0]->value;
      attr_list[0]->value = new AttributeIntegerValue( i + 1 );
      attr_string = GraphAttributes::make_string( attr_list );
      s_out << attr_string << "\n";
    }
    
    //save edges
    for ( i = 0; i < m_nodes.size(); i++ )
    {
      links = m_nodes[i]->get_links();
      for ( NodeLinkList::iterator p = links->begin(); p != links->end(); p++ )
      {
        s_out << i + 1 << "-> " << (*p + 1) << "\n";
      }
    }
    s_out << "\n"; //block separator
  }
  else
  {
    m_attr_name_storage->write_to( s_out, write_options );
    
    //save nodes
    for ( i = 0; i < m_nodes.size(); i++ )
    {
      m_nodes[i]->get_attributes()->get_parsed_attributes( attr_list );
      delete attr_list[0]->value;
      attr_list[0]->value = new AttributeIntegerValue( i + 1 );
      attr_list.write_bin_to( s_out, m_attr_name_storage, write_options );
    }
    
    //save edges
    for ( i = 0; i < m_nodes.size(); i++ )
    {
      links = m_nodes[i]->get_links();
      for ( NodeLinkList::iterator p = links->begin(); p != links->end(); p++ )
      {
        write_bin_identifier( s_out, (long)BIN_EDGE_ID, write_options);
        write_4bytes( s_out, (long)(i + 1) );
        write_4bytes( s_out, (long)(*p + 1) );
      }
    }
    
    write_bin_identifier( s_out, (long)BIN_BLOCK_SEPARATOR, write_options);
  }
}

bool Graph::read_from( istream& s_in, unsigned long read_options )
{
  bool res = 0;
  unsigned  int ind = 0;
  vector<GraphAttributes*> node_attributes; //sorted by node id
  vector<NodeLinkList> node_edges; //sorted by node id
  GraphAttributes *attr = 0;
  unsigned long curr_offset = s_in.tellg();
  string data;
  string data_id;
    
  node_attributes.resize(0);
  node_edges.resize(0);
	
  if ( read_options & ASCII_STORAGE )
  {
    char c;
    bool is_attr_correct = false;
        	        
    res = read_min_ASCII_block( s_in, data );
    
    if ( data.empty() || !res ) //end of block - graph absent or fail
    {
      if ( !res )
        set_error( "Cannot load graph file" );
      return res;
    }
    
    s_in.seekg( curr_offset );

    remove_all_nodes( );
    
    //read storage/templates
    res = m_attr_name_storage->read_from( s_in, read_options );
		
    if ( res )
    {
      while ( !s_in.eof() && !s_in.fail() && res )
      {
        res = read_min_ASCII_block( s_in, data, data_id );
        
        if ( res )
        {
          if ( !data_id.empty() && !data.empty() )// node
          {  
            istringstream str_in( data_id);
          
            str_in >> ind;
            
            if ( str_in.fail() )
              res = false;
            else
            {
              if ( node_attributes.size() < ind )
                node_attributes.resize( ind , 0 );
              else
              {
                if ( node_attributes[ind - 1] != 0 )
                {
                  delete node_attributes[ind - 1];
                  node_attributes[ind - 1] = 0;
                }
              }
              
              attr = new GraphAttributes( m_attr_name_storage );
              is_attr_correct = attr->set_attribute_string( data );
             
              if ( !is_attr_correct )
              {	
                delete attr;
                res = false;
              }
              else
                node_attributes[ind - 1] = attr;
            }
            
          }
          else if ( !data.empty() )//edge
          {
            istringstream str_in( data );
  
            str_in >> ind;
            
            if ( str_in.fail() )
              res = false;
            else
            {
              str_in.get( c );
              
              if ( c != '-' )
                res = false;
              else
              {
                str_in.get( c );
                if ( c != '>')
                  res = false;
                else
                {
                  unsigned int edge_end = 0;
                  str_in >> edge_end;
                    
                  if ( str_in.fail() || edge_end <= 0 || node_attributes[ind - 1] == 0 )
                    res = false;
                  else
                  {
                    if ( edge_end > node_attributes.size() || node_attributes[edge_end - 1] == 0 )
                      res = false;
                    else 
                    {
                      if ( node_edges.size() < ind )
                        node_edges.resize( ind, NodeLinkList(0) );
                      
                      node_edges[ind - 1].push_back( edge_end - 1 );
                    }
                  }
                }
              }
            }
          }
          else 
          {  
            if ( !data_id.empty() ) //all empty - end of block
              res = false;
            break;
          }
        }
      }
      
      if ( !res )
         set_error( "Cannot read graph information" );
    }
  }
  else
  {
    long id;
    NodeAttributeList attr_list;
    res = read_bin_identifier( s_in, id, read_options );
    
    if ( id == (long)BIN_BLOCK_SEPARATOR || !res )
    {
      if ( !res )
        set_error( "Cannot load graph file" );
      return res;
    }
    
    s_in.seekg( curr_offset );
    
    remove_all_nodes( );
    
    //read storage/templates
    res = m_attr_name_storage->read_from( s_in, read_options );
		
    if ( res )
    {
      while ( !s_in.eof() && !s_in.fail() && res )
      {
        res = read_bin_identifier( s_in, id, read_options );
        
        if ( id < 0 )
        {
          if ( id == (long)BIN_BLOCK_SEPARATOR )//name set is first
          {
            if ( last_offset != 0 )
              res = false;
            break;
          }
          else if ( id == (long)BIN_INTERNAL_SEPARATOR1 )//name set is last
          {
            if ( last_offset != 0 )
            {
              s_in.seekg( last_offset );
              last_offset = 0;
            }
            else
              res = false;
            break;
          }
          else if ( id == (long)BIN_EDGE_ID )//edge
          {
            NodeIndex edge_start = 0;
            NodeIndex edge_end = 0;
            res = read_4bytes( s_in, edge_start );
            
            if ( res )
              res = read_4bytes( s_in, edge_end );
            
            if ( res )
            {            
              if ( node_attributes[edge_start - 1] == 0 )
                res = false;
              else
              {
                if ( edge_end > (NodeIndex)node_attributes.size() || node_attributes[edge_end - 1] == 0 )
                  res = false;
                else 
                {
                  bool duple = false;
                  if ( (NodeIndex)node_edges.size() < edge_start )
                    node_edges.resize( edge_start, NodeLinkList(0) );
                        
                  for ( unsigned int i = 0; i < node_edges[edge_start - 1].size() && !duple; i++ )
                    duple = (node_edges[edge_start - 1][i] == edge_end - 1);
                  
                  if ( !duple )
                    node_edges[edge_start - 1].push_back( edge_end - 1 );
                }
              }
            }
          }
          else
            res = false;
        }
        else //node
        {
          char c;
          NodeAttribute *n_attr = 0;
          res = read_1bytes( s_in, c );
          
          if ( c != Storable::BIN_START_NODE_TYPE )
            res = false;
          else
          {
            long value;
            n_attr = new NodeAttribute();
            n_attr->name = m_attr_name_storage->get_id_name( id - 1 );
            n_attr->type = NodeAttribute::ATTR_START;
            res = read_4bytes( s_in, value );
            n_attr->value_op = AttributeValue::EQUAL;
            n_attr->value = new AttributeIntegerValue( value );
            ind = value;
            
            if ( !res )
              delete n_attr;
            else
            {
              attr_list.push_back( n_attr );
              res = attr_list.read_bin_from( s_in, m_attr_name_storage, read_options ); 
              
              if ( res )
              {
                attr = new GraphAttributes( m_attr_name_storage );
                res = attr->set_attribute_list( attr_list );
                
                if ( !res )
                  delete attr;
                else
                {
                  if ( node_attributes.size() < ind )
                    node_attributes.resize( ind , 0 );
                  else
                  {
                    if ( node_attributes[ind - 1] != 0 )
                    {
                      delete node_attributes[ind - 1];
                      node_attributes[ind - 1] = 0;
                    }
                  }
                  
                  node_attributes[ind - 1] = attr;
                }
              }  
            }
          }
          attr_list.clear();
        }
      }
      
      if ( !res )
        set_error( "Cannot read graph information" );
    }
  }
	
  if ( s_in.eof() || s_in.fail() )	
    res = false;
		
  if ( res )
  {
    unsigned int i = 0;
    unsigned int attr_ind = 0;
    vector<unsigned int> real_indx;
    real_indx.resize( node_attributes.size(), (unsigned int)-1 );
      
    for ( i = 0; i < node_attributes.size(); i++ )
    {
      if ( node_attributes[i] != 0 )
      {
        real_indx[i] = attr_ind;
        attr_ind++;
      }
    }        
      
    for ( ind = 0; ind < node_attributes.size() && res; ind++ )
    {
      if ( node_attributes[ind] == 0 || real_indx[ind] == (unsigned int)-1 )
        continue;
        
      node_attributes[ind]->set_highlevel_index( real_indx[ind] );
      add_node( node_attributes[ind] );
        
      if ( ind < node_edges.size() )
        for ( i = 0; i < node_edges[ind].size(); i++ )
        {
           if ( node_edges[ind][i] >= (long)node_attributes.size() || real_indx[node_edges[ind][i]] == (unsigned int)-1 )
           {
              res = false;
              break;
           }
           add_edge( real_indx[ind], real_indx[node_edges[ind][i]] );
        }
    }
    
  }
    
  if ( !res )
  {
    for ( ind = 0; ind < node_attributes.size(); ind++ )
      if ( node_attributes[ind] != 0 )
         delete node_attributes[ind];
      
    m_attr_name_storage->clear_name_set();
    remove_all_nodes(  );     
    
    set_error_level( Storable::CRITICAL );
  }      
    
  node_attributes.clear();
    
  for (ind = 0; ind < node_edges.size(); ind++ )
    node_edges[ind].clear();
    
  node_edges.clear();
  
  return res;
}

/*************************************************************/
void GraphView::write_to( ostream& s_out, unsigned long write_options )
{
  GraphViewPropertyGroup group;
  string name;
    
  if ( write_options & ASCII_STORAGE )
  {
    //graph vis. block
    //filters
    for( name = m_filters->get_first_template_name(); name != AttrStringTemplateStorage::INVALID_TEMPLATE_NAME; 
               name =  m_filters->get_next_template_name( name) )
    {
      s_out << ASCII_FILTER_ID << name << "\n";
      s_out << m_filters->get_template( name ) << "\n";
    }
    name = m_filters->get_current_template_name();
    if ( name != AttrStringTemplateStorage::INVALID_TEMPLATE_NAME && 
         m_filters->is_template_name_exist( name ) )
      s_out << ASCII_DEF_FILTER_ID << name << "\n";
    
    s_out << ASCII_INTERNAL_SEPARATOR1 << "\n";
    //rendering options
    s_out << ASCII_NODE_RANGE_ID << " " << m_radius << "\n";
    s_out << ASCII_V_DISTANCE_ID << " " << m_y_separation << "\n";
    s_out << ASCII_H_DISTANCE_ID << " " << m_x_separation << "\n";
    s_out << ASCII_SPLINE_ID << " " << (m_spline_interpltn? "true": "false") << "\n";
    s_out << ASCII_INTERNAL_SEPARATOR1 << "\n";
    
    //general drawing options
    s_out << ASCII_BG_COLOR_ID << " ";
    m_view_groups.get_background_color().write_to( s_out, write_options ); s_out << "\n";
    s_out << ASCII_SEL_COLOR_ID << " ";
    m_view_groups.get_selected_node_color().write_to( s_out, write_options ); s_out << "\n";
    s_out << ASCII_EDGE_COLOR_ID << " ";
    m_view_groups.get_edge_color().write_to( s_out, write_options ); s_out << "\n";
    s_out << ASCII_EDGE_WIDTH_ID << " " << m_view_groups.get_edge_pen().get_line_width() << "\n";
    s_out << ASCII_INTERNAL_SEPARATOR1 << "\n";
  }
  else
  {
    //graph vis. block
    //filters
    string attr_template;
    for( name = m_filters->get_first_template_name(); name != AttrStringTemplateStorage::INVALID_TEMPLATE_NAME; 
               name =  m_filters->get_next_template_name( name) )
    {
      write_bin_identifier( s_out, (long)BIN_FILTER_ID, write_options );
      write_bin_identifier( s_out, name.length(), write_options );
      write_bytes( s_out, (char*) name.c_str(), name.length() );
      attr_template = m_filters->get_template( name );
      
      write_bin_identifier( s_out, attr_template.length(), write_options );
            
      if ( attr_template.length() > 0 )
        write_bytes( s_out, (char*) attr_template.c_str(), attr_template.length() );
      
    }
    name = m_filters->get_current_template_name();
    if ( name != AttrStringTemplateStorage::INVALID_TEMPLATE_NAME && 
         m_filters->is_template_name_exist( name ) )
    {
      write_bin_identifier( s_out, (long)BIN_DEF_FILTER_ID, write_options );
      write_bin_identifier( s_out, name.length(), write_options );
      write_bytes( s_out, (char*) name.c_str(), name.length() );
    }
    
    write_bin_identifier( s_out, (long)BIN_INTERNAL_SEPARATOR1, write_options );
            
    //rendering options
    write_bin_identifier( s_out, (long)BIN_NODE_RANGE_ID, write_options );
    write_bin_identifier( s_out, 4, write_options );
    write_4bytes( s_out, m_radius );
    write_bin_identifier( s_out, (long)BIN_V_DISTANCE_ID, write_options );
    write_bin_identifier( s_out, 4, write_options );
    write_4bytes( s_out, m_y_separation );
    write_bin_identifier( s_out, (long)BIN_H_DISTANCE_ID, write_options );
    write_bin_identifier( s_out, 4, write_options );
    write_4bytes( s_out, m_x_separation );
    write_bin_identifier( s_out, (long)BIN_SPLINE_ID, write_options );
    write_bin_identifier( s_out, 1, write_options );
    if ( m_spline_interpltn )
      write_1bytes( s_out, 1 );
    else
      write_1bytes( s_out, 0 );
    write_bin_identifier( s_out, (long)BIN_INTERNAL_SEPARATOR1, write_options );
        
    //general drawing options
    write_bin_identifier( s_out, (long)BIN_BG_COLOR_ID, write_options );
    write_bin_identifier( s_out, 4, write_options );
    m_view_groups.get_background_color().write_to( s_out, write_options );
    write_bin_identifier( s_out, (long)BIN_SEL_COLOR_ID, write_options );
    write_bin_identifier( s_out, 4, write_options );
    m_view_groups.get_selected_node_color().write_to( s_out, write_options );
    write_bin_identifier( s_out, (long)BIN_EDGE_COLOR_ID, write_options );
    write_bin_identifier( s_out, 4, write_options );
    m_view_groups.get_edge_color().write_to( s_out, write_options );
    write_bin_identifier( s_out, (long)BIN_EDGE_WIDTH_ID, write_options );
    write_bin_identifier( s_out, 1, write_options );
    char c = (unsigned char)(m_view_groups.get_edge_pen().get_line_width());
    write_1bytes( s_out, c );
    
    write_bin_identifier( s_out, (long)BIN_INTERNAL_SEPARATOR1, write_options );
  }
  
  //groups
  group = m_view_groups.get_view_group( GraphViewPropertyGroup::DEFAULT );
  group.write_to( s_out, write_options );
    
  for ( StyleFlags type = m_view_groups.get_first_view_group_id(); type != GraphViewPropertyGroup::DEFAULT; type = m_view_groups.get_next_view_group_id( type ) )
  {
    group = m_view_groups.get_view_group( type );
    group.write_to( s_out, write_options);
  }
  
  if ( write_options & ASCII_STORAGE )  
  {
    if ( !m_label_templates.empty() )
    {
      s_out << ":\n";
      for ( unsigned int i = 0; i < m_label_templates.size(); i++ )
      {
        s_out << "{:\n";
        s_out << m_label_templates[i].node_attributes << "\n";
        s_out << m_label_templates[i].node_label_template << "\n";
        s_out << "}:\n";
      }
    }
    s_out << "\n";
  }
  else
  {
    if ( !m_label_templates.empty() )
    {
      write_bin_identifier( s_out, (long)BIN_INTERNAL_SEPARATOR1, write_options );
      for ( unsigned int i = 0; i < m_label_templates.size(); i++ )
      {
        write_bin_identifier( s_out, (long)BIN_START_ID, write_options ); // start id
        write_bin_identifier( s_out, m_label_templates[i].node_attributes.size(), write_options );
        write_bytes( s_out, (char*)m_label_templates[i].node_attributes.c_str(), (unsigned short) m_label_templates[i].node_attributes.size() ); 
        
        write_bin_identifier( s_out, m_label_templates[i].node_label_template.size(), write_options );
        write_bytes( s_out, (char*)m_label_templates[i].node_label_template.c_str(), (unsigned short) m_label_templates[i].node_label_template.size() ); 
        write_bin_identifier( s_out, (long)BIN_INTERNAL_SEPARATOR2, write_options );
      }
    }
    write_bin_identifier( s_out, (long)BIN_BLOCK_SEPARATOR, write_options );
  }
}

bool GraphView::read_from( istream& s_in, unsigned long read_options )
{
  bool res = true;
  bool critical = false;
  Graph *graph = get_app_context()->get_graph();
  
  if (read_options & ASCII_STORAGE )
  {
    string data;
    string data_id;
    
    res = graph->read_from( s_in, read_options ); //read graph info block
    critical = !res;
            
    // read graph vis.block
    if ( res )
      res = read_min_ASCII_block( s_in, data, data_id );
    
    if ( (!data.empty() || !data_id.empty()) && res )
    {
      //filters
      if ( !(data.empty() && data_id == ASCII_INTERNAL_SEPARATOR1) )
      {
        AttrStringTemplateStorage *filters = new AttrStringTemplateStorage();
        string attr;
        
        while( res )
        {
          if ( data.empty() && data_id == ASCII_INTERNAL_SEPARATOR1 )
            break;
          
          if ( data.empty() && data_id.empty() )
            res = false;
          else if ( data_id == ASCII_FILTER_ID )
          {
            res = read_min_ASCII_block( s_in, attr );
            
            if ( res && !attr.empty() && !data.empty() )
              filters->set_template( data, attr );
          }
          else if ( data_id == ASCII_DEF_FILTER_ID )
          {
            if ( filters->is_template_name_exist( data ) )
              filters->set_current_template_name( data );
          }
          else
            res = false;
          
          if ( res )
            res = read_min_ASCII_block( s_in, data, data_id );
        }
        
        if ( res )
        {
          if ( m_filters != 0 )
            delete m_filters;
          
          m_filters = filters;
        }
        else
        {
          delete filters;
          set_error( "Cannot read filters. All visual options are not changed" );
        }
      }      
    }    
        
    if ( (!data.empty() || !data_id.empty()) && res )
    {
      //rendering options
      res = read_min_ASCII_block( s_in, data, data_id );
      
      while ( res ) 
      {
        if ( data.empty() && data_id == ASCII_INTERNAL_SEPARATOR1 )
          break;
         
        istringstream str_in( data );
         
        if ( data.empty() && data_id.empty() )
          res = false;
        else if ( data_id == ASCII_NODE_RANGE_ID )
          str_in >> m_radius;
        else if ( data_id == ASCII_V_DISTANCE_ID )
          str_in >> m_y_separation;
        else if ( data_id == ASCII_H_DISTANCE_ID )
          str_in >> m_x_separation;
        else if ( data_id == ASCII_SPLINE_ID )
          m_spline_interpltn = ( data == "true" );
        //else skip
        
        res = !str_in.fail();
        
        if ( res )
          res = read_min_ASCII_block( s_in, data, data_id );
      }
      
      if ( !res )
        set_error( "Cant load rendering options" );
    }
    
    if ( (!data.empty() || !data_id.empty()) && res )
    {
      //general drawing options
      Color color;
      res = read_min_ASCII_block( s_in, data, data_id );
      
      while ( res ) 
      {
        if ( data.empty() && data_id == ASCII_INTERNAL_SEPARATOR1 )
          break;
         
        istringstream str_in( data );
         
        if ( data.empty() && data_id.empty() )
          res = false;
        else if ( data_id == ASCII_BG_COLOR_ID )
        {
          res = color.read_from( str_in, read_options );
          if ( res )
            m_view_groups.set_background_color( color );
        }        
        else if ( data_id == ASCII_SEL_COLOR_ID )
        {
          res = color.read_from( str_in, read_options );
          if ( res )
            m_view_groups.set_selected_node_color( color );
        }
        else if ( data_id == ASCII_EDGE_COLOR_ID )
        {
          res = color.read_from( str_in, read_options );
          if ( res )
            m_view_groups.set_edge_color( color );
        }
        else if ( data_id == ASCII_EDGE_WIDTH_ID )
        {
          OneDValue v;
          str_in >> v;
          res = !str_in.fail();
          if ( res )
            m_view_groups.set_edge_pen( DrawingPen(DrawingPen::SOLID, (unsigned char)v ) );
        }
        //else skip
        
        if ( res )
          res = read_min_ASCII_block( s_in, data, data_id );
      }
      
      if ( !res )
        set_error( "Cannot read general drawing options" );
    }
    
    if ( (!data.empty() || !data_id.empty()) && res )
    {
      //groups
      GraphViewPropertyGroup group;
      res = read_min_ASCII_block( s_in, data, data_id );
      
      if ( !data.empty() || !data_id.empty() )
      {
        m_view_groups.remove_all_view_groups();
        
        while ( res && !s_in.eof() )
        {
          if ( data.empty() && 
              (data_id.empty() || data_id == ASCII_INTERNAL_SEPARATOR1))
            break;
          
          if ( data_id != GraphViewPropertyGroup::ASCII_START_ID )
            res = false;
          else
          {
            res = group.read_from( s_in, read_options );
            
            if ( res )
            {
              if ( group.get_name() == GraphViewPropertyGroup::ASCII_DEFAULT_NAME )
                m_view_groups.change_view_group( GraphViewPropertyGroup::DEFAULT, group );
              else
                m_view_groups.add_view_group( group.get_assigned_attributes(), group, false );
            }
          }
          
          if ( res )
            res = read_min_ASCII_block( s_in, data, data_id );
        }
        
        if ( !res )
          set_error( "Cannot read node properties group" );
      }
    }
    
    if ( (!data.empty() || !data_id.empty()) && res )
    {
      //read label templates
      res = read_min_ASCII_block( s_in, data, data_id );
      
      if ( !data.empty() || !data_id.empty() )
      {
        string attr_string;
        
        remove_node_label_templates();
        
        while ( res && !s_in.eof() )
        {
          if ( data.empty() && data_id.empty() )
            break;
          
          if ( data_id != GraphViewPropertyGroup::ASCII_START_ID )
            res = false;
          else
          {
            res = read_min_ASCII_block( s_in, data, data_id );
            
            if ( res )
            {
              if ( data_id.empty() && data.empty() || 
                   data_id == GraphViewPropertyGroup::ASCII_END_ID)
                res = false;
              else 
              {
                attr_string = data_id + data; 
                res = read_min_ASCII_block( s_in, data );
                
                if ( res )
                {
                  if ( data.empty() )
                    res = false;
                  else
                  {
                    NodeAttributeList attr_list;
                                        
                    res = GraphAttributes::parse_string( attr_string, attr_list );
                    if ( res )
                    {
                      add_node_label_template( attr_string, data );
                      
                      res = read_min_ASCII_block( s_in, data, data_id );
                      
                      if ( res )
                        res = (data_id == GraphViewPropertyGroup::ASCII_END_ID);
                      
                      if ( res )
                        res = read_min_ASCII_block( s_in, data, data_id );
                    }
                  }
                }
              }
            }
          }
        }
        if ( !res )
          set_error( "Cannot read node label templates" );
      }
    }
  }
  else
  {
    long id = 0;
    long size = 0;
    
    res = graph->read_from( s_in, read_options ); //read graph info block
    critical = !res;
    
    //read graph vis. info block
    if ( res )
      res = read_bin_identifier( s_in, id, read_options);
      
    if ( id != (long)BIN_BLOCK_SEPARATOR && res )
    {
      //filters
      if ( id != (long)BIN_INTERNAL_SEPARATOR1 )
      {
        AttrStringTemplateStorage *filters = new AttrStringTemplateStorage();
        string attr;
        string name;
        char * buff= 0;
        
        while( res && id != (long)BIN_INTERNAL_SEPARATOR1 )
        {
          res = read_bin_identifier( s_in, size, read_options);
          if ( size < 0 )
            res = false;
            
          if ( res )
            switch( id )
            {
            case BIN_FILTER_ID:
              buff = new (char)[size + 1];
              buff[size] = 0;
              res = read_bytes( s_in, buff, size );
            
              if ( res )
              {
                name = string( buff );
                res = read_bin_identifier( s_in, size, read_options);
              }
              delete[] buff;
              
              if ( res )
              {
                buff = new (char)[size + 1];
                buff[size] = 0;
                res = read_bytes( s_in, buff, size );
                
                if ( res )
                  attr = string( buff );
                
                delete[] buff;  
              }
              
              if ( res && !name.empty() && !attr.empty() )
                 filters->set_template( name, attr );
                            
              break;
              
            case BIN_DEF_FILTER_ID:
              buff = new (char)[size + 1];
              buff[size] = 0;
              res = read_bytes( s_in, buff, size );
            
              if ( res && buff[0] != 0 )
              {
                name = string(buff);
                if ( filters->is_template_name_exist( name ) )
                  filters->set_current_template_name( name );
              }
              delete[] buff;
              break;
            
            default:
              res = false;
            }
            
          if ( res )
            res = read_bin_identifier( s_in, id, read_options );
        }
        
        if ( res )
        {
          if ( m_filters != 0 )
            delete m_filters;
          
          m_filters = filters;
        }
        else
        {
          delete filters;
          set_error( "Cannot read filters. All visual options are not changed" );
        }
      }      
    }
    
    if ( id != (long)BIN_BLOCK_SEPARATOR && res )
    {
      res = read_bin_identifier( s_in, id, read_options );
      //rendering options
      while ( id != (long)BIN_INTERNAL_SEPARATOR1 && res )
      {
        if ( id < 0 )
        {
          res = false;
          break;
        }
        
        res = read_bin_identifier( s_in, size, read_options );
        
        if ( res )
        {
          switch ( id )
          {
          case BIN_NODE_RANGE_ID:
            if ( size != 4 )
              res = false;
            else
              res = read_4bytes( s_in, m_radius );
            break;
          
          case BIN_V_DISTANCE_ID:
            if ( size != 4 )
              res = false;
            else
              res = read_4bytes( s_in, m_y_separation );
            break;
          
          case BIN_H_DISTANCE_ID:
            if ( size != 4 )
              res = false;
            else
              res = read_4bytes( s_in, m_x_separation );
            break;
            
          case BIN_SPLINE_ID:
            {
              char b = 0;
              if ( size != 1 )
                res = false;
              else
              {
                res = read_1bytes( s_in, b );
                m_spline_interpltn = ( b == 1 );
              }
            }
            break;
          
          default:            
            char *buff = new (char)[size];
            res = read_bytes( s_in, buff, size );
            delete[] buff;
          }
        }
        
        if ( res )
          res = read_bin_identifier( s_in, id, read_options );
      } 
      
      if ( !res )
        set_error( "Cannot read rendering options" );
    }
    
    if ( id != (long)BIN_BLOCK_SEPARATOR && res )
    {
      Color color;
      //general drawing properties
      res = read_bin_identifier( s_in, id, read_options );
      
      while ( id != (long)BIN_INTERNAL_SEPARATOR1 && res )
      {
        if ( id < 0 )
        {
          res = false;
          break;
        }
        
        res = read_bin_identifier( s_in, size, read_options );
        
        if ( res )
        {
          switch ( id )
          {
          case BIN_BG_COLOR_ID:
            if ( size != 4 )
              res = false;
            else
            {
              res = color.read_from( s_in, read_options );
              if ( res )
                m_view_groups.set_background_color( color );
            }
            break;
          
          case BIN_SEL_COLOR_ID:
            if ( size != 4 )
              res = false;
            else
            {
              res = color.read_from( s_in, read_options );
              if ( res )
                m_view_groups.set_selected_node_color( color );
            }
            break;
          
          case BIN_EDGE_COLOR_ID:
            if ( size != 4 )
              res = false;
            else
            {
              res = color.read_from( s_in, read_options );
              if ( res )
                m_view_groups.set_edge_color( color );
            }
            break;
          
          case BIN_EDGE_WIDTH_ID:
            {
              char b = 0;
              if ( size != 1 )
                res = false;
              else
              {
                res = read_1bytes( s_in, b );
                if ( res )
                  m_view_groups.set_edge_pen( DrawingPen( DrawingPen::SOLID, (unsigned char)b ) );
              }
            }
            break;
          
          default:            
            if ( size < 0 )
              res = false;
            else
            {
              char *buff = new (char)[size];
              res = read_bytes( s_in, buff, size );
              delete[] buff;
            }
          }
        }

        
        if ( res )
          res = read_bin_identifier( s_in, id, read_options );
      }
      
      if ( !res )
        set_error( "Cannot read general drawing options" );
    }
    
    if ( id != (long)BIN_BLOCK_SEPARATOR && res )
    {
      //groups
      GraphViewPropertyGroup group;
      res = read_bin_identifier( s_in, id, read_options );
      
      if ( id != (long)BIN_BLOCK_SEPARATOR && id != (long)BIN_INTERNAL_SEPARATOR1 )
      {
        m_view_groups.remove_all_view_groups();
        
        while ( id != (long)BIN_BLOCK_SEPARATOR && id != (long)BIN_INTERNAL_SEPARATOR1 
                && res && !s_in.eof() )
        {
          if ( id != (long)GraphViewPropertyGroup::BIN_START_ID )
            res = false;
          else
          {
            res = group.read_from( s_in, read_options );
            
            if ( res )
            {
              if ( group.get_name() == GraphViewPropertyGroup::ASCII_DEFAULT_NAME )
                m_view_groups.change_view_group( GraphViewPropertyGroup::DEFAULT, group );
              else
                m_view_groups.add_view_group( group.get_assigned_attributes(), group, false );
            }
          }
          
          if ( res )
            res = read_bin_identifier( s_in, id, read_options );
        }
        
        if ( !res )
        {
          set_error( "Cannot read node properties group" );
        }
      }
    }
    
    if ( id != (long)BIN_BLOCK_SEPARATOR && res )
    {
      //node label templates
      res = read_bin_identifier( s_in, id, read_options );
      
      if ( id != (long)BIN_BLOCK_SEPARATOR )
      {
        string attr_string;
        string label_template;
        remove_node_label_templates();
        
        while ( id != (long)BIN_BLOCK_SEPARATOR && res && !s_in.eof() )
        {
          if ( id != (long)GraphViewPropertyGroup::BIN_START_ID )
            res = false;
          else
          {
            res = read_bin_identifier( s_in, id, read_options );
            
            if ( id <= 0 )
              res = false;
            
            if ( res )
            {
              char *buff = new char[ id + 1 ];
              res = read_bytes( s_in, buff, id );
              
              if ( res )
              {
                buff[id] = 0;
                attr_string = string( buff );
              }
              delete[] buff;
            }
            
            if ( res )
            {
              res = read_bin_identifier( s_in, id, read_options );
            
              if ( id <= 0 )
                res = false;
              
              if ( res )
              {
                char *buff = new char[ id + 1 ];
                res = read_bytes( s_in, buff, id );
                
                if ( res )
                {
                  buff[id] = 0;
                  label_template = string( buff );
                }
                delete[] buff;
              } 
            }
            
            if ( res )
            {
              NodeAttributeList attr_list;
              
              if ( res = GraphAttributes::parse_string( attr_string, attr_list ) )
              {                
                add_node_label_template( attr_string, label_template );
                
                res = read_bin_identifier( s_in, id, read_options );
                
                if ( id != (long)BIN_INTERNAL_SEPARATOR2 )
                  res = false;
                
                if ( res )
                  res = read_bin_identifier( s_in, id, read_options );
              }
            }
          }
        }
        
        if ( !res )
        {
          set_error( "Cannot read node label templates" );
        }
      }
    }
  }

  unselect_all_nodes( 0 );
  mark_all_edges( 0 );
  //m_selected_edge = 0;
  //m_selected_scenario.clear();
  //m_selected_scenario.resize(0);
    
  delete_nodes_edges();
  
  if (m_gl != 0)
  {
    delete m_gl;
    m_gl = 0;  
  }
  m_view_rect = m_bounds = Rectangle( 0, 0, 0, 0 );
  
  if ( !res )
    set_error_level( Storable::CRITICAL );
  
  return res;
}

bool GraphViewContext::save_file( const GraphFileInfo &file_info )
{
  bool res = true;
  Graph *graph = get_graph();
  GraphView *view = get_view();
  ofstream s_out( file_info.file_name.c_str() );
  
  Storable::set_error( "" );
  Storable::set_error_level( Storable::OK_STATUS );
  unsigned long write_options = (( file_info.file_format & GraphFileInfo::ASCII_FORMAT )? Storable::ASCII_STORAGE: 
                                 ( file_info.file_format & GraphFileInfo::BIN32_FORMAT )?Storable::BIN32_STORAGE: 
                                  Storable::BIN_STORAGE );
  
  if ( s_out.fail() || s_out.eof() )
  {
    res = false;
    Storable::set_error( "Cannot open file" );
  }
  else
  {
    //write file header
    s_out << "graph";
    s_out << file_info.version << "\n";
    s_out << file_info.graph_name << "\n";
    s_out << (( file_info.file_format == GraphFileInfo::ASCII_FORMAT )? "ASC\n": 
               ( file_info.file_format == GraphFileInfo::BIN32_FORMAT )?"B32\n": "B16\n" );
    
    //write extra info block
    if ( file_info.file_format == GraphFileInfo::ASCII_FORMAT )
    {
      if ( !file_info.comment.empty() )
        s_out << Storable::ASCII_COMMENT_ID << " " << file_info.comment << "\n";
      
      if ( !file_info.model_file.empty() )
        s_out << Storable::ASCII_MODEL_FILE_ID << " " << file_info.model_file << "\n";
      s_out << "\n";
    }
    else
    {
      if ( !file_info.comment.empty() )
      {
        Storable::write_bin_identifier( s_out, (long)Storable::BIN_COMMENT_ID, write_options );
        Storable::write_bin_identifier( s_out, file_info.comment.length(), write_options );
        Storable::write_bytes( s_out, (char*)file_info.comment.c_str(), file_info.comment.length() );
      }
      
      if ( !file_info.model_file.empty() )
      {
        Storable::write_bin_identifier( s_out, (long)Storable::BIN_MODEL_FILE_ID, write_options );
        Storable::write_bin_identifier( s_out, file_info.model_file.length(), write_options );
        Storable::write_bytes( s_out, (char*)file_info.model_file.c_str(), file_info.model_file.length() );
      }
      Storable::write_bin_identifier( s_out, (long)Storable::BIN_BLOCK_SEPARATOR, write_options );
    }
    
    //write the rest
    //graph info block
    if ( file_info.save_graph )
      graph->write_to( s_out, write_options );
    else
    {
      if ( file_info.file_format == GraphFileInfo::ASCII_FORMAT )
        s_out << "\n";
      else
        Storable::write_bin_identifier( s_out, (long)Storable::BIN_BLOCK_SEPARATOR, write_options );
    }
    
    //graph vis info block
    if ( file_info.save_view )
      view->write_to( s_out, write_options );
    else
    {
      if ( file_info.file_format == GraphFileInfo::ASCII_FORMAT )
        s_out << "\n";
      else
        Storable::write_bin_identifier( s_out, (long)Storable::BIN_BLOCK_SEPARATOR, write_options );
    }
    
    s_out.close();
  }
  
  return res;
}

bool GraphViewContext::load_file( const string &file_name )
{
  bool res = true;
  char signature[5];
  GraphFileInfo info;
  GraphView *view = get_view();
  string v;
  ifstream s_in( file_name.c_str() );
  Storable::set_error( "" );
  Storable::set_error_level( Storable::OK_STATUS );
  
  info.file_name = file_name;
  
  if ( s_in.fail() || s_in.eof() )
  {
    res = false;
    Storable::set_error( "Cannot open file" );
    Storable::set_error_level( Storable::NOT_GRAPH_FILE );
  }
  else
  {
    s_in.read( signature, sizeof( signature ) );
    
    if ( s_in.fail() || memcmp( signature, "graph", 5 ) != 0 )
    {
      res = false;
      Storable::set_error( "It is not graph file" );
      Storable::set_error_level( Storable::NOT_GRAPH_FILE );
    }
    else
    {
      res = Storable::read_min_ASCII_block( s_in, v );
      
      info.version = v;
      
      //if ( v != "001.000" ) //currently supported
      //  res = false;
            
      if ( res )
      {
        res = Storable::read_min_ASCII_block( s_in, v ); //model name
        
        info.graph_name = v;
      }
      
      if ( res )
      {
        res = Storable::read_min_ASCII_block( s_in, v ); //file format type
        
        if ( v == "ASC" )
          info.file_format = GraphFileInfo::ASCII_FORMAT;
        else if ( v == "BIN" || v == "B16" )
          info.file_format = GraphFileInfo::BIN16_FORMAT;
        else if ( v == "B32" )
          info.file_format = GraphFileInfo::BIN32_FORMAT;
        else
          res = false;

      }
      
      if ( !res )
      {
        Storable::set_error_level( Storable::NOT_GRAPH_FILE );
        Storable::set_error( "Cannot read file header" );
      }
      else
      {
        if ( v == "ASC" ) //ascii format
        {
          string data, data_id;
          res = Storable::read_min_ASCII_block( s_in, data, data_id );
          while ( (!data.empty() || !data_id.empty()) && res ) // read extra info block
          {
            res = Storable::read_min_ASCII_block( s_in, data, data_id );
            
            if ( data_id == Storable::ASCII_COMMENT_ID )
              info.comment = data;
            else if ( data_id == Storable::ASCII_MODEL_FILE_ID )
              info.model_file = data;
          }
       
          if ( !res )
          {
            Storable::set_error_level( Storable::NOT_GRAPH_FILE );
            Storable::set_error( "Cannot read extra info block. File is not loaded" );
          }
          else
            res = view->read_from( s_in, Storable::ASCII_STORAGE ); // read the rest
        }
        else //binary format
        {
          long id;
          long size;
          char *buff;
          unsigned long read_options = ((info.file_format == GraphFileInfo::BIN32_FORMAT)? 
                                        Storable::BIN32_STORAGE: Storable::BIN_STORAGE);
          
          res = Storable::read_bin_identifier( s_in, id, read_options );
          
    
          while ( id != (long)Storable::BIN_BLOCK_SEPARATOR && res ) // read extra info block
          {
            res = Storable::read_bin_identifier( s_in, size, read_options );
            
            if ( res )
              if ( size < 0 )
                res = false;
                        
            if ( res && size > 0 )
            {
              buff = new (char)[size + 1];
              buff[size] = 0;
              res = Storable::read_bytes( s_in, buff, size );
                            
              switch ( id )
              {
              case Storable::BIN_COMMENT_ID:
                info.comment = string( buff );
                break;
              case Storable::BIN_MODEL_FILE_ID:
                info.model_file = string( buff );
                break;
              }
              
              delete[] buff;
            }
            
            if ( res )
              res = Storable::read_bin_identifier( s_in, id, read_options );
          }
          
          if ( !res )
          {
            Storable::set_error_level( Storable::NOT_GRAPH_FILE );
            Storable::set_error( "Cannot read extra info block. File is not loaded" );
          }
          else
            res = view->read_from( s_in, read_options ); //read the rest
        }
      }
    }
   
    s_in.close();
  }
  
  set_load_info( info, true );
  
  return res;
}
