// UnixGui.cpp: implementation of Unix version of the GraphGUI class.
//
// ////////////////////////////////////////////////////////////////////

#include <pthread.h>
#include <gtk/gtk.h>
#include "../OS_indep/graphics.h"
#include "UnixNodeSearchGUI.h"
#include "UnixGraphicArea.h"
#include "UnixAttrTreeView.h"
#include "UnixGroupGUI.h"
#include "UnixPropertiesDialog.h"
#include "UnixPreferenceDialog.h"
#include "UnixGraphSaveGUI.h"
#include "UnixGenerateGUI.h"
#include "UnixAttrFilterGUI.h"
#include "UnixCompManGUI.h"
#include "GTKInformationWindow.h"
#include "GTKScenarioGroupGUI.h"

namespace GraphGraphics
{
class NodeAttrCloseHandler: public GUICloseHandler
{
private:
  GraphGUI *m_gui;
  GraphNodeAttributesGUI *m_window;

public:
  NodeAttrCloseHandler( GraphGUI *gui, GraphNodeAttributesGUI *window ) 
  {
    m_gui = gui;
    m_window = window;
  }
  
  virtual ~NodeAttrCloseHandler() {}
  
  void operator () ( )
  {
    if ( m_gui->m_last_attr_window == m_window && m_window != 0 )
    {
      UnixGraphAttrView *dialog = dynamic_cast<UnixGraphAttrView *>(m_window);
      
      if ( dialog != 0 )
        dialog->get_expanded_pathes(m_gui->m_last_attr_expanded_rows);
      m_gui->m_last_attr_window = 0;
    }
    
    m_window = 0;
  }
};

class InfoWindowCloseHandler: public GUICloseHandler
{
private:
  GraphGUI *m_gui;
  GraphInformationGUI *m_window;

public:
  InfoWindowCloseHandler( GraphGUI *gui, GraphInformationGUI *window ) 
  {
    m_gui = gui;
    m_window = window;
  }
  
  virtual ~InfoWindowCloseHandler() {}
  
  void operator () ( )
  {
    if ( m_gui->m_last_info_window == m_window )
      m_gui->m_last_info_window = 0;
    
    m_window = 0;
  }
};
};
  
using namespace GraphGraphics;

pthread_mutex_t graph_load_mutex;

void GraphGUI::initialize_globals() 
{ 
  pthread_mutex_init(&graph_load_mutex, 0); 
}


/* The os_data should always be (GtkWidget *) */
GraphicArea* GraphGUI::get_graphic_area( void *os_data, bool screen_window )
{
	GraphicArea *res = 0;
	GtkWidget 	*widget = (GtkWidget *)os_data;
	
	if ( screen_window )
	{
		res = new UnixGraphicArea( widget->window, false );
	}
	else
	{
		gint          width = 0;
		gint					height = 0;
		GdkPixmap			*pixmap = 0;
	
		gdk_drawable_get_size( widget->window, &width, &height );
	
		pixmap = gdk_pixmap_new( widget->window, width, height, -1 );
		
		res = new UnixGraphicArea( pixmap, true );
	}
	
	return res; 
}

GraphNodeAttributesGUI* GraphGUI::create_nodes_info_dialog( GraphView *view, NodeIndex node_id, bool ignore_modes )
{
  Graph           *graph = view->get_app_context()->get_graph();
  vector<NodeAttributeList>   attr_lists;
  string                      filter_str = view->get_node_attr_filters()->get_current_template_name();
  NodeAttributeList           attr_filter;
  NodeIndexVector  node_ids;
  GraphAttributes *attr;
  GraphNodeAttributesGUI* res = 0;
    
  attr_lists.resize(0);
  node_ids.resize(0);
  
  if ( filter_str.empty() || !view->get_node_attr_filters()->is_template_name_exist(filter_str) )
  {
    show_message_dialog( "Warning: current node attributes filter is incorrect" );
    filter_str = "{ }";
  }
  else
    filter_str = view->get_node_attr_filters()->get_template( filter_str );
  
  if ( node_id == Graph::INVALID_NODE_ID )
    view->get_selected_nodes( node_ids );
  else
    node_ids.push_back( node_id );
  
  GraphAttributes::parse_string( filter_str, attr_filter );

  
  if ( node_ids.size() > 0 )
  {
    sort( node_ids.begin(), node_ids.end() );
    
    attr_lists.resize( node_ids.size() );
    
    for ( unsigned int i = 0; i < node_ids.size(); i++ )
    {
      Node *node = graph->get_node( node_ids[i] );
      
      if ( node != 0 )
      {
        attr = node->get_attributes();
        attr->get_parsed_attributes_filtered( attr_lists[i], attr_filter );
      }
    }
    
    bool create_win = true;
    
    if ( m_app != 0 && !ignore_modes && m_last_attr_window != 0 )
      create_win = ( ((m_app->get_behavior_modes() & ScenarioApp::SINGLE_ATTR_WINDOW) == 0 &&
                     (m_app->get_behavior_modes() & ScenarioApp::SHOW_ATTR_ON_MOUSE_OVER) == 0) );
    
    
    if ( m_last_attr_window != 0 )
    {
      UnixGraphAttrView *existed = dynamic_cast<UnixGraphAttrView *>(m_last_attr_window);
      existed->get_expanded_pathes( m_last_attr_expanded_rows );
    }
    
    if ( create_win )
    {
      UnixGraphAttrView *unix_res = new UnixGraphAttrView( graph->get_graph_attr_name_storage(), UnixGraphAttrView::ATTRIBUTE_DIALOG );
      unix_res->set_expanded_pathes( m_last_attr_expanded_rows );
      res = unix_res;
      res->register_close_handler( new NodeAttrCloseHandler( this, res ) );
      res->set_dockable_type( GraphGUI::DOCKABLE_NODE_INFO );
           
      if ( !ignore_modes )
        m_last_attr_window = res;
    }
    else
      res = m_last_attr_window; 
  
    res->set_attributes( attr_lists );
    
    generate_create_dockable_event( res, create_win );
  }
  else
    show_message_dialog( "Please select nodes" );
  
  return res;
}
 
GraphNodeSearchGUI* GraphGUI::create_node_search_dialog( GraphView *view )
{
  Graph              *graph  = view->get_app_context()->get_graph();
  GraphNodeSearchGUI *dialog = new GTKNodeSearchDialog( graph->get_graph_attr_name_storage() );
    
  GraphAttrNameStorage *attr_name_storage = graph->get_graph_attr_name_storage();
  
  dialog->set_attr_template_string( attr_name_storage->get_search_format_string() );
  
  return dialog;
}

GraphAttrFilterGUI* GraphGUI::create_node_attr_filter_dialog( GraphView *view )
{
  Graph             * graph  = view->get_app_context()->get_graph();
  GraphAttrFilterGUI* dialog = new GTKAttrFilterGUI( graph->get_graph_attr_name_storage() );
  GraphAttrNameStorage *attr_name_storage = graph->get_graph_attr_name_storage();
  
  dialog->set_attr_template( attr_name_storage->get_search_format_string() );
  dialog->set_filters( view->get_node_attr_filters() );
  
  return dialog;
}

GraphViewPropertiesGUI* GraphGUI::create_group_properties_dialog( GraphView *view )
{
  GTKPropertiesGroupGUI *dialog = new GTKPropertiesGroupGUI( view, this );
  return dialog;
}

GraphViewPropertiesGUI* GraphGUI::create_view_properties_dialog( GraphView *view )
{
  GTKViewPropertiesGUI *dialog = new GTKViewPropertiesGUI( view, this );
  
  return dialog;
}

GraphGUIModalWindow* GraphGUI::create_view_preferences_dialog( GraphView *view, bool apply_changes_after_close )
{
  GTKPreferenceGUI* dialog = new GTKPreferenceGUI( m_app, view, apply_changes_after_close );
  
  return dialog;
}

void GraphGUI::show_message_dialog( const string &message )
{
  GtkWidget *message_box;
  GtkWidget *dialog_vbox4;
  GtkWidget *message_label;

  message_box = gtk_dialog_new ();
  
  gtk_dialog_add_button( GTK_DIALOG(message_box), GTK_STOCK_QUIT, GTK_RESPONSE_CLOSE );
  g_signal_connect_swapped( GTK_OBJECT(message_box), "response", G_CALLBACK(gtk_widget_destroy), GTK_OBJECT(message_box) );
  gtk_window_set_title (GTK_WINDOW (message_box), "Message...");
  gtk_window_set_position (GTK_WINDOW (message_box), GTK_WIN_POS_CENTER);
  gtk_window_set_modal (GTK_WINDOW (message_box), TRUE);

  dialog_vbox4 = GTK_DIALOG (message_box)->vbox;
  gtk_widget_show (dialog_vbox4);

  message_label = gtk_label_new ( message.c_str() );
  gtk_widget_show (message_label);
  gtk_box_pack_start (GTK_BOX (dialog_vbox4), message_label, TRUE, TRUE, 20);
//  gtk_label_set_line_wrap (GTK_LABEL (message_label), TRUE);

  
  gtk_widget_show( message_box );
  
  if ( m_os_data != 0 )
    if ( ((GtkWidget*)m_os_data)->window != NULL || message_box->window != NULL )
      gdk_window_set_transient_for( message_box->window, ((GtkWidget*)m_os_data)->window );
}

Color GraphGUI::run_choose_color_dialog( const Color &c )
{
  GtkWidget *colorselectiondialog1;
  GtkWidget *dialog_vbox1;
  GtkWidget *color_selection1;
  GdkColor  color;
  Color	    res_color = c;

  colorselectiondialog1 = gtk_dialog_new_with_buttons( "Select Color", NULL, GTK_DIALOG_MODAL, 
                                                                GTK_STOCK_OK, GTK_RESPONSE_OK, 
                                                                GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL, NULL );
  gtk_window_set_resizable (GTK_WINDOW (colorselectiondialog1), FALSE);
  
  dialog_vbox1 = GTK_DIALOG (colorselectiondialog1)->vbox;
  gtk_widget_show (dialog_vbox1);
  
  color_selection1 = gtk_color_selection_new();
  gtk_widget_show (color_selection1);
  gtk_color_selection_set_has_opacity_control (GTK_COLOR_SELECTION (color_selection1), FALSE);
  gtk_box_pack_start (GTK_BOX (dialog_vbox1), color_selection1, TRUE, TRUE, 0);
  
	color.red = c.get_RGB().red << 8;
	color.green = c.get_RGB().green << 8;
	color.blue = c.get_RGB().blue << 8;	
	
	gtk_color_selection_set_current_color( GTK_COLOR_SELECTION( color_selection1 ), &color );	
	
	gboolean result = gtk_dialog_run( GTK_DIALOG(colorselectiondialog1) );
  
  switch( result )
  {
  case GTK_RESPONSE_OK:
    gtk_color_selection_get_current_color( GTK_COLOR_SELECTION( color_selection1 ) , &color );
    res_color.set_RGB( RGB(color.red >> 8, color.green >> 8, color.blue >> 8) );
    break;
  
  default:;
  }
    
  gtk_widget_destroy( colorselectiondialog1 );
  
  return res_color;  
}

Font GraphGUI::run_choose_font_dialog( const Font &f )
{
  GtkWidget *fontselectiondialog1;
  GtkWidget *dialog_vbox1;
  GtkWidget *font_selection1;
  Font	    res_font = f;
  char      *font_name;
  
  fontselectiondialog1 = gtk_dialog_new_with_buttons( "Select Font", NULL, GTK_DIALOG_MODAL, 
                                                                GTK_STOCK_OK, GTK_RESPONSE_OK, 
                                                                GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL, NULL );
  gtk_window_set_resizable (GTK_WINDOW (fontselectiondialog1), FALSE);
  
  dialog_vbox1 = GTK_DIALOG (fontselectiondialog1)->vbox;
  gtk_widget_show (dialog_vbox1);
  
  font_selection1 = gtk_font_selection_new();
  gtk_widget_show (font_selection1);
  gtk_box_pack_start (GTK_BOX (dialog_vbox1), font_selection1, TRUE, TRUE, 0);

  if ( f.get_face_name().size() > 0 )
  {    
     gtk_font_selection_set_font_name( GTK_FONT_SELECTION( font_selection1 ), f.get_face_name().c_str() );	
  }
	
	gboolean result = gtk_dialog_run( GTK_DIALOG(fontselectiondialog1) );
  
  switch( result )
  {
  case GTK_RESPONSE_OK:
    font_name = gtk_font_selection_get_font_name( GTK_FONT_SELECTION( font_selection1 ) );
    if ( font_name != 0 )
    {
      string    font( font_name );
      int       ind = 0;
      string    face_name;
      StyleFlags style = 0;
      int       size = 8;
      string    tmp;
      
      ind = font.find( ',' );
      if ( ind > 0 )
      {
        face_name = font.substr( 0, ind );
        font = font.substr( ind + 2 );
      }
      else
      {
        ind = font.find( ' ' );
        face_name = font.substr( 0, ind );
        font = font.substr( ind + 1 );
      }

      ind = font.find( ' ' );
      tmp = font.substr( 0, ind );
      font = font.substr( ind + 1 );
      
//      while ( tmp != "Bold" && tmp != "Italic" && ind > 0 )
//      {
//        face_name += " " + tmp;
//        ind = font.find( ' ' );
//        tmp = font.substr( 0, ind );
//        font = font.substr( ind + 1 );
//      }
      
      if ( tmp == "Bold" )
      {
        style |= Font::BOLD;
        ind = font.find( ' ' );
        tmp = font.substr( 0, ind );
        font = font.substr( ind + 1 );
      }
      
      if ( tmp == "Italic" )
      {
        style |= Font::ITALIC;
        ind = font.find( ' ' );
        tmp = font.substr( 0, ind );
        font = font.substr( ind + 1 );
      }
      
      sscanf( tmp.c_str(), "%d", &size );
      
      res_font.set_face_name( font_name );
      res_font.set_style( style );
      res_font.set_size( size );
            
    }
    break;
  
  default:;
  }
    
  gtk_widget_destroy( fontselectiondialog1 );
  
  return res_font; 
}

void GraphGUI::show_about_dialog()
{
  GtkWidget *about_dialog;
  GtkWidget *dialog_vbox1;
  GtkWidget *vbox1;
  GtkWidget *label1;
  GtkWidget *label2;

  about_dialog = gtk_dialog_new_with_buttons ( "About...", NULL, GTK_DIALOG_MODAL, 
                                                  GTK_STOCK_CLOSE, GTK_RESPONSE_OK, NULL);

  dialog_vbox1 = GTK_DIALOG (about_dialog)->vbox;
  gtk_widget_show (dialog_vbox1);

  vbox1 = gtk_vbox_new (FALSE, 20);
  gtk_widget_show (vbox1);
  gtk_box_pack_start (GTK_BOX (dialog_vbox1), vbox1, TRUE, TRUE, 0);
  gtk_container_set_border_width (GTK_CONTAINER (vbox1), 30);

  label1 = gtk_label_new ( "Graph Visualization" );
  gtk_widget_show (label1);
  gtk_box_pack_start (GTK_BOX (vbox1), label1, FALSE, FALSE, 0);

  label2 = gtk_label_new ( "Version: 1.0.0 " );
  gtk_widget_show (label2);
  gtk_box_pack_start (GTK_BOX (vbox1), label2, FALSE, FALSE, 0);
  gtk_label_set_justify (GTK_LABEL (label2), GTK_JUSTIFY_LEFT);
  
  gtk_dialog_run( GTK_DIALOG(about_dialog) );
  
  gtk_widget_destroy( about_dialog );
}

gboolean on_wait_window_delete_event( GtkWidget *widget, GdkEvent *event, gpointer user_data)
{
  return TRUE;
}

gboolean on_wait_window_destroy_event( GtkWidget *widget, GdkEvent *event, gpointer user_data )
{
  GraphGUI *gui = (GraphGUI *)user_data;
  gui->hide_wait_dialog(  );
  return TRUE;
}

void GraphGUI::show_wait_dialog( const string &message )
{
  GtkWidget *wait_dialog;
  
  hide_wait_dialog(  );

  wait_dialog = gtk_window_new ( GTK_WINDOW_TOPLEVEL);
  gtk_container_set_border_width (GTK_CONTAINER (wait_dialog), 1);
  gtk_window_set_title (GTK_WINDOW (wait_dialog), message.c_str());
  
  gtk_widget_set_size_request( wait_dialog, 290, 0);
  gtk_window_set_position (GTK_WINDOW (wait_dialog), GTK_WIN_POS_CENTER);
  gtk_window_set_resizable( GTK_WINDOW (wait_dialog), FALSE );
  gtk_window_set_transient_for( GTK_WINDOW (wait_dialog), GTK_WINDOW (m_os_data) );

  gtk_signal_connect (GTK_OBJECT (wait_dialog), "destroy_event",
                      GTK_SIGNAL_FUNC (on_wait_window_destroy_event),
                      (gpointer)this);
  gtk_signal_connect (GTK_OBJECT (wait_dialog), "delete_event",
                      GTK_SIGNAL_FUNC (on_wait_window_delete_event),
                      NULL);
  
  gtk_widget_show_now( wait_dialog );
  
  
  m_wait = wait_dialog;
}

void GraphGUI::hide_wait_dialog(  )
{
  if ( m_wait != 0 )
  {
    gtk_widget_destroy( GTK_WIDGET( m_wait ) );
    m_wait = 0;
  }
}

int GraphGUI::set_cursor( StyleFlags cursor_type )
{
  if ( m_os_data != 0 )
  {
    GtkWidget* main_window = (GtkWidget *) m_os_data;
    GdkCursor    *cursor = NULL;
    
    switch( cursor_type )
    {
    case CROSS_CURSOR:
      cursor = gdk_cursor_new( GDK_TCROSS );
      break;
    default:;
    }
    
    gdk_window_set_cursor( main_window->window, cursor );
    
  }
  
  return 0;
}

string GraphGUI::run_fileselection_dialog ( const string &title, const string &file_name, const string &pattern )
{
  GtkWidget *fileselection;
  GtkWidget *ok_button1;
  GtkWidget *file_cancel_button;
  string    res_file_name = "";

  fileselection = gtk_file_selection_new ( title.c_str() );
  gtk_container_set_border_width (GTK_CONTAINER (fileselection), 10);
  gtk_window_set_default_size (GTK_WINDOW (fileselection), 200, 200);

  ok_button1 = GTK_FILE_SELECTION (fileselection)->ok_button;
  gtk_widget_show (ok_button1);
  GTK_WIDGET_SET_FLAGS (ok_button1, GTK_CAN_DEFAULT);

  file_cancel_button = GTK_FILE_SELECTION (fileselection)->cancel_button;
  gtk_widget_show (file_cancel_button);
  GTK_WIDGET_SET_FLAGS (file_cancel_button, GTK_CAN_DEFAULT);
     
  gtk_window_set_modal( GTK_WINDOW (fileselection), TRUE );
  
  if ( !file_name.empty() )
    gtk_file_selection_set_filename( GTK_FILE_SELECTION(fileselection), file_name.c_str() );
  
  if ( !pattern.empty() )
    gtk_file_selection_complete( GTK_FILE_SELECTION(fileselection), pattern.c_str() );
  
  gint result = gtk_dialog_run( GTK_DIALOG(fileselection) );

  switch( result )
  {
  case GTK_RESPONSE_OK:
  {
    const char *file = gtk_file_selection_get_filename( GTK_FILE_SELECTION(fileselection) );
    
    if ( file != 0 )
      res_file_name = string( file );
    
    break;
  }    
  default:;
  }
  
  gtk_widget_destroy(fileselection);
  
  return res_file_name;
}

GraphSaveGUI* GraphGUI::create_graph_save_dialog( )
{
  GTKGraphSaveGUI *dialog = new GTKGraphSaveGUI( m_app );
  
  return dialog;
}

GraphGenerationGUI* GraphGUI::create_graph_generation_window( const string &title )
{
  return new GTKUnixGraphGenerationWindow( m_app, title );
}

GraphInformationGUI* GraphGUI::create_information_window( StyleFlags type, InfoStorageSubLevel *root_level_info, bool ignore_modes )
{
  GraphInformationGUI *res = 0;
  bool                 created_new = false;
    
  if ( !ignore_modes )
    res = m_last_info_window;  
    
  if ( (m_app->get_behavior_modes() & ScenarioApp::SINGLE_INFO_WINDOW) == 0 || 
        res == 0 )
  {
    res = new GTKInformationGUI();
    created_new = true;
      
    res->set_dockable_type( type );
      
    if ( !ignore_modes )
    {
      m_last_info_window = res;  
      res->register_close_handler( new InfoWindowCloseHandler( this, res ) );
    }
  }
  
  if ( root_level_info != 0 )
    res->set_root_level( root_level_info );
  
  generate_create_dockable_event( res, created_new );
    
  return res;
}

bool GraphGUI::run_promt_dialog( const string &message )
{
  GtkWidget *dialog_vbox;
  GtkWidget *message_label;
  GtkWidget *promt_dialog = gtk_dialog_new_with_buttons ( "Warning...", NULL, GTK_DIALOG_MODAL, 
                                                          GTK_STOCK_OK, GTK_RESPONSE_OK, 
                                                          GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL, NULL);

  dialog_vbox = GTK_DIALOG (promt_dialog)->vbox;
  gtk_widget_show (dialog_vbox);

  message_label = gtk_label_new ( message.c_str() );
  gtk_widget_show (message_label);
  gtk_box_pack_start (GTK_BOX (dialog_vbox), message_label, TRUE, TRUE, 20);
  
  gint result = gtk_dialog_run( GTK_DIALOG(promt_dialog) );
  
  gtk_widget_destroy( promt_dialog );
  
  return ((result == GTK_RESPONSE_OK)? true: false);
}

GraphGUIModalWindow* GraphGUI::create_comp_manager_dialog( ComponentManager *comp_manager )
{
  GTKComponentManagerGUI* dialog = new GTKComponentManagerGUI( comp_manager );
  
  return dialog;
}

GraphScenarioGroupRestrictionGUI* GraphGUI::create_scenario_group_restriction_dialog( Graph *graph )
{
  GraphScenarioGroupRestrictionGUI* dialog = new GTKScenarioGroupGUI( graph->get_graph_attr_name_storage() );
  
  return dialog;
}

GraphInformationGUI* GraphGUI::create_graph_statistics_info_window( const GraphStatistics &statistics, StyleFlags type, bool ignore_modes )
{
  InfoStorageSubLevel  *root_level = new InfoStorageSubLevel( "root" );
  InfoStorageItemLevel *item_level;
  InfoStorageSubLevel  *level;
  ostringstream         s_stream( "" );
  
  if ( statistics.calc_nodes_edges )
  {
    item_level = new InfoStorageItemLevel( "Number of graph nodes" );
    s_stream << statistics.node_count;
    item_level->add_value( new InfoValue( s_stream.str().c_str() ) );
    root_level->add_sublevel( item_level );
    
    s_stream.str("");
    item_level = new InfoStorageItemLevel( "Number of graph edges" );
    s_stream << statistics.edge_count;
    item_level->add_value(new InfoValue( s_stream.str().c_str()) );
    root_level->add_sublevel( item_level );
    
    s_stream.str("");
    item_level = new InfoStorageItemLevel( "Maximum node out-degree" );
    s_stream << statistics.max_nodes_out_degree;
    item_level->add_value(new InfoValue( s_stream.str().c_str()) );
    root_level->add_sublevel( item_level );
    
    s_stream.str("");
    item_level = new InfoStorageItemLevel( "Maximum node in-degree" );
    s_stream << statistics.max_nodes_in_degree;
    item_level->add_value(new InfoValue( s_stream.str().c_str()) );
    root_level->add_sublevel( item_level );
  }    
  
  if ( statistics.calc_path_depth )
  {
    s_stream.str("");
    item_level = new InfoStorageItemLevel( "Number of DFS pathes" );
    s_stream << statistics.path_count;
    item_level->add_value(new InfoValue( s_stream.str().c_str()) );
    root_level->add_sublevel( item_level );
    
    s_stream.str("");
    item_level = new InfoStorageItemLevel( "Longest DFS path length (the graph depth)" );
    s_stream << statistics.graph_depth;
    item_level->add_value( new InfoValue( s_stream.str().c_str() ) );
    root_level->add_sublevel( item_level );
    
    s_stream.str("");
    item_level = new InfoStorageItemLevel( "Shortest DFS path length" );
    s_stream << statistics.min_path_length;
    item_level->add_value( new InfoValue( s_stream.str().c_str() ) );
    root_level->add_sublevel( item_level );
  }
  
  if ( !statistics.nodes_to_count.empty() )
  {
    NodeValuesMap::const_iterator n_it = statistics.nodes_to_count.begin();
    for ( ; n_it != statistics.nodes_to_count.end(); n_it++ )
    {
      level = new InfoStorageSubLevel( n_it->second.name );
      
      s_stream.str("");
      item_level = new InfoStorageItemLevel( "Number of the nodes" );
      s_stream << n_it->second.value1;
      item_level->add_value( new InfoValue( s_stream.str().c_str() ) );
      level->add_sublevel( item_level );
      
      s_stream.str("");
      item_level = new InfoStorageItemLevel( "Maximum out-degree" );
      s_stream << n_it->second.value2;
      item_level->add_value( new InfoValue( s_stream.str().c_str() ) );
      level->add_sublevel( item_level );
      
      s_stream.str("");
      item_level = new InfoStorageItemLevel( "Maximum in-degree" );
      s_stream << n_it->second.value3;
      item_level->add_value( new InfoValue( s_stream.str().c_str() ) );
      level->add_sublevel( item_level );
      
      root_level->add_sublevel( level );
    }
  }
  
  if ( !statistics.pathes_to_count.empty() )
  {
    PathValuesMap::const_iterator p_it = statistics.pathes_to_count.begin();
    
    for ( ; p_it != statistics.pathes_to_count.end(); p_it++ )
    {
      level = new InfoStorageSubLevel( p_it->second.name );
      
      s_stream.str("");
      item_level = new InfoStorageItemLevel( "Count" );
      s_stream << p_it->second.value1;
      item_level->add_value( new InfoValue( s_stream.str().c_str() ) );
      level->add_sublevel( item_level );
      
      s_stream.str("");
      item_level = new InfoStorageItemLevel( "Longest" );
      s_stream << p_it->second.value2;
      item_level->add_value( new InfoValue( s_stream.str().c_str() ) );
      level->add_sublevel( item_level );
      
      s_stream.str("");
      item_level = new InfoStorageItemLevel( "Shortest" );
      s_stream << p_it->second.value3;
      item_level->add_value( new InfoValue( s_stream.str().c_str() ) );
      level->add_sublevel( item_level );
      
      root_level->add_sublevel( level );
    }
  }
  
  return create_information_window( type, root_level, ignore_modes );
}
