
#include "UnixAttrTreeView.h"

using namespace GraphGraphics;

const string UnixGraphAttrView::EMPTY_META_NAME = "{ }";
const string UnixGraphAttrView::EXPR_PARENTHESES = "( )";

GtkTreeIter UnixGraphAttrView::add_to_storage( GtkTreeStore *store, GtkTreeIter *parent_iter, const string &op, bool negation, const string &name, const string &value_op, const string &value, int type )
{
  GtkTreeIter iter;
  gtk_tree_store_append( store, &iter, parent_iter );
  
  gtk_tree_store_set( store, &iter, TYPE_COLUMN, type, -1 );
  
  if ( m_mode == ATTRIBUTE_EXPRESSION )
  {
    gtk_tree_store_set( store, &iter, EXPR_BIN_OPERATOR_COLUMN, op.c_str(), -1 ); 
    if ( negation )
      gtk_tree_store_set( store, &iter, EXPR_UN_OPERATOR_COLUMN, "!", -1 );
    else
      gtk_tree_store_set( store, &iter, EXPR_UN_OPERATOR_COLUMN, "", -1 );
  }
  
  gtk_tree_store_set( store, &iter, ATTRIBUTE_COLUMN, name.c_str(), -1 );
  
  if ( m_mode == ATTRIBUTE_EXPRESSION || m_mode == SEARCH_TEMPLATE )
    gtk_tree_store_set( store, &iter, OPERATION_COLUMN, value_op.c_str(), -1 );
  
  if ( m_mode != ATTRIBUTE_TEMPLATE )
    gtk_tree_store_set( store, &iter, VALUE_COLUMN, value.c_str(), -1 );
  
  return iter;
}

void  UnixGraphAttrView::add_to_storage( AttrExprView* attr_view, GtkTreeStore *store, GtkTreeIter *parent_iter )
{
  GtkTreeIter iter;
  
  switch (attr_view->meta_type)
  {
  case AttrExprView::PARENTHESES:
    iter = add_to_storage( store, parent_iter, attr_view->op, attr_view->negation, EXPR_PARENTHESES, "", "" );
    break;
  
  case AttrExprView::META_ATTR:
    iter = add_to_storage( store, parent_iter, attr_view->op, attr_view->negation, attr_view->name, "", "" );
    break;
  
  default:
    iter = add_to_storage( store, parent_iter, attr_view->op, attr_view->negation, attr_view->name, attr_view->value_op, attr_view->value, (attr_view->is_string_type? STRING_VALUE: INTEGER_VALUE) );
  }
  
  if ( attr_view->meta_type != AttrExprView::ATTR )
  {
    if ( attr_view->attr_list.empty() )
      add_to_storage( store, &iter, "", false, "", "", "" );
    else
    {
      for ( AttrExprListIt it = attr_view->attr_list.begin(); it != attr_view->attr_list.end(); it++ )
        add_to_storage( *it, store, &iter );
    }
  }
}

unsigned int UnixGraphAttrView::add_to_storage( GtkTreeStore *store, GtkTreeIter *parent_iter, const NodeAttributeList &attr_lists, unsigned int attr_ind, const string &end_cond )
{
  GtkTreeIter iter;
  bool negation = false; 
  string op = "";
  
  for ( ;attr_ind < attr_lists.size() && attr_lists[attr_ind]->type != end_cond ; attr_ind++ )
  {
    if ( attr_lists[attr_ind]->type == NodeAttribute::ATTR_PRED_AND )
    {
      if ( m_mode == ATTRIBUTE_EXPRESSION )
        op = "&";
    }
    else if ( attr_lists[attr_ind]->type == NodeAttribute::ATTR_PRED_OR )
    {
      if ( m_mode == ATTRIBUTE_EXPRESSION )
        op = "|";
    }
    else if ( attr_lists[attr_ind]->type == NodeAttribute::ATTR_PRED_NOT )
    {
      if ( m_mode == ATTRIBUTE_EXPRESSION )
        negation = true;
    }
    else 
    {
      m_stream.str("");
      gtk_tree_store_append( store, &iter, parent_iter );
      
      if ( m_mode == ATTRIBUTE_EXPRESSION )
      {
        gtk_tree_store_set( store, &iter, EXPR_BIN_OPERATOR_COLUMN, op.c_str(), -1 ); 
        if ( negation )
          gtk_tree_store_set( store, &iter, EXPR_UN_OPERATOR_COLUMN, "!", -1 );
        
        negation = false;
        op = "";
      }
      
      if ( attr_lists[attr_ind]->type == NodeAttribute::ATTR_PRED_OPEN_PARENTHESIS )
      {
        gtk_tree_store_set( store, &iter, TYPE_COLUMN, INTEGER_VALUE, ATTRIBUTE_COLUMN, EXPR_PARENTHESES.c_str(), -1 );
        
        attr_ind = add_to_storage( store, &iter, attr_lists, attr_ind + 1, NodeAttribute::ATTR_PRED_CLOSE_PARENTHESIS );
      }
      else if ( attr_lists[attr_ind]->type == NodeAttribute::ATTR_START )
      {
        if ( attr_lists[attr_ind]->value->get_integer() >= 0 && m_mode != ATTRIBUTE_TEMPLATE )
          m_stream << attr_lists[attr_ind]->value->get_string() << ": ";
        
        m_stream << ((attr_lists[attr_ind]->name.empty())? EMPTY_META_NAME: attr_lists[attr_ind]->name);
        
        gtk_tree_store_set( store, &iter, TYPE_COLUMN, INTEGER_VALUE, ATTRIBUTE_COLUMN, m_stream.str().c_str(), -1 );  
        
        attr_ind = add_to_storage( store, &iter, attr_lists, attr_ind + 1 );
      }
      else
      {
        int value_type = INTEGER_VALUE; 
        if ( attr_lists[attr_ind]->value->get_actual_type() == AttributeValue::STRING )
        {
          m_stream << "\"" << attr_lists[attr_ind]->value->get_string() << "\"";
          value_type = STRING_VALUE;
        }
        else
          m_stream << attr_lists[attr_ind]->value->get_string();
        
        gtk_tree_store_set( store, &iter, TYPE_COLUMN, value_type, -1 );
        
        if ( m_mode == ATTRIBUTE_TEMPLATE )
          gtk_tree_store_set( store, &iter, ATTRIBUTE_COLUMN, attr_lists[attr_ind]->name.c_str(), -1 );
        else
        {
          if ( m_mode == ATTRIBUTE_DIALOG )
            gtk_tree_store_set( store, &iter, ATTRIBUTE_COLUMN, (attr_lists[attr_ind]->type + ": " + attr_lists[attr_ind]->name).c_str(), 
                                          VALUE_COLUMN, m_stream.str().c_str(), -1 );
          else
          {
            string value_op = "=";
            switch( attr_lists[attr_ind]->value_op )
            {
            case AttributeValue::GREATER:
              value_op =  ">";
              break;
            
            case AttributeValue::LESS:
              value_op =  "<";
              break;
            
            case AttributeValue::NOT_LESS:
              value_op =  ">=";
              break;
            
            case AttributeValue::NOT_GREATER:
              value_op =  "<=";
              break;
            
            case AttributeValue::NOT_EQUAL:
              value_op =  "<>";
              break;
            
            default:;
            }
            
            gtk_tree_store_set( store, &iter, ATTRIBUTE_COLUMN, attr_lists[attr_ind]->name.c_str(), OPERATION_COLUMN, value_op.c_str(),
                                          VALUE_COLUMN, m_stream.str().c_str(), -1 );
          }
        }
      }
    }
  }  
  
  return attr_ind;
}

unsigned int UnixGraphAttrView::make_alias_attr_map( MetaAttrView &res_meta, StringContainer &full_attr_name, const NodeAttributeList &attr_list, unsigned int ind, bool opened )
{
  for ( ; ind < attr_list.size() - 1 && attr_list[ind]->type != NodeAttribute::ATTR_END; ind++ )
  {
    full_attr_name.push_back( attr_list[ind]->name );
    
    if ( attr_list[ind]->type == NodeAttribute::ATTR_START )
    {
      bool new_opened = opened;
      if ( new_opened )
        new_opened =  m_name_storage->open_attr_level( attr_list[ind]->name );
      
      ind = make_alias_attr_map( res_meta, full_attr_name, attr_list, ++ind, new_opened );
      
      if ( new_opened )
        m_name_storage->close_attr_level( );
    }
    else
    {
      bool                      alias_exists = opened;
      MetaAttrView             *meta = &res_meta;
      MetaAttrViewMap::iterator meta_it;
      unsigned int              i = 1;
      StringContainer           alias;
      string                    name = full_attr_name[full_attr_name.size() - 1];
      
      if ( alias_exists )
        alias_exists = m_name_storage->get_attr_name_alias( name, alias );
      
      if ( !alias_exists )
        alias = full_attr_name;
          
      for (  ; i < alias.size() - 1; i++ )
      {
        if ( (meta_it = meta->meta.find( alias[i] )) != meta->meta.end() )
          meta = meta_it->second;
        else
        {
          MetaAttrView *new_meta = new MetaAttrView( );
          new_meta->name = alias[i];
          
          meta->meta[ alias[i] ] = new_meta;
          meta->meta_order.push_back( new_meta );
          meta = new_meta;
        }
      }
      
      AttrView attr_view;
      attr_view.name = alias[alias.size() - 1];
      attr_view.is_string_type = (attr_list[ind]->value->get_actual_type() == AttributeValue::STRING);
      
      if ( attr_view.is_string_type )
        attr_view.value = "\"" + attr_list[ind]->value->get_string() + "\"";
      else
      {
        attr_view.value = m_name_storage->get_attr_value_alias( name, attr_list[ind]->type, attr_list[ind]->value->get_integer() );
      
        if ( attr_view.value.empty() )
          attr_view.value = attr_list[ind]->value->get_string();
      }
      
      attr_view.value_op = "=";
      switch( attr_list[ind]->value_op )
      {
      case AttributeValue::GREATER:
        attr_view.value_op =  ">";
        break;
      case AttributeValue::LESS:
        attr_view.value_op =  "<";
        break;
      case AttributeValue::NOT_LESS:
        attr_view.value_op =  ">=";
        break;
      case AttributeValue::NOT_GREATER:
        attr_view.value_op =  "<=";
        break;
      case AttributeValue::NOT_EQUAL:
        attr_view.value_op =  "<>";
        break;
      default:;
      }
      
      meta->attr.push_back( attr_view );
      
    }
    full_attr_name.pop_back( );
  }
  
  return ind;
}

void UnixGraphAttrView::set_alias_meta( MetaAttrView *meta, GtkTreeIter *parent_iter )
{
  GtkTreeIter iter = add_to_storage( m_store, parent_iter, "", false, 
                                     ((meta->name.empty())? EMPTY_META_NAME: meta->name), 
                                     "", "" );
  if ( meta->attr.empty() && meta->meta_order.empty() )
    add_to_storage( m_store, &iter, "", false, "", "", "" );
  else
  {
    for ( AttrViewVector::const_iterator vit = meta->attr.begin(); vit != meta->attr.end(); vit++ )
      add_to_storage( m_store, &iter, "", false, vit->name, vit->value_op, vit->value, ((vit->is_string_type)? STRING_VALUE: INTEGER_VALUE) );
  
    for ( MetaAttrViewVector::const_iterator mit = meta->meta_order.begin(); mit != meta->meta_order.end(); mit++ )
      set_alias_meta( *mit, &iter );
  }
}

unsigned int UnixGraphAttrView::make_attr_view( AttrExprView *attr_view, StringContainer &full_attr_name, const NodeAttributeList &attr_list, unsigned int ind, 
                             const string &parent_meta_value, bool opened )
{
  unsigned int len = attr_list.size();
  NodeAttribute *attr;
  bool           negation = false;
  string         op = "";
  
  for ( ;ind < len; ind++ )
  {
    attr = attr_list[ind];
    
    if ( attr->type == NodeAttribute::ATTR_PRED_CLOSE_PARENTHESIS ||
         attr->type == NodeAttribute::ATTR_END )
      break;
    
    if ( attr->type == NodeAttribute::ATTR_PRED_OR )
      op = "|";
    else if ( attr->type == NodeAttribute::ATTR_PRED_AND )
      op = "&";
    else if ( attr->type == NodeAttribute::ATTR_PRED_NOT )
      negation = true;
    else if ( attr->type == NodeAttribute::ATTR_PRED_OPEN_PARENTHESIS )
    {
      AttrExprView *paren = new AttrExprView();
      paren->meta_type = AttrExprView::PARENTHESES;
      paren->name = EXPR_PARENTHESES;
      paren->op = op;
      paren->negation = negation;
      paren->value = "";
      paren->value_op = "";
      
      ind = make_attr_view( paren, full_attr_name, attr_list, ++ind, parent_meta_value, opened );
      attr_view->attr_list.push_back( paren );
      
      op = "";
      negation = false;
    }
    else 
    {
      full_attr_name.push_back( attr->name );
      if ( attr->type == NodeAttribute::ATTR_START )
      {
        string meta_value = parent_meta_value;
        bool   new_opened = opened;
        
        if ( full_attr_name.size() == 1 ) // first meta
        {
          m_name_storage->open_top_attr_level( );          
          new_opened = true;
          if ( attr->value->get_integer() != -1 ) 
            meta_value = attr->value->get_string();
        }
        else if ( new_opened )
          new_opened = m_name_storage->open_attr_level( attr->name );
        
        AttrExprView *paren = new AttrExprView();
        paren->meta_type = AttrExprView::PARENTHESES;
        paren->name = EXPR_PARENTHESES;
        paren->op = op;
        paren->negation = negation;
        paren->value = "";
        paren->value_op = "";
        
        ind = make_attr_view( paren, full_attr_name, attr_list, ++ind, meta_value, new_opened );
        attr_view->attr_list.push_back( paren );
        
        if ( paren->attr_list.empty() )
        {
          StringContainer meta_alias;
          if ( m_name_storage->get_meta_attr_name_alias( meta_alias ) )
          {
            AttrExprView *last_meta = paren;
            last_meta->meta_type = AttrExprView::META_ATTR;
            
            if ( full_attr_name.size() == 1 && !meta_value.empty() ) //first meta has index
              last_meta->name = meta_value + ": " + meta_alias[0];
            else
              last_meta->name = meta_alias[0];
            
            for ( unsigned int i = 1; i < meta_alias.size(); i++ )
            {
              AttrExprView *new_meta = new AttrExprView( );
              new_meta->name = meta_alias[i];
              new_meta->meta_type = AttrExprView::META_ATTR;
              new_meta->op = "";
              new_meta->negation = false;
              new_meta->value = "";
              new_meta->value_op = "";
                  
              last_meta->attr_list.push_back( new_meta );
              last_meta = new_meta;
            }
          }
        }
        
        if ( new_opened )
          m_name_storage->close_attr_level( );
      }
      else //attribute
      {
        AttrExprView             *last_meta;
        bool                      alias_exists = opened;
        unsigned int              i = 1;
        StringContainer           alias;
        string                    name = attr->name;
        stringstream              s_str;
        
        if ( alias_exists )
          alias_exists = m_name_storage->get_attr_name_alias( name, alias );
        
        if ( !alias_exists )
          alias = full_attr_name;
                      

        last_meta = new AttrExprView( );
        last_meta->meta_type = AttrExprView::META_ATTR;
        last_meta->op = op;
        
        if ( parent_meta_value.empty() )
          last_meta->name = alias[0];
        else
          last_meta->name = parent_meta_value + ": " + alias[0];
        
        last_meta->negation = false;
        
        last_meta->value = "";
        last_meta->value_op = "";
        attr_view->attr_list.push_back( last_meta );
        
          
        for ( ; i < alias.size() - 1; i++ )
        {
          AttrExprView *new_meta = new AttrExprView( );
          new_meta->name = alias[i];
          new_meta->meta_type = AttrExprView::META_ATTR;
          new_meta->op = "";
          new_meta->negation = false;
          new_meta->value = "";
          new_meta->value_op = "";
              
          last_meta->attr_list.push_back( new_meta );
          last_meta = new_meta;
        }
        
        AttrExprView *vattr = new AttrExprView( );
        vattr->name = alias[alias.size() - 1];
        vattr->meta_type = AttrExprView::ATTR;
        vattr->op = "";
        vattr->negation = negation;
        vattr->value = "";
        vattr->is_string_type = (attr->value->get_actual_type() == AttributeValue::STRING);
        
        if ( vattr->is_string_type )
        {
          vattr->value = "\"" + attr->value->get_string() + "\"";
        }          
        else if ( attr->value->get_actual_type() != AttributeValue::INT_RANGE )
        {
          vattr->value = m_name_storage->get_attr_value_alias( name, "bool", attr->value->get_integer() );
          if ( vattr->value.empty() )
            vattr->value = attr->value->get_string();
        }
        else
        {
          AttributeIntegerRange *range_value = dynamic_cast<AttributeIntegerRange *>(attr->value);
          ValueSet values;
          string   value_alias;
          
          range_value->get_int_values( values );
          
          for ( ValueSet::const_iterator vit = values.begin(); vit != values.end(); vit++ )
          {
            value_alias = m_name_storage->get_attr_value_alias( name, attr->type, *vit );
            
            if ( !value_alias.empty() )
              vattr->value += value_alias;
            else
            {
              s_str.str("");
              s_str << *vit;
              vattr->value += s_str.str();  
            }
            vattr->value += '\n';
          }            
          
          if ( vattr->value.empty() )
            vattr->value = "0";
          else
            vattr->value = vattr->value.substr( 0, vattr->value.size() - 1 );
        }
        
        switch( attr->value_op )
        {
        case AttributeValue::GREATER:
          vattr->value_op =  ">";
          break;
        case AttributeValue::LESS:
          vattr->value_op =  "<";
          break;
        case AttributeValue::NOT_LESS:
          vattr->value_op =  ">=";
          break;
        case AttributeValue::NOT_GREATER:
          vattr->value_op =  "<=";
          break;
        case AttributeValue::NOT_EQUAL:
          vattr->value_op =  "<>";
          break;
        default:
          vattr->value_op =  "=";
        }
              
        last_meta->attr_list.push_back( vattr );
      }
      
      full_attr_name.pop_back( );
      
      op = "";
      negation = false;
    }
  }
  
  return ind;
}

void UnixGraphAttrView::set_alias_attributes( const NodeAttributeList &attr_list )
{
  if ( GraphAttributes::is_predicate(attr_list) )
  {
    AttrExprView attr_view;
    StringContainer full_attr_name;
    
    attr_view.value = "";
    attr_view.value_op = "";
    attr_view.name = "";
    attr_view.op = "";
    attr_view.negation = false;
    
    make_attr_view( &attr_view, full_attr_name, attr_list, 1, "", false );
    
    attr_view.meta_type = AttrExprView::ATTR;
    optimize_attr_expr( &attr_view );
    attr_view.meta_type = AttrExprView::PARENTHESES;
    
    add_to_storage( &attr_view, m_store, NULL );
  }
  else
  {
    MetaAttrView meta;
    StringContainer full_attr_name;
    string       name;
  
    m_name_storage->open_top_attr_level();
    
    if ( !attr_list[0]->name.empty())
      name = meta.name = attr_list[0]->name;
    else
    {
      StringContainer meta_alias;
      m_name_storage->open_top_alias_level();
      if ( m_name_storage->get_meta_attr_name( meta_alias ) )
        name = meta_alias[0];
      
      if ( m_name_storage->get_meta_attr_name_alias( meta_alias ) )
        meta.name = meta_alias[0];
      else
        meta.name = name;
    }
    
    if ( attr_list[0]->value->get_integer() != -1 )
      meta.name = attr_list[0]->value->get_string() + ": " + meta.name;  
    
    full_attr_name.push_back( name );
    
    make_alias_attr_map( meta, full_attr_name, attr_list, 1, true );
    
    set_alias_meta( &meta, NULL );
  }
}

void UnixGraphAttrView::expand_rows()
{
  if ( m_expanded_rows.empty() )
  {
    GtkTreePath *path = gtk_tree_path_new_first();
    gtk_tree_view_expand_row( GTK_TREE_VIEW( m_tree ), path, false );
    gtk_tree_path_free( path );
  }
  else
  {
    bool wrong_path = false;
    for ( set<string>::iterator p = m_expanded_rows.begin(); p != m_expanded_rows.end() && !wrong_path; p++ )
    {
      GtkTreePath *path = gtk_tree_path_new_from_string( p->c_str() );
      wrong_path = !gtk_tree_view_expand_row( GTK_TREE_VIEW( m_tree ), path, false );
      gtk_tree_path_free( path );
    }
    
    if ( wrong_path )
    {
      m_expanded_rows.clear();
      
      gtk_tree_view_collapse_all( GTK_TREE_VIEW( m_tree ) );
      
      GtkTreePath *path = gtk_tree_path_new_first();
      gtk_tree_view_expand_row( GTK_TREE_VIEW( m_tree ), path, false );
      gtk_tree_path_free( path );
    }
  }
}

void UnixGraphAttrView::set_attribute_string( const string& attr_string )
{
  bool              parse_res;
  NodeAttributeList attr_list;
  bool              aliases = m_name_storage != 0;
  
  if ( aliases )
    aliases = (m_name_storage->get_curr_alias_set() >= 0);
  
  m_attr_string_list.clear();
  
  m_attr_string_list.push_back( attr_string );
  
  parse_res = GraphAttributes::parse_string( attr_string, attr_list );  
    
  gtk_tree_store_clear( m_store );
  
  if ( parse_res )
  {
    if ( aliases )
      set_alias_attributes( attr_list );
    else if ( m_mode == ATTRIBUTE_TEMPLATE &&
         attr_list[0]->type == NodeAttribute::ATTR_START && 
         attr_list[1]->type == NodeAttribute::ATTR_END )
    {
      GtkTreeIter iter;
      GtkTreeIter iter2;
      string      name = ((attr_list[0]->name.empty())? EMPTY_META_NAME: attr_list[0]->name);
      gtk_tree_store_append( m_store, &iter, NULL );
      gtk_tree_store_set( m_store, &iter, ATTRIBUTE_COLUMN, name.c_str(), -1 );  
      gtk_tree_store_append( m_store, &iter2, &iter );
      gtk_tree_store_set( m_store, &iter2, ATTRIBUTE_COLUMN, "", -1 );  
    }
    else
      add_to_storage( m_store, NULL, attr_list, 0, (GraphAttributes::is_predicate( attr_list )? NodeAttribute::ATTR_PRED_CLOSE_PARENTHESIS: NodeAttribute::ATTR_END) );
  }
  
  expand_rows();
}

void UnixGraphAttrView::set_attributes( const StringContainer &attr_lists )
{
  NodeAttributeList attr_list;
  unsigned int      i;
  bool              parse_res;
  StringContainer   temp_attr_lists;
  bool              aliases = m_name_storage != 0;
  
  if ( aliases )
    aliases = (m_name_storage->get_curr_alias_set() >= 0);
  
  temp_attr_lists = attr_lists;
  m_attr_string_list.clear();
  
  m_attr_string_list = temp_attr_lists;
  
  gtk_tree_store_clear( m_store );
  for ( i = 0; i < m_attr_string_list.size(); i++ )
  {
    parse_res = GraphAttributes::parse_string( m_attr_string_list[i], attr_list );  
    
    if ( parse_res )
    {
      if ( aliases )
        set_alias_attributes( attr_list );
      else if ( m_mode == ATTRIBUTE_TEMPLATE && 
           attr_list[0]->type == NodeAttribute::ATTR_START && 
           attr_list[1]->type == NodeAttribute::ATTR_END )
      {
          GtkTreeIter iter;
          m_stream << ((attr_list[0]->name.empty())? EMPTY_META_NAME: attr_list[0]->name);
          
          gtk_tree_store_append( m_store, &iter, NULL );
          gtk_tree_store_set( m_store, &iter, ATTRIBUTE_COLUMN, m_stream.str().c_str(), -1 );  
          gtk_tree_store_append( m_store, &iter, &iter );
          gtk_tree_store_set( m_store, &iter, ATTRIBUTE_COLUMN, "", -1 );  
      }
      else
        add_to_storage( m_store, NULL, attr_list, 0, (GraphAttributes::is_predicate( attr_list )? NodeAttribute::ATTR_PRED_CLOSE_PARENTHESIS: NodeAttribute::ATTR_END) );
    }
  }
  
  if ( m_attr_string_list.size() == 1 )
    expand_rows();
  else
    gtk_tree_view_collapse_all( GTK_TREE_VIEW( m_tree ) );
}

void UnixGraphAttrView::set_attributes( const vector<NodeAttributeList> &attr_lists )
{
  unsigned int      i;
  bool              aliases = m_name_storage != 0;
  
  if ( aliases )
    aliases = (m_name_storage->get_curr_alias_set() >= 0);
  
  m_attr_string_list.clear();
  
  for ( i = 0; i < attr_lists.size(); i++ )
    m_attr_string_list.push_back( GraphAttributes::make_string( attr_lists[i] ) );
  
  gtk_tree_store_clear( m_store );
  
  for ( i = 0; i < attr_lists.size(); i++ )
  {
    if ( aliases )
      set_alias_attributes( attr_lists[i] );
    else 
      add_to_storage( m_store, NULL, attr_lists[i], 0, (GraphAttributes::is_predicate( attr_lists[i] )? NodeAttribute::ATTR_PRED_CLOSE_PARENTHESIS: NodeAttribute::ATTR_END) );
  }
  
  if ( attr_lists.size() == 1 )
    expand_rows();
  else
    gtk_tree_view_collapse_all( GTK_TREE_VIEW( m_tree ) );
}

static void op_sell_data_func( GtkTreeViewColumn *tree_column, GtkCellRenderer *cell, GtkTreeModel *model, GtkTreeIter *iter, gpointer data )
{
  GdkColor color;
  color.red = 0;
  color.green = 0;
  color.blue = 255 << 8;
  g_object_set( G_OBJECT(cell), "foreground_gdk", &color, NULL );
}

static void not_sell_data_func( GtkTreeViewColumn *tree_column, GtkCellRenderer *cell, GtkTreeModel *model, GtkTreeIter *iter, gpointer data )
{
  GdkColor color;
  color.red = 255 << 8;
  color.green = 0;
  color.blue = 0;
  g_object_set( G_OBJECT(cell), "foreground_gdk", &color, NULL );
}

void UnixGraphAttrView::handle_treeview_row_expanded_collapsed_event( GtkTreePath *path, bool expand )
{
  string str_path = string( gtk_tree_path_to_string( path ) );
  if ( expand )
    m_expanded_rows.insert( str_path );
  else
    m_expanded_rows.erase( str_path );
}

static void on_treeview_row_expanded(GtkTreeView *treeview, GtkTreeIter *arg1, GtkTreePath *arg2, gpointer user_data )
{
  UnixGraphAttrView *dialog = (UnixGraphAttrView *)user_data;
  dialog->handle_treeview_row_expanded_collapsed_event( arg2, true );
}

static void on_treeview_row_collapsed(GtkTreeView *treeview, GtkTreeIter *arg1, GtkTreePath *arg2, gpointer user_data )
{
  UnixGraphAttrView *dialog = (UnixGraphAttrView *)user_data;
  dialog->handle_treeview_row_expanded_collapsed_event( arg2, false );
}

void UnixGraphAttrView::create()
{
  GtkTreeViewColumn *column;
  GtkCellRenderer   *render;
  
  switch (m_mode)
  {
  case ATTRIBUTE_DIALOG:
    m_store = gtk_tree_store_new( 3, G_TYPE_INT, G_TYPE_STRING, G_TYPE_STRING );
    break;
  
  case ATTRIBUTE_TEMPLATE:
    m_store = gtk_tree_store_new( 2, G_TYPE_INT, G_TYPE_STRING );
    break;
  
  case SEARCH_TEMPLATE:
    m_store = gtk_tree_store_new( 4, G_TYPE_INT, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING );
    break;
  
  case ATTRIBUTE_EXPRESSION:
    m_store = gtk_tree_store_new( 6, G_TYPE_INT, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING );
    break;
  }
  
  m_tree = gtk_tree_view_new_with_model( GTK_TREE_MODEL(m_store) );
  
  switch( m_mode )
  {
  case ATTRIBUTE_EXPRESSION:
    render = gtk_cell_renderer_text_new( );
    column = gtk_tree_view_column_new_with_attributes( "", render, "text", EXPR_BIN_OPERATOR_COLUMN, NULL );
    gtk_tree_view_append_column( GTK_TREE_VIEW( m_tree ), column );
  
    gtk_tree_view_column_set_cell_data_func( column, render, (GtkTreeCellDataFunc)op_sell_data_func, NULL, NULL);
  
    render = gtk_cell_renderer_text_new( );
    column = gtk_tree_view_column_new_with_attributes( "", render, "text", EXPR_UN_OPERATOR_COLUMN, NULL );
    gtk_tree_view_append_column( GTK_TREE_VIEW( m_tree ), column );
    gtk_tree_view_column_set_cell_data_func( column, render, (GtkTreeCellDataFunc)not_sell_data_func, NULL, NULL);
  
  case ATTRIBUTE_TEMPLATE:
  case ATTRIBUTE_DIALOG:
  case SEARCH_TEMPLATE:
    render = gtk_cell_renderer_text_new( );
    column = gtk_tree_view_column_new_with_attributes( "Attribute", render, "text", ATTRIBUTE_COLUMN, NULL );
    gtk_tree_view_append_column( GTK_TREE_VIEW( m_tree ), column );
    gtk_tree_view_set_expander_column( GTK_TREE_VIEW( m_tree ), column );
  
    if ( m_mode != ATTRIBUTE_TEMPLATE )
    {
      if (m_mode != ATTRIBUTE_DIALOG )
      {
        render = gtk_cell_renderer_text_new( );
        column = gtk_tree_view_column_new_with_attributes( " ", render, "text", OPERATION_COLUMN, NULL );
        gtk_tree_view_append_column( GTK_TREE_VIEW( m_tree ), column );
        gtk_tree_view_column_set_cell_data_func( column, render, (GtkTreeCellDataFunc)op_sell_data_func, NULL, NULL);
      }    
    
      render = gtk_cell_renderer_text_new( );
      column = gtk_tree_view_column_new_with_attributes( "Value", render, "text", VALUE_COLUMN, NULL );
      gtk_tree_view_append_column( GTK_TREE_VIEW( m_tree ), column );
    }
  }
  
  gtk_tree_view_set_headers_clickable( GTK_TREE_VIEW( m_tree ), FALSE );
    
  if ( m_mode != ATTRIBUTE_DIALOG )
    gtk_widget_set_size_request( m_tree, 250, 320 );
  else
    gtk_widget_set_size_request( m_tree, 250, 350 );
  
  gtk_signal_connect (GTK_OBJECT (m_tree), "row_collapsed",
                      GTK_SIGNAL_FUNC (on_treeview_row_collapsed),
                      (gpointer)this);    
  gtk_signal_connect (GTK_OBJECT (m_tree), "row_expanded",
                      GTK_SIGNAL_FUNC (on_treeview_row_expanded),
                      (gpointer)this);    
  
  dialog = NULL;
  scrolledwindow = NULL;
}

UnixGraphAttrView::UnixGraphAttrView( GraphAttrNameStorage *name_storage, AttrViewModes mode )
{
  m_mode = mode;
  m_docked = false;
  m_name_storage = name_storage;
  
  TYPE_COLUMN = 0;
    
  switch( mode )
  {
  case ATTRIBUTE_DIALOG:
    EXPR_BIN_OPERATOR_COLUMN = -1;
    EXPR_UN_OPERATOR_COLUMN = -1;
    ATTRIBUTE_COLUMN = 1;
    OPERATION_COLUMN = -1;
    VALUE_COLUMN = 2;
    break;
  
  case ATTRIBUTE_TEMPLATE:
    EXPR_BIN_OPERATOR_COLUMN = -1;
    EXPR_UN_OPERATOR_COLUMN = -1;
    ATTRIBUTE_COLUMN = 1;
    OPERATION_COLUMN = -1;
    VALUE_COLUMN = -1;
    break;
  
  case SEARCH_TEMPLATE:
    EXPR_BIN_OPERATOR_COLUMN = -1;
    EXPR_UN_OPERATOR_COLUMN = -1;
    ATTRIBUTE_COLUMN = 1;
    OPERATION_COLUMN = 2;
    VALUE_COLUMN = 3;
    break;
  
  default:
    EXPR_BIN_OPERATOR_COLUMN = 1;
    EXPR_UN_OPERATOR_COLUMN = 2;
    ATTRIBUTE_COLUMN = 3;
    OPERATION_COLUMN = 4;
    VALUE_COLUMN = 5;
    break;
  }
  
  create();
}

void UnixGraphAttrView::handle_close_event(  )
{
  if ( dialog != NULL || scrolledwindow != NULL )
  {
    CloseHandlerIter  iter = m_close_handler_list.begin();
    
    for ( ;iter != m_close_handler_list.end(); iter++ )
      (**iter)( );
    
    dialog = NULL;
    scrolledwindow = NULL;
  }
}

void UnixGraphAttrView::handle_dock_menu_event( bool docked )
{
  list<GUIDockMenuHandler*>::iterator  iter = m_dock_menu_handler_list.begin();
        
  for (; iter != m_dock_menu_handler_list.end(); ++iter)
    (**iter)( docked );
}

static void on_dock_menu_activate(GtkMenuItem *menuitem, gpointer user_data )
{
  UnixGraphAttrView *dialog = (UnixGraphAttrView *)user_data;
  dialog->handle_dock_menu_event( (bool)( gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(menuitem)) ) );
}

static gboolean on_button_release_event( GtkWidget *widget, GdkEventButton  *event, gpointer user_data )
{
  UnixGraphAttrView *dialog = (UnixGraphAttrView *)user_data;
  
  if ( event->button == 3 && dialog->is_dock_handler_registered() )
  {
    GtkWidget *dock_menu;
    GtkWidget *popup_menu = gtk_menu_new ();
    gtk_widget_show (popup_menu);
    
    dock_menu = gtk_check_menu_item_new_with_mnemonic( "Docked" );
    gtk_widget_show (dock_menu);
    gtk_container_add (GTK_CONTAINER (popup_menu), dock_menu);
    gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM(dock_menu), dialog->is_docked() );
    
    gtk_signal_connect( GTK_OBJECT (dock_menu), "activate",
                          GTK_SIGNAL_FUNC (on_dock_menu_activate),
                          (gpointer)dialog);
    
    gtk_menu_popup (GTK_MENU (popup_menu), NULL, NULL, NULL, NULL,
                      0, event->time);
  }
  
  return FALSE;
}

static void on_response_event( GtkWidget *widget )
{
  UnixGraphAttrView *dialog = (UnixGraphAttrView *)g_object_get_data( G_OBJECT(widget), "dialog_object" );
  dialog->handle_close_event(  );
  
  gtk_widget_destroy( widget );
  
  delete dialog;
}

void UnixGraphAttrView::show_dialog()
{
  dialog = gtk_dialog_new();
  gtk_window_set_title( GTK_WINDOW(dialog), "Attributes...");
  //g_signal_connect_swapped( GTK_OBJECT(information_window), "response", G_CALLBACK(on_close_dialog), GTK_OBJECT(information_window) );
  gtk_dialog_add_button( GTK_DIALOG(dialog), GTK_STOCK_QUIT, GTK_RESPONSE_CLOSE );
  g_signal_connect_swapped( GTK_OBJECT(dialog), "response", G_CALLBACK(on_response_event), GTK_OBJECT(dialog) );
  
  scrolledwindow = gtk_scrolled_window_new (NULL, NULL);
  gtk_widget_show (scrolledwindow);
  gtk_box_pack_start (GTK_BOX( GTK_DIALOG(dialog)->vbox ), scrolledwindow, TRUE, TRUE, 0);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolledwindow), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (scrolledwindow), GTK_SHADOW_ETCHED_IN);

  gtk_widget_show( m_tree );
  gtk_container_add (GTK_CONTAINER (scrolledwindow), m_tree);  

  gtk_widget_show( dialog );
  
  g_object_set_data( G_OBJECT(dialog), "dialog_object", (gpointer)this );
  
  gtk_signal_connect( GTK_OBJECT (dialog), "button_release_event",
                      GTK_SIGNAL_FUNC (on_button_release_event),
                      (gpointer)this);
}

void UnixGraphAttrView::show( bool show_window )
{
  if ( !m_docked )
  {
    if ( dialog != NULL )
    {
      if ( show_window )
        gtk_widget_show( dialog );
      else
        gtk_widget_hide( dialog );
    }
    else if ( show_window )
      show_dialog();
  }
}

void UnixGraphAttrView::get_path( GtkTreeModel *model, GtkTreeIter *iter, vector<string> &path )
{
  GtkTreeIter parent;
  gchar       *attr_name;
  
  if ( gtk_tree_model_iter_parent( model, &parent, iter) )
  {
    get_path( model, &parent, path );
  }
  
  gtk_tree_model_get( model, iter, ATTRIBUTE_COLUMN, &attr_name, -1 );
  
  path.push_back( string( attr_name ) );
  g_free( attr_name );
  
}

bool UnixGraphAttrView::get_selected_path( vector<string> &path )
{
  int               value_type = INTEGER_VALUE;
  GtkTreeIter       sel_iter;
  GtkTreeModel     *model;
  GtkTreeSelection *selection = gtk_tree_view_get_selection( GTK_TREE_VIEW( m_tree ) );
  
  if ( gtk_tree_selection_get_selected( selection, &model, &sel_iter) )
  {
    path.clear();
    path.resize(0);
    get_path( model, &sel_iter, path );
    
    if ( gtk_tree_model_iter_has_child( model, &sel_iter) )
    {
      path.push_back( "" );
    }
    gtk_tree_model_get( model, &sel_iter, TYPE_COLUMN, &value_type, -1 );
  }
  
  return (value_type == STRING_VALUE);
}

void UnixGraphAttrView::insert_path_value( const vector<string> &path, const string &value, const string &relation, bool negation, const string &op, bool is_string )
{
  unsigned int      path_ind = 0, ind = 0;
  GtkTreeIter       sel_iter;
  GtkTreeIter       parent_iter;
  GtkTreeIter       child_iter;
  GtkTreeIter      iter;
  GtkTreeModel     *model;
  GtkTreeSelection *selection = gtk_tree_view_get_selection( GTK_TREE_VIEW( m_tree ) );
  bool              selected  = gtk_tree_selection_get_selected( selection, &model, &sel_iter);
  bool              has_root = false;
  bool              has_child = true;
  GtkTreePath      *selected_path = NULL;
  gchar            *attr_name;
  gint              n_children = 0;
  gint              i;
  string            str_attr_name = "";
  string            str2_attr_name = "";
  unsigned int      colon_ind = 0;
  gint             *sel_path_indexes = 0;
  gint              sel_path_depth = 0;
  int               insert_value_ind = -1;
  
  if ( path.size() == 1 && path[0] == EXPR_PARENTHESES )
  {
    if ( !gtk_tree_model_get_iter_first( model, &parent_iter ) )
      gtk_tree_store_insert( m_store, &iter, NULL, 0 );
    else
    {
      if ( selected )
      {
        gtk_tree_model_iter_parent( model, &parent_iter, &sel_iter );
        selected_path = gtk_tree_model_get_path( model, &sel_iter );
        sel_path_depth = gtk_tree_path_get_depth( selected_path );
        sel_path_indexes = gtk_tree_path_get_indices( selected_path );
        
        insert_value_ind = sel_path_indexes[sel_path_depth - 1] + 1; //add after selection          
      }
      else
        insert_value_ind = gtk_tree_model_iter_n_children( model, &parent_iter ); // add to the end
      
      if ( insert_value_ind > 0 )
      {
        gtk_tree_model_iter_nth_child( model, &child_iter, &parent_iter, insert_value_ind - 1 );
        
        gtk_tree_model_get( model, &child_iter, ATTRIBUTE_COLUMN, &attr_name, -1 );
            
        if ( string( attr_name ) == "" )
        {
          gtk_tree_store_remove( m_store, &child_iter );
          insert_value_ind = 0;
        }
        g_free( attr_name );
      }      
      gtk_tree_store_insert( m_store, &iter, &parent_iter, insert_value_ind );
      
    }
    
    gtk_tree_store_set( m_store, &iter, TYPE_COLUMN, ((is_string)? STRING_VALUE: INTEGER_VALUE), 
                                        ATTRIBUTE_COLUMN, EXPR_PARENTHESES.c_str(), -1 );
    
    if ( insert_value_ind > 0 )
      gtk_tree_store_set( m_store, &iter, EXPR_BIN_OPERATOR_COLUMN, op.c_str(), -1 );
    
    if ( insert_value_ind >= 0 && negation )
      gtk_tree_store_set( m_store, &iter, EXPR_UN_OPERATOR_COLUMN, "!", -1 );
    
    gtk_tree_store_insert( m_store, &parent_iter, &iter, 0 );
    
    gtk_tree_store_set( m_store, &parent_iter, ATTRIBUTE_COLUMN, "", -1 );
    
    if ( selected_path != NULL )
    gtk_tree_path_free( selected_path );
  
    gtk_tree_view_expand_all( GTK_TREE_VIEW( m_tree ) );  
    gtk_tree_selection_select_iter( selection, &parent_iter );
    
    return;
  }
  
  if ( gtk_tree_model_get_iter_first( model, &parent_iter ) )
  {
    has_root    = true;
    iter = parent_iter;
    
    gtk_tree_model_get( model, &parent_iter, ATTRIBUTE_COLUMN, &attr_name, -1 );
      
    while ( string( attr_name ) == EXPR_PARENTHESES )
    {
      n_children = gtk_tree_model_iter_n_children( model, &parent_iter );
      if ( n_children <= 0 )
        break;
      
      gtk_tree_model_iter_nth_child( model, &child_iter, &parent_iter, 0 );
      
      g_free( attr_name );
      
      parent_iter = child_iter;
      gtk_tree_model_get( model, &parent_iter, ATTRIBUTE_COLUMN, &attr_name, -1 );
    }
    
    g_free( attr_name );
    
    if ( selected )
    {
      GtkTreePath  *iter_path;
      
      //get selection tree
      selected_path = gtk_tree_model_get_path( model, &sel_iter );
      sel_path_depth = gtk_tree_path_get_depth( selected_path );
      sel_path_indexes = gtk_tree_path_get_indices( selected_path );
      iter_path = gtk_tree_path_new();
      
      //find first attr. level that should be added
      for( i = 0, ind = 0; (i < sel_path_depth - 1) && (ind < path.size() - 1); i++, ind++ )
      {
        gtk_tree_path_append_index( iter_path, sel_path_indexes[i] );
        gtk_tree_model_get_iter( model, &iter, iter_path );
        gtk_tree_model_get( model, &iter, ATTRIBUTE_COLUMN, &attr_name, -1 );
        
        //skip all parentheses
        if ( string( attr_name ) == EXPR_PARENTHESES )
        {
          for ( i++; i < sel_path_depth - 1; i++ )
          {
            g_free( attr_name );
            
            gtk_tree_path_append_index( iter_path, sel_path_indexes[i] );
            gtk_tree_model_get_iter( model, &iter, iter_path );
            gtk_tree_model_get( model, &iter, ATTRIBUTE_COLUMN, &attr_name, -1 );
            
            if ( string( attr_name ) != EXPR_PARENTHESES )
              break;
          }
        }
        
        if ( i >= sel_path_depth )
          break;
        
        if ( (colon_ind = path[ ind ].find(':')) != string::npos )
          str_attr_name = path[ ind ].substr( colon_ind + 2 );
        else
          str_attr_name = path[ ind ];
      
        str2_attr_name = string( attr_name );
        if ( (colon_ind = str2_attr_name.find(':')) != string::npos )
          str2_attr_name = str2_attr_name.substr( colon_ind + 2 );
          
        g_free( attr_name );
        
        // 
        if ( !(str_attr_name == str2_attr_name || str2_attr_name == EMPTY_META_NAME) )
          break;
      }
      
      if ( i >= sel_path_depth - 1 && i > 0 )
      {
        parent_iter = iter;  
        path_ind = ind;
      }
      else if ( i < sel_path_depth - 1 )
      {
        if ( ind < path.size() - 1 )
          i++;
        gtk_tree_path_append_index( iter_path, sel_path_indexes[i] );
        gtk_tree_model_get_iter( model, &iter, iter_path );
        gtk_tree_model_get( model, &iter, ATTRIBUTE_COLUMN, &attr_name, -1 );
          
         //skip all parentheses
        if ( string( attr_name ) == EXPR_PARENTHESES )
        {
          path_ind = ind;
          parent_iter = iter;
            
          for ( i++; i < sel_path_depth - 1; i++ )
          {
            g_free( attr_name );
              
            gtk_tree_path_append_index( iter_path, sel_path_indexes[i] );
            gtk_tree_model_get_iter( model, &iter, iter_path );
            gtk_tree_model_get( model, &iter, ATTRIBUTE_COLUMN, &attr_name, -1 );
              
            if ( string( attr_name ) != EXPR_PARENTHESES )
              break;
              
            parent_iter = iter;
          }
        }
        else
          path_ind = 1;
      }
      else
        path_ind = 1;
      
      gtk_tree_path_free( iter_path );    
      
    }
    
    for( ; path_ind < path.size() - 1 && has_child && has_root; path_ind++ )
    {
      str_attr_name = path[ path_ind ];
      
      n_children = gtk_tree_model_iter_n_children( model, &parent_iter );
      if ( n_children > 0 )
      {
        for ( i = 0; i < n_children; i++ )
        {
          gtk_tree_model_iter_nth_child( model, &child_iter, &parent_iter, i );
          gtk_tree_model_get( model, &child_iter, ATTRIBUTE_COLUMN, &attr_name, -1 );
  
          str2_attr_name = string( attr_name );
  
          g_free( attr_name );
              
          if ( str_attr_name == str2_attr_name )
            break;
        }
            
        if ( i >= n_children )
          break;
        parent_iter = child_iter;
      }        
      else
      {
        child_iter = parent_iter;
        has_child = gtk_tree_model_iter_next( model, &child_iter);
        if ( has_child )
          parent_iter = child_iter;
      }
    }
  }
  
  if ( path_ind < path.size() && path.size() > 0 )
  {
    bool is_first_at_level = true;
    int last_child_without_children = -1;

    n_children = 0;
    
    if ( has_root )
      n_children = gtk_tree_model_iter_n_children( model, &parent_iter );
    
    if ( !(n_children > 0 && path[path_ind] == "") )
    {
      if ( n_children > 0 )
      {
        for ( i = 0; i < n_children; i++ )
        {
          gtk_tree_model_iter_nth_child( model, &child_iter, &parent_iter, i );
          gtk_tree_model_get( model, &child_iter, ATTRIBUTE_COLUMN, &attr_name, -1 );
          
          if ( !gtk_tree_model_iter_has_child( model, &child_iter ) )
            last_child_without_children = i;
                    
          if ( (path[ path_ind ] == string( attr_name ) && path_ind == (path.size() - 1)) || string( attr_name ) == "" )
          {
            g_free( attr_name );
            gtk_tree_store_remove( m_store, &child_iter );
            
            insert_value_ind = i;
            
            break;
          }
          g_free( attr_name );
        }
        is_first_at_level = (i == 0);
      }
      
      if ( has_root )
      {
        if ( insert_value_ind < 0 && selected )
        {
          GtkTreePath *parent_path = gtk_tree_model_get_path( model, &parent_iter );
          
          if ( sel_path_depth - 1 > 0 )
          {
            gtk_tree_path_up( selected_path );
            if ( gtk_tree_path_compare( selected_path, parent_path ) == 0 ) //add new attr. to the same level where selection is
              insert_value_ind = sel_path_indexes[sel_path_depth - 1] + 1; //add after selection
          }
          
          gtk_tree_path_free( parent_path );
        }
        
        if ( insert_value_ind < 0 )
        {
          if ( path[path.size() - 1] == "" || path_ind != path.size() - 1 )
            insert_value_ind = gtk_tree_model_iter_n_children( model, &parent_iter ) + 1; // add meta-attr. to the end
          else
            insert_value_ind = last_child_without_children + 1; // add attr. before meta-attr.
        }
        iter = parent_iter;
        gtk_tree_store_insert( m_store, &parent_iter, &iter, insert_value_ind );
      }
      else
        gtk_tree_store_insert( m_store, &parent_iter, NULL, 0 );
  
      bool op_is_set = is_first_at_level; // ignore op. if it is first attr. at level 
            
      if ( path_ind < path.size() - 1 )
      {      
        gtk_tree_store_set( m_store, &parent_iter, TYPE_COLUMN, ((is_string)? STRING_VALUE: INTEGER_VALUE),
                                                   ATTRIBUTE_COLUMN, path[path_ind++].c_str(), -1 );
        
        if ( m_mode == ATTRIBUTE_EXPRESSION )
        {
          if ( !op_is_set )
          {
            gtk_tree_store_set( m_store, &parent_iter, EXPR_BIN_OPERATOR_COLUMN, op.c_str(), -1 );
            op_is_set = true;
          }
        }
      
        for ( ; path_ind < path.size() - 1; path_ind++ )
        {
          iter = parent_iter;
          gtk_tree_store_insert( m_store, &parent_iter, &iter, 0 );
          gtk_tree_store_set( m_store, &parent_iter, ATTRIBUTE_COLUMN, path[path_ind].c_str(), -1 );
        }
        iter = parent_iter;      
        gtk_tree_store_insert( m_store, &parent_iter, &iter, 0 );
      }
      
      if ( m_mode == ATTRIBUTE_EXPRESSION )   
      {
        if ( !op_is_set )
          gtk_tree_store_set( m_store, &parent_iter, EXPR_BIN_OPERATOR_COLUMN, op.c_str(), -1 );
        else
          gtk_tree_store_set( m_store, &parent_iter, EXPR_BIN_OPERATOR_COLUMN, "", -1 );
          
        if ( negation )
          gtk_tree_store_set( m_store, &parent_iter, EXPR_UN_OPERATOR_COLUMN, "!", -1 );
        else
          gtk_tree_store_set( m_store, &parent_iter, EXPR_UN_OPERATOR_COLUMN, "", -1 );
      }
      
      gtk_tree_store_set( m_store, &parent_iter, ATTRIBUTE_COLUMN, path[path_ind].c_str(), -1 );
      
      if ( path[path_ind] != "" && m_mode != ATTRIBUTE_TEMPLATE )
      { 
        if ( m_mode == SEARCH_TEMPLATE || m_mode == ATTRIBUTE_EXPRESSION )
          gtk_tree_store_set( m_store, &parent_iter, OPERATION_COLUMN, relation.c_str(), -1 );
        
        gtk_tree_store_set( m_store, &parent_iter, VALUE_COLUMN, value.c_str(), -1 );
      }
    }      
  }
  
  if ( selected_path != NULL )
    gtk_tree_path_free( selected_path );
  
  gtk_tree_view_expand_all( GTK_TREE_VIEW( m_tree ) );  
  gtk_tree_selection_select_iter( selection, &parent_iter );  
}

void UnixGraphAttrView::remove_selected()
{
  GtkTreeIter      iter;
  GtkTreeIter      sel_iter;
  GtkTreeIter      parent_iter;
  GtkTreeModel     *model;
  GtkTreeSelection *selection = gtk_tree_view_get_selection( GTK_TREE_VIEW( m_tree ) );
  bool              selected  = gtk_tree_selection_get_selected( selection, &model, &sel_iter);
  bool              has_parent;
    
  if ( selected )
  {
    has_parent = gtk_tree_model_iter_parent( model, &parent_iter, &sel_iter );
    iter = sel_iter;
    
    if ( !gtk_tree_model_iter_next( model, &sel_iter ) )
    {
      GtkTreePath *path = gtk_tree_model_get_path( model, &iter );
      
      if ( gtk_tree_path_prev( path ) )
        gtk_tree_model_get_iter( model, &sel_iter, path );
      
      gtk_tree_path_free( path );
    }
    
    gtk_tree_store_remove( m_store, &iter );
    
    if ( has_parent )
    {
      if ( !gtk_tree_model_iter_has_child( model, &parent_iter ) )
      {
        gtk_tree_store_insert( m_store, &sel_iter, &parent_iter, 0 );
        gtk_tree_store_set( m_store, &sel_iter, ATTRIBUTE_COLUMN, "", -1 );
      }
      else
      {
        if ( m_mode == ATTRIBUTE_EXPRESSION )
        {
          gtk_tree_model_iter_children( model, &iter, &parent_iter );
          gtk_tree_store_set( m_store, &iter,  EXPR_BIN_OPERATOR_COLUMN, "", -1 );
        }
      }
      
      gtk_tree_view_expand_all( GTK_TREE_VIEW( m_tree ) );  
      gtk_tree_selection_select_iter( selection, &sel_iter );
    }
  }
}

void UnixGraphAttrView::make_attr_expr_view( AttrExprView* meta, StringContainer &full_name, GtkTreeModel *model, GtkTreeIter *parent_iter, const string &parent_meta_value, bool opened )
{
  GtkTreeIter  iter;
  gchar        *bin_op = NULL;
  gchar        *un_op = NULL;
  gchar        *value_op = NULL;
  gchar        *value = NULL;
  gchar        *str_data = NULL;
  string       str_bin_op;
  string       str_un_op;
  string       str_value_op;
  string       str_value;
  gboolean     valid = gtk_tree_model_iter_children( model, &iter, parent_iter );
  string       name;
  gboolean     is_level;
  int          value_type = INTEGER_VALUE;
  
  while ( valid )
  {
    bin_op = 0;
    un_op = 0;
    value_op = 0;
    value = 0;
    str_data = 0;
    str_bin_op = "";
    str_un_op = "";
    str_value_op = "";
    str_value = "";
    
    gtk_tree_model_get( model, &iter, ATTRIBUTE_COLUMN, &str_data, -1 );
    name = string( str_data );
    
    if ( !name.empty() )
    {
      gtk_tree_model_get( model, &iter, TYPE_COLUMN, &value_type, -1 );
      gtk_tree_model_get( model, &iter, VALUE_COLUMN, &value, -1 );
      
      if ( value != 0 )
      {
        str_value = string(value);
        g_free( value );
      }
      
      gtk_tree_model_get( model, &iter, OPERATION_COLUMN, &value_op, -1 );
      if ( value_op != 0 )
      {
        str_value_op = string(value_op);
        g_free( value_op );
      }
      gtk_tree_model_get( model, &iter, EXPR_BIN_OPERATOR_COLUMN, &bin_op, -1 );
      if ( bin_op != 0 )
      {
        str_bin_op = string( bin_op );
        g_free( bin_op );
      }
      gtk_tree_model_get( model, &iter, EXPR_UN_OPERATOR_COLUMN, &un_op, -1 );
      if ( un_op != 0)
      {
        str_un_op = string( un_op );
        g_free( un_op );
      }
      
      is_level = gtk_tree_model_iter_has_child( model, &iter );
      if ( name == EXPR_PARENTHESES && is_level ) // parentheses level
      {
        AttrExprView* attr_paren = new AttrExprView();
        attr_paren->meta_type = AttrExprView::PARENTHESES;
        attr_paren->op = str_bin_op;
        attr_paren->negation = !(str_un_op.empty());
        attr_paren->value = "";
        attr_paren->value_op = "";
        
        make_attr_expr_view( attr_paren, full_name, model, &iter, parent_meta_value, opened );
        
        meta->attr_list.push_back( attr_paren );
      }
      else
      {
        full_name.push_back( name );
        
        if ( is_level ) //meta attr. level
        {
          bool new_opened = opened;
          string  meta_value = parent_meta_value;
          AttrExprView* attr_meta = new AttrExprView();
          attr_meta->meta_type = AttrExprView::PARENTHESES;
          attr_meta->op = str_bin_op;
          if ( !attr_meta->op.empty() )
            attr_meta->op = " " + attr_meta->op + " ";
          attr_meta->negation = !(str_un_op.empty());
          attr_meta->value = "";
          attr_meta->value_op = "";
        
          if ( full_name.size() == 1 ) //first meta
          {
            unsigned int pos;
            m_name_storage->open_top_alias_level();
            new_opened = true;
            
            if ( (pos = name.find(':')) != string::npos )
            {
              meta_value = name.substr( 0, pos );
              name = name.substr( pos + 1 );
              
              full_name.pop_back();
              full_name.push_back( name );
            }
          }
          else if ( new_opened )
            new_opened = m_name_storage->open_alias_level( name );
          
          make_attr_expr_view( attr_meta, full_name, model, &iter, meta_value, new_opened );
          
          if ( new_opened )
            m_name_storage->close_alias_level( );
          
          meta->attr_list.push_back( attr_meta );
          
          if ( attr_meta->attr_list.empty() )
          {
            StringContainer meta_name;
            if ( m_name_storage->get_meta_attr_name( meta_name ) )
            {
              AttrExprView *last_meta = attr_meta;
              last_meta->meta_type = AttrExprView::META_ATTR;
              
              if ( full_name.size() == 1 && !meta_value.empty() ) //first meta has index
                last_meta->name = meta_value + ": " + meta_name[0];
              else
                last_meta->name = meta_name[0];
              
              for ( unsigned int i = 1; i < meta_name.size(); i++ )
              {
                AttrExprView *new_meta = new AttrExprView( );
                new_meta->name = meta_name[i];
                new_meta->meta_type = AttrExprView::META_ATTR;
                new_meta->op = "";
                new_meta->negation = false;
                new_meta->value = "";
                new_meta->value_op = "";
                    
                last_meta->attr_list.push_back( new_meta );
                last_meta = new_meta;
              }
            }
          }
        }
        else // attr
        {
          bool                      alias_exists = opened;
          AttrExprView             *last_meta;
          unsigned int              i = 0, pos = 0;
          StringContainer           attr_name;
          long                      lvalue = 0;
          string                    value_alias = str_value;
          stringstream              s_str;
                    
          if ( alias_exists )
            alias_exists = m_name_storage->get_attr_name( name, attr_name );
          
          if ( !alias_exists )
            attr_name = full_name;

          last_meta = new AttrExprView( );
          last_meta->meta_type = AttrExprView::META_ATTR;
          last_meta->op = str_bin_op;
            
          if ( !last_meta->op.empty() )
            last_meta->op = " " + last_meta->op + " ";
            
          if ( parent_meta_value.empty() )
            last_meta->name = attr_name[0];
          else
            last_meta->name = parent_meta_value + ": " + attr_name[0];
          last_meta->negation = false;
          last_meta->value = "";
          last_meta->value_op = "";
          meta->attr_list.push_back( last_meta );
          
          for ( i = 1; i < attr_name.size() - 1; i++ )
          {
            AttrExprView *new_meta = new AttrExprView( );
            new_meta->name = attr_name[i];
            new_meta->meta_type = AttrExprView::META_ATTR;
            new_meta->op = "";
            new_meta->negation = false;
            new_meta->value = "";
            new_meta->value_op = "";
              
            last_meta->attr_list.push_back( new_meta );
            last_meta = new_meta;
          }
          
          if ( (pos = value_alias.find( '\n' )) != string::npos ) // aliased range
          {
            vector<long> values;
            string single_value;
            
            while ( pos != string::npos )
            {            
              single_value = value_alias.substr( 0, pos); 
              value_alias = value_alias.substr( pos + 1 ); 
              
              if ( !m_name_storage->get_attr_value( name, "bool", single_value, lvalue ) )
                sscanf( single_value.c_str(), "%ld", &lvalue );
               
              values.push_back( lvalue );            
              pos = value_alias.find( '\n' );
            }
            
            if ( !m_name_storage->get_attr_value( name, "bool", value_alias, lvalue ) )
              sscanf( value_alias.c_str(), "%ld", &lvalue );
               
            values.push_back( lvalue );
            
            unsigned int lind = 0;
            for ( ; lind < values.size() - 1; lind++ )
              s_str << values[lind] << ",";
            
            s_str << values[lind];
            value_alias = s_str.str();
          }
          else if ( value_alias == "" || 
               m_name_storage->get_attr_value( name, "bool", value_alias, lvalue ) )
          {
            s_str << lvalue;
            value_alias = s_str.str();
          }
                  
          AttrExprView *attr = new AttrExprView( );
          attr->name = attr_name[ attr_name.size() - 1 ];
          attr->meta_type = AttrExprView::ATTR;
          attr->op = "";
          attr->negation = !(str_un_op.empty());
          attr->value = value_alias;
          attr->value_op = " " + str_value_op + " ";
          attr->is_string_type = (value_type == STRING_VALUE);
              
          last_meta->attr_list.push_back( attr );

        }
        
        full_name.pop_back( );
      }
    }
    
    g_free( str_data );
    
    valid = gtk_tree_model_iter_next( model, &iter );
  }
}

bool UnixGraphAttrView::unite_meta_attr( AttrExprView *meta_attr1,  AttrExprView *meta_attr2, bool first )
{
  bool res = false;
  int meta1_op = (meta_attr1->op.find('&') != string::npos || meta_attr1->op.empty())? 1: 2;
  int meta2_op = (meta_attr2->op.find('&') != string::npos || meta_attr2->op.empty())? 1: 2;
    
  // (a &/| b), & (a & b), |(a &/| b)
  if ( first || meta1_op == meta2_op || meta2_op == 1 )
  {
    if ( meta_attr1->attr_list.size() == 1 && 
         meta_attr2->attr_list.size() == 1 )
    {
      
        AttrExprView *attr = *(meta_attr2->attr_list.begin());
        AttrExprView *attr1 = *(meta_attr1->attr_list.begin());
        
        attr->negation = attr->negation || meta_attr2->negation;
        attr1->negation = attr1->negation || meta_attr1->negation;
        meta_attr1->negation = false;
          
        attr->op = meta_attr2->op;
        
        meta_attr2->attr_list.clear();
        meta_attr1->attr_list.push_back( attr );
        
        res = true;
    }
    else 
    {
      AttrExprList::iterator it;
      int attr1_op = 0;
      int attr2_op = 0;
      AttrExprView *attr;
      
      it = meta_attr1->attr_list.begin();
      
      for ( it++; it != meta_attr1->attr_list.end() && attr1_op != -1; it++ )
      {
        if ( (*it)->op.find('&') != string::npos || (*it)->op.empty() )
          attr1_op = ((attr1_op == 0 || attr1_op == 1)? 1: -1);
        else 
          attr1_op = ((attr1_op == 0 || attr1_op == 2)? 2: -1);
      }
      
      it = meta_attr2->attr_list.begin();
      
      for ( it++; it != meta_attr2->attr_list.end() && attr2_op != -1; it++ )
      {
        if ( (*it)->op.find('&') != string::npos || (*it)->op.empty() )
          attr2_op = ((attr2_op == 0 || attr2_op == 1)? 1: -1);
        else 
          attr2_op = ((attr2_op == 0 || attr2_op == 2)? 2: -1);
      }
      
      // 
      if ( !(attr1_op == 0 || meta2_op == 2 || attr1_op == meta2_op) || 
            (meta_attr1->negation && attr1_op != 0) || meta_attr1->attr_list.empty() )
      {
        attr = new AttrExprView();
        attr->meta_type = AttrExprView::PARENTHESES;
        attr->op = "";
        attr->value = "";
        attr->value_op = "";
        attr->negation = false;
        attr->name = "";
        attr->is_string_type = false;
        attr->attr_list = meta_attr1->attr_list;
        meta_attr1->attr_list.clear();
        meta_attr1->attr_list.push_back( attr );
      }
      
      if ( meta_attr1->negation )
      {
        attr = *(meta_attr1->attr_list.begin());
        attr->negation = !attr->negation;
        meta_attr1->negation = false;
      }        
      
      if ( !(attr2_op == 0 || meta2_op == 2 || attr2_op == meta2_op) ||
            (meta_attr2->negation && attr2_op != 0) || meta_attr2->attr_list.empty() )
      {
        attr = new AttrExprView();
        attr->meta_type = AttrExprView::PARENTHESES;
        attr->op = "";
        attr->value = "";
        attr->value_op = "";
        attr->negation = false;
        attr->name = "";
        attr->is_string_type = false;
        attr->attr_list = meta_attr2->attr_list;
        meta_attr2->attr_list.clear();
        meta_attr2->attr_list.push_back( attr );
      }
      
      if ( meta_attr1->negation )
      {
        attr = *(meta_attr2->attr_list.begin());
        attr->negation = !attr->negation;
        meta_attr2->negation = false;
      }
      
      it = meta_attr2->attr_list.begin();
      attr = *it;
      attr->op = meta_attr2->op;
      meta_attr1->attr_list.push_back( attr );
      
      for ( it++; it != meta_attr2->attr_list.end(); it++ )
        meta_attr1->attr_list.push_back( *it );
      
       meta_attr2->attr_list.clear();
      
      res = true;
    }
  }  
  
  if ( res && !meta_attr1->attr_list.empty() )
    optimize_attr_ored_expr( meta_attr1->attr_list, meta_attr1->attr_list.begin(), meta_attr1->attr_list.end() );
  
  return res;
}

void UnixGraphAttrView::remove_redundant_parentheses( AttrExprView *attr )
{
  if ( attr->attr_list.size() == 1 )
  {
    AttrExprList::iterator it = attr->attr_list.begin();
    AttrExprView *sub_attr = *it;
    
    remove_redundant_parentheses( sub_attr );
      
    if ( attr->meta_type == AttrExprView::PARENTHESES )
    {
      attr->name = sub_attr->name;
      attr->meta_type = sub_attr->meta_type;
      if ( sub_attr->negation )
        attr->negation = !attr->negation;      
        
      attr->attr_list = sub_attr->attr_list;
      sub_attr->attr_list.clear();
        
      if ( sub_attr->meta_type == AttrExprView::ATTR )
      {
        attr->value = sub_attr->value;
        attr->value_op = sub_attr->value_op;
        attr->is_string_type = sub_attr->is_string_type;
      }
      delete sub_attr;
    }
    else if ( attr->meta_type == AttrExprView::META_ATTR && 
              sub_attr->meta_type == AttrExprView::PARENTHESES )
    {
      if ( sub_attr->negation )
        attr->negation = !attr->negation;      
        
      attr->attr_list = sub_attr->attr_list;
      sub_attr->attr_list.clear();
        
      delete sub_attr;
    }
  }
}

AttrExprListIt UnixGraphAttrView::optimize_attr_anded_expr( AttrExprList &attr_list, AttrExprListIt first_it, AttrExprListIt last_it )
{
  AttrExprListIt res_it = first_it;
  if ( first_it != last_it )
  {
    AttrExprView *prev_attr;
    AttrExprView *sub_attr = *first_it;
    bool          united;
    bool          first = true;
    
    if ( !sub_attr->attr_list.empty() )
    {
      optimize_attr_ored_expr( sub_attr->attr_list, sub_attr->attr_list.begin(), sub_attr->attr_list.end() );
      remove_redundant_parentheses( sub_attr );
    }
    
    prev_attr = sub_attr;
    
    for ( first_it++; first_it != last_it; )
    {
      united = false;
      sub_attr = *first_it;
      
      if ( sub_attr->op.find('|') != string::npos )
        break;
      
            
      if ( !sub_attr->attr_list.empty() )
      {
        optimize_attr_ored_expr( sub_attr->attr_list, sub_attr->attr_list.begin(), sub_attr->attr_list.end() );
        remove_redundant_parentheses( sub_attr );
      }
      
      
      if ( prev_attr->meta_type == AttrExprView::META_ATTR && 
           sub_attr->meta_type == AttrExprView::META_ATTR  &&
           prev_attr->name == sub_attr->name ) 
      {
        if ( (united = unite_meta_attr( prev_attr, sub_attr, first )) )
        {
          first_it = attr_list.erase( first_it );
          delete sub_attr;
        }
      }
      
      if ( !united )
      {
        first_it++;
        prev_attr = sub_attr;
        first = false;
      }
    }
  }
  
  return res_it;
}

AttrExprListIt UnixGraphAttrView::optimize_attr_ored_expr( AttrExprList &attr_list, AttrExprListIt first_it, AttrExprListIt last_it )
{
  if ( first_it != last_it )
  {
    AttrExprView *prev_attr;
    AttrExprView *sub_attr;
    bool          united;
    bool          first = true;
      
    first_it = optimize_attr_anded_expr( attr_list, first_it, last_it );
    
    prev_attr = *first_it;
      
    for ( first_it++; first_it != last_it; )
    {
      united = false;
        
      first_it = optimize_attr_anded_expr( attr_list, first_it, last_it );
        
      sub_attr = *first_it;      
        
      if ( prev_attr->meta_type == AttrExprView::META_ATTR && 
           sub_attr->meta_type == AttrExprView::META_ATTR  &&
           prev_attr->name == sub_attr->name ) 
      {
        if ( (united = unite_meta_attr( prev_attr, sub_attr, first )) )
        {
          first_it = attr_list.erase( first_it );
          delete sub_attr;
        }
      }
          
      if ( !united )
      {
        first_it++;
        prev_attr = sub_attr;
        first = false;
      }
    }
  }
  
  return first_it;
}

void UnixGraphAttrView::optimize_attr_expr( AttrExprView *attr )
{
  if ( !attr->attr_list.empty() )
    optimize_attr_ored_expr( attr->attr_list, attr->attr_list.begin(), attr->attr_list.end() );
}

void UnixGraphAttrView::make_string( AttrExprView *attr, string &res_str )
{
  res_str += attr->op;
  if ( attr->negation )
    res_str += "!";
  
  switch( attr->meta_type )
  {
  case AttrExprView::PARENTHESES:
  case AttrExprView::META_ATTR:
    if ( attr->meta_type == AttrExprView::PARENTHESES )
      res_str += "( ";
    else
      res_str += attr->name + "{ ";
    
    for ( AttrExprList::iterator it = attr->attr_list.begin(); it != attr->attr_list.end(); it++ )
      make_string( *it, res_str );
    
    if ( attr->meta_type == AttrExprView::PARENTHESES )
      res_str += ") ";
    else
      res_str += "} ";
    
    break;
    
  default:
    res_str += attr->name + attr->value_op + attr->value + "; ";
  }
}

void UnixGraphAttrView::make_attr_alias_map( MetaAttrView &res_meta, StringContainer &full_alias_name, GtkTreeModel *model, GtkTreeIter *parent_iter, bool opened )
{
  GtkTreeIter  iter;
  gchar        *str_data;
  gboolean     valid = gtk_tree_model_iter_children( model, &iter, parent_iter );
  string       name;
  int          value_type;
  
    
  while ( valid )
  {
    gtk_tree_model_get( model, &iter, ATTRIBUTE_COLUMN, &str_data, -1 );
    gtk_tree_model_get( model, &iter, TYPE_COLUMN, &value_type, -1 );
    
    name = string( str_data );
    
    if ( !name.empty() )
    {
      full_alias_name.push_back( name );
      
      if ( gtk_tree_model_iter_has_child( model, &iter ) )
      {
        bool new_opened = opened;
        
        if ( new_opened )
          new_opened = m_name_storage->open_alias_level( name );
        
        make_attr_alias_map( res_meta, full_alias_name, model, &iter, new_opened );
        
        if ( new_opened )
          m_name_storage->close_alias_level( );
      }
      else
      {
        bool                      alias_exists = opened;
        MetaAttrView             *meta = &res_meta;
        MetaAttrViewMap::iterator meta_it;
        unsigned int              i = 1, pos = 0;
        StringContainer           attr_name;
        long                      value = 0;
        string                    value_alias = "";
        stringstream              s_str;
        
        if ( alias_exists )
          alias_exists = m_name_storage->get_attr_name( name, attr_name );
        
        if ( !alias_exists )
          attr_name = full_alias_name;
            
        for (  ; i < attr_name.size() - 1; i++ )
        {
          if ( (meta_it = meta->meta.find( attr_name[i] )) != meta->meta.end() )
            meta = meta_it->second;
          else
          {
            MetaAttrView *new_meta = new MetaAttrView( );
            new_meta->name = attr_name[i];
            
            meta->meta[ attr_name[i] ] = new_meta;
            meta->meta_order.push_back( new_meta );
            meta = new_meta;
          }
        }
        
        if ( m_mode != ATTRIBUTE_TEMPLATE )
        {
          g_free( str_data );
          gtk_tree_model_get( model, &iter, VALUE_COLUMN, &str_data, -1 );
          value_alias = string( str_data );
        }
        
        if ( value_type != STRING_VALUE )
        {
          if ( (pos = value_alias.find( '\n' )) != string::npos ) // aliased range
          {
            vector<long> values;
            string single_value;
            
            while ( pos != string::npos )
            {            
              single_value = value_alias.substr( 0, pos); 
              value_alias = value_alias.substr( pos + 1 ); 
              
              if ( !m_name_storage->get_attr_value( name, "bool", single_value, value ) )
                sscanf( single_value.c_str(), "%ld", &value );
               
              values.push_back( value );            
              pos = value_alias.find( '\n' );
            }
            
            if ( !m_name_storage->get_attr_value( name, "bool", value_alias, value ) )
              sscanf( value_alias.c_str(), "%ld", &value );
               
            values.push_back( value );
            
            unsigned int lind = 0;
            for ( ; lind < values.size() - 1; lind++ )
              s_str << values[lind] << ",";
            
            s_str << values[lind];
            value_alias = s_str.str();
          }
          else if ( value_alias == "" || 
               m_name_storage->get_attr_value( name, "bool", value_alias, value ) )
          {
            s_str << value;
            value_alias = s_str.str();
          }
        }        
        
        AttrView attr_view;
        attr_view.name = attr_name[attr_name.size() - 1];
        attr_view.value = value_alias;
        attr_view.value_op = "=";
        attr_view.is_string_type = (value_type == STRING_VALUE);
        
        if ( m_mode == SEARCH_TEMPLATE ||
             m_mode == ATTRIBUTE_EXPRESSION )
        {
          g_free( str_data );
          gtk_tree_model_get( model, &iter, OPERATION_COLUMN, &str_data, -1 );
          attr_view.value_op = string( str_data );
        }
        meta->attr.push_back( attr_view );
      }
      
      full_alias_name.pop_back( );
    }
    g_free( str_data );
    valid = gtk_tree_model_iter_next( model, &iter );
  }
}

void UnixGraphAttrView::make_string( MetaAttrView *meta, string &res_str )
{
  res_str += meta->name + "{ ";
    
  for ( AttrViewVector::const_iterator vit = meta->attr.begin(); vit != meta->attr.end(); vit++ )
    res_str += vit->name + " " + vit->value_op + " " + vit->value + "; ";
  
  for ( MetaAttrViewVector::const_iterator mit = meta->meta_order.begin(); mit != meta->meta_order.end(); mit++ )
  {
    make_string( *mit, res_str );
  }
  res_str += "}";
}

string UnixGraphAttrView::make_aliased_attr_to_string( GtkTreeModel *model )
{
  string          res = "";
  GtkTreeIter     iter;
  gboolean        valid = gtk_tree_model_get_iter_first( model, &iter );
    
  if ( valid )
  {
    StringContainer alias;
    gchar          *str_data = 0;
    string          name;
    unsigned int    pos;
    
    gtk_tree_model_get( model, &iter, ATTRIBUTE_COLUMN, &str_data, -1 );
    
    name = string( str_data );
      
    g_free( str_data );
            
    if ( name == EXPR_PARENTHESES )
    {
      AttrExprView  attr;
      
      attr.negation = false;
      attr.name = "";
      attr.op = "";
      attr.value = "";
      attr.value_op = "";
      attr.meta_type = AttrExprView::ATTR;
      
      make_attr_expr_view( &attr, alias, model, &iter, "", false );
      
      optimize_attr_expr( &attr );
      
      attr.meta_type = AttrExprView::PARENTHESES;
                  
      make_string( &attr, res );
    }
    else
    {
      MetaAttrView    meta;
      
      meta.name = name;
      
      if ( (pos = name.find(":")) != string::npos )
        name = name.substr( pos + 1 );
      
      m_name_storage->open_top_alias_level();      
      alias.push_back( name );
    
      make_attr_alias_map( meta, alias, model, &iter, true );
      
      make_string( &meta, res );
    }
  }
  
  return res;
}

void UnixGraphAttrView::make_string( GtkTreeModel *model, GtkTreeIter *iter, string &str )
{
//  gboolean          valid = TRUE;
  gchar            *str_data;
  GtkTreeIter       child_iter;
  gint              n_children = 0;
  gint              i;
  string            attr_name;
  unsigned int      pos; 
  string            attr_value;
  
  if ( m_mode == ATTRIBUTE_EXPRESSION )
  {
    gtk_tree_model_get( model, iter, EXPR_BIN_OPERATOR_COLUMN, &str_data, -1 );
    
    if ( str_data != 0 )
    {
      attr_name = string( str_data );
      
      if ( !attr_name.empty() )
        str += attr_name + " ";
    }
    g_free( str_data );
    
    gtk_tree_model_get( model, iter, EXPR_UN_OPERATOR_COLUMN, &str_data, -1 );
    
    if ( str_data != 0 )
    {
      attr_name = string( str_data );
      
      if ( !attr_name.empty() )
        str += attr_name;
    }
    g_free( str_data );
  }
  
  gtk_tree_model_get( model, iter, ATTRIBUTE_COLUMN, &str_data, -1 );
  attr_name = string(str_data);
  g_free( str_data );
  
  if ( attr_name != EXPR_PARENTHESES )
  {
    pos = attr_name.find( EMPTY_META_NAME );
    if ( pos != string::npos )
      attr_name.replace( pos, EMPTY_META_NAME.length(), "" );
  
    if ( !str.empty() )
      str += " ";
    
    str += attr_name;
  }
  
  n_children = gtk_tree_model_iter_n_children( model, iter );
  if ( n_children > 0 )
  {
    if ( attr_name == EXPR_PARENTHESES )
      str += "(";
    else
      str += "{";
    
    for( i = 0; i < n_children; i++ )
    {
      gtk_tree_model_iter_nth_child( model, &child_iter, iter, i );
      make_string( model, &child_iter, str );
    }
    
    if ( attr_name == EXPR_PARENTHESES )
      str += ") ";
    else
      str += "} ";
  }
  else 
  {
    if ( !attr_name.empty() )
    {
      int value_type;
      gtk_tree_model_get( model, iter, TYPE_COLUMN, &value_type, -1 );
      
      switch( m_mode )
      {
      case ATTRIBUTE_TEMPLATE:
        
        if ( value_type == STRING_VALUE )
          str += "=\"\"; ";
        else
          str += "=0; ";
        break;
      
      default:
        gtk_tree_model_get( model, iter, VALUE_COLUMN, &str_data, -1 );
      
        if (str_data != NULL)
        {
          attr_value = string(str_data);
          
          if ( m_mode == ATTRIBUTE_DIALOG )
            str += "=" + attr_value + "; ";
          else
          {
            gtk_tree_model_get( model, iter, OPERATION_COLUMN, &str_data, -1 );
            
            if ( str_data == NULL )
              str += "=" + attr_value + "; ";
            else
              str += string(str_data) + attr_value + "; ";
          }
          g_free( str_data );
        }
      }
    }
  }
}

string UnixGraphAttrView::get_string()
{
  string            res = "";
  GtkTreeModel     *model = gtk_tree_view_get_model( GTK_TREE_VIEW( m_tree ) );
  GtkTreeIter       parent_iter;
  
  
  if ( gtk_tree_model_get_iter_first( model, &parent_iter ) )
  {
    bool aliases = m_name_storage != 0;
  
    if ( aliases )
      aliases = (m_name_storage->get_curr_alias_set() >= 0);
    
    if ( aliases )
      res = make_aliased_attr_to_string( model );
    else
      make_string( model, &parent_iter, res );
  }
  
  return res;
}

string UnixGraphAttrView::get_label()
{
  return "Node Info";
}

void UnixGraphAttrView::make_docked( bool docked )
{
  if ( !m_docked && docked)
  {
    if ( dialog != NULL ) 
    {
      gtk_widget_destroy( dialog );
      dialog = NULL;  
      scrolledwindow = NULL;
      create();
      set_attributes( m_attr_string_list );
    }
      
    scrolledwindow = gtk_scrolled_window_new (NULL, NULL);
    gtk_widget_show (scrolledwindow);
    gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolledwindow), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
    gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (scrolledwindow), GTK_SHADOW_ETCHED_IN);

    gtk_widget_show( m_tree );
    gtk_container_add (GTK_CONTAINER (scrolledwindow), m_tree);
        
    gtk_signal_connect( GTK_OBJECT (scrolledwindow), "button_release_event",
                      GTK_SIGNAL_FUNC (on_button_release_event),
                      (gpointer)this);     
    
  }
  else if ( m_docked && !docked )
  {
    if ( scrolledwindow != NULL )
    {
      //Need to be destroyed outside
//      if ( GTK_IS_WIDGET(scrolledwindow) )
//        gtk_widget_destroy(scrolledwindow);
      
      scrolledwindow = NULL;
      create();
      
      set_attributes( m_attr_string_list );
    }
  }    
        
  m_docked = docked;
}

bool UnixGraphAttrView::is_docked()
{
  return m_docked;
}

void* UnixGraphAttrView::get_dockable()
{
  void * res = 0;
  
  if ( m_docked )
    res = scrolledwindow;
  
  return res;
}

bool UnixGraphAttrView::is_empty()
{
  bool res = (m_store == 0);
  
  if ( !res )
  {
    GtkTreeIter iter;
    res = !gtk_tree_model_get_iter_first( GTK_TREE_MODEL(m_store), &iter );
  }
  
  return res;
}
