/*
 *  
 * I2C routines
 *
 */
 
#ifndef _I2C_H_
#define _I2C_H_

#define I2C_MASTER 0

// initializes MSSP for I2C
void i2c_init(char addr) {
	
	if (addr != I2C_MASTER) {
		// enable SSP, I2C slave mode, 7 bit addr
		SSPCON = 00110110b;
		SSPADD = addr;
		clear_bit(SSPADD,7); // ensure
	}
	else {
		// enable SSP, I2C master mode
		SSPCON = 00101000b;
		// SSPADD = (Fosc/(4*bitrate)) - 1
		// Fosc = 20MHz, bitrate = 400kHz
		SSPADD = 12;
	}
	
	// slew rate control enabled, i2c spec input levels
	SSPSTAT = 0;

	// I/O pins as inputs (RC3,4)
	TRISC |= 0x18;
}

// initiates read/write to slave at addr
// if reading, goes into receive mode upon ACK
char i2c_open(char addr, char read) {

	set_bit(SSPCON2,SEN); // assert start
	
	// prepare address
	addr = addr << 1;
	if (read) addr |= 1;
	
	while (SSPCON2 & 1); // wait for start to finish
	
	SSPBUF = addr; // send address
	
	while (SSPSTAT & 4); // wait for ack bit
	
	if (SSPCON2 & 64) {	// nack!
		return 1;
	}
	else { // ack...
		return 0;
	}
}

// rcen, waits for bf to set, then returns sspbuf
// remember to call i2c_nack after this!!!
char i2c_rx(void) {
	
	set_bit(SSPCON2,RCEN); // start receiving

	while (!(SSPSTAT & 1)); // wait for buffer full
	
	return SSPBUF;	
}

// to nack or not to nack
// has to be called after calling i2c_rx()
void i2c_nack(char nack) {
	
	// nack or ack?
	if (nack) set_bit(SSPCON2,ACKDT);
	else clear_bit(SSPCON2,ACKDT);
	
	// send it
	set_bit(SSPCON2,ACKEN);
	
	// wait for it to finish sending
	while (SSPCON2 & 0x10);
}

// transmit data, MUST open connection to destination first!
char i2c_tx(char data) {
	
	SSPBUF = data; // send data

	while (SSPSTAT & 4); // wait for ack bit
	
	if (SSPCON2 & 64) {	// nack!
		return 1;
	}
	else { // ack...
		return 0;
	}
}

// sends stop bit
char i2c_close(void) {
	
	set_bit(SSPCON2,PEN); // STOP
	
	while (SSPCON2 & 4); // wait for stop to finish
	
}

#endif