///////////////////////////////////////////////////////
//
// FileName: dbmain.c
// Author: kwanjee@andrew.cmu.edu
//
// Daughterboard firmware
//
////////////////////////////////////////////////////////

#include "16f877.h"

#include "877reg.h"
#include "sercomm.h"
#include "dbpwm.h"
#include "encupdn.h"
#include "i2cs.h"

//#use delay(clock = 20000000)

#define putstring(x) ser_tx(x)

#define I2C_PREFIX 0xf8

//#define BLINK_PORT 	PORTB
//#define BLINK_PIN  	PORTB.0

signed Kp = 16;
signed Ki = 8;
signed Kd = 13;

signed vel; // desired velocity
signed v; // instantaneous velocity
signed e; // error in velocity
signed pe; // previous error
signed de; // change in error over time
signed ie; // sum of error over time
signed pie; // previous sum of error
signed long Y; // PID output

// Timer2 overflows at 20kHz
// PID @ 300Hz == once every 65 overflows 
char pid_period = 128;
char t2_count = 0;
char pid_update = 0;

char modYl, modYh;
char sigY;

signed long modY;

signed long disp;

char _dir_swap;
#bit dir_swap = _dir_swap.0

char i2c_addr;
char buf[10];
char txlen;
char bufcnt;
char bufptr;

char dump[50];
char dp = 0;
char dumpit;

char sat_add(char a, char b, char sub) {
	char res;
	
	if (sub) b = ~b + 1;

	res = a + b;

	if (!((a ^ b) & 0x80)) {
		if ((res ^ a) & 0x80) {
			if (res & 0x80) res = 0x7f;
			else res = 0x81;
		}
	}
	return res;
}

#INT_TIMER2
void tmr2intr(void) {

	// timer2 overflow
		
	// is it time to update PID?
	if (t2_count < pid_period) t2_count++;
	
	else {
		// update PID...
		t2_count = 0;
		
		// 16 bit signed arithmetic!!!!!

		v = enc_read();		
		enc_reset();

		/*if (enc_swap) {
			v = (~v) + 1;
		}*/

		// cumulate displacement
		disp = disp + (signed long)v;
	
		if (dumpit) {
			dump[dp++] = v;
			if (dp == 50) dp = 0;
		}
	
		// must saturate dv and d2v as well, 
		// else cannot handle large changes in vel
	
		pe = e;
		// e = vel - v;
		e = sat_add(vel,v,1);
	
		pie = ie;
		// ie = pie + e
		ie = sat_add(pie,e,0);
	
		//de = e - pe;
		de = sat_add(e,pe,1);
	
		Y = (signed long)Kp * e;
	
		Y = Y + (signed long)Ki * ie;
	
		Y = Y + (signed long)Kd * de;
			
		// saturate xh:xl to 10 bits magnitude
		// xh: chop off top 6 bits
		// if xh < 128, x is positive (sigY = 0)
		// else x is negative (sigY = 1, flip xl:xh)
		// at this point: xl:xh = magnitude. saturate it if necessary
		// if xh > 3, overflow, saturate
		// now, shift 2 bits right, setvel.
	
		if (Y > 1023) Y = 1023;
		if (Y < -1023) Y = -1023;
		if (Y > 0) {
			sigY = 0;
			modY = Y;
		}
		else {
			sigY = 1;
			modY = -Y;
		}
		
		modYh = modY >> 2;
		modYl = modY & 0xff;
		modYl = modYl << 6;
		
		// swap?
		if (dir_swap) {
			sigY = (~sigY) & 0x01;
		}

		pwm_setvel(sigY,modYh,modYl);
		
		pid_update = 1;		
	}	

	TMR2IF = 0; //interrupt handled.
}

void do_cmd(void) {
	char inp, i;

	txlen = 0;

	i = 0;
	ser_tx('.');
	switch (buf[0]) {
	case 0: // motor off
			if (bufcnt != 1) {
				// too many params!
				buf[0] = 2;
				txlen = 1;
				break;
			}
			pwm_setvel(0,0,0); // zero motor
			TMR2IE = 0; // controller off
			vel = 0;
			//ser_tx('0');
			break;
	case 2: // controller off
			if (bufcnt != 1) {
				// too many params!
				buf[0] = 2;
				txlen = 1;
				break;
			}
			TMR2IE = 0;
			vel = 0;
			pwm_setvel(0,0,0);
			//ser_tx('2');
			break;
	case 3: // controller on
			if (bufcnt != 1) {
				// too many params!
				buf[0] = 2;
				txlen = 1;
				break;
			}
			TMR2IE = 1;
			//ser_tx('3');
			break;
	case 4: // set pwm
			if (bufcnt != 4) {
				// wrong no. of params
				buf[0] = 2;
				txlen = 1;
				break;
			}
			if (TMR2IE) {
				// controller active, cannot mess with pwm
				buf[0] = 3;
				txlen = 1;
				break;
			}
			pwm_setvel(buf[1],buf[2],0);
//			ser_tx(':');
//			putdec(d[0]);
//			putdec(d[1]);
//			putdec(d[2]);
			break;
	case 5: // set velocity
			if (bufcnt != 2) {
				// wrong no. of params
				buf[0] = 2;
				txlen = 1;
				break;
			}
			vel = buf[1];
			break;
	case 6: // reset disp count
			if (bufcnt != 1) {
				// too many params!
				buf[0] = 2;
				txlen = 1;
				break;
			}
			disp = 0;
			break;
	case 7: // set encoder polarity
			if (bufcnt != 2) {
				// wrong no. of params
				buf[0] = 2;
				txlen = 1;
				break;
			}
			if (buf[1]) enc_swap = 1;
			else enc_swap = 0;
	case 8: // set motor polarity
			if (bufcnt != 2) {
				// wrong no. of params
				buf[0] = 2;
				txlen = 1;
				break;
			}
			if (buf[1]) dir_swap = 1;
			else dir_swap = 0;
	case 9: // set constants
			if (bufcnt != 4) {
				// too many/few params!
				buf[0] = 2;
				txlen = 1;
				break;
			}
			Kp = buf[1];
			Ki = buf[2];
			Kd = buf[3];
			break;
	case 10: // read vel
			if (bufcnt != 1) {
				// too many/few params!
				buf[0] = 2;
				txlen = 1;
				break;
			}
			buf[0] = 0;
			buf[1] = v;
			txlen = 2;
			break;
	case 11: // read disp
			if (bufcnt != 1) {
				// too many/few params!
				buf[0] = 2;
				txlen = 1;
				break;
			}
			buf[0] = 0;
			buf[1] = disp >> 8;
			buf[2] = disp & 0xff;
			txlen = 3;
			break;
	case 12: // read constants
			if (bufcnt != 1) {
				// too many/few params!
				buf[0] = 2;
				txlen = 1;
				break;
			}
			buf[0] = 0;
			buf[1] = Kp;
			buf[2] = Ki;
			buf[3] = Kd;
			txlen = 4;
			break;
	case 13: dumpit = 1;
			break;
	case 14: dumpit = 0;
			break;
	case 15: // dump the dump
			for (dp=0; dp<50; dp++) {
				//putdec(dump[dp]);
				ser_tx('\r');
			}
			break;
	default: // huh? unknown command
			buf[0] = 1;
			txlen = 1;
			break;
	}
	
	if (txlen == 0) { // all OK
		buf[0] = 0;
		txlen = 1;
	}
	
	return;
}

void write_buf(void) {
	char i;
	
	i2c_tx(txlen);
	
	for (i=0; i<txlen; i++) {
		i2c_tx(buf[i]);
	}
}

void main(void) {
	char inp;
	char i;

	TSB3 = 0;

	OPTION_REG = 0x05; // TMR0 @ 1:64, prescaler to timer0 (pid loop freq)

	RBPU = 0; // pullups on port b
	TRISB |= 0x07; // B0-2 as inputs
	i2c_addr = (~PORTB) & 0x07;
	i2c_addr |= I2C_PREFIX;
    
	// init...
	ser_init(SER_115200); // init rs-232 comms
	pwm_init(); // init CCP module
	enc_init(); // init encoder counters
	i2c_init(i2c_addr); // initialize i2c module

	vel = 0; // init to 0
	e = 0;
	de = 0;
	ie = 0;
	disp = 0;

	dir_swap = 0; // powerup default to no swap

	TMR2IE = 0; // controller off
	PEIE = 1;
	GIE = 1;

	RB4 = 1; // brakes off
	RB5 = 0; // mode = 0

	ser_tx("I2C:\r");
	
	while (1) {
		inp = i2c_rx();
		switch (inp) {
		case I2C_SLAVE_READ:
				ser_tx('r');
				bufcnt = 255; // some invalid value
				bufptr = 0;
				break;
		case I2C_DATA_READ:
				//ser_tx(i2c_data);
				if (bufcnt == 255) {
					bufcnt = i2c_data;
					break;
				}
				if (bufptr < bufcnt) {
					//ser_tx('!');
					buf[bufptr++] = i2c_data;
				}
				if (bufptr == bufcnt) do_cmd();
				break;
		case I2C_SLAVE_WRITE:
				ser_tx('w');
				write_buf();
				break;
		}
//		if (pid_update) {
//			pid_update = 0;
//			ser_tx('#');
//		}
	}
}


