/*
 *  
 * I2C master mode routines
 *
 */
 
#ifndef _I2CS_H_
#ifndef _I2CM_H_
#define _I2CM_H_

#define I2C_WRITE 0 // master read, slave write
#define I2C_READ 1 // master write, slave read

// initializes MSSP for I2C
void i2c_init(void) {
	
	// enable SSP, I2C master mode
	SSPCON = 0b00101000;
	// SSPADD = (Fosc/(4*bitrate)) - 1
	// Fosc = 20MHz, bitrate = 400kHz
	SSPADD = 12;
	
	// slew rate control enabled, i2c spec input levels
	SSPSTAT = 0;

	// I/O pins as inputs (RC3,4)
	TSC3 = 1;
	TSC4 = 1;
}

// initiates read/write to slave at addr
// if reading, goes into receive mode upon ACK
char i2c_open(char addr, char read) {

	SEN = 1; // assert start
	
	// prepare address
	addr = addr << 1;
	if (read) addr |= 1;
	
	while (SEN); // wait for start to finish
	
	SSPBUF = addr; // send address
	
	while (RW); // wait for ack bit
	
	if (ACKSTAT) {	// nack!
		return 1;
	}
	else { // ack...
		return 0;
	}
}

// rcen, waits for bf to set, then returns sspbuf
// remember to call i2c_nack after this!!!
char i2c_rx(void) {
	
	RCEN = 1; // start receiving
	while (!BF); // wait for buffer full
	
	return SSPBUF;	
}

// to nack or not to nack
// has to be called after calling i2c_rx()
void i2c_nack(char nack) {
	
	// nack or ack?
	if (nack) ACKDT = 1;
	else ACKDT = 0;
	
	// send it
	ACKEN = 1;
	
	// wait for it to finish sending
	while (ACKEN);
}

// transmit data, MUST open connection to destination first!
char i2c_tx(char data) {
	
	SSPBUF = data; // send data

	while (RW); // wait for ack bit
	
	if (ACKSTAT) {	// nack!
		return 1;
	}
	else { // ack...
		return 0;
	}
}

// sends stop bit
char i2c_close(void) {
	
	PEN = 1; // STOP
	while (PEN); // wait for stop to finish
}

#endif
#endif