/*
 *  
 * I2C slave mode routines
 * 
 * Bit Bangin' Style (for the 16F62x)
 *
 */
 
#ifndef _I2CBB_H_
#define _I2CBB_H_

#define I2C_PORT PORTB
#define I2C_TRIS TRISB
#define I2C_CLK 0x02
#define I2C_CLK_BIT 1
#define I2C_DATA 0x04
#define I2C_DATA_BIT 2

#define I2C_RX_FLAG 0x01
#define I2C_RX_FLAG_BIT 0
#define I2C_SLAVE_WRITE 0x02
#define I2C_SLAVE_WRITE_BIT 1
#define I2C_IS_ADDR 0x04
#define I2C_IS_ADDR_BIT 2

#define I2C_SLAVE_READ 0
#define I2C_DATA_READ 3

#define I2C_MASTER_ACK 0
#define I2C_MASTER_NACK 1

char i2c_shift;
char i2c_data;
char i2c_addr;
char i2c_status;
char i2c_rx_state;

void i2c_init(char addr) {

	// set clock line to input
	set_bit(I2C_TRIS,I2C_CLK_BIT);

	// set data line to input
	set_bit(I2C_TRIS,I2C_DATA_BIT);

	// lsb of addr byte is R/W bit...
	i2c_addr = addr << 1;

	i2c_rx_state = 0;
	i2c_status = 0;
}

// resets the i2c receiver fsm
void i2c_reset(void) {
	i2c_rx_state = 0;
}

void i2c_rx_go(void) {
	// non-blocking receiver fsm
	// must poll non-stop while checking i2c_status

	char inp;

	set_bit(PORTB,0);

	//inp = 0;
	//if (I2C_PORT & I2C_DATA) set_bit(inp,1);
	//if (I2C_PORT & I2C_CLK) set_bit(inp,0);
	asm {
		clrf _inp_i2c_rx_go
		btfsc PORTB,2
		bsf _inp_i2c_rx_go,1
		btfsc PORTB,1
		bsf _inp_i2c_rx_go,0
	}

	clear_bit(PORTB,0);
	
	switch (i2c_rx_state) {
	case 0: // confuesd state, wait for hi-hi
			if (inp == 3) i2c_rx_state = 1;
			break;
	case 1: // wait for start
			//set_bit(PORTB,0);
			if (inp == 1) i2c_rx_state = 2;
			else if (inp != 3) i2c_rx_state = 0;
			break;
	case 2: // started
			//clear_bit(PORTB,0);
			if (inp == 0) {
				set_bit(i2c_status,I2C_IS_ADDR_BIT); // next byte is addr
				i2c_rx_state = 3; // clockin address byte
			}
			else if (inp == 3) i2c_rx_state = 1; // stop
			else if (inp == 2) i2c_rx_state = 0; // confused
			break;
	case 3: // data can change, wait for posedge clk
			if (i2c_shift == 8) {
				i2c_shift = 0; // reset shift count
				if (i2c_status & I2C_IS_ADDR) {
					// it was an addr byte, check...
					if ((i2c_data & 0xFE) != i2c_addr) {
						i2c_rx_state = 0; // addr mismatch, reset
						break;
					}
					// it's me!
					else if (i2c_data & 1) {
						set_bit(i2c_status,I2C_SLAVE_WRITE_BIT);
					}
					else {
						clear_bit(i2c_status,I2C_SLAVE_WRITE_BIT);
					}
				}
				// either data, or address match, so ack
				clear_bit(I2C_TRIS,I2C_DATA_BIT); // data to output
				clear_bit(I2C_PORT,I2C_DATA_BIT); // pull to 0
				i2c_rx_state = 6; // bf
			}
			else if (inp & 1) {
				i2c_shift++;
				i2c_data = i2c_data << 1;
				if (inp == 1) {
					i2c_rx_state = 4;
				}
				else {
					set_bit(i2c_data,0);
					i2c_rx_state = 5;
				}
			}
			break;
	case 4: // data was 0
			if (inp == 0) i2c_rx_state = 3; // negedge clk
			else if (inp == 2) i2c_rx_state = 0; // confused
			else if (inp == 3) i2c_rx_state = 1; // stop
			break;
	case 5: // data was 1
			if (inp == 0) i2c_rx_state = 0; // confused
			else if (inp == 1) i2c_rx_state = 2; // (re?)start
			else if (inp == 2) i2c_rx_state = 3; // negedge clk
			break;
	case 6: // ack bit, posedge clk
			if (inp & 1) i2c_rx_state = 7;
			break;
	case 7: // ack bit, negedge clk
			if (!(inp & 1)) {
				// release data line
				set_bit(I2C_TRIS,I2C_DATA_BIT); // data to input
				if (i2c_status & I2C_SLAVE_WRITE) {
					// slave has to tx after this
					// hold clk lo to stall for time
					clear_bit(I2C_TRIS,I2C_CLK_BIT); // clk to output
					clear_bit(I2C_PORT,I2C_CLK_BIT); // clk lo
					
					// when we come back, we want to be resetted
					i2c_rx_state = 0;
				}
				else {
					i2c_rx_state = 8;
				}
				// simulate SSPIF
				set_bit(i2c_status, I2C_RX_FLAG_BIT);
			}
			break;
	case 8: // last byte was address, next byte is data
			clear_bit(i2c_status, I2C_IS_ADDR_BIT);
			i2c_rx_state = 3; // grab next data byte
			break;
	}

	return;
}


// slave receive
char i2c_rx(void) {

	clear_bit(i2c_status,I2C_RX_FLAG_BIT);
	while (!(i2c_status & I2C_RX_FLAG)) {
		i2c_rx_go();
	}
		
	if (i2c_status & I2C_SLAVE_WRITE) { // check R/W bit
		// if he wants to read, we have to write
		// CKP cleared in hardware to stall for time
		return I2C_SLAVE_WRITE;
	}
	else {
		// master write, slave read
		
		// this is either our address or the incoming data
		if (i2c_status & I2C_IS_ADDR) {
			// just got address with write command
			// exit, then get called again to rx incoming data
			return I2C_SLAVE_READ;
		}
		else {
			// just got the incoming data
			return I2C_DATA_READ;
		}
	}

}

// transmit data
// returns 1 if master NACK, else returns 0
char i2c_tx(char data) {
	
	char nack, i;
	
	i2c_data = data;
	
	clear_bit(I2C_TRIS,I2C_DATA_BIT); // data to output
	
	for (i = 0; i<8; i++) {
		if (i2c_data & 0x80) set_bit(I2C_PORT,I2C_DATA);
		else clear_bit(I2C_PORT,I2C_DATA);
	
		if (i==0) { // first bit
			set_bit(I2C_TRIS,I2C_CLK_BIT); // release clk line	
		}
	
		i2c_data = i2c_data << 1;	
		
		while (!(I2C_PORT & I2C_CLK)); // wait for posedge clk
		while (I2C_PORT & I2C_CLK); // wait for negedge clk
	}
	
	set_bit(I2C_TRIS,I2C_DATA_BIT); // release data line

	while (!(I2C_PORT & I2C_CLK)); // wait for posedge clk
	
	if (I2C_PORT & I2C_DATA) {
		nack = 1;
	}
	else {
		nack = 0;	
	}
	
	while (I2C_PORT & I2C_CLK); // wait for negedge clk
	
	return nack;
}

#endif
