///////////////////////////////////////////////////////
//
// FileName: sui.c
// Author: kwanjee@andrew.cmu.edu
//
// Serial user interface (implementing v0.2)
//
////////////////////////////////////////////////////////

//Timing settings
#pragma CLOCK_FREQ 20000000

#include "16f877.h"
#include "sercomm.h"
#include "adc.h"
#include "pwm.h"
#include "servo.h"
#include "i2cm.h"

#define BLINK_PORT 	PORTB
#define BLINK_PIN  	0

#define BUFMAX 50

#define RETCHAR 13

#define SIGN_POS 0
#define SIGN_NEG 1
#define SIGN_NONE 2

char buf[BUFMAX];
char bufcnt;
char bufptr;
char proc_err;

void interrupt(void) {

	if (servo_state == 1) {
		// next intr in 0.6 ms
		TMR0 = 180;

		if (servo_pos[servo_curr] != 0)
			SERVO_PORT = SERVO_PORT | servo_switch; // output hi
		
		servo_switch = servo_switch << 1;
		if (servo_switch == 0) servo_switch = 1;
		
		servo_state = servo_state + 1;
		servo_count = 0;
	}
	else if (servo_state == 2) {
		TMR0 = 255 - servo_pos[servo_curr];
		servo_state = 3;
	}
	else if (servo_state == 3) {
		SERVO_PORT = 0; // output lo
		TMR0 = servo_pos[servo_curr];
		servo_curr = servo_curr + 1;
		servo_curr = servo_curr & 0x07;
		servo_state = 1;
	}

    clear_bit( INTCON, T0IF );  //interrupt handled.
}

void getbuf(void) {
	char ch;

	bufcnt = 0;
	ch = 0;
	while (ch != '\r') {
		ch = ser_rx();
		if (ch == '\n') ch = '\r';
		else if ((ch == 8)) {
			if (bufcnt > 0) {
				ser_tx(ch);
				bufcnt--;
			}
		}
		else if (bufcnt < 50) {
			ser_tx(ch);
			buf[bufcnt++] = ch;
		}
	}
	ser_tx(ch);
}

void putbuf(void) {
	char i;
	
	for (i=0; i<bufcnt; i++) {
		ser_tx(buf[i]);
	}	
	
}

// pre: bufptr @ 1st suspected space
// post: bufptr @ 1st non-space
void proc_space(void) {
	while (buf[bufptr] == ' ') {
		bufptr++;
		if (bufptr == bufcnt) break;
	}
}

// pre: bufptr @ 1st digit
// post: if no error, bufptr @ 1st non-digit
char proc_num(void) {
	char ch, val, count, pval;
	
	val = 0;
	proc_err = 1;
	ch = buf[bufptr] - '0';
	while (ch < 10) {
		if (val > 25) {
			proc_err = 2;
			return 0;
		}
		else {
			//val = val * 10;
			asm {
				bcf		STATUS,C
				rlf		_val_proc_num,f
				movf	_val_proc_num,w
				rlf		_val_proc_num,f
				bcf		STATUS,C
				rlf		_val_proc_num,f
				addwf	_val_proc_num,f
			}
		}
		val = val + ch;
		if (STATUS & 1) {
			proc_err = 2;
			return 0;
		}
		proc_err = 0;
		bufptr++;
		if (bufptr == bufcnt) {
			proc_err = 3;
			return 0;
		}
		ch = buf[bufptr] - '0';
	}
	return val;
}

// pre: bufptr @ suspected sign char
// post: if sign char seen, bufptr @ next char
char proc_sign(void) {
	
	if (bufptr == bufcnt) return SIGN_NONE;
	if (buf[bufptr] == '+') {
		bufptr++;
		return SIGN_POS;
	}
	if (buf[bufptr] == '-') {
		bufptr++;
		return SIGN_NEG;
	}
	return SIGN_NONE;
}

void do_an(char ch) {
	char val;
	
	val = adc_read(ch);
	putdec(val);
	ser_tx(RETCHAR);
}

char do_mt(void) {
	// MT[0|1] [+|-]n
	// precondition: bufptr is at char after MT
	
	char ch;
	char v0,v1;
	char d0,d1;
	
	ch = buf[bufptr];
	if (ch == ' ') {
		// conseq mode	
		proc_space();
		d0 = proc_sign();
		if (d0 == SIGN_NONE) d0 = SIGN_POS;
		v0 = proc_num();
		if (proc_err != 0) return 0;
		proc_space();
		d1 = proc_sign();
		if (d1 == SIGN_NONE) d1 = SIGN_POS;
		v1 = proc_num();
		if (proc_err != 0) return 0;
		// still ok?
		pwm_setvel8(0,d0,v0);
		pwm_setvel8(1,d1,v1);
	}
	else if ((ch == '0')||(ch == '1')) {
		ch = ch - '0';
		bufptr++;
		proc_space();
		d0 = proc_sign();
		if (d0 == SIGN_NONE) d0 = SIGN_POS;
		v0 = proc_num();
		if (proc_err != 0) return 0;
		putstring("mt");
		putdec(ch);
		ser_tx(',');
		putdec(d0);
		ser_tx(',');
		putdec(v0);
		ser_tx('\r');
		pwm_setvel8(ch,d0,v0);
	}	
	else return 0;

	return 1;
}

char proc_baud(void) {
	char i;
	
/*	i = bufptr;
	if (buf[i++] == '9')
	if (buf[i++] == '6')
	if (buf[i++] == '0')
	if (buf[i++] == '0') {
		bufptr = bufptr + 3;
		return 0;
	}
	
	i = bufptr;
	if (buf[i++] == '1')
	if (buf[i++] == '9')
	if (buf[i++] == '2')
	if (buf[i++] == '0')
	if (buf[i++] == '0') {
		bufptr = bufptr + 4;
		return 1;
	}

	i = bufptr;
	if (buf[i++] == '3')
	if (buf[i++] == '8')
	if (buf[i++] == '4')
	if (buf[i++] == '0')
	if (buf[i++] == '0') {
		bufptr = bufptr + 4;
		return 2;
	}

	i = bufptr;
	if (buf[i++] == '5')
	if (buf[i++] == '7')
	if (buf[i++] == '6')
	if (buf[i++] == '0')
	if (buf[i++] == '0') {
		bufptr = bufptr + 4;
		return 3;
	}

	i = bufptr;
	if (buf[i++] == '1')
	if (buf[i++] == '1')
	if (buf[i++] == '5')
	if (buf[i++] == '2')
	if (buf[i++] == '0')
	if (buf[i++] == '0') {
		bufptr = bufptr + 5;
		return 4;
	}*/
	
	return 5;
}

void procbuf(void) {
	char ch;
	char val;
	char state;
	
	bufptr = 0;
	proc_err = 0;
	state = 0;

	while (bufptr < bufcnt) {
		ch = buf[bufptr];
		if (ch >96) ch = ch - 32; // toupper
		
		putdec(state); ser_tx(':');
		putdec(ch); ser_tx(' ');
		
		switch (state) {
		case 0: // first char
				if (ch == 'A') state = 1;
				else if (ch == 'B') state = 2;
				else if (ch == 'D') state = 3;
				else if (ch == 'M') state = 4;
				else if (ch == 'S') state = 5;
				else if (ch == 'T') state = 6;
				else if (ch == 'V') state = 7;
				else if (ch == '?') state = 80;
				else state = 99;
				break;
		case 1: // A...
				if (ch == 'N') state = 8;
				else state = 99;
				break;
		case 2: // B...
				if (ch == 'R') state = 9;
				else state = 99;
				break;
		case 3: // D...
				if (ch == 'B') state = 10;
				else if (ch == 'H') state = 11;
				else if (ch == 'I') state = 12;
				else if (ch == 'L') state = 13;
				else state = 99;
				break;				
		case 4: // M...
				if (ch == 'T') state = 14;
				else state = 99;
				break;
		case 5: // S...
				if (ch == 'S') state = 15;
				else if (ch == 'V') state = 16;
				else state = 99;
				break;
		case 6: // T...
				if (ch == 'O') state = 17;
				else state = 99;
				break;
		case 7: // V...
				if (ch == 'E') state = 18;
				else state = 99;
				break;
		case 8: // AN...
				proc_space();
				val = proc_num();
				if (val > 7) state = 99;
				else if (proc_err == 0) {
					do_an(val);
					state = 88;
				}
				else state = 99;
				break;
		case 9: // BR...
				if (ch == 'S') {
					state = 19;
					break;
				}
				else {
					val = proc_baud();
					if (val > 4) state = 99;
					else {
						//set_baud(val);
						state = 88;
					}
				}
		case 10: // DB...
/*				if (do_db(val)) state = 88;
				else state = 99;
				break;*/
		case 11: // DH...
/*				proc_space();
				val = proc_num();
				if (val > 7) state = 99;
				else if (proc_err == 0) {
					do_dout(val,1);
					state = 88;
				}
				else state = 99;
				break;*/
		case 12: // DI...
/*				proc_space();
				val = proc_num();
				if (val > 7) state = 99;
				else if (proc_err == 0) {
					do_din(val);
					state = 88;
				}
				else state = 99;
				break;*/
		case 13: // DL...
/*				proc_space();
				val = proc_num();
				if (val > 7) state = 99;
				else if (proc_err == 0) {
					do_dout(val,0);
					state = 88;
				}
				else state = 99;
				break;*/
				putstring("NOTYET\r");
				bufptr = bufcnt; // we're done
				break;
		case 14: // MT...
				if (do_mt()) {
					state = 88;
				}
				else {
					state = 99;
				}
				break;
		case 15: // SS...
		case 16: // SV...
		case 17: // TO...
		case 18: // VE...
				putstring("NOTYET\r");
				bufptr = bufcnt; // we're done
				break;
		case 88: // ok
				putstring("OK\r");
				bufptr = bufcnt; // we're done
				break;
		case 99: // error
				putstring("ERROR\r");
				bufptr = bufcnt; // we're done
				break;
		}
		bufptr++;
	}
}

void main(void) {
	char inp, num;

	// init...
	ser_init(SER_115200); // init rs-232 comms
	pwm_init(); // initialize CCP module
	servo_init();
	i2c_init();
	servo_state = 1; // activate servo driver
	ADCON1 = 0; // all channels analog, output MSBs

    enable_interrupt(GIE);
    enable_interrupt(T0IE);  //enable TMR0 overflow bit

	putstring("\rCerebellum v0.2\r");
	
	while (1) {
		ser_tx('>');
		getbuf();
		ser_tx(':');
		procbuf();
		/*inp = ser_rx();
		putdec(inp);
		ser_tx('\r');*/
	}
}

