///////////////////////////////////////////////////////
//
// FileName: sui.c
// Author: kwanjee@andrew.cmu.edu
//
// Serial user interface (implementing v0.2)
//
////////////////////////////////////////////////////////

//#opt 9

#include <16f877.h>
#include "877reg.h"
#include "sercom2.h"
#include "sersoft.h"
#include "servo.h"
#include "adc.h"
#include "pwm.h"
#include "i2cm.h"

#define VERSION_STR "Cerebellum Serial Interface v0.9b "

#define BUFMAX 50

#define RETCHAR 13

#define I2C_PREFIX 0xf8

#define SIGN_POS 0
#define SIGN_NEG 1
#define SIGN_NONE 2

char buf[BUFMAX];
char bufcnt;
char bufptr;
char servo_buf[8];
char proc_err;

/*int save_w;
#locate save_w=0x7f
int save_status;
#locate save_status=0x20*/

// All ISRs *must* use dedicated global variables

/*#INT_GLOBAL
void isr(void) {
   	#asm
   	//store current state of processor
   	MOVWF save_w
   	SWAPF status,W
   	BCF   status,5
   	BCF   status,6
   	MOVWF save_status
	#endasm
	
	if (PEIE) {
		if (RBIE && RBIF) rx_isr();
		//if (TMR1IE && TMR1IF) timeout_isr();
	}
	if (T0IE && T0IF) servo_isr();

	#asm
	// restore processor state
   	SWAPF save_status,W
   	MOVWF status
   	SWAPF save_w,F
   	SWAPF save_w,W
   	#endasm
}
*/

void getbuf(void) {
	char ch;

	bufcnt = 0;
	ch = 0;

	//ch = hw_getc();
	ch = sw_getc();
	while (ch != '\r') {
		//if (ch == '\n') ch = '\r';
		if ((ch == 8)) {
			if (bufcnt > 0) {
				//hw_putc(ch);
				//sw_putc(ch);
				bufcnt--;
			}
		}
		else if (bufcnt < 50) {
			//hw_putc(ch);
			//sw_putc(ch);
			buf[bufcnt++] = ch;
		}
		//ch = hw_getc();
		ch = sw_getc();
	}
	//sw_putc(ch);
}

void putbuf(void) {
	char i;
	
	for (i=0; i<bufcnt; i++) {
		//hw_putc(buf[i]);
		sw_putc(buf[i]);
	}	
	
}

// pre: bufptr @ 1st suspected space
// post: bufptr @ 1st non-space
void proc_space(void) {
	while (buf[bufptr] == ' ') {
		if (bufptr == bufcnt) break;
		bufptr++;
	}
}

// pre: bufptr @ 1st digit
// post: if no error, bufptr @ 1st non-digit
char proc_num(void) {
	char ch, val, count, pval;
	
	val = 0;
	proc_err = 1;
	ch = buf[bufptr] - '0';
	while (ch < 10) {
		if (bufptr == bufcnt) {
			return val;
		}
		if (val > 25) {
			proc_err = 2;
			return 0;
		}
		else {
			//val = val * 10;
			#asm
				bcf		STATUS,0
				rlf		val,f
				movf	val,w
				rlf		val,f
				bcf		STATUS,0
				rlf		val,f
				addwf	val,f
			#endasm
		}
		val = val + ch;
		if (STATUS & 1) {
			proc_err = 2;
			return 0;
		}
		proc_err = 0;
		bufptr++;
		ch = buf[bufptr] - '0';
	}
	return val;
}

// pre: bufptr @ suspected sign char
// post: if sign char seen, bufptr @ next char
char proc_sign(void) {
	
	if (bufptr == bufcnt) return SIGN_NONE;
	if (buf[bufptr] == '+') {
		bufptr++;
		return SIGN_POS;
	}
	if (buf[bufptr] == '-') {
		bufptr++;
		return SIGN_NEG;
	}
	return SIGN_NONE;
}

void do_an(char ch) {
	// <tested>
	
	char val;
	
	val = adc_read(ch);
	sw_putdec(val);
	//hw_putc(RETCHAR);
	sw_putc(RETCHAR);
}

#separate
void do_dout(char port, char op) {
	#bit out = op.0

	switch (port) {
	case 0: if (out) RB2 = 1;
			else RB2 = 0;
			TSB2 = 0;
			break;
	case 1: if (out) RA4 = 1;
			else RA4 = 0;
			TSA4 = 0;
			break;
	case 2: if (out) RB1 = 1;
			else RB1 = 0;
			TSB1 = 0;
			break;
	case 3: if (out) RB0 = 1;
			else RB0 = 0;
			TSB0 = 0;
			break;
	}	
}

#separate
void do_din(char port) {
	char out;

	switch (port) {
	case 0: TSB2 = 1;
			if (RB2) out = 1;
			else out = 0;
			break;
	case 1: TSA4 = 1;
			if (RA4) out = 1;
			else out = 0;
			break;
	case 2: TSB1 = 1;
			if (RB1) out = 1;
			else out = 0;
			break;
	case 3: TSB0 = 1;
			if (RB0) out = 1;
			else out = 0;
			break;
	}
	//hw_putc('0' + out);
	sw_putc('0' + out);
	//hw_putc('\r');
	sw_putc('\r');
}

#separate
char do_db(void) {
	// DBn [d ...] <tested>
	
	char i, j;
	char addr;
	
	// db number?
	addr = buf[bufptr] - '0';
	if (addr > 7) return 0;
	addr |= I2C_PREFIX;

	if ((++bufptr) > bufcnt) return 0;
	for (i=0; bufptr < bufcnt; i++) {
		proc_space();
		buf[i] = proc_num();
		if (proc_err != 0) return 0;
	}
	
	// data all ok, tx!
//	hw_putc(':');
	if (i2c_open(addr,I2C_WRITE)) {
		// nack, nobody home
		i2c_close();
		return 0;
	}

	// if no data to send, then we're done
	if (i == 0) {
		i2c_close();
		return 1;
	}
	
	// send length
	if (i2c_tx(i)) {
		i2c_close();
		return 0;
	}
	//send packet
	for (j=0; j<i; j++) {
		if (i2c_tx(buf[j])) {
			// nack?!
			i2c_close();
			return 0;
		}
		//hw_putc('!');
	}
	
	// get reply packet
	i2c_close();
	if (i2c_open(addr,I2C_READ)) {
		// nack...
		i2c_close();
		return 0;
	}
	i = i2c_rx(); // length byte
	i2c_nack(0);
	j=0;
	while (i>0) {
		buf[j++] = i2c_rx();
		if (i == 1) i2c_nack(1);
		else i2c_nack(0);
		i--;
	}
	i2c_close(); // done with i2c...
//	hw_putc('.');
		
	// display returned data
	for (i=0; i<j; i++) {
		sw_putdec(buf[i]);
		//hw_putc(' ');
		sw_putc(' ');
	}
	//hw_putc('\r');
	sw_putc('\r');
	
	return 1;
}

#separate
char do_mt(void) {
	// MT[0|1] [+|-]n <tested>
	// precondition: bufptr is at char after MT
	
	char ch;
	char v0,v1;
	char d0,d1;
	
	ch = buf[bufptr];
	if (ch == ' ') {
		// conseq mode	
		proc_space();
		d0 = proc_sign();
		if (d0 == SIGN_NONE) d0 = SIGN_POS;
		v0 = proc_num();
		if (proc_err != 0) return 0;
		proc_space();
		d1 = proc_sign();
		if (d1 == SIGN_NONE) d1 = SIGN_POS;
		v1 = proc_num();
		if (proc_err != 0) return 0;
		// still ok?
		pwm_setvel8(0,d0,v0);
		pwm_setvel8(1,d1,v1);
	}
	else if ((ch == '0')||(ch == '1')) {
		ch = ch - '0';
		bufptr++;
		proc_space();
		d0 = proc_sign();
		if (d0 == SIGN_NONE) d0 = SIGN_POS;
		v0 = proc_num();
		if (proc_err != 0) return 0;
		pwm_setvel8(ch,d0,v0);
	}	
	else return 0;

	return 1;
}

#separate
char do_ss(void) {
	// SS ["camera command"] [Rn] <tested>
	// precondition: bufptr is at char after SS

	char i,j;
	char lines = 1;
	
	proc_space();
	
	i=0;
	if (buf[bufptr] == '"') {
		// grab string
		if ((++bufptr) == bufcnt) return 0;
		while (buf[bufptr] != '"') {
			buf[i++] = buf[bufptr];
			if ((++bufptr) == bufcnt) return 0;
		}
		bufptr++;
	}
	// i == length of command string
	
	proc_space();

	if ((buf[bufptr] == 'R')||(buf[bufptr] == 'r')) {
		if ((++bufptr) == bufcnt) {
			return 0;
		}
		proc_space();
		lines = proc_num();
		if (proc_err != 0) {
			return 0;
		}
	}
	proc_space();
	if (bufptr != bufcnt) {
		return 0;
	}
	
	// rx a minimum of 1 line	
	if (lines < 1) lines = 1;
	
	//tx string
	for (bufptr=0; bufptr<i; bufptr++) {
		hw_putc(buf[bufptr]);
		//sw_putc(buf[bufptr]);
	}
	if (i != 0) hw_putc('\r');

	RBIE = 0;
	i = j = 0;
	to_reset();
	while (lines > 0) {
		while (i != '\r') {
			if (ser_gotchar) {
				//i = getc();
				//hw_putc(i);
				i = hw_getc();
				buf[j++] = i;
				if (j > BUFMAX) j = BUFMAX;
				to_reset();
			}
			if (timeout) {
				TMR1IE = 0; // timer off
				RBIE = 1;
				hw_putc('\r');
				//sw_putc('\r');
				return 2;
			}
		}
		for (i=0; i<j; i++) {
			sw_putc(buf[i]);
		}
		j = 0;
	
		i = 0;
		lines--;
	}
	TMR1IE = 0; // timer off*/
	RBIE = 1;

	return 1;
}

#separate
char do_sv(void) {
	// SV[0..7] [+|-]n [[+|-]n] <tested>
	// precondition: bufptr is at char after SV
	
	char pos;
	char d, i;
	
	i = buf[bufptr] - '0';
	if (i == (' ' - '0')) {
		// conseq mode
		i = 0;
		while (bufptr < bufcnt) {
			proc_space();
			d = proc_sign();
			pos = proc_num();
			if (proc_err != 0) return 0;
			if (d == SIGN_NONE) {
				servo_buf[i] = pos;
			}
			else if (servo_pos[i] == 0) {
				servo_buf[i] = 0;
			}
			else {
				if (d == SIGN_POS) {
					servo_buf[i] = servo_pos[i] + pos;
					if (servo_buf[i] < pos) {
						servo_buf[i] = 255;
					}
				}
				else {
					servo_buf[i] = servo_pos[i] - pos;
					if ((servo_buf[i] == 0)||(servo_buf[i] > servo_pos[i])) {
						servo_buf[i] = 1;
					}
				}
			}

			i++;
		}
		
		while (i > 0) {
			servo_pos[--i] = servo_buf[i];
		}
	}
	else if (i < 8) {
		bufptr++;
		proc_space();
		d = proc_sign();
		pos = proc_num();
		if (proc_err != 0) return 0;
		if (d == SIGN_POS) {
			if ((servo_pos[i] != 0)&&(servo_pos[i] + pos > pos)) {
				servo_pos[i] += pos;
			}
		}
		else if (d == SIGN_NEG) {
			if ((servo_pos[i] != 0)&&(servo_pos[i] > (pos + 1))) {
				servo_pos[i] -= pos;
			}
		}
		else {
			servo_pos[i] = pos;
		}
	}
	else return 0;
	
	return 1;
}

#separate
char do_to(void) {
	// TO n
	// precondition: bufptr is at char after TO

	char num;

	bufptr++;
	proc_space();
	num = proc_num();
	if (proc_err != 0) {
		// show current timeout
		sw_putdec(to_val);
		sw_putc('\r');
		return 1;
	}
	
	if (num == 255) num = 100; // reset to default
	write_eeprom(TO_ADDR,num);
	to_val = num;
	
	return 1;
}

#separate
char proc_baud(void) {
	char i;
	
	proc_space();
	
	i = bufptr;
	if (buf[i++] == '9')
	if (buf[i++] == '6')
	if (buf[i++] == '0')
	if (buf[i++] == '0') {
		bufptr = bufptr + 3;
		return 0x20;
	}
	
	i = bufptr;
	if (buf[i++] == '1')
	if (buf[i++] == '9')
	if (buf[i++] == '2')
	if (buf[i++] == '0')
	if (buf[i++] == '0') {
		bufptr = bufptr + 4;
		return 0x10;
	}

	i = bufptr;
	if (buf[i++] == '3')
	if (buf[i++] == '8')
	if (buf[i++] == '4')
	if (buf[i++] == '0')
	if (buf[i++] == '0') {
		bufptr = bufptr + 4;
		return 0x00;
	}

	return 0xff;
}

void show_info(void) {
	sw_putc(VERSION_STR);
	sw_putc(__DATE__);
	//hw_putc('\r');
	sw_putc('\r');
	sw_putc("SER1 timeout = ");
	sw_putdec(to_val);
	sw_putc('\r');
}

void procbuf(void) {
	char ch;
	char val;
	char state;
	#bit done = state.7
	
	bufptr = 0;
	proc_err = 0;
	state = 0;

	while (!done) {
		ch = buf[bufptr];
		if (ch >96) ch = ch - 32; // toupper
		
		switch (state) {
		case 0: // first char
				if (ch == 'A') state = 1;
				else if (ch == 'B') state = 2;
				else if (ch == 'D') state = 3;
				else if (ch == 'M') state = 4;
				else if (ch == 'S') state = 5;
				else if (ch == 'T') state = 6;
				else if (ch == 'V') state = 7;
				else if (ch == '?') state = 80;
				else state = 99;
				break;
		case 1: // A...
				if (ch == 'N') state = 8;
				else state = 99;
				break;
		case 2: // B...
				if (ch == 'R') state = 9;
				else state = 99;
				break;
		case 3: // D...
				if (ch == 'B') state = 10;
				else if (ch == 'H') state = 11;
				else if (ch == 'I') state = 12;
				else if (ch == 'L') state = 13;
				else state = 99;
				break;				
		case 4: // M...
				if (ch == 'T') state = 14;
				else state = 99;
				break;
		case 5: // S...
				if (ch == 'S') state = 15;
				else if (ch == 'V') state = 16;
				else state = 99;
				break;
		case 6: // T...
				if (ch == 'O') {
					if (do_to()) state = 90;
					else state = 99;
				}
				else state = 99;
				break;
		case 7: // V...
				if (ch == 'E') state = 18;
				else state = 99;
				break;
		case 8: // AN...
				proc_space();
				val = proc_num();
				if (val > 7) state = 99;
				else if (proc_err == 0) {
					do_an(val);
					state = 90;
				}
				else state = 99;
				break;
		case 9: // BR...
				val = proc_baud();
				if (val > 0x20) state = 99;
				else {
					write_eeprom(BR_ADDR,val);
					state = 90;
				}
		case 10: // DB...
				if (do_db()) state = 90;
				else state = 99;
				break;
		case 11: // DH...
				proc_space();
				val = proc_num();
				if (val > 7) state = 99;
				else if (proc_err == 0) {
					do_dout(val,1);
					state = 90;
				}
				else state = 99;
				break;
		case 12: // DI...
				proc_space();
				val = proc_num();
				if (val > 7) state = 99;
				else if (proc_err == 0) {
					do_din(val);
					state = 90;
				}
				else state = 99;
				break;
		case 13: // DL...
				proc_space();
				val = proc_num();
				if (val > 7) state = 99;
				else if (proc_err == 0) {
					do_dout(val,0);
					state = 90;
				}
				else state = 99;
				break;
				state = 98;
				break;
		case 14: // MT...
				if (do_mt()) state = 90;
				else state = 99;
				break;
		case 15: // SS...
				val = do_ss();
				if (val == 1) state = 90;
				else if (val == 2) state = 97;
				else state = 99;
				break;
		case 16: // SV...
				if (do_sv()) state = 90;
				else state = 99;
				break;				
		case 18: // VE...
				if (ch == 'R') {
					show_info();
					state = 90;
				}
				else state = 99;
				break;
		case 80: // help 
				state = 98; break;
		case 90: // ok
				sw_putc('O'); sw_putc('K'); sw_putc('\r');
				state = 128;
				break;
		case 97: // timeout
				sw_putc("TIMEOUT\r");
				state = 128;
				break;
		case 98: // not implemented
				sw_putc("NOTYET\r");
				state = 128;
				break;
		case 99: // error
				sw_putc("ERROR\r");
				state = 128;
				break;
		}
		bufptr++;
		
		if ((bufptr == bufcnt)&&(state < 90)) state = 99;
	}
}

void main(void) {
	char inp, num;

	TSB3 = 0;
	TSB2 = 0; // RB2 as output
	RB2 = 0;
	TSB1 = 0;
	RB1 = 0;

	// init...
	ser_init(SER_115200); // init rs-232 comms
	sersoft_init(); // init software serial port
	pwm_init(); // initialize CCP module
	servo_init();
	i2c_init();
	adc_init(0); // all channels analog, output MSBs

	servo_state = 1; // activate servo driver

	GIE = 1;

	inp = read_eeprom(TO_ADDR);
	if (inp != 255) to_val = inp;
	
	show_info();

	//hw_putc('O'); hw_putc('K'); hw_putc('\r');
	sw_putc('O'); sw_putc('K'); sw_putc('\r');
	while (1) {
		getbuf();
		procbuf();
	}

/*	putc('!');	
	while (1) {
		num = sw_getc();
		while(servo_free);
		while (!servo_free);
		putc(num);
	}*/

}

