#lang racket

(require (prefix-in audit: "../audit/audit.rkt"))
(require "model.rkt")

(define audit audit:auditMDPapprox-valueIter)

;; the base used for exponentially incresing or decresing bounds
(define base 10.0)

;; Given an Ex4 model, we determine how large the reward for reading a patient's 
;; record must be before it makes more sense than studying.

;; The functions below all use a predicate test?.
;; test? is over positive numbers and such that 
;; (test? m) = false for all m < n and
;; (test? m) = true for all m >= n
;; we call n the crossover point.


;; given a value from-below such that not (test? from-below),
;; it will return a value m such that (test? m).
(define (find-over-approx test? from-below)
  (if (test? from-below) ; has from-below gotten high enough?
      from-below
      (find-over-approx test? (* from-below base))))

;; given a value from-above such that (test? from-above)
;; it will return a valu m such that not (test? m)
(define (find-under-approx test? from-above)
  (if (test? from-above)
      (find-under-approx test? (/ from-above base))
      from-above))

;; approximates the crossover point to within 1% of its true value
;; returns an overapproximation, underapproximation, and their average
;; all three are within 1% (the average is the closest at about 0.5%)
(define (zero-in test? from-below from-above) 
  (let ([avg (/ (+ from-below from-above) 2.0)])
    ; the crossover point is somewhere between from-below and from-above
    ; if their difference is less than 1% of from-below, 
    ; then their average is is within 1% of the crossover point. 
    (if (< (- from-above from-below) (* 0.01 from-below))
        (list from-below from-above avg)
        (if (test? avg) ; run returns #t if its better to study
            (zero-in test? from-below avg)
            (zero-in test? avg from-above)))))

;; This function will find high and low such that low < n < high
;; and high-low < low/100 (i.e., n is with 1% of high and low)
;; This function additionally assumes that 1.0 is an under-approximation.
;; i.e., (test? 1.0) must be false.
(define (find-threshold test?)
  (let ([high (find-over-approx test? 1.0)])
    ; given how find-over-approx works, we know it tried high/base, 
    ; but it was too low.  Thus, we may use it as an under approximation
    (zero-in test? (/ high base) high)))

;(find-threshold (lambda (i) (> i 10000)))

(define (find-threshold-from-guess test? guess)
  (if (test? guess) ; is the guess an over or under approximation?
      ; it is an over-approximation, we divid by two since guess is too large
      (zero-in test? (find-under-approx test? (/ guess 2.0)) guess) 
      ; it is an under-approximation
      (zero-in test? guess (find-over-approx test? (* guess 2.0)))))


(define (make-printing-test port test?)
  (lambda (m)
    (fprintf port "(test (testing ~a) " m)
    (let-values ([(res t1 t2 t3) (time-apply test? (list m))])
      (fprintf port "(result ~a) (time ~a ~a ~a))\n" (first res) t1 t2 t3)
      (flush-output port)
      (first res))))

(define (find-threshold-from-guess-printing port test? guess)
  (fprintf port "(find-threshold-from-guess (guess ~a) (tests \n" guess)
  (let-values ([(res t1 t2 t3) (time-apply find-threshold-from-guess 
                                           (list (make-printing-test port test?) guess))])
    (fprintf port ")\n (result ~a) (time ~a ~a ~a))\n" (first res) t1 t2 t3)
    (first res)))

(define (find-threshold-printing port test?)
  (fprintf port "(find-threshold (tests \n")
  (let-values ([(res t1 t2 t3) (time-apply find-threshold 
                                           (list (make-printing-test port test?)))])
    (fprintf port ")\n (result ~a) (time ~a ~a ~a))\n" (first res) t1 t2 t3)
    (first res)))


(define (expand-param s p_other p_i r_base R r_study gamma)
  (list s
        s
        (let ([p_not-none (cons p_other (make-list s p_i))])
          (cons (- 1 (apply + p_not-none))
                p_not-none))
        (make-list (+ s 1) r_base)
        (make-list s R)
        r_study
        gamma))


(define (make-allowed-test epsilon s p_other p_i r_base r_study gamma)
  (lambda (R)
    (audit epsilon
           (apply make-multi-step-nmdp 
                  (expand-param s p_other p_i r_base R r_study gamma))
           (list [list (list-state->state-symbol (cons 'n (make-list s 't)))
                       'a.s]))))

;; using the single step model, we guess the read threshold point.
(define (args->aert s p_other p_i r_base r_study gamma)
  (* (/ (+ p_other (* s p_i)) p_i)
     r_study))


;; Finds the SERT'
(define (run-around-test epsilon s p_other p_i r_base r_study gamma) 
  ;(let ([output-port (current-output-port)])
  (call-with-output-file 
      (format "results/around-~a-~a-~a-~a-~a-~a-~a.rkt" epsilon s p_other p_i r_base r_study gamma)
    (lambda (output-port)
      (fprintf output-port 
               "(run-around-test (args ~a ~a ~a ~a ~a ~a ~a)\n" 
               epsilon s p_other p_i r_base r_study gamma)
      (begin0
        (find-threshold-from-guess-printing output-port 
                                               (make-allowed-test epsilon s p_other p_i r_base r_study gamma)
                                               (args->aert s p_other p_i r_base r_study gamma))
        (fprintf output-port ")\n")))
    #:exists 'replace))


;; Find the SERT
(define (run-fresh-test epsilon s p_other p_i r_base r_study gamma)
  (call-with-output-file 
      (format "results/fresh-~a-~a-~a-~a-~a-~a-~a.rkt" epsilon s p_other p_i r_base r_study gamma)
    (lambda (output-port)
      (fprintf output-port 
               "(run-fresh-test (args ~a ~a ~a ~a ~a ~a ~a)\n" 
               epsilon s p_other p_i r_base r_study gamma)
      (begin0 (find-threshold-printing output-port 
                                  (make-allowed-test epsilon s p_other p_i r_base r_study gamma))
              (fprintf output-port ")\n")))
    #:exists 'replace))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; The Tests
;; Uncomment any of the tests you would like to run or write your own.

;; around tests on s=2
;
;;                s p_other p_i r_base r_study gamma
;(run-around-test 0.001 2 0.95 0.01 1000 1 0.01)
;(run-around-test 0.001 2 0.95 0.01 1000 1 0.1)
;(run-around-test 0.001 2 0.95 0.01 1000 1 0.9)
;
;(run-around-test 0.001 2 0.9698 0.0001 1000 1 0.01)
;(run-around-test 0.001 2 0.9698 0.0001 1000 1 0.1)
;(run-around-test 0.001 2 0.9698 0.0001 1000 1 0.9)
;
;(run-around-test 0.001 2 0.9698 0.0001 10 1 0.01)
;(run-around-test 0.001 2 0.9698 0.0001 10 1 0.1)
;(run-around-test 0.001 2 0.9698 0.0001 10 1 0.9)
;
;(run-around-test 0.001 2 0.9698 0.0001 1 1 0.01)
;(run-around-test 0.001 2 0.9698 0.0001 1 1 0.1)
;(run-around-test 0.001 2 0.9698 0.0001 1 1 0.9)
;
;(run-around-test 0.001 2 0.95 0.0001 1000 1 0.01)
;(run-around-test 0.001 2 0.95 0.0001 1000 1 0.1)
;(run-around-test 0.001 2 0.95 0.0001 1000 1 0.9)
;
;(run-around-test 0.001 2 0.8 0.01 1000 1 0.01)
;(run-around-test 0.001 2 0.8 0.01 1000 1 0.1)
;(run-around-test 0.001 2 0.8 0.01 1000 1 0.9)
;
;(run-around-test 0.001 2 0.95 0.01 10000 1 0.01)
;(run-around-test 0.001 2 0.95 0.01 10000 1 0.1)
;(run-around-test 0.001 2 0.95 0.01 10000 1 0.9)
;
;(run-around-test 0.001 2 0.95 0.01 100 1 0.01)
;(run-around-test 0.001 2 0.95 0.01 100 1 0.1)
;(run-around-test 0.001 2 0.95 0.01 100 1 0.9)
;
;(run-around-test 0.001 2 0.95 0.01 10 1 0.01)
;(run-around-test 0.001 2 0.95 0.01 10 1 0.1)
;(run-around-test 0.001 2 0.95 0.01 10 1 0.9)
;
;(run-around-test 0.001 2 0.95 0.01 1 1 0.01)
;(run-around-test 0.001 2 0.95 0.01 1 1 0.1)
;(run-around-test 0.001 2 0.95 0.01 1 1 0.9)
;
;(run-around-test 0.001 2 0.8 0.01 1000 1 0.01)
;(run-around-test 0.001 2 0.8 0.01 1000 1 0.1)
;(run-around-test 0.001 2 0.8 0.01 1000 1 0.9)
;
;;; fresh tests on s=2
;
;(run-fresh-test 0.001 2 0.95 0.01 1000 1 0.01)
;(run-fresh-test 0.001 2 0.95 0.01 1000 1 0.1)
;(run-fresh-test 0.001 2 0.95 0.01 1000 1 0.9)
;
;(run-fresh-test 0.001 2 0.9698 0.0001 1000 1 0.01)
;(run-fresh-test 0.001 2 0.9698 0.0001 1000 1 0.1)
;(run-fresh-test 0.001 2 0.9698 0.0001 1000 1 0.9)
;
;(run-fresh-test 0.001 2 0.9698 0.0001 10 1 0.01)
;(run-fresh-test 0.001 2 0.9698 0.0001 10 1 0.1)
;(run-fresh-test 0.001 2 0.9698 0.0001 10 1 0.9)
;
;(run-fresh-test 0.001 2 0.9698 0.0001 1 1 0.01)
;(run-fresh-test 0.001 2 0.9698 0.0001 1 1 0.1)
;(run-fresh-test 0.001 2 0.9698 0.0001 1 1 0.9)
;
;(run-fresh-test 0.001 2 0.95 0.0001 1000 1 0.01)
;(run-fresh-test 0.001 2 0.95 0.0001 1000 1 0.1)
;(run-fresh-test 0.001 2 0.95 0.0001 1000 1 0.9)
;
;(run-fresh-test 0.001 2 0.8 0.01 1000 1 0.01)
;(run-fresh-test 0.001 2 0.8 0.01 1000 1 0.1)
;(run-fresh-test 0.001 2 0.8 0.01 1000 1 0.9)
;
;(run-fresh-test 0.001 2 0.95 0.01 10000 1 0.01)
;(run-fresh-test 0.001 2 0.95 0.01 10000 1 0.1)
;(run-fresh-test 0.001 2 0.95 0.01 10000 1 0.9)
;
;(run-fresh-test 0.001 2 0.95 0.01 100 1 0.01)
;(run-fresh-test 0.001 2 0.95 0.01 100 1 0.1)
;(run-fresh-test 0.001 2 0.95 0.01 100 1 0.9)
;
;(run-fresh-test 0.001 2 0.95 0.01 10 1 0.01)
;(run-fresh-test 0.001 2 0.95 0.01 10 1 0.1)
;(run-fresh-test 0.001 2 0.95 0.01 10 1 0.9)
;
;(run-fresh-test 0.001 2 0.95 0.01 1 1 0.01)
;(run-fresh-test 0.001 2 0.95 0.01 1 1 0.1)
;(run-fresh-test 0.001 2 0.95 0.01 1 1 0.9)




;; s=3 tests

;(run-around-test 0.001 3 0.94 0.01 1000 1 0.01)
;(run-around-test 0.001 3 0.94 0.01 1000 1 0.1);
;(run-around-test 0.001 3 0.94 0.01 1000 1 0.9)

;(run-fresh-test 0.001 3 0.94 0.01 1000 1 0.01)
;(run-fresh-test 0.001 3 0.94 0.01 1000 1 0.1);
;(run-fresh-test 0.001 3 0.94 0.01 1000 1 0.9)

