/*
 *     INCLUDE HEADERS
 */
#include <stdio.h>
#include <math.h>
#include <time.h>
#include "mex.h"

/*
 *     DEFINES ASSOCIATED WITH MATRIX MANIPULATION 
 */
#define ON 1
#define OFF 0
#define RUNCHK ON             /* Run-time checks switched on/off. */

/* "Inline" functions.                                                 */
/* ( No run-time checks is performed, when inline functions are used ) */
#define nof_rows(ptm)                      (ptm->row)                           /* See getrows */
#define nof_cols(ptm)                      (ptm->col)                           /* See getcols */
#define vec_len(ptv)                       (ptv->row+ptv->col-1)                /* See length  */
#define get_val(ptm,row_pos,col_pos)       (ptm->mat[row_pos][col_pos])         /* See mget    */
#define put_val(ptm,row_pos,col_pos,value) ((ptm->mat[row_pos][col_pos])=value) /* See mput    */
#define rvget(ptv,element)                 (ptv->mat[0][element])               /* See vget    */
#define cvget(ptv,element)                 (ptv->mat[element][0])               /* See vget    */
#define rvput(ptv,element,value)           ((ptv->mat[0][element])=value)       /* See vput    */
#define cvput(ptv,element,value)           ((ptv->mat[element][0])=value)       /* See vput    */


/* Declaration of the "abstract" data-type. */

typedef struct {               /* Matrix structure for C library  */
	int row;               /* These are meant to be "private" */
	int col;               /* and should only be accessed via */
	double **mat;          /* the "member functions" below.   */
} matrix;


typedef struct {               /* Matrix structure for C library  */
	int row;               /* These are meant to be "private" */
	int col;               /* and should only be accessed via */
	int **mat;             /* the "member functions" below.   */
} intmatrix;

typedef struct {               /* Optional initializations        */
	matrix *wmean;         /* Mean of measurement noise       */
	matrix *vmean;         /* Mean of process noise           */
	matrix *init;          /* Initialization parameters       */
} optpar;


typedef struct {               /* Observation data structure       */
        int    ny;             /* Dimension of observation vector  */
	int    nobs;           /* Number of observations in stream */
	int    nw;             /* Dimension of obs. noise vector   */
	matrix *y;             /* Mean of process noise            */
        intmatrix *tidx;       /* Time stamps for obs. in .y       */
	matrix *wmean;         /* Mean of measurement noise        */
	matrix *Sw;            /* Root of noise covariance matrix  */
	matrix *y0;            /* Storage of current outp. estimate*/
        matrix *y02;           /* Storage of 2 * output estimate   */
	matrix *ybar;          /* Storage of scaled output estimate*/
	matrix *Sy;            /* Root of outp. covariance matrix  */
	matrix *Sy0;           /* Rectangular root of outp. cov mat*/
	matrix *Syx;           /* Root of cross-covariance matrix  */
	matrix *Syw;           /* Root of cross-covariance matrix  */
	matrix *Syx2;          /* Root of cross-covariance matrix  */
	matrix *Syw2;          /* Root of cross-covariance matrix  */
	matrix *K;             /* Kalman gain                      */
	matrix *Sxlong;        /* Long root of covariance matrix   */
	matrix *hSwt;          /* Sw multiplied by h and transposed*/
	int    yidx;           /* Index to next observation        */
	int    lasttime;       /* Sample no. for last observation  */
	matrix *wtmp;          /* temp vector                      */
	matrix *wtmp2;         /* temp vector                      */
	matrix *syp;           /* temp vector                      */
	matrix *sym;           /* temp vector                      */
} obsstruct;


/* Declaration of the "member functions".   */
matrix *mmake( int, int );
void mfree( matrix* );
void mprint( matrix* );
void merror( char* );
int getrows( matrix* );
int getcols( matrix* );
void minit( matrix* );
void madd( matrix*, matrix*, matrix* );
void subvec( matrix*, matrix*, int, int);
void mset( matrix*, matrix*);
intmatrix *intmmake( int, int );
void intmfree( intmatrix* );
int triagold(matrix*, matrix*);
void triag(matrix*, matrix*, matrix*);
int fbsubst(matrix*, matrix*);
int makerect(matrix*, matrix*, matrix*, matrix*, matrix*);
int makerect2(matrix*, matrix*, matrix*, matrix*, matrix*, matrix*, matrix*);
int thmat(matrix*, matrix*, double);


/*
 *     PROTOTYPE DECLARATION
 */
matrix* mat2sm(const mxArray*);
void sm2mat(mxArray*, matrix*);
intmatrix* mat2intsm(const mxArray*);
void intsm2mat(mxArray*, intmatrix*);
int dd1filt(int (*xfct)(matrix*, matrix*, matrix*, matrix*, int),
            int (*yfct)(matrix*, matrix*, matrix*, int),
	    matrix*, matrix*, matrix*, matrix*, matrix*, matrix*, 
            matrix*, matrix*, intmatrix*, optpar*);
int dd1filtm(int (*xfct)(matrix*, matrix*, matrix*, matrix*, int),
            int (*yfct)(matrix*, matrix*, matrix*, int),
	    matrix*, matrix*, matrix*, matrix*, matrix*, matrix*, 
            obsstruct*, int, optpar*);
int dd2filt(int (*xfct)(matrix*, matrix*, matrix*, matrix*, int),
            int (*yfct)(matrix*, matrix*, matrix*, int),
	    matrix*, matrix*, matrix*, matrix*, matrix*, matrix*, 
            matrix*,matrix*, intmatrix*, optpar*);
int dd2filtm(int (*xfct)(matrix*, matrix*, matrix*, matrix*, int),
            int (*yfct)(matrix*, matrix*, matrix*, int),
	    matrix*, matrix*, matrix*, matrix*, matrix*, matrix*, 
            obsstruct*, int, optpar*);


/* DD1FILT     This function implements the DD1-filter; a state estimator for
 *  nonlinear systems that is based on first-order polynomial approximations 
 *  of the nonlinear mappings. The approximations are derived by using a 
 *  multivariable extension of Stirling's interpolation formula. 
 *  The model of the nonlinear system must be specified in the form:
 *              x(k+1) = f[x(k),u(k),v(k)]
 *              y(k+1) = g[x(k+1),w(k+1)]
 *  where 'x' is the state vector, 'u' is a possible input, and 'v' and 'w'
 *  are (white) noise sources.
 *
 * Call
 *     dd1filt(xfunc,yfunc,xhat,Smat,x0,P0,q,r,u,y,tidx,optpar);
 *
 *
 * Outputs
 *  xfunc   - Name of function containing state equations.
 *  yfunc   - Name of function containing observation equations.
 *  xhat    - State estimates. Dimension is [samples+1-by-states].
 *  Smat    - Matrix for which each row contains elements of (the upper triangular
 *            part of) the Cholesky factor of the covariance matrix. The 
 *            dimension is [samples+1-by-0.5*states*(states+1)]. The individual
 *            covariance matrices can later be extracted with SMAT2COV.
 *
 * Inputs
 *  x0      - Initial state vector.
 *  P0      - Root of initial covariance matrix (symmetric, nonnegative 
 *            definite).
 *  q,r     - Root of covariance matrices for v and w, respectively.
 *  u       - Input signal. Dimension is [samples-by-inputs].
 *            Should be empty if there are no inputs.
 *  y       - Output signal. Dimension is [observations-by-outputs].
 *  timeidx - Vector containing time stamps (in samples) for the 
 *            observations in y.
 *  optpar  - Data structure containing optional parameters:
 *            .vmean: Mean of process noise vector.
 *            .wmean: Mean of measurement noise vector.
 *            .init : Initial parameters for 'xfile', 'yfile'
 *                    (use an arbitrary format).
 *
 * Written by: Magnus Norgaard, IAU/IMM, Technical University of Denmark
 * LastEditDate: Apr. 15, 2000
 */
int dd1filt(int (*xfct)(matrix*, matrix*, matrix*, matrix*, int),
            int (*yfct)(matrix*, matrix*, matrix*, int),
            matrix *xhat_data, matrix *Smat, matrix *xbar, 
            matrix *P0, matrix *q, matrix *r, matrix *u, matrix *y, 
	    intmatrix *tidx, optpar *opt)
{
   register int i, j, l;
   int samples, nu, ny, nx, nw, nv, kx, kw, kv, k, yidx;
   int lasttime, ktmp, counter=0;
   double h, h2, scal1;
   double *ptm1, *ptm2, *ptm3, *ptm4, *Smatptm, dum;
   matrix *Syx, *Syw, *Sy, *K, *xhat, *ybar, *Sx, *Sy0;
   matrix *hSxt, *hSvt, *hSwt, *Sxbar, *Sxlong, *uk1;
   matrix *syp, *sym, *swp, *swm, *sxp, *sxm, *svp, *svm;
   matrix *xtmp, *xtmp2, *wtmp, *wtmp2, *v, *vtmp, *vtmp2;

   h2    = 3.0;               /* Squared divided difference step */
   h     = sqrt(h2);          /* Divided difference step         */
   scal1 = 0.5/h;             /* Scaling factor                  */
   if (getrows(u)==0){        /* No external input present       */
      nu = 0;
      samples = cvget(tidx,nof_rows(tidx)-1);
      uk1 = mmake(1,1);
   }
   else{
      samples = getrows(u);   /* # of samples and inputs */
      nu      = getcols(u);
      uk1     = mmake(nu,1);
   }
   nx  = getrows(P0);         /* # of states  */
   ny  = getcols(y);          /* # of outputs */
   nv  = getrows(q);          /* # of process noise sources */
   nw  = getrows(r);          /* # of measurement noise sources */
   Smatptm = Smat->mat[0];    /* Pointer to first element in Smat */


   /* Allocate space for vectors and matrices */
   xhat= mmake(nx,1);
   ybar= mmake(ny,1);
   K   = mmake(nx,ny);
   Sy  = mmake(ny,nx+nw);
   Sx  = mmake(nx,nx);
   Sy0 = mmake(ny,ny);
   Sxbar = mmake(nx,nx+nv); hSxt = mmake(nx,nx);
   hSwt = mmake(nw,nw); hSvt = mmake(nv,nv);
   Syx = mmake(nx,ny); Syw = mmake(nw,ny);
   syp = mmake(ny,1); sym=mmake(ny,1); swp=mmake(ny,1); swm=mmake(ny,1);
   sxm = mmake(nx,1); sxp = mmake(nx,1); svp = mmake(nx,1); svm=mmake(nx,1);
   xtmp= mmake(nx,1); wtmp = mmake(nw,1); xtmp2=mmake(nx,1);
   wtmp2 = mmake(nw,1); vtmp = mmake(nv,1);vtmp2 = mmake(nv,1);
   Sxlong = mmake(nx,nx+nw);
   v   = mmake(1,nx+nw+ny+nv);
   
   /* Transpose covariance square roots and multiply by h */
   thmat(hSwt,r,h);
   thmat(hSvt,q,h);

   /* Put P0 into Sx */   
   minit(Sx);minit(Sy0);
   triag(Sx,P0,v);
   thmat(hSxt,Sx,h);

   yidx= 0;                   /* Index into y-vector */ 
   lasttime = cvget(tidx,nof_rows(tidx)-1); /* Time for last observation */


   /* Initialize state update and observation functions */
   (*xfct)(opt->init,xhat,uk1,opt->vmean,-1);
   (*yfct)(opt->init,xbar,opt->wmean,-1);

   printf("Filtering in progress\n");
   
   
   /* >>>>>>>>>>>>>>>>>>>>>>>>>>>> FILTERING <<<<<<<<<<<<<<<<<<<<<<<<<<< */
   for(k=0;k<=samples;k++){
         
      /* --- Measurement update (a posteriori update) --- */
      (*yfct)(ybar,xbar,opt->wmean,0);
      if(k<=lasttime && cvget(tidx,yidx)==k){
         ptm3=hSxt->mat[0];
         for(kx=0;kx<nx;kx++){

	    /* Calculate syp */
            ptm1 = xtmp->mat[0]; ptm2=xbar->mat[0]; ptm4=xtmp2->mat[0];
            for(i=0;i<nx;i++){
	       *(ptm1++) = *(ptm2) + *ptm3;
	       *(ptm4++) = *(ptm2++) - *(ptm3++);
	    }
            (*yfct)(syp,xtmp,opt->wmean,0);
	    
	    /* Calculate sym */
            (*yfct)(sym,xtmp2,opt->wmean,0);
	    
	    /* Update Syx */
	    ptm1 = syp->mat[0]; ptm2=sym->mat[0]; ptm4=Syx->mat[kx];
            for(i=0;i<ny;i++){
	       dum = scal1*(*(ptm1++) - *(ptm2++));
	       put_val(Sy,i,kx,dum);
	       *(ptm4++) = dum;
	    }
         }
	 ptm3 = hSwt->mat[0];
	 kx   = nx-1;
         for(kw=0;kw<nw;kw++){
	 
	    /* Calculate swp */
            ptm1 = wtmp->mat[0]; ptm2=(opt->wmean)->mat[0]; 
	    ptm4 = wtmp2->mat[0];
            for(i=0;i<nw;i++){
	       *(ptm1++) = *ptm2 + *ptm3;
	       *(ptm4++) = *(ptm2++) - *(ptm3++);
	    }
            (*yfct)(swp,xbar,wtmp,0);
	    
	    /* Calculate swm */
            (*yfct)(swm,xbar,wtmp2,0);
	    
	    /* Update Syw */
	    ptm1 = swp->mat[0]; ptm2=swm->mat[0]; ptm4=Syw->mat[kw];
	    kx++;
            for(i=0;i<ny;i++){
	       dum = scal1*(*(ptm1++) - *(ptm2++));
	       put_val(Sy,i,kx,dum);
	       *(ptm4++) = dum;
	    }
         }

	 /* Cholesky factor of a posteriori output estimate of covariance */
         ptm1 = K->mat[0];                 /* Calculate Sxbar*Syx'   */
	 for(i=0;i<nx;i++){                /* Used later for calc. K */
	    for(j=0;j<ny;j++){
	       ptm2 = Sx->mat[i]+i;
	       ptm3 = Sy->mat[j]+i;
	       for(l=i,dum=0.0;l<nx;l++)
	          dum+=*(ptm2++)* *(ptm3++);
	       *(ptm1++) = dum;
	    }
	 }

         /* Calculate Sy */
	 triag(Sy0,Sy,v);
	 for(i=0;i<ny;i++){
	    if(get_val(Sy0,i,i)==0)
                mexErrMsgTxt("Output covariance matrix singular.");
         }

	 /* Calculate Kalman gain */
	 fbsubst(K,Sy0);
	 	 
	 /* Update state estimate, xhat */
	 ptm1=K->mat[0];
	 for(i=0;i<nx;i++){
	    dum=cvget(xbar,i);
	    ptm2 = y->mat[yidx];
	    for(j=0;j<ny;j++)
	       dum+=*(ptm1++) * (*(ptm2++)-cvget(ybar,j));
	    cvput(xhat,i,dum);
	 }
	 
	 /* Cholesky factor of a'posteriori estimation error covariance */
	 makerect(Sxlong,Sx,K,Syx,Syw);
	 triag(Sx,Sxlong,v);
         ++yidx; 
	 
	 /* Transpose Sx and multiply h */
	 for(i=0;i<nx;i++){
	    ptm1=hSxt->mat[i];
            for(j=0;j<=i;j++)
               *(ptm1++) = h*get_val(Sx,j,i);
         }	 
      }
      else{
         mset(xhat,xbar);
      }

      /* Store covariance estimate */
      ktmp = nx-1;
      for(i=0;i<nx;i++,ktmp--){
	 ptm1=Sx->mat[i]+i;
           for(j=0;j<=ktmp;j++)
           *(Smatptm++) = *(ptm1++);
      }

      
      /* --- Time update (a'priori update) of state and covariance --- */
      if(k<samples){

         /* Copy current u to u-vector */
         for(i=0;i<nu;i++)
	    cvput(uk1,i,get_val(u,k,i));

         /* --- Measurement update (a posteriori update) --- */
         (*xfct)(xbar,xhat,uk1,opt->vmean,0);

         ptm3=hSxt->mat[0];
         for(kx=0;kx<nx;kx++){

	    /* Calculate sxp */
            ptm1 = xtmp->mat[0]; ptm2=xhat->mat[0]; ptm4=xtmp2->mat[0];
            for(i=0;i<nx;i++){
	       *(ptm1++) = *(ptm2) + *ptm3;
	       *(ptm4++) = *(ptm2++) - *(ptm3++);
	    }
            (*xfct)(sxp,xtmp,uk1,opt->vmean,0);
	    
	    /* Calculate sxm */
            (*xfct)(sxm,xtmp2,uk1,opt->vmean,0);
	    
	    /* Update Sxx */
	    ptm1 = sxp->mat[0]; ptm2=sxm->mat[0];
            for(i=0;i<nx;i++){
	       dum = scal1*(*(ptm1++) - *(ptm2++));
	       put_val(Sxbar,i,kx,dum);
	    }
         }

	 ptm3 = hSvt->mat[0];
	 kx   = nx-1;
         for(kv=0;kv<nv;kv++){
	 
	    /* Calculate swp */
            ptm1 = vtmp->mat[0]; ptm2=(opt->vmean)->mat[0]; 
	    ptm4 = vtmp2->mat[0];
            for(i=0;i<nv;i++){
	       *(ptm1++) = *ptm2 + *ptm3;
	       *(ptm4++) = *(ptm2++) - *(ptm3++);
	    }
            (*xfct)(svp,xhat,uk1,vtmp,0);
	    
	    /* Calculate swm */
            (*xfct)(svm,xhat,uk1,vtmp2,0);
	    
	    /* Update Sxv */
	    ptm1 = svp->mat[0]; ptm2=svm->mat[0];
	    kx++;
            for(i=0;i<nx;i++){
	       dum = scal1*(*(ptm1++) - *(ptm2++));
	       put_val(Sxbar,i,kx,dum);
	    }
         }

	 /* Cholesky factor of a'priori estimation error covariance */
         /* Sxbar = triag([Sxx Sxv]); */
	 triag(Sx,Sxbar,v);

	 /* Transpose Sx and multiply h */
         for(i=0;i<nx;i++){
	    ptm1=hSxt->mat[i];
            for(j=0;j<=i;j++)
               *(ptm1++) = h*get_val(Sx,j,i);
         }
      }
      
      
      /* --- Store estimates --- */
      for(i=0;i<nx;i++)
         put_val(xhat_data,k,i,cvget(xhat,i));


      /* --- How much longer? --- */
      if ((counter+1)<= (int)(100*k/samples)){
        ++counter;
        printf(".");
      }
   }
   printf("\nFiltering completed\n");

   
   /* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> CLEAN UP <<<<<<<<<<<<<<<<<<<<<<<<<<< */
   (*xfct)(xbar,xhat,uk1,opt->vmean,-2);
   (*yfct)(ybar,xbar,opt->wmean,-2);
   mfree(K); mfree(xhat); mfree(ybar); mfree(Sy);
   mfree(Syx);mfree(Syw); mfree(uk1); mfree(syp); mfree(sym); mfree(swp);
   mfree(swm); mfree(sxp); mfree(sxm); mfree(svp); mfree(svm);
   mfree(xtmp); mfree(xtmp2); mfree(wtmp); mfree(wtmp2);
   mfree(vtmp); mfree(vtmp2); mfree(v); mfree(Sxbar); mfree(hSxt);
   mfree(Sxlong); mfree(hSvt); mfree(hSwt); mfree(Sx); mfree(Sy0);
   return 0;
}


/* DD1FILTM     This function implements the DD1-filter; a state estimator for
 *  nonlinear systems that is based on first-order polynomial approximations 
 *  of the nonlinear mappings. The approximations are derived by using a 
 *  multivariable extension of Stirling's interpolation formula. The
 *  function is implemented to handle multiple observation streams.
 *  The model of the nonlinear system must be specified in the form:
 *              x(k+1) = f[x(k),u(k),v(k)]
 *              y1(k)  = g1[x(k),w1(k)]
 *                       :
 *              yn(k)  = gn[x(k),wn(k)]
 *
 *  where 'x' is the state vector, 'u' is a possible input, and 'v' and 'w'
 *  are (white) noise sources.
 *
 * Call
 *     dd1filtm(xfunc,yfunc,xhat,Smat,x0,P0,q,u,obs,streams,optpar);
 *
 *
 * Outputs
 *  xfunc   - Name of function containing state equations.
 *  yfunc   - Name of function containing observation equations.
 *  xhat    - State estimates. Dimension is [samples+1-by-states].
 *  Smat    - Matrix for which each row contains elements of (the upper triangular
 *            part of) the Cholesky factor of the covariance matrix. The 
 *            dimension is [samples+1-by-0.5*states*(states+1)]. The individual
 *            covariance matrices can later be extracted with SMAT2COV.
 *
 * Inputs
 *  x0      - Initial state vector.
 *  P0      - Root of initial covariance matrix (symmetric, nonnegative 
 *            definite).
 *  q       - Covariance matrices for process noise.
 *  u       - Input signal. Dimension is [samples-by-inputs].
 *            Should be empty (have zero rows) if there are no inputs.
 *  obs     - An array of data structures with stream specific parameters. The
 *            following elements must exist in each structure upon call:
 *            .ny    - Stream width (outputs in the stream).
 *            .nobs  - Observations in the stream.
 *            .y     - Matrix containing observations in the stream [ny x nobs].
 *            .tidx  - Vecor containing the time stamps for the observations.
 *            .nw    - Number of observation noise sources in stream.
 *            .wmean - Mean of measurement noise vector.
 *            .Sw    - Square root of measurement noise covariance.
 *  optpar  - Data structure containing optional parameters:
 *            .vmean: Mean of process noise vector.
 *            .init : Initial parameters for 'xfile', 'yfile'
 *                    (use an arbitrary format).
 *
 * Written by: Magnus Norgaard, IAU/IMM, Technical University of Denmark
 * LastEditDate: Apr. 15, 2000
 */
int dd1filtm(int (*xfct)(matrix*, matrix*, matrix*, matrix*, int),
            int (*yfct)(matrix*, matrix*, matrix*, int),
	    matrix *xhat_data, matrix *Smat, matrix *xbar, matrix *P0, 
            matrix *q, matrix *u, obsstruct *obs, int streams, optpar *opt)
{
   register int i, j, l, n;
   int samples, nu, ny, nx, nv, nw, kx, kw, kv, k, counter=0;
   int lasttime, idum, ktmp;
   double h, h2, scal1;
   double *ptm1, *ptm2, *ptm3, *ptm4, *Smatptm, dum;
   matrix *xhat, *sxp, *sxm, *hSxt, *hSvt, *Sx;
   matrix *Sxbar, *uk1, *xtmp, *xtmp2, *v, *vtmp, *vtmp2;
   intmatrix *idx1;

   
   h2    = 3;                 /* Squared divided difference step */
   h     = sqrt(h2);          /* Divided difference step         */
   scal1 = 0.5/h;             /* Scaling factor                  */
   if (getrows(u)==0){        /* No external input present       */
      nu = 0;
      uk1 = mmake(1,1);
   }
   else{
      nu      = getcols(u);
      uk1     = mmake(nu,1);
   }
   samples = getrows(xhat_data)-1; /* # of samples */
   nx  = getrows(P0);         /* # of states  */
   nv  = getrows(q);          /* # of process noise sources */
   Smatptm = Smat->mat[0];    /* Pointer to first element in Smat */


   /* Index vectors and samples */
   idx1 = intmmake(1,streams);
   ny = 0;
   nw = 0;
   for(n=0;n<streams;n++){
      ny +=obs[n].ny;            /* Total number of observations */
      nw +=obs[n].nw;            /* Total number of process noise sources */
      rvput(idx1,n,ny-obs[n].ny);/* Index to the start of each stream */
      obs[n].yidx = 0;           /* Index points to first observation */
      obs[n].lasttime = cvget(obs[n].tidx,obs[n].nobs-1);/* Time for last obs.*/
      obs[n].K   = mmake(nx,obs[n].ny);
      obs[n].Sy  = mmake(obs[n].ny,nx+obs[n].nw);
      obs[n].Sy0 = mmake(obs[n].ny,obs[n].ny);
      minit(obs[n].Sy0);
      obs[n].Syx = mmake(nx,obs[n].ny);
      obs[n].Syw = mmake(obs[n].nw,obs[n].ny);
      obs[n].hSwt = mmake(obs[n].nw,obs[n].nw);    
      minit(obs[n].hSwt);
      thmat(obs[n].hSwt,obs[n].Sw,h);        /* Transpose Sw and multiply h */
      obs[n].wtmp = mmake(obs[n].nw,1);
      obs[n].wtmp2 = mmake(obs[n].nw,1);
      obs[n].syp = mmake(obs[n].ny,1);
      obs[n].sym = mmake(obs[n].ny,1);
      obs[n].y0  = mmake(obs[n].ny,1);
      obs[n].Sxlong = mmake(nx,nx+obs[n].nw);
   }

   /* Allocate space for vectors and matrices */
   xhat= mmake(nx,1);
   Sxbar = mmake(nx,nx+nv); hSxt = mmake(nx,nx);
   Sx  = mmake(nx,nx);
   hSvt = mmake(nv,nv);
   sxm = mmake(nx,1); sxp = mmake(nx,1);
   xtmp= mmake(nx,1); xtmp2=mmake(nx,1);
   vtmp = mmake(nv,1);vtmp2 = mmake(nv,1);
   v   = mmake(1,nx+nw+ny+nv);
   
   /* Transpose covariance square roots and multiply by h */
   minit(hSvt); thmat(hSvt,q,h);

   /* Put P0 into Sx */   
   minit(Sx);
   triag(Sx,P0,v);
   thmat(hSxt,Sx,h);
    

   /* Initialize state update and observation functions */
   (*xfct)(opt->init,xhat,uk1,opt->vmean,-1);
   (*yfct)(opt->init,xbar,xbar,-1);
   
   printf("Filtering in progress\n");
   
   /* >>>>>>>>>>>>>>>>>>>>>>>>>>>> FILTERING <<<<<<<<<<<<<<<<<<<<<<<<<<< */
   for(k=0;k<=samples;k++){

      /* --- Measurement update (a posteriori update) --- */
      for(n=0;n<streams;n++){
	 (*yfct)(obs[n].y0,xbar,obs[n].wmean,n);
	 if(k<=obs[n].lasttime && cvget(obs[n].tidx,obs[n].yidx)==k){
            ptm3=hSxt->mat[0];
            for(kx=0;kx<nx;kx++){

	       /* Calculate syp */
               ptm1 = xtmp->mat[0]; ptm2=xbar->mat[0]; ptm4=xtmp2->mat[0];
               for(i=0;i<nx;i++){
		  *(ptm1++) = *(ptm2) + *ptm3;
		  *(ptm4++) = *(ptm2++) - *(ptm3++);
	       }
               (*yfct)(obs[n].syp,xtmp,obs[n].wmean,n);

	       /* Calculate sym */
               (*yfct)(obs[n].sym,xtmp2,obs[n].wmean,n);

	       /* Update Syx */
	       ptm1 = obs[n].syp->mat[0];
	       ptm2 = obs[n].sym->mat[0];
	       ptm4 = obs[n].Syx->mat[kx];
               for(i=0;i<obs[n].ny;i++){
		  dum = scal1*(*(ptm1++) - *(ptm2++));
		  put_val(obs[n].Sy,i,kx,dum);
		  *(ptm4++) = dum;
	       }
            }
	    ptm3 = obs[n].hSwt->mat[0];
	    kx   = nx-1;
            for(kw=0;kw<obs[n].nw;kw++){

	       /* Calculate swp */
               ptm1 = obs[n].wtmp->mat[0];
	       ptm2 = obs[n].wmean->mat[0]; 
	       ptm4 = obs[n].wtmp2->mat[0];
               for(i=0;i<obs[n].nw;i++){
		  *(ptm1++) = *ptm2 + *ptm3;
		  *(ptm4++) = *(ptm2++) - *(ptm3++);
	       }
               (*yfct)(obs[n].syp,xbar,obs[n].wtmp,n);

	       /* Calculate swm */
               (*yfct)(obs[n].sym,xbar,obs[n].wtmp2,n);

	       /* Update Syw */
	       ptm1 = obs[n].syp->mat[0];
	       ptm2 = obs[n].sym->mat[0];
	       ptm4 = obs[n].Syw->mat[kw];
	       kx++;
               for(i=0;i<obs[n].ny;i++){
		  dum = scal1*(*(ptm1++) - *(ptm2++));
		  put_val(obs[n].Sy,i,kx,dum);
		  *(ptm4++) = dum;
	       }
            }

	    /* Cholesky factor of a posteriori output estimate of covariance */
            ptm1 = obs[n].K->mat[0];          /* Calculate Sxbar*Syx'   */
	    for(i=0;i<nx;i++){                /* Used later for calc. K */
	       for(j=0;j<obs[n].ny;j++){
		  ptm2 = Sx->mat[i]+i;
		  ptm3 = obs[n].Sy->mat[j]+i;
		  for(l=i,dum=0.0;l<nx;l++)
	             dum+=*(ptm2++)* *(ptm3++);
		  *(ptm1++) = dum;
	       }
	    }

            /* Calculate Sy */
	    triag(obs[n].Sy0,obs[n].Sy,v);
	    for(i=0;i<obs[n].ny;i++){
	       if(get_val(obs[n].Sy0,i,i)==0)
                   mexErrMsgTxt("Output covariance matrix singular.");
            }

	    /* Calculate Kalman gain */
	    fbsubst(obs[n].K,obs[n].Sy0);

	    /* Update state estimate, xhat */
	    ptm1=obs[n].K->mat[0];
	    for(i=0;i<nx;i++){
	       dum=cvget(xbar,i);
	       ptm2 = obs[n].y->mat[obs[n].yidx];
	       for(j=0;j<obs[n].ny;j++)
		  dum+=*(ptm1++) * (*(ptm2++)-cvget(obs[n].y0,j));
	       cvput(xbar,i,dum);
	    }

	    /* Cholesky factor of a'posteriori estimation error covariance */
	    makerect(obs[n].Sxlong,Sx,obs[n].K,obs[n].Syx,obs[n].Syw);
            triag(Sx,obs[n].Sxlong,v);
            ++(obs[n].yidx); 

	    /* Transpose Sx and multiply h */
	    for(i=0;i<nx;i++){
	       ptm1=hSxt->mat[i];
               for(j=0;j<=i;j++)
        	  *(ptm1++) = h*get_val(Sx,j,i);
            }	
	 }
      }
      mset(xhat,xbar);

      /* Store covariance estimate */
      ktmp = nx-1;
      for(i=0;i<nx;i++,ktmp--){
	 ptm1=Sx->mat[i]+i;
           for(j=0;j<=ktmp;j++)
           *(Smatptm++) = *(ptm1++);
      }

      
      /* --- Time update (a'priori update) of state and covariance --- */
      if(k<samples){

         /* Copy current u to u-vector */
         for(i=0;i<nu;i++)
	    cvput(uk1,i,get_val(u,k,i));

         /* --- Measurement update (a posteriori update) --- */
         (*xfct)(xbar,xhat,uk1,opt->vmean,0);
         ptm3=hSxt->mat[0];
         for(kx=0;kx<nx;kx++){

	    /* Calculate sxp */
            ptm1 = xtmp->mat[0]; ptm2=xhat->mat[0]; ptm4=xtmp2->mat[0];
            for(i=0;i<nx;i++){
	       *(ptm1++) = *(ptm2) + *ptm3;
	       *(ptm4++) = *(ptm2++) - *(ptm3++);
	    }
            (*xfct)(sxp,xtmp,uk1,opt->vmean,0);
	    
	    /* Calculate sxm */
            (*xfct)(sxm,xtmp2,uk1,opt->vmean,0);
	    
	    /* Update Sxx */
	    ptm1 = sxp->mat[0]; ptm2=sxm->mat[0];
            for(i=0;i<nx;i++){
	       dum = scal1*(*(ptm1++) - *(ptm2++));
	       put_val(Sxbar,i,kx,dum);
	    }
         }

	 ptm3 = hSvt->mat[0];
	 kx   = nx-1;
         for(kv=0;kv<nv;kv++){
	 
	    /* Calculate svp */
            ptm1 = vtmp->mat[0]; ptm2=(opt->vmean)->mat[0]; 
	    ptm4 = vtmp2->mat[0];
            for(i=0;i<nv;i++){
	       *(ptm1++) = *ptm2 + *ptm3;
	       *(ptm4++) = *(ptm2++) - *(ptm3++);
	    }
            (*xfct)(sxp,xhat,uk1,vtmp,0);
	    
	    /* Calculate svm */
            (*xfct)(sxm,xhat,uk1,vtmp2,0);
	    
	    /* Update Sxv */
	    ptm1 = sxp->mat[0]; ptm2=sxm->mat[0];
	    kx++;
            for(i=0;i<nx;i++){
	       dum = scal1*(*(ptm1++) - *(ptm2++));
	       put_val(Sxbar,i,kx,dum);
	    }
         }

	 /* Cholesky factor of a'priori estimation error covariance */
         /* Sxbar = triag([Sxx Sxv]); */
	 triag(Sx,Sxbar,v);

	 /* Transpose Sx and multiply h */
         for(i=0;i<nx;i++){
	    ptm1=hSxt->mat[i];
            for(j=0;j<=i;j++)
               *(ptm1++) = h*get_val(Sx,j,i);
         }
      }
      
      
      /* --- Store estimates --- */
      for(i=0;i<nx;i++)
         put_val(xhat_data,k,i,cvget(xhat,i));
      

      /* --- How much longer? --- */
      if ((counter+1)<= (int)(100*k/samples)){
        ++counter;
        printf(".");
      }
   }
   printf("\nFiltering completed\n");


   /* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> CLEAN UP <<<<<<<<<<<<<<<<<<<<<<<<<<< */
   (*xfct)(xbar,xhat,uk1,opt->vmean,-2);
   (*yfct)(obs[n].y0,xbar,xbar,-2);
   for(n=0;n<streams;n++){       /* Free matrices in obs-structure */
      mfree(obs[n].y); intmfree(obs[n].tidx); mfree(obs[n].wmean); 
      mfree(obs[n].Sw); mfree(obs[n].y0); mfree(obs[n].Sy);
      mfree(obs[n].Sy0);
      mfree(obs[n].Syx); mfree(obs[n].Syw); mfree(obs[n].K); 
      mfree(obs[n].Sxlong); mfree(obs[n].hSwt); mfree(obs[n].wtmp); 
      mfree(obs[n].wtmp2); mfree(obs[n].syp); mfree(obs[n].sym);
   }
   mfree(xhat); mfree(sxp); mfree(sxm); 
   mfree(xtmp); mfree(xtmp2); mfree(vtmp); mfree(vtmp2); mfree(v); 
   mfree(Sxbar); mfree(hSxt); mfree(uk1); mfree(hSvt); mfree(Sx);
   intmfree(idx1);
   return 0;
}


/* DD2FILT     This function implements the DD2-filter; a state estimator for
 *  nonlinear systems that is based on second-order polynomial approximations 
 *  of the nonlinear mappings. The approximations are derived by using a 
 *  multivariable extension of Stirling's interpolation formula. 
 *  The model of the nonlinear system must be specified in the form:
 *              x(k+1) = f[x(k),u(k),v(k)]
 *              y(k+1) = g[x(k+1),w(k+1)]
 *  where 'x' is the state vector, 'u' is a possible input, and 'v' and 'w'
 *  are (white) noise sources.
 *
 * Call
 *     dd2filt(xfunc,yfunc,xhat,Smat,x0,P0,q,r,u,y,tidx,optpar);
 *
 *
 * Outputs
 *  xfunc   - Name of function containing state equations.
 *  yfunc   - Name of function containing observation equations.
 *  xhat    - State estimates. Dimension is [samples+1-by-states].
 *  Smat    - Matrix for which each row contains elements of (the upper triangular
 *            part of) the Cholesky factor of the covariance matrix. The 
 *            dimension is [samples+1-by-0.5*states*(states+1)]. The individual
 *            covariance matrices can later be extracted with SMAT2COV.
 *
 * Inputs
 *  x0      - Initial state vector.
 *  P0      - Root of initial covariance matrix (symmetric, nonnegative 
 *            definite).
 *  q,r     - Root of covariance matrices for v and w, respectively.
 *  u       - Input signal. Dimension is [samples-by-inputs].
 *            Should be empty if there are no inputs.
 *  y       - Output signal. Dimension is [observations-by-outputs].
 *  timeidx - Vector containing time stamps (in samples) for the 
 *            observations in y.
 *  optpar  - Data structure containing optional parameters:
 *            .vmean: Mean of process noise vector.
 *            .wmean: Mean of measurement noise vector.
 *            .init : Initial parameters for 'xfile', 'yfile'
 *                    (use an arbitrary format).
 *
 * Written by: Magnus Norgaard, IAU/IMM, Technical University of Denmark
 * LastEditDate: Apr. 15, 2000
 */
int dd2filt(int (*xfct)(matrix*, matrix*, matrix*, matrix*, int),
            int (*yfct)(matrix*, matrix*, matrix*, int),
            matrix *xhat_data, matrix *Smat, matrix *xbar, 
            matrix *P0,matrix *q, matrix *r, matrix *u, matrix *y, 
	    intmatrix *tidx, optpar *opt)
{
   register int i, j, l;
   int samples, nu, ny, nx, nw, nv, kx, kw, kv, k, ktmp, yidx;
   int lasttime, idum, counter=0; 
   double h, h2, h22, scal1, scal2, scal3, scal4, *Smatptm;
   double *ptm1, *ptm2, *ptm3, *ptm4, *ptm5, *ptm6, *ptm7, dum, dum2;
   matrix *Syx, *Syw, *Syx2, *Syw2, *Sy, *K, *xhat, *ybar;
   matrix *hSxt, *hSvt, *hSwt, *Sxbar, *Sxlong, *uk1, *y0, *y02, *Sx, *Sy0;
   matrix *syp, *sym, *swp, *swm, *sxp, *sxm, *svp, *svm, *fxbar2;
   matrix *xtmp, *xtmp2, *wtmp, *wtmp2, *v, *vtmp, *vtmp2;
   
   h2    = 3.0;               /* Squared divided difference step */
   h     = sqrt(h2);          /* Divided difference step         */
   h22   = 1/(2.0*h2);        /* 1/2h2                           */
   scal1 = 0.5/h;             /* Scaling factor                  */
   scal2 = sqrt((h2-1)/(4*h2*h2));/* Another scaling factor      */
   if (getrows(u)==0){        /* No external input present       */
      nu = 0;
      samples = cvget(tidx,nof_rows(tidx)-1);
      uk1 = mmake(1,1);
   }
   else{
      samples = getrows(u);   /* # of samples and inputs */
      nu      = getcols(u);
      uk1     = mmake(nu,1);
   }
   nx  = getrows(P0);         /* # of states                    */
   ny  = getcols(y);          /* # of outputs                   */
   nv  = getrows(q);          /* # of process noise sources     */
   nw  = getrows(r);          /* # of measurement noise sources */
   scal3 = (h2-nx-nw)/h2;     /* And a third scaling factor     */
   scal4 = (h2-nx-nv)/h2;     /* And a fourth scaling factor    */
   Smatptm = Smat->mat[0];    /* Pointer to first element in Smat */


   /* Allocate space for vectors and matrices */
   xhat= mmake(nx,1);
   ybar= mmake(ny,1);
   y0  = mmake(ny,1);
   y02 = mmake(ny,1);
   fxbar2 = mmake(nx,1);
   K   = mmake(nx,ny);
   Sy  = mmake(ny,2*(nx+nw));
   Sx  = mmake(nx,nx);
   Sy0 = mmake(ny,ny);
   Sxbar = mmake(nx,2*(nx+nv)); hSxt = mmake(nx,nx);
   hSwt = mmake(nw,nw); hSvt = mmake(nv,nv);
   Syx  = mmake(nx,ny); Syw  = mmake(nw,ny);
   Syx2 = mmake(nx,ny); Syw2 = mmake(nw,ny);
   syp = mmake(ny,1); sym=mmake(ny,1); swp=mmake(ny,1); swm=mmake(ny,1);
   sxm = mmake(nx,1); sxp = mmake(nx,1); svp = mmake(nx,1); svm=mmake(nx,1);
   xtmp= mmake(nx,1); wtmp = mmake(nw,1); xtmp2=mmake(nx,1);
   wtmp2 = mmake(nw,1); vtmp = mmake(nv,1);vtmp2 = mmake(nv,1);
   Sxlong = mmake(nx,2*(nx+nw));
   v   = mmake(1,2*(nx+nw+ny+nv));
   
   /* Transpose covariance square roots and multiply by h */
   thmat(hSwt,r,h);
   thmat(hSvt,q,h);

   /* Put P0 into Sx */   
   minit(Sx);minit(Sy0);
   triag(Sx,P0,v);
   thmat(hSxt,Sx,h);

   yidx= 0;                   /* Index into y-vector */ 
   lasttime = cvget(tidx,nof_rows(tidx)-1); /* Time for last observation */


   /* Initialize state update and observation functions */
   (*xfct)(opt->init,xhat,uk1,opt->vmean,-1);
   (*yfct)(opt->init,xbar,opt->wmean,-1);
   
   printf("Filtering in progress\n");
   
   
   /* >>>>>>>>>>>>>>>>>>>>>>>>>>>> FILTERING <<<<<<<<<<<<<<<<<<<<<<<<<<< */
   for(k=0;k<=samples;k++){

      /* --- Measurement update (a posteriori update) --- */
      (*yfct)(y0,xbar,opt->wmean,0);
      if(k<=lasttime && cvget(tidx,yidx)==k){
      
         /* Initialize 'ybar' and set y02=2*y0*/
	 ptm1 = y0->mat[0];
	 ptm2 = ybar->mat[0];
	 ptm3 = y02->mat[0];
         for(i=0;i<ny;i++){
	   *(ptm2++) = scal3* *ptm1;    /* ybar = scal3*y0 */
	   *(ptm3++) = 2.0* *(ptm1++);  /* y02 = 2y0       */
	 }
	 
         ptm3=hSxt->mat[0];
         for(kx=0;kx<nx;kx++){

	    /* Calculate syp */
            ptm1 = xtmp->mat[0]; ptm2=xbar->mat[0]; ptm4=xtmp2->mat[0];
            for(i=0;i<nx;i++){
	       *(ptm1++) = *(ptm2) + *ptm3;
	       *(ptm4++) = *(ptm2++) - *(ptm3++);
	    }
            (*yfct)(syp,xtmp,opt->wmean,0);
	    
	    /* Calculate sym */
            (*yfct)(sym,xtmp2,opt->wmean,0);
	    
	    /* Update Syx, Syx2 and ybar */
	    ptm1 = syp->mat[0]; ptm2=sym->mat[0]; ptm4=Syx->mat[kx];
	    ptm5 = y02->mat[0]; ptm6 = Syx2->mat[kx]; ptm7 = ybar->mat[0];
	    ktmp = kx+nx+nw;
            for(i=0;i<ny;i++){
	       dum = scal1*(*ptm1 - *ptm2);
	       dum2 = *(ptm1++) + *(ptm2++);
	       *(ptm7++) += dum2*h22;           /* Update ybar  */
	       dum2 = scal2*(dum2 - *(ptm5++));
	       put_val(Sy,i,kx,dum);            /* Update Sy    */
	       put_val(Sy,i,ktmp,dum2);
	       *(ptm4++) = dum;                 /* Update Syx'  */
	       *(ptm6++) = dum2;                /* Update Syx2' */
	    }
         }
	 ptm3 = hSwt->mat[0];
	 kx   = nx-1;
         for(kw=0;kw<nw;kw++){
	 
	    /* Calculate swp */
            ptm1 = wtmp->mat[0]; ptm2=(opt->wmean)->mat[0]; 
	    ptm4 = wtmp2->mat[0];
            for(i=0;i<nw;i++){
	       *(ptm1++) = *ptm2 + *ptm3;
	       *(ptm4++) = *(ptm2++) - *(ptm3++);
	    }
            (*yfct)(swp,xbar,wtmp,0);
	    
	    /* Calculate swm */
            (*yfct)(swm,xbar,wtmp2,0);
	    
	    /* Update Syw, Syw2 and ybar */
	    ptm1 = swp->mat[0]; ptm2 = swm->mat[0]; ptm4 = Syw->mat[kw];
	    ptm5 = y02->mat[0]; ptm6 = Syw2->mat[kw]; ptm7 = ybar->mat[0];
	    kx++;
	    ktmp = kx+nx+nw;
            for(i=0;i<ny;i++){
	       dum = scal1*(*ptm1 - *ptm2);
	       dum2 = *(ptm1++) + *(ptm2++);
	       *(ptm7++) += dum2*h22;           /* Update ybar  */
	       dum2 = scal2*(dum2 - *(ptm5++));
	       put_val(Sy,i,kx,dum);            /* Update Sy    */
	       put_val(Sy,i,ktmp,dum2);
	       *(ptm4++) = dum;                 /* Update Syw'  */
	       *(ptm6++) = dum2;                /* Update Syw2' */
	    }
         }

	 /* Cholesky factor of a posteriori output estimate of covariance */
         ptm1 = K->mat[0];                 /* Calculate Sxbar*Syx'   */
	 for(i=0;i<nx;i++){                /* Used later for calc. K */
	    for(j=0;j<ny;j++){
	       ptm2 = Sx->mat[i]+i;
	       ptm3 = Sy->mat[j]+i;
	       for(l=i,dum=0.0;l<nx;l++)
	          dum+=*(ptm2++)* *(ptm3++);
	       *(ptm1++) = dum;
	    }
	 }
 
         /* Calculate Sy */
	 triag(Sy0,Sy,v);
	 for(i=0;i<ny;i++){
	    if(get_val(Sy0,i,i)==0)
                mexErrMsgTxt("Output covariance matrix singular.");
         }

	 /* Calculate Kalman gain */
	 fbsubst(K,Sy0);
	 	 
	 /* Update state estimate, xhat */
	 ptm1=K->mat[0];
	 for(i=0;i<nx;i++){
	    dum  = cvget(xbar,i);
	    ptm2 = y->mat[yidx];
	    for(j=0;j<ny;j++)
	       dum+=*(ptm1++) * (*(ptm2++)-cvget(ybar,j));
	    cvput(xhat,i,dum);
	 }

	 /* Cholesky factor of a'posteriori estimation error covariance */
	 makerect2(Sxlong,Sx,K,Syx,Syw,Syx2,Syw2);
	 triag(Sx,Sxlong,v);
         ++yidx; 
	 
	 /* Transpose Sx and multiply h */
	 for(i=0;i<nx;i++){
	    ptm1=hSxt->mat[i];
            for(j=0;j<=i;j++)
               *(ptm1++) = h*get_val(Sx,j,i);
         }
      }
      else{
         mset(xhat,xbar);
      }

      /* Store covariance estimate */
      ktmp = nx-1;
      for(i=0;i<nx;i++,ktmp--){
	 ptm1=Sx->mat[i]+i;
           for(j=0;j<=ktmp;j++)
           *(Smatptm++) = *(ptm1++);
      }

            
      /* --- Time update (a'priori update) of state and covariance --- */
      if(k<samples){

         /* Copy current u to u-vector */
         for(i=0;i<nu;i++)
	    cvput(uk1,i,get_val(u,k,i));

         /* --- "Nominal" a priori update --- */
         (*xfct)(fxbar2,xhat,uk1,opt->vmean,0);
	 
         /* Initialize 'xbar' and set fxbar2=2*fxbar2 */
	 ptm1 = fxbar2->mat[0];
	 ptm2 = xbar->mat[0];
         for(i=0;i<nx;i++,ptm1++){
	   *(ptm2++) = scal4* *ptm1;    /* xbar   = scal4*fxbar */
	   *ptm1 = 2.0* *ptm1;          /* fxbar2 = 2fxbar      */
	 }


         ptm3=hSxt->mat[0];
	 idum = nx+nv;
         for(kx=0;kx<nx;kx++){

	    /* Calculate sxp */
            ptm1 = xtmp->mat[0]; ptm2=xhat->mat[0]; ptm4=xtmp2->mat[0];
            for(i=0;i<nx;i++){
	       *(ptm1++) = *(ptm2) + *ptm3;
	       *(ptm4++) = *(ptm2++) - *(ptm3++);
	    }
            (*xfct)(sxp,xtmp,uk1,opt->vmean,0);
	    
	    /* Calculate sxm */
            (*xfct)(sxm,xtmp2,uk1,opt->vmean,0);
	    
	    /* Update Sxx and Sxx2 */
	    ptm1 = sxp->mat[0]; ptm2=sxm->mat[0];
	    ptm5 = fxbar2->mat[0]; ptm7 = xbar->mat[0];
            for(i=0;i<nx;i++){
	       dum = scal1*(*(ptm1) - *(ptm2));
	       dum2 = (*(ptm1++) + *(ptm2++));
	       *(ptm7++) += dum2*h22;           /* Update xbar  */
	       dum2 = scal2*(dum2 - *(ptm5++));
	       put_val(Sxbar,i,kx,dum);         /* Update Sxbar (Sxx)   */
	       put_val(Sxbar,i,idum,dum2);      /* Update Sxbar (Sxx2)  */
	    }
	    idum++;
         }

	 ptm3 = hSvt->mat[0];
	 kx   = nx-1;
	 idum = nx+kx+nv;
         for(kv=0;kv<nv;kv++){
	 
	    /* Calculate svp */
            ptm1 = vtmp->mat[0]; ptm2=(opt->vmean)->mat[0]; 
	    ptm4 = vtmp2->mat[0];
            for(i=0;i<nv;i++){
	       *(ptm1++) = *ptm2 + *ptm3;
	       *(ptm4++) = *(ptm2++) - *(ptm3++);
	    }
            (*xfct)(svp,xhat,uk1,vtmp,0);
	    
	    /* Calculate svm */
            (*xfct)(svm,xhat,uk1,vtmp2,0);
	    
	    /* Update Sxv and Sxv2 */
	    ptm1 = svp->mat[0]; ptm2=svm->mat[0];
	    ptm5 = fxbar2->mat[0]; ptm7 = xbar->mat[0];
	    kx++; idum++;
            for(i=0;i<nx;i++){
	       dum = scal1*(*(ptm1) - *(ptm2));
	       dum2 = (*(ptm1++) + *(ptm2++));
	       *(ptm7++) += dum2*h22;           /* Update xbar  */
	       dum2 = scal2*(dum2 - *(ptm5++));
	       put_val(Sxbar,i,kx,dum);         /* Update Sxbar (Sxv)  */
	       put_val(Sxbar,i,idum,dum2);      /* Update Sxbar (Sxv2) */
	    }
         }

	 /* Cholesky factor of a'priori estimation error covariance */
         /* Sxbar = triag([Sxx Sxv Sxx2 Sxv2]); */
	 triag(Sx,Sxbar,v);

	 /* Transpose Sx and multiply by h */
         for(i=0;i<nx;i++){
	    ptm1=hSxt->mat[i];
            for(j=0;j<=i;j++)
               *(ptm1++) = h*get_val(Sx,j,i);
         }
      }
      
      /* --- Store estimates --- */
      for(i=0;i<nx;i++)
         put_val(xhat_data,k,i,cvget(xhat,i));


      /* --- How much longer? --- */
      if ((counter+1)<= (int)(100*k/samples)){
        ++counter;
        printf(".");
      }
   }
   printf("\nFiltering completed\n");

   
   /* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> CLEAN UP <<<<<<<<<<<<<<<<<<<<<<<<<<< */
   (*xfct)(xbar,xhat,uk1,opt->vmean,-2);
   (*yfct)(ybar,xbar,opt->wmean,-2);
   mfree(K); mfree(xhat); mfree(ybar); mfree(Sy);
   mfree(Syx);mfree(Syw); mfree(uk1); mfree(syp); mfree(sym); mfree(swp);
   mfree(Syx2);mfree(Syw2); mfree(y0); mfree(y02); mfree(fxbar2);
   mfree(swm); mfree(sxp); mfree(sxm); mfree(svp); mfree(svm);
   mfree(xtmp); mfree(xtmp2); mfree(wtmp); mfree(wtmp2);
   mfree(vtmp); mfree(vtmp2); mfree(v); mfree(Sxbar); mfree(hSxt);
   mfree(Sxlong);mfree(hSvt);mfree(hSwt); mfree(Sx); mfree(Sy0);
   return 0;
}



/* DD2FILTM     This function implements the DD2-filter; a state estimator for
 *  nonlinear systems that is based on second-order polynomial approximations 
 *  of the nonlinear mappings. The approximations are derived by using a 
 *  multivariable extension of Stirling's interpolation formula. The
 *  function is implemented to handle multiple observation streams.
 *  The model of the nonlinear system must be specified in the form:
 *              x(k+1) = f[x(k),u(k),v(k)]
 *              y1(k)  = g1[x(k),w1(k)]
 *                       :
 *              yn(k)  = gn[x(k),wn(k)]
 *
 *  where 'x' is the state vector, 'u' is a possible input, and 'v' and 'w'
 *  are (white) noise sources.
 *
 * Call
 *     dd2filtm(xfunc,yfunc,xhat,Smat,x0,P0,q,u,obs,streams,optpar);
 *
 *
 * Outputs
 *  xfunc   - Name of function containing state equations.
 *  yfunc   - Name of function containing observation equations.
 *  xhat    - State estimates. Dimension is [samples+1-by-states].
 *  Smat    - Matrix for which each row contains elements of (the upper triangular
 *            part of) the Cholesky factor of the covariance matrix. The 
 *            dimension is [samples+1-by-0.5*states*(states+1)]. The individual
 *            covariance matrices can later be extracted with SMAT2COV.
 *
 * Inputs
 *  x0      - Initial state vector.
 *  P0      - Root of initial covariance matrix (symmetric, nonnegative 
 *            definite).
 *  q       - Covariance matrices for process noise.
 *  u       - Input signal. Dimension is [samples-by-inputs].
 *            Should be empty (have zero rows) if there are no inputs.
 *  obs     - An array of data structures with stream specific parameters. The
 *            following elements must exist in each structure upon call:
 *            .ny    - Stream width (outputs in the stream).
 *            .nobs  - Observations in the stream.
 *            .y     - Matrix containing observations in the stream [ny x nobs].
 *            .tidx  - Vecor containing the time stamps for the observations.
 *            .nw    - Number of observation noise sources in stream.
 *            .wmean - Mean of measurement noise vector.
 *            .Sw    - Square root of measurement noise covariance.
 *  optpar  - Data structure containing optional parameters:
 *            .vmean: Mean of process noise vector.
 *            .init : Initial parameters for 'xfile', 'yfile'
 *                    (use an arbitrary format).
 *
 * Written by: Magnus Norgaard, IAU/IMM, Technical University of Denmark
 * LastEditDate: Apr. 15, 2000
 */
int dd2filtm(int (*xfct)(matrix*, matrix*, matrix*, matrix*, int),
             int (*yfct)(matrix*, matrix*, matrix*, int),
             matrix *xhat_data, matrix *Smat, matrix *xbar, 
             matrix *P0,matrix *q, matrix *u, obsstruct *obs, int streams, 
	     optpar *opt)
{
   register int i, j, l, n;
   int samples, nu, ny, nx, nv, nw, kx, kw, kv, k;
   int trcode, lasttime, idum, ktmp, counter=0;
   double h, h2, h22, scal1, scal2, scal4, *Smatptm;
   double *ptm1, *ptm2, *ptm3, *ptm4, *ptm5, *ptm6, *ptm7, dum, dum2;
   matrix *xhat, *sxp, *sxm, *hSxt, *hSvt, *Sx;
   matrix *Sxbar, *uk1, *xtmp, *xtmp2, *v, *vtmp, *vtmp2, *fxbar2;
   intmatrix *idx1;
   
   h2    = 3.0;               /* Squared divided difference step */
   h     = sqrt(h2);          /* Divided difference step         */
   h22   = 1/(2.0*h2);        /* 1/2h2                           */
   scal1 = 0.5/h;             /* Scaling factor                  */
   scal2 = sqrt((h2-1)/(4*h2*h2));/* Another scaling factor      */
   if (getrows(u)==0){        /* No external input present       */
      nu = 0;
      uk1 = mmake(1,1);
   }
   else{
      nu      = getcols(u);
      uk1     = mmake(nu,1);
   }
   samples = getrows(xhat_data)-1; /* # of samples */
   nx  = getrows(P0);         /* # of states  */
   nv  = getrows(q);          /* # of process noise sources */
   scal4 = (h2-nx-nv)/h2;     /* And a fourth scaling factor    */
   Smatptm = Smat->mat[0];    /* Pointer to first element in Smat */


   /* Index vectors and samples */
   idx1 = intmmake(1,streams);
   ny = 0;
   nw = 0;
   for(n=0;n<streams;n++){
      ny +=obs[n].ny;            /* Total number of observations */
      nw +=obs[n].nw;            /* Total number of process noise sources */
      rvput(idx1,n,ny-obs[n].ny);/* Index to the start of each stream */
      obs[n].yidx = 0;           /* Index points to first observation */
      obs[n].lasttime = cvget(obs[n].tidx,obs[n].nobs-1);/* Time for last obs.*/
      obs[n].K   = mmake(nx,obs[n].ny);
      obs[n].Sy  = mmake(obs[n].ny,2*(nx+obs[n].nw));
      obs[n].Sy0 = mmake(obs[n].ny,obs[n].ny);
      minit(obs[n].Sy0);
      obs[n].Syx = mmake(nx,obs[n].ny);
      obs[n].Syw = mmake(obs[n].nw,obs[n].ny);
      obs[n].Syx2= mmake(nx,obs[n].ny);
      obs[n].Syw2= mmake(obs[n].nw,obs[n].ny);
      obs[n].hSwt = mmake(obs[n].nw,obs[n].nw);
      thmat(obs[n].hSwt,obs[n].Sw,h);        /* Transpose Sw and multiply h */
      obs[n].wtmp = mmake(obs[n].nw,1);
      obs[n].wtmp2 = mmake(obs[n].nw,1);
      obs[n].syp = mmake(obs[n].ny,1);
      obs[n].sym = mmake(obs[n].ny,1);
      obs[n].y0  = mmake(obs[n].ny,1);
      obs[n].y02 = mmake(obs[n].ny,1);
      obs[n].ybar= mmake(obs[n].ny,1);
      obs[n].Sxlong = mmake(nx,2*(nx+obs[n].nw));
   }

   /* Allocate space for vectors and matrices */
   xhat  = mmake(nx,1);
   fxbar2= mmake(nx,1);
   Sxbar = mmake(nx,2*(nx+nv));
   Sx    = mmake(nx,nx);
   hSxt  = mmake(nx,nx);hSvt  = mmake(nv,nv);
   sxm   = mmake(nx,1); sxp   = mmake(nx,1);
   xtmp  = mmake(nx,1); xtmp2 = mmake(nx,1);
   vtmp  = mmake(nv,1); vtmp2 = mmake(nv,1);
   v     = mmake(1,2*(nx+nw+ny+nv));
   
   /* Transpose covariance square roots and multiply by h */
   thmat(hSvt,q,h);
   thmat(hSxt,P0,h);

   /* Put P0 into Sx */   
   minit(Sx);
   triag(Sx,P0,v);
   thmat(hSxt,Sx,h);

   /* Initialize state update and observation functions */
   (*xfct)(opt->init,xhat,uk1,opt->vmean,-1);
   (*yfct)(opt->init,xbar,xbar,-1);
   
   printf("Filtering in progress\n");
   
   
   /* >>>>>>>>>>>>>>>>>>>>>>>>>>>> FILTERING <<<<<<<<<<<<<<<<<<<<<<<<<<< */
   for(k=0;k<=samples;k++){

      /* --- Measurement update (a posteriori update) --- */
      for(n=0;n<streams;n++){
	 (*yfct)(obs[n].y0,xbar,obs[n].wmean,n);
	 
	 if(k<=obs[n].lasttime && cvget(obs[n].tidx,obs[n].yidx)==k){
 
 	    /* Initialize 'ybar' and set y02=2*y0*/
	    ptm1 = obs[n].y0->mat[0];
	    ptm2 = obs[n].ybar->mat[0];
	    ptm3 = obs[n].y02->mat[0];
            for(i=0;i<obs[n].ny;i++){
	      *(ptm2++) = (h2-nx-obs[n].nw)/h2* *ptm1; /* ybar = scal3*y0 */
	      *(ptm3++) = 2.0* *(ptm1++);              /* y02 = 2y0       */
	    }

            ptm3=hSxt->mat[0];
            for(kx=0;kx<nx;kx++){

	       /* Calculate syp */
               ptm1 = xtmp->mat[0]; ptm2=xbar->mat[0]; ptm4=xtmp2->mat[0];
               for(i=0;i<nx;i++){
		  *(ptm1++) = *(ptm2) + *ptm3;
		  *(ptm4++) = *(ptm2++) - *(ptm3++);
	       }
               (*yfct)(obs[n].syp,xtmp,obs[n].wmean,n);

	       /* Calculate sym */
               (*yfct)(obs[n].sym,xtmp2,obs[n].wmean,n);

	       /* Update Syx, Syx2 and ybar */
	       ptm1 = obs[n].syp->mat[0];
	       ptm2 = obs[n].sym->mat[0];
	       ptm4 = obs[n].Syx->mat[kx];
	       ptm5 = obs[n].y02->mat[0];
	       ptm6 = obs[n].Syx2->mat[kx];
	       ptm7 = obs[n].ybar->mat[0];
	       ktmp = kx+nx+obs[n].nw;
               for(i=0;i<obs[n].ny;i++){
	          dum = scal1*(*ptm1 - *ptm2);
	          dum2 = *(ptm1++) + *(ptm2++);
	          *(ptm7++) += dum2*h22;           /* Update ybar  */
	          dum2 = scal2*(dum2 - *(ptm5++));
	          put_val(obs[n].Sy,i,kx,dum);     /* Update Sy    */
	          put_val(obs[n].Sy,i,ktmp,dum2);
	          *(ptm4++) = dum;                 /* Update Syx'  */
	          *(ptm6++) = dum2;                /* Update Syx2' */
	       }
            }


	    ptm3 = obs[n].hSwt->mat[0];
	    kx   = nx-1;
            for(kw=0;kw<obs[n].nw;kw++){

	       /* Calculate swp */
               ptm1 = obs[n].wtmp->mat[0];
	       ptm2 = obs[n].wmean->mat[0]; 
	       ptm4 = obs[n].wtmp2->mat[0];
               for(i=0;i<obs[n].nw;i++){
		  *(ptm1++) = *ptm2 + *ptm3;
		  *(ptm4++) = *(ptm2++) - *(ptm3++);
	       }
               (*yfct)(obs[n].syp,xbar,obs[n].wtmp,n);

	       /* Calculate swm */
               (*yfct)(obs[n].sym,xbar,obs[n].wtmp2,n);

	       /* Update Syw, Syw2 and ybar */
	       ptm1 = obs[n].syp->mat[0];
	       ptm2 = obs[n].sym->mat[0];
	       ptm4 = obs[n].Syw->mat[kw];
	       ptm5 = obs[n].y02->mat[0];
	       ptm6 = obs[n].Syw2->mat[kw];
	       ptm7 = obs[n].ybar->mat[0];
	       kx++;
	       ktmp = kx+nx+obs[n].nw;
               for(i=0;i<obs[n].ny;i++){
	          dum = scal1*(*ptm1 - *ptm2);
	          dum2 = *(ptm1++) + *(ptm2++);
	          *(ptm7++) += dum2*h22;           /* Update ybar  */
	          dum2 = scal2*(dum2 - *(ptm5++));
	          put_val(obs[n].Sy,i,kx,dum);     /* Update Sy    */
	          put_val(obs[n].Sy,i,ktmp,dum2);
	          *(ptm4++) = dum;                 /* Update Syw'  */
	          *(ptm6++) = dum2;                /* Update Syw2' */
	       }
            }

	    /* Cholesky factor of a posteriori output estimate of covariance */
            ptm1 = obs[n].K->mat[0];          /* Calculate Sxbar*Syx'   */
	    for(i=0;i<nx;i++){                /* Used later for calc. K */
	       for(j=0;j<obs[n].ny;j++){
		  ptm2 = Sx->mat[i]+i;
		  ptm3 = obs[n].Sy->mat[j]+i;
		  for(l=i,dum=0.0;l<nx;l++)
	             dum+=*(ptm2++)* *(ptm3++);
		  *(ptm1++) = dum;
	       }
	    }

            /* Calculate Sy */
	    triag(obs[n].Sy0,obs[n].Sy,v);
	    for(i=0;i<obs[n].ny;i++){
	       if(get_val(obs[n].Sy0,i,i)==0)
                   mexErrMsgTxt("Output covariance matrix singular.");
            }

	    /* Calculate Kalman gain */
	    fbsubst(obs[n].K,obs[n].Sy0);

	    /* Update state estimate, xhat */
	    ptm1=obs[n].K->mat[0];
	    for(i=0;i<nx;i++){
	       dum=cvget(xbar,i);
	       ptm2 = obs[n].y->mat[obs[n].yidx];
	       ptm3 = obs[n].ybar->mat[0];
	       for(j=0;j<obs[n].ny;j++)
		  dum+=*(ptm1++) * (*(ptm2++)-*(ptm3++));
	       cvput(xbar,i,dum);
	    }

	    /* Cholesky factor of a'posteriori estimation error covariance */
	    makerect2(obs[n].Sxlong,Sx,obs[n].K,obs[n].Syx,
	                  obs[n].Syw,obs[n].Syx2,obs[n].Syw2);
            triag(Sx,obs[n].Sxlong,v);
            ++(obs[n].yidx); 

	    /* Transpose Sx and multiply h */
	    for(i=0;i<nx;i++){
	       ptm1=hSxt->mat[i];
               for(j=0;j<=i;j++)
        	  *(ptm1++) = h*get_val(Sx,j,i);
            }	
	 }
      }
      mset(xhat,xbar);

      /* Store covariance estimate */
      ktmp = nx-1;
      for(i=0;i<nx;i++,ktmp--){
	 ptm1=Sx->mat[i]+i;
           for(j=0;j<=ktmp;j++)
           *(Smatptm++) = *(ptm1++);
      }


      
      /* --- Time update (a'priori update) of state and covariance --- */
      if(k<samples){

         /* Copy current u to u-vector */
         for(i=0;i<nu;i++)
	    cvput(uk1,i,get_val(u,k,i));

         /* --- "Nominal" a priori update --- */
         (*xfct)(fxbar2,xhat,uk1,opt->vmean,0);
	 
         /* Initialize 'xbar' and set fxbar2=2*fxbar2 */
	 ptm1 = fxbar2->mat[0];
	 ptm2 = xbar->mat[0];
         for(i=0;i<nx;i++,ptm1++){
	   *(ptm2++) = scal4* *ptm1;    /* xbar   = scal4*fxbar */
	   *ptm1 = 2.0* *ptm1;          /* fxbar2 = 2fxbar      */
	 }

         ptm3=hSxt->mat[0];
	 idum = nx+nv;
         for(kx=0;kx<nx;kx++){

	    /* Calculate sxp */
            ptm1 = xtmp->mat[0]; ptm2=xhat->mat[0]; ptm4=xtmp2->mat[0];
            for(i=0;i<nx;i++){
	       *(ptm1++) = *(ptm2) + *ptm3;
	       *(ptm4++) = *(ptm2++) - *(ptm3++);
	    }
            (*xfct)(sxp,xtmp,uk1,opt->vmean,0);
	    
	    /* Calculate sxm */
            (*xfct)(sxm,xtmp2,uk1,opt->vmean,0);
	    
	    /* Update Sxx and Sxx2 */
	    ptm1 = sxp->mat[0]; ptm2=sxm->mat[0];
	    ptm5 = fxbar2->mat[0]; ptm7 = xbar->mat[0];
            for(i=0;i<nx;i++){
	       dum = scal1*(*(ptm1) - *(ptm2));
	       dum2 = (*(ptm1++) + *(ptm2++));
	       *(ptm7++) += dum2*h22;           /* Update xbar  */
	       dum2 = scal2*(dum2 - *(ptm5++));
	       put_val(Sxbar,i,kx,dum);         /* Update Sxbar (Sxx)   */
	       put_val(Sxbar,i,idum,dum2);      /* Update Sxbar (Sxx2)  */
	    }
	    idum++;
         }

	 ptm3 = hSvt->mat[0];
	 kx   = nx-1;
	 idum = nx+kx+nv;
         for(kv=0;kv<nv;kv++){
	 
	    /* Calculate svp */
            ptm1 = vtmp->mat[0]; ptm2=(opt->vmean)->mat[0]; 
	    ptm4 = vtmp2->mat[0];
            for(i=0;i<nv;i++){
	       *(ptm1++) = *ptm2 + *ptm3;
	       *(ptm4++) = *(ptm2++) - *(ptm3++);
	    }
            (*xfct)(sxp,xhat,uk1,vtmp,0);
	    
	    /* Calculate svm */
            (*xfct)(sxm,xhat,uk1,vtmp2,0);
	    
	    /* Update Sxv and Sxv2 */
	    ptm1 = sxp->mat[0]; ptm2=sxm->mat[0];
	    ptm5 = fxbar2->mat[0]; ptm7 = xbar->mat[0];
	    kx++; idum++;
            for(i=0;i<nx;i++){
	       dum = scal1*(*(ptm1) - *(ptm2));
	       dum2 = (*(ptm1++) + *(ptm2++));
	       *(ptm7++) += dum2*h22;           /* Update xbar  */
	       dum2 = scal2*(dum2 - *(ptm5++));
	       put_val(Sxbar,i,kx,dum);         /* Update Sxbar (Sxv)  */
	       put_val(Sxbar,i,idum,dum2);      /* Update Sxbar (Sxv2) */
	    }
         }

	 /* Cholesky factor of a'priori estimation error covariance */
         /* Sxbar = triag([Sxx Sxv Sxx2 Sxv2]); */
         triag(Sx,Sxbar,v);

	 /* Transpose Sx and multiply h */
         for(i=0;i<nx;i++){
	    ptm1=hSxt->mat[i];
            for(j=0;j<=i;j++)
               *(ptm1++) = h*get_val(Sx,j,i);
         }
      }

      
      
      /* --- Store estimates --- */
      for(i=0;i<nx;i++)
         put_val(xhat_data,k,i,cvget(xhat,i));


      /* --- How much longer? --- */
      if ((counter+1)<= (int)(100*k/samples)){
        ++counter;
        printf(".");
      }
   }
   printf("\nFiltering completed\n");


   /* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> CLEAN UP <<<<<<<<<<<<<<<<<<<<<<<<<<< */
   (*xfct)(xbar,xhat,uk1,opt->vmean,-2);
   (*yfct)(obs[n].y0,xbar,xbar,-2);
   for(n=0;n<streams;n++){       /* Free matrices in obs-structure */
      mfree(obs[n].y); intmfree(obs[n].tidx); mfree(obs[n].wmean); 
      mfree(obs[n].Sw); mfree(obs[n].y0); mfree(obs[n].y02); 
      mfree(obs[n].ybar); mfree(obs[n].Sy0);
      mfree(obs[n].Sy); mfree(obs[n].Syx); mfree(obs[n].Syw); 
      mfree(obs[n].Syx2); mfree(obs[n].Syw2); mfree(obs[n].K); 
      mfree(obs[n].Sxlong); mfree(obs[n].hSwt); mfree(obs[n].wtmp); 
      mfree(obs[n].wtmp2); mfree(obs[n].syp); mfree(obs[n].sym);
   }
   mfree(xhat); mfree(sxp); mfree(sxm); mfree(Sx);
   mfree(xtmp); mfree(xtmp2); mfree(vtmp); mfree(vtmp2); mfree(v); 
   mfree(Sxbar); mfree(hSxt); mfree(uk1); mfree(hSvt); mfree(fxbar2);
   intmfree(idx1);
   return 0;
}



/********************************************************************************
 *                                                                              *
 *                      MISCELLANEOUS MATRIX FUNCTIONS                          *
 *                      ------------------------------                          *
 *                                                                              *
 ********************************************************************************/

/*
 * MMAKE :
 * -------
 * This function allocates memory for a matrix of the specified size
 * and assigns the specified dimensions to the allocated matrix.
 * Inputs  : rows - number of rows.
 *           cols - number of columns.
 * Outputs : *ptm - pointer to the new matrix structure.
 */
matrix *mmake( int rows, int cols )
{
	matrix *ptm;
	double **row_pointers;
	register int i;

#if RUNCHK

	/* Check that no dimension is zero. */
	if ((rows==0)||(cols==0)) merror("Invalid dimension error in mmake!");
#endif

	/* Memory allocation for matrix structure. */
	ptm = (matrix*)malloc(sizeof(matrix));

	/* Memory for array of pointers to rows. */;
	row_pointers = (double**)malloc(rows*sizeof(double*));

	/* Memory for all rows, initialize row pointers. */
	row_pointers[0] = (double*)malloc(rows*cols*sizeof(double));
	for (i=1;i<rows;i++) {
		row_pointers[i] = row_pointers[i-1] + cols;
	}

#if RUNCHK

	/* Check if last allocation was ok ! */
	if (!row_pointers[0]) {
		merror("Memory allocation error in mmmake!");
	}
#endif

	ptm->row = rows;             /* Initialize matrix structure */
	ptm->col = cols;
	ptm->mat = row_pointers;     /* Pointer to row pointers     */

	return ptm;           /* Return pointer to matrix structure */
}


/*
 * MFREE :
 * -------
 * This function deallocates the memory that was used for a matrix.
 * Input : Pointer to the matrix to be deallocated.
 */
void mfree( matrix *ptm )
{
	/* Deallocate rows */
	free(ptm->mat[0]);

	/* Deallocate row pointer array */
	free(ptm->mat);

	/* Deallocate matrix structure */
	free(ptm);
}


/*
 * INTMMAKE :
 * ----------
 * This function allocates memory for a matrix of the specified size
 * and assigns the specified dimensions to the allocated matrix.
 * Inputs  : rows - number of rows.
 *           cols - number of columns.
 * Outputs : *ptm - pointer to the new matrix structure.
 */
intmatrix *intmmake( int rows, int cols )
{
	intmatrix *ptm;
	int **row_pointers;
	register int i;

#if RUNCHK

	/* Check that no dimension is zero. */
	if ((rows==0)||(cols==0)) merror("Invalid dimension error in intmmake!");
#endif

	/* Memory allocation for matrix structure. */
	ptm = (intmatrix*)malloc(sizeof(intmatrix));

	/* Memory for array of pointers to rows. */;
	row_pointers = (int**)malloc(rows*sizeof(int*));

	/* Memory for all rows, initialize row pointers. */
	row_pointers[0] = (int*)malloc(rows*cols*sizeof(int));
	for (i=1;i<rows;i++) {
		row_pointers[i] = row_pointers[i-1] + cols;
	}

#if RUNCHK

	/* Check if last allocation was ok ! */
	if (!row_pointers[0]) {
		merror("Memory allocation error in intmmake!");
	}
#endif

	ptm->row = rows;             /* Initialize matrix structure */
	ptm->col = cols;
	ptm->mat = row_pointers;     /* Pointer to row pointers     */

	return ptm;           /* Return pointer to matrix structure */
}



/*
 * INTMFREE :
 * ----------
 * This function deallocates the memory that was used for a matrix.
 * Input : Pointer to the matrix to be deallocated.
 */
void intmfree( intmatrix *ptm )
{
	/* Deallocate rows */
	free(ptm->mat[0]);

	/* Deallocate row pointer array */
	free(ptm->mat);

	/* Deallocate matrix structure */
	free(ptm);
}



/*
 * GETROWS
 * -------
 *
 * This function gets the number of rows in a matrix
 *
 * Input:   *ptm - Pointer to matrix
 *
 * Output:  rows - Number of rows in matrix
 *
 */
int getrows(matrix *ptm)
{
	return ptm->row;
}



/*
 * GETCOLS
 * -------
 *
 * This function gets the number of columns in a matrix
 *
 * Input:   *ptm - Pointer to matrix
 *
 * Output:  columns - Number of columns in matrix
 *
 */
int getcols(matrix *ptm)
{
	return ptm->col;
}


/*
 * MERROR :
 * --------
 * This function displays the specified error message and exits to server
 * operating system.
 * Input : Text string with the error message.
 */
void merror( char error_text[] )
{

#ifndef NO_IO     /* For compilation by the DSP ANSI-C compiler. */

	printf("--- Runtime error in matrix library:\n");
	printf("--- ");
	printf("%s\n\n",error_text);
#else
	error_text=error_text; /* Dummy command to avoid warnings on DSP */ 
#endif
	/*exit(1);*/
	mexErrMsgTxt("AN ERROR OCCURED IN A CMEX-PROGRAM");
}


/*
 * MPRINT :
 * --------
 * This is a simple function to print a matrix on the screen.
 * Input : ptm - Pointer to the matrix that will be printed.
 */
void mprint( matrix *ptm )
{
#ifndef NO_IO     /* For compilation by the DSP ANSI-C compiler. */

	int i,j;
	for ( i=0; i < ptm->row; i++ )
	{
		for( j=0; j < ptm->col; j++ )
		{
			printf("%7e %s",ptm->mat[i][j]," ");
			/* printf("%5f %s",ptm->mat[i][j]," "); */
		}
		printf("\n");
	}
#else
	ptm=ptm;
	exit(0); /* printf is not available in DSP compiler. */
#endif
}



/*
 * MINIT :
 * -------
 * This function initializes a matrix to all zeros.
 * Inputs : ptm - Pointer to the matrix that will be initialized.
 */
void minit( matrix *ptm )
{
	register unsigned char zero_byte = 0;
	register int bytes;
	
	/* Number of bytes in matrix data area. */
	bytes = 8*(ptm->row)*(ptm->col);

	/* Set data area to all zeroes. */
	memset( ptm->mat[0], zero_byte, bytes );
}



/*
 * MADD
 * ----
 *
 * Matrix addition
 *
 * Input:  *ptm1 - Pointer to result matrix
 *                 (can be equal to *ptm1 and/or *ptm2)
 *         *ptm2 - Pointer to first argument matrix
 *         *ptm3 - Pointer to second argument matrix
 */
void madd( matrix *ptm1, matrix *ptm2, matrix *ptm3 )
{
	register int i,j;

#if RUNCHK

	if (!((ptm1->row == ptm2->row) && (ptm2->row == ptm3->row))) \
	merror("Dimension mismatch error in madd!");
	if (!((ptm1->col == ptm2->col) && (ptm2->col == ptm3->col))) \
	merror("Dimension mismatch error in madd!");

#endif

	/* Add the two matrices element by element */
	for ( i=0; i < ptm2->row; i++ )
	{
		for ( j=0; j < ptm2->col; j++ )
		{
			ptm1->mat[i][j] = ptm2->mat[i][j] + ptm3->mat[i][j];
		}
	}
}



/*
 * SUBVEC
 * ------
 *
 * Inputs:        *ptm1   - Pointer to the result vector
 *                *ptm2   - Pointer to input vector
 *                elem1   - Number of the first element
 *                elem2   - Number of the last element
 */
void subvec( matrix *ptm1, matrix *ptm2, int elem1, int elem2 )
{
	register int i,j;

#if RUNCHK

	/* Check whether elem2 is inside bound */
	if ( elem2 > ptm2->row + ptm2->col - 2 ) \
	merror("Argument out of range in function subvec!");

	/* Check whether sub-vector fits output-vector. */
	if ((ptm1->row + ptm1->col) != ( elem2 - elem1 + 2 )) \
	merror("Wrong number of elements error in subvec!");


#endif

	if ( ptm2->col == 1 )                                 /* Column vector */
	{
		for ( i=0,j=elem1; j<=elem2; i++,j++ ) \
		ptm1->mat[i][0] = ptm2->mat[j][0];
	}

	else if ( ptm2->row == 1 )                            /* Row vector    */
	{
		for ( i=0,j=elem1; j<=elem2; i++,j++ ) \
		ptm1->mat[0][i] = ptm2->mat[0][j];
	}
	else merror("Dimension mismatch in function subvec"); /* Not a vector  */
}


/*
 * MSET :
 * ------
 * This function copies the content of a matrix to another matrix.
 * ( A := B ) == mset(a,b); Assignment of matrix.
 * Inputs : ptm1, ptm2 - Pointers to matrices.
 */
void mset( matrix *ptm1, matrix *ptm2 )
{
	register int bytes;

#if RUNCHK

	if ( !( (ptm1->row == ptm2->row) && (ptm1->col == ptm2->col) ) ) \
	merror("Dimension mismatch error in mset!");

#endif

	bytes = 8*(ptm1->row)*(ptm1->col);
	memcpy( ptm1->mat[0], ptm2->mat[0], bytes );
}




/*
 * MAT2SM
 * ------
 *
 * Conversion of a (real) Matlab matrix into the SteffenMagnus format
 *
 * INPUT : MatlabMatrix - Matrix in Matlab form
 * OUTPUT: SMmatrix     - Matrix in SM form
 *
 */
matrix *mat2sm(const mxArray *MatlabMatrix)
{
  int rows, cols, i, j;
  double *M;
  matrix *SMmatrix;

  rows = mxGetM(MatlabMatrix);
  cols = mxGetN(MatlabMatrix);
  M    = mxGetPr(MatlabMatrix);

  SMmatrix = mmake(rows,cols);
  for(i=0;i<cols;i++)
  {
    for(j=0;j<rows;j++)
    {
      put_val(SMmatrix,j,i,M[j+i*rows]);
    }
  }
  return SMmatrix;
}


/*
 * SM2MAT
 * ------
 *
 * Conversion of a SteffenMagnus matrix to a (real) Matlab matrix format
 *
 * INPUTS: MatlabMatrix - Pointer to Matlab matrix 
 *         SMmatrix     - Pointer to SM matrix
 *
 */
void sm2mat(mxArray *MatlabMatrix, matrix *SMmatrix)
{
  int rows, cols, i, j;
  double *M;
  rows = mxGetM(MatlabMatrix);
  cols = mxGetN(MatlabMatrix);
  M    = mxGetPr(MatlabMatrix);

  for(i=0;i<cols;i++)
  {
    for(j=0;j<rows;j++)
    {
      M[j+i*rows]=get_val(SMmatrix,j,i);
    }
  }
  mfree(SMmatrix);
}


/*
 * MAT2INTSM
 * ---------
 *
 * Conversion of a (real) Matlab matrix into the SteffenMagnus integer format
 *
 * INPUT : MatlabMatrix - Matrix in Matlab form
 * OUTPUT: SMmatrix     - Matrix in SM integer form
 *
 */
intmatrix *mat2intsm(const mxArray *MatlabMatrix)
{
  int rows, cols, i, j;
  double *M;
  intmatrix *SMmatrix;

  rows = mxGetM(MatlabMatrix);
  cols = mxGetN(MatlabMatrix);
  M    = mxGetPr(MatlabMatrix);

  SMmatrix = intmmake(rows,cols);
  for(i=0;i<cols;i++){
    for(j=0;j<rows;j++){
      SMmatrix->mat[j][i]=(int)M[j+i*rows];
    }
  }
  return SMmatrix;
}


/*
 * INTSM2MAT
 * ---------
 *
 * Conversion of a SteffenMagnus matrix in integer form to
 * a (real) Matlab matrix format
 *
 * INPUTS: MatlabMatrix - Pointer to Matlab matrix 
 *         SMmatrix     - Pointer to integer SM matrix
 */
void intsm2mat(mxArray *MatlabMatrix, intmatrix *SMmatrix)
{
  int rows, cols, i, j;
  double *M;
  rows = mxGetM(MatlabMatrix);
  cols = mxGetN(MatlabMatrix);
  M    = mxGetPr(MatlabMatrix);

  for(i=0;i<cols;i++){
    for(j=0;j<rows;j++)
      M[j+i*rows]=SMmatrix->mat[j][i];
  }
  intmfree(SMmatrix);
}


/*
 * TRIAG
 * -----
 * Use a Householder transformation on a rectangular matrix A to
 * produce a square and upper triangular matrix S with the property
 *     S*S' = A*A'
 *
 * Call
 *     triag(S,A,v)
 *
 * A should have at least as many columns as rows.
 * v is a vector used for intermediate results. Its dimension should
 * correspond to (at least) the number of columns+rows in A.
 * 
 * NB! A will be modified and should not be used afterwards!
 *
 * Literature: Grewal & Andrews: "Kalman Filtering - Theory and Practice"
 *    Prentice-Hall, 1993, pp. 234. However, there 
 *    are certain cases in which their implementation fails. These have been
 *    handled as described in G.W. Stewart: "Matrix Algorithms, 
 *    Vol. 1: Basic Decompositions", SIAM 1988.
 */
void triag(matrix *S, matrix *A, matrix *v)
{
   register int i, j, k;
   int r, rk, n;
   double ny, d, *ptm1, *ptm2, *ptm3;
   n = nof_rows(A);
   r = nof_cols(A)-n;

   for(k=n-1;k>=0;k--){
      rk = r+k; ny = 0.0;
      ptm1 = A->mat[k];
      ptm2 = v->mat[0];
      for(j=0;j<=rk;j++){
         d = *(ptm1++);
	 *(ptm2++) = d;
	 ny += d*d;
      }  

      ny = sqrt(ny);

/*      if(ny==0.0)*/
      if (ny<1e-15)          /* <1e-15 is used instead of 0 */
         *(--ptm2)=sqrt(2);
      else{
         d = 1/ny;
	 ptm2 = v->mat[0];
	 for(j=0;j<=rk;j++)
            *(ptm2++)*=d;
	 if(*(--ptm2)>=0.0){
	    *ptm2+=1.0;
	    ny = -ny;
	 }
	 else
	    *ptm2-=1.0;
	 d = 1.0/sqrt(fabs(*ptm2));

	 ptm2 = v->mat[0];
	 for(j=0;j<=rk;j++)
            *(ptm2++)*=d;
      }
      
      put_val(S,k,k,ny);
      for(i=0;i<=k-1;i++){
         ptm1 = A->mat[i];
	 ptm2 = v->mat[0];
         for(j=0, d=0.0;j<=rk;j++)
	    d += *(ptm1++) * *(ptm2++);
	 ptm1 = A->mat[i];
	 ptm2 = v->mat[0];
	 for(j=0;j<=rk;j++)
	    *(ptm1++) -= d* *(ptm2++);
	 put_val(S,i,k,*(--ptm1));
      }
   }
}


/*
 * TRIAGOLD
 * --------
 * Use a Householder transformation on a rectangular matrix A to
 * produce a square and upper triangular matrix S such that
 *     S*S' = A*A'
 * Call
 *     code = triag(A,v)
 *
 * S is returned as the upper triangular part to the right in A.
 * A should have more columns than rows.
 *
 * v is a vector used for intermediate results. Its dimension should
 * correspond to (at least) the number of columns in A.
 *
 * code = 0: Operation performed.
 * code = 1: Operation terminated due to "divide by zero".
 * 
 * Literature: Grewal & Andrews: "Kalman Filtering - Theory and Practice"
 *             Prentice-Hall, 1993, pp. 234.
 */
int triagold(matrix *S, matrix *v)
{
   register int i, j;
   int k, r, rk, n;
   double sigma, a, d, *ptm1, *ptm2;
   n = nof_rows(S);
   r = nof_cols(S)-n;
   
   for(k=n-1;k>=0;k--){
      rk = r+k; sigma = 0.0;
      ptm1 = S->mat[k];
      for(j=0;j<=rk;j++){
         d = *(ptm1++);
	 sigma += d*d;
      }
      a = sqrt(sigma);
   
      ptm2 = v->mat[0];
      ptm1 = S->mat[k];
      sigma = 0.0;
      for(j=0;j<rk;j++){
         d = *(ptm1++);
         *(ptm2++) = d;
	 sigma += d*d;
      }
      d = *ptm1-a;
      *ptm2 = d;
      sigma += d*d;
      if(0.5*sigma==0.0) return 1;
      a  = 2.0/sigma;           /* Possible division by 0 */
      
      for(i=0;i<=k;i++){
         sigma = 0.0;
	 ptm2 = v->mat[0];
         ptm1 = S->mat[i];
	 for(j=0;j<=rk;j++)
	    sigma += *(ptm1++) * *(ptm2++);
	 d = a*sigma;
	 ptm2 = v->mat[0];
         ptm1 = S->mat[i];
	 for(j=0;j<=rk;j++,ptm1++)      /*!!!!!!!!!!!!!!!!!!!!*/
	    *(ptm1) -= d* *(ptm2++);
      }
   }
   return 0;
}



/*
 * FBSUBST
 * -------
 * Use forward and backsubstitution to solve K(S*S')=A
 * S is upper triangular. K is returned in A.
 *
 */
int fbsubst(matrix *K, matrix *S)
{
   register int i,j,l;
   int nx, ny, ny1;
   double dum, *ptm1, *ptm2, *ptm3;
   
   nx = nof_rows(K); ny = nof_cols(K); ny1 = ny-1;

   /*--- First substitution pass ---*/
   for(i=0; i<nx; i++){
      ptm1 = K->mat[i]+ny1;
      for(j=ny1;j>=0; j--){
         ptm3 = S->mat[j]+j+1;
	 ptm2 = ptm1+1;
	 dum = *ptm1;
         for(l=j+1; l<=ny1; l++)
	    dum -= *(ptm2++) * *(ptm3++);
         *(ptm1--)=dum/(S->mat[j][j]);
      }
   }

   /*--- Second substitution pass ---*/
   for(i=0; i<nx; i++){
      for(j=0;j<ny; j++){
	 ptm2 = K->mat[i];
	 dum  =  get_val(K,i,j);
         for(l=0; l<j; l++)
	    dum -= *(ptm2++)*get_val(S,l,j);
	 put_val(K,i,j,dum/get_val(S,j,j));
      }
   }
   return 0;
}


/* MAKERECT
 * --------
 *
 * Make the compund matrix [Sxbar-K*Syx K*Syw]
 */
int makerect(matrix *Sxlong,matrix *Sxbar,matrix *K,matrix *Syx,matrix *Syw)
{
   register int i, j, k;
   int nx, ny, nw;
   double dum1, *ptm1, *ptm2, *ptm3, *ptm4, *ptm5, *ptm6;
   
   nx = nof_rows(Sxbar); ny = nof_cols(K);
   nw = nof_rows(Syw);
   for(i=0;i<nx;i++){
      ptm1 = Sxlong->mat[i];
      ptm2 = Sxlong->mat[i]+nx;
      ptm3 = Sxbar->mat[i];
      ptm5 = Syx->mat[0];
      ptm6 = Syw->mat[0];
      for(j=0;j<nx;j++){
         ptm4 = K->mat[i];
	 for(k=0, dum1=*(ptm3++);k<ny;k++){
	    dum1-=*(ptm4++) * *(ptm5++);
	 }
	 *(ptm1++) = dum1;
      }
      for(j=0;j<nw;j++){
         ptm4 = K->mat[i];
	 for(k=0, dum1=0.0;k<ny;k++){
	    dum1 +=*(ptm4++) * *(ptm6++);
	 }
	 *(ptm2++) = dum1;
      }
   }
}


/* MAKERECT2
 * ---------
 *
 * Make the compund matrix [Sxbar-K*Syx K*Syw K*Syx2 K*Syw2]
 */
int makerect2(matrix *Sxlong,matrix *Sxbar,matrix *K,matrix *Syx,
              matrix *Syw, matrix *Syx2, matrix *Syw2)
{
   register int i, j, k;
   int nx, ny, nw, nv;
   double dum1, *ptm1, *ptm2, *ptm3, *ptm4, *ptm5, *ptm6;
   double dum2, *ptm7, *ptm8, *ptm9, *ptm10;
   
   nx = nof_rows(Sxbar); ny = nof_cols(K);
   nw = nof_rows(Syw);
   for(i=0;i<nx;i++){
      ptm1 = Sxlong->mat[i];
      ptm2 = Sxlong->mat[i]+nx;
      ptm3 = Sxbar->mat[i];
      ptm5 = Syx->mat[0];
      ptm6 = Syw->mat[0];
      
      ptm7 = Sxlong->mat[i]+nx+nw;
      ptm8 = Sxlong->mat[i]+nx+nx+nw;
      ptm9 = Syx2->mat[0];
      ptm10= Syw2->mat[0];
      for(j=0;j<nx;j++){
         ptm4 = K->mat[i];
	 for(k=0, dum1=*(ptm3++),dum2 = 0.0;k<ny;k++){
	    dum1-=*(ptm4) * *(ptm5++);
	    dum2+=*(ptm4++) * *(ptm9++);
	 }
	 *(ptm1++) = dum1;
	 *(ptm7++) = dum2;
      }
      for(j=0;j<nw;j++){
         ptm4 = K->mat[i];
	 for(k=0, dum1=0.0, dum2=0.0;k<ny;k++){
	    dum1 +=*(ptm4) * *(ptm6++);
	    dum2 +=*(ptm4++) * *(ptm10++);
	 }
	 *(ptm2++) = dum1;
	 *(ptm8++) = dum2;
      }
   }
}


/* THMAT
 * -----
 * Transpose quadratic matrix and multiply h to all diagonal elements.
 *
 */
int thmat(matrix *A, matrix *B,double h)
{
   int nx, i, j;
   double *ptm1;
   nx = nof_cols(A);
   ptm1=A->mat[0];
   for(i=0;i<nx;i++){
      for(j=0;j<=i;j++)
         *(ptm1++) = h*get_val(B,j,i);
      for(j=i+1; j<nx;j++)
         *(ptm1++) = 0.0;
   }
}
