/* KALMFILE is the name of the file where XFUNC and YFUNC can be
 *          found. Remember to use " " around the name.
 * XFUNC is the name of the function implementing the state equation.
 * YFUNC is the name of the function implementing the observation equation.
 */
#define KALMFILE "agvfctm.c"
#define XFUNC agvtu
#define YFUNC agvobs


/*
 *     INCLUDE HEADERS
 */
#include <stdio.h>
#include <math.h>
#include <time.h>
#include "mex.h"


/*
 *     DEFINES ASSOCIATED WITH MATRIX MANIPULATION 
 */
#define ON 1
#define OFF 0
#define RUNCHK ON             /* Run-time checks switched on/off. */


/* "Inline" functions.                                                 */
/* ( No run-time checks is performed, when inline functions are used ) */
#define nof_rows(ptm)                      (ptm->row)                           /* See getrows */
#define nof_cols(ptm)                      (ptm->col)                           /* See getcols */
#define vec_len(ptv)                       (ptv->row+ptv->col-1)                /* See length  */
#define get_val(ptm,row_pos,col_pos)       (ptm->mat[row_pos][col_pos])         /* See mget    */
#define put_val(ptm,row_pos,col_pos,value) ((ptm->mat[row_pos][col_pos])=value) /* See mput    */
#define rvget(ptv,element)                 (ptv->mat[0][element])               /* See vget    */
#define cvget(ptv,element)                 (ptv->mat[element][0])               /* See vget    */
#define rvput(ptv,element,value)           ((ptv->mat[0][element])=value)       /* See vput    */
#define cvput(ptv,element,value)           ((ptv->mat[element][0])=value)       /* See vput    */


/* Declaration of the "abstract" data-type. */
typedef struct {               /* Matrix structure for C library  */
	int row;               /* These are meant to be "private" */
	int col;               /* and should only be accessed via */
	double **mat;          /* the "member functions" below.   */
} matrix;


typedef struct {               /* Matrix structure for C library  */
	int row;               /* These are meant to be "private" */
	int col;               /* and should only be accessed via */
	int **mat;             /* the "member functions" below.   */
} intmatrix;

typedef struct {               /* Optional initializations        */
	matrix *wmean;         /* Mean of measurement noise       */
	matrix *vmean;         /* Mean of process noise           */
	matrix *init;          /* Initialization parameters       */
} optpar;


typedef struct {               /* Observation data structure       */
        int    ny;             /* Dimension of observation vector  */
	int    nobs;           /* Number of observations in stream */
	int    nw;             /* Dimension of obs. noise vector   */
	matrix *y;             /* Mean of process noise            */
        intmatrix *tidx;       /* Time stamps for obs. in .y       */
	matrix *wmean;         /* Mean of measurement noise        */
	matrix *Sw;            /* Root of noise covariance matrix  */
	matrix *y0;            /* Storage of current outp. estimate*/
        matrix *y02;           /* Storage of 2 * output estimate   */
	matrix *ybar;          /* Storage of scaled output estimate*/
	matrix *Sy;            /* Root of outp. covariance matrix  */
	matrix *Sy0;           /* Rectangular root of outp. cov mat*/
	matrix *Syx;           /* Root of cross-covariance matrix  */
	matrix *Syw;           /* Root of cross-covariance matrix  */
	matrix *Syx2;          /* Root of cross-covariance matrix  */
	matrix *Syw2;          /* Root of cross-covariance matrix  */
	matrix *K;             /* Kalman gain                      */
	matrix *Sxlong;        /* Long root of covariance matrix   */
	matrix *hSwt;          /* Sw multiplied by h and transposed*/
	int    yidx;           /* Index to next observation        */
	int    lasttime;       /* Sample no. for last observation  */
	matrix *wtmp;          /* temp vector                      */
	matrix *wtmp2;         /* temp vector                      */
	matrix *syp;           /* temp vector                      */
	matrix *sym;           /* temp vector                      */
} obsstruct;


/* Declaration of the "member functions".   */
matrix *mmake( int, int );
void mfree( matrix* );
void mprint( matrix* );
void merror( char* );
int getrows( matrix* );
int getcols( matrix* );
void minit( matrix* );
void madd( matrix*, matrix*, matrix* );
void mset( matrix*, matrix*);
intmatrix *intmmake( int, int );
void intmfree( intmatrix* );


/*
 *     PROTOTYPE DECLARATION
 */
matrix* mat2sm(const mxArray*);
void sm2mat(mxArray*, matrix*);
intmatrix* mat2intsm(const mxArray*);
void intsm2mat(mxArray*, intmatrix*);
int dd2filtm(int (*xfct)(matrix*, matrix*, matrix*, matrix*, int),
            int (*yfct)(matrix*, matrix*, matrix*, int),
	    matrix*, matrix*, matrix*, matrix*, matrix*, matrix*, 
            obsstruct*, int, optpar*);

#include KALMFILE

/*********************************************************************************
 *                                                                               *
 *    dd2mc gateway routine                                                      *
 *    ---------------------                                                      *
 *                                                                               *
 *    This is a C-version of the Matlab function 'dd2m'.                         *
 *    Type 'help dd2m' from Matlab for information on                            *
 *    how to call the function.                                                  *
 *                                                                               *
 *                                                                               *
 *    Programmed by: Magnus Norgaard                                             *
 *    LastEditDate : Jan. 12, 1999                                               *
 *                                                                               *
 *********************************************************************************/


/*********************************************************************************
 *                                                                               *
 *                           G A T E W A Y   R O U T I N E                       *
 *                                                                               *
 *********************************************************************************/
void mexFunction(int nlhs, mxArray *plhs[], int nrhs, const mxArray *prhs[])
{
  /*
   >>>>>>>>>>>>>>>>>>           VARIABLE DECLARATIONS          <<<<<<<<<<<<<<<<<<<
   */
   matrix *xhat_data, *Smat, *xbar, *Sxbar, *Sv, *u;
   optpar *opt;
   int a, samples=0, nx, ny=0, *ptmi, streams, i;
   mxArray  *Matmatrix, *M;
   obsstruct *obs;
   const char *fieldnames[] = {"wmean", "vmean", "init"};

  /*
   >>>>>>>>>>>>>>>>      CHECK FOR PROPER NUMBER OF ARGUMENTS      <<<<<<<<<<<<<<<
   */
   if (nrhs<7 || nrhs>8)
      mexErrMsgTxt("Wrong number of input arguments");
   else if (nlhs > 2)
       mexErrMsgTxt("Too many output arguments");
 
  /*
   >>>>>>>>>>>>>>>>>     CONVERT INPUT ARGUMENTS TO SM FORMAT     <<<<<<<<<<<<<<<<
   */
  /* Convert Sxbar */
  a = 1;

  nx = mxGetN(prhs[a]);
  if (nx==0 || nx!=mxGetM(prhs[a]))
     mexErrMsgTxt("Dimension mismatch in P0");
  else
     Sxbar = mat2sm(prhs[a]);
     
  /* Convert xbar and initialize if necessary */
  a = 0;
  if (mxGetN(prhs[a])==0 || mxGetM(prhs[a])==0){
     xbar = mmake(nx,1);
     minit(xbar);
  }
  else
     xbar = mat2sm(prhs[a]);
    
  /* Convert Sv */
  a = 2;
  if (mxGetN(prhs[a])!=mxGetM(prhs[a]))
     mexErrMsgTxt("Dimension mismatch in Q");
  else
     Sv = mat2sm(prhs[a]);


  /* Convert y */
  a  = 5;
  if(!mxIsCell(prhs[a]))
     mexErrMsgTxt("Argument 'y' must be a cell array");
  if(mxGetM(prhs[a])!=1 || mxGetM(prhs[a])>mxGetN(prhs[a]))
     mexErrMsgTxt("Argument 'y' must be a 'row' vector of cells");
  streams = mxGetN(prhs[a]);          /* Observation streams */

  /* Allocate memory for array of 'obs' structures */
  obs = (obsstruct*) malloc(streams*sizeof(obsstruct));
  
  /* Extract each 'y' matrix from cell array */
  for(i=0;i<streams;i++){
     M = mxGetCell(prhs[a],i);
     obs[i].ny = mxGetN(M);
     obs[i].nobs = mxGetM(M);
     if (obs[i].ny==0 || obs[i].nobs==0)
     mexErrMsgTxt("Observation matrix is empty");
     obs[i].y = mat2sm(M);
     ny += obs[i].ny;
  }


  /* Convert timeidx */
  a = 6;
  if(!mxIsCell(prhs[a]))
     mexErrMsgTxt("Argument 'timeidx' must be a cell array");
  if(mxGetM(prhs[a])!=1 || mxGetN(prhs[a])!=streams)
     mexErrMsgTxt("'timeidx' must have the same dimension as 'y'");

  /* Extract each 'timeidx' matrix from cell array */
  for(i=0;i<streams;i++){
     M = mxGetCell(prhs[a],i);
     if (mxGetN(M)==0 || mxGetM(M)==0)
        mexErrMsgTxt("No time stamps - 'timeidx' is empty");
     if (mxGetM(M)!=obs[i].nobs)
        mexErrMsgTxt("Dimension mismatch between dimension of cells in 'y' and 'timeidx'");
     obs[i].tidx = mat2intsm(M);
     if(samples<cvget(obs[i].tidx,obs[i].nobs-1))
         samples=cvget(obs[i].tidx,obs[i].nobs-1);
  }
  
  /* Convert Sw */
  a = 3;
  if(!mxIsCell(prhs[a]))
     mexErrMsgTxt("Argument 'Sw' must be a cell array");
  if(mxGetM(prhs[a])!=1 || mxGetN(prhs[a])!=streams)
     mexErrMsgTxt("'Sw' must have the same dimension as 'y'");

  /* Extract each 'Sw' matrix from cell array */
  for(i=0;i<streams;i++){
     M = mxGetCell(prhs[a],i);
     if (mxGetN(M)!=mxGetM(M))
        mexErrMsgTxt("Cell in 'Sw' not quadratic");
     obs[i].Sw = mat2sm(M);
     obs[i].nw = mxGetN(M);
  }
  

  /* Convert u */
  a = 4;
  if (mxGetN(prhs[a])==0 || mxGetM(prhs[a])==0){
     u = mmake(1,1);
     u->row = 0;
  }
  else{
     u = mat2sm(prhs[a]);
     samples = mxGetM(prhs[a]);
  }


  /* Convert optpar */
  opt = (optpar*) malloc(sizeof(optpar)); /* Allocate mem for par structure */
  if (nrhs==8){
    a  = 7;
    Matmatrix = mxGetField(prhs[a], 0, "vmean");
    if(Matmatrix==NULL){
       opt->vmean = mmake(getrows(Sv),1);
       minit(opt->vmean);
    }
    else
       opt->vmean = mat2sm(Matmatrix);
     
    Matmatrix = mxGetField(prhs[a], 0, "init");
    if(Matmatrix==NULL){
       opt->init = mmake(1,1);
       minit(opt->init);
    }
    else
       opt->init = mat2sm(Matmatrix);
       
    Matmatrix = mxGetField(prhs[a], 0, "wmean");
    if(Matmatrix==NULL){
       for(i=0;i<streams;i++){
          obs[i].wmean = mmake(getrows(obs[i].Sw),1);
          minit(obs[i].wmean);
       }
    }
    else{
       if(!mxIsCell(Matmatrix))
            mexErrMsgTxt("Argument 'opt.wmean' must be a cell array");
       for(i=0;i<streams;i++){
          if(mxGetM(Matmatrix)!=0 && mxGetM(Matmatrix)!=0){
	     M = mxGetCell(Matmatrix,i);
             if(mxGetM(M)!=obs[i].nw)
	        mexErrMsgTxt("Dimension mismatch between cells in 'Sw' and 'wmean'");
             else
	        obs[i].wmean = mat2sm(M);
	  }
	  else{
             obs[i].wmean = mmake(obs[i].nw,1);
             minit(obs[i].wmean);
	  }
       }
    }
  }
  else{
     opt->vmean = mmake(getrows(Sv),1);
     minit(opt->vmean);
     opt->init = mmake(1,1);
     minit(opt->init);
     for(i=0;i<streams;i++){
          obs[i].wmean = mmake(obs[i].nw,1);
          minit(obs[i].wmean);
     }
  }


  /* Allocate memory for output matrices */
  xhat_data = mmake(samples+1,nx);
  Smat      = mmake(samples+1,floor(0.5*(nx*(nx+1))+0.5));


  /*
   >>>>>>>>>>>>>>>>>>>>>>         CALL THE C-ROUTINE         <<<<<<<<<<<<<<<<<<<<<
   */
  dd2filtm(XFUNC,YFUNC, xhat_data, Smat, xbar, Sxbar, Sv, u, obs, streams,opt);


  /*
   >>>>>>>>>>>>>>>>>>>         CREATE OUTPUT MATICES            <<<<<<<<<<<<<<<<<<
   */
  if(nlhs>0){
     plhs[0] = mxCreateDoubleMatrix(getrows(xhat_data),nx,mxREAL);
     sm2mat(plhs[0],xhat_data);
  }

  if(nlhs>1){
     plhs[1] = mxCreateDoubleMatrix(getrows(Smat),getcols(Smat),mxREAL);
     sm2mat(plhs[1],Smat);
  }


  /*
   >>>>>>>>>>>>>>>>>>>>        FREE ARGUMENT MATRICES        <<<<<<<<<<<<<<<<<<<<<
   */
  mfree(xbar); mfree(Sxbar); mfree(Sv); mfree(u);
  mfree(opt->vmean); mfree(opt->init); free(opt); free(obs);
}
