functor UnionFindFun(structure Hash: HASH):> UNION_FIND =
  struct

    (* Implements the famous union-find algorithm for merging finite disjoint
       sets.  Very useful for constructing equivalence relations by defining
       their equivalence classes as sets, and merging sets by putting two
       representatives in the same class.

       Note, that most of the functions destructively modify the collection,
       even the lookup ones like `sameSubset. *)

    structure Hash = Hash
    open Hash

    (* Raised when an element is expected to be in the collection and is not
       (in the sameSubset and mergeSubset functions). *)
    exception UnionFindError of string

    datatype 'a Node = 
	Node of { value: 'a,
		  (* Depth of the tree; the root has it 0, and it
		     "owns" the reference.  When the root is added to
		     another tree (and stops being a root), it injects
		     the new parent's "int ref" reference into its
		     subtree and updates it if necessary. *)

		  (* Actually, ignore the depth, it's hardly important anyway *)
		  (* depth: int ref ref, *)
		  parent: 'a Node option ref }

    type 'a Collection = { hash: ('a, 'a Node) Hash,
			   eq: 'a * 'a -> bool,
			   v2str: 'a -> string }

    fun node2value(Node{value=v,...}) = v

    fun nodeEq({eq=eq, ...}: 'a Collection) (n1, n2) = eq(node2value n1, node2value n2)

    (* Create a new collection *)
    fun makeCollection(pair as (eq, v2str)) = 
	{hash=makeHashDefault pair,
	 eq=eq,
	 v2str=v2str}

    (* Add an element as a new singleton set to the collection, if not already there. *)
    fun addToCollection (coll as {hash=hash, ...}: 'a Collection) v =
	(case findHash(hash, v) of
	     NONE => (insertHashDestructive(hash, v, Node{value=v,
							  parent=ref NONE}); ())
	   | SOME _ => ())

    (* Walk up until we find a parent, then return it and shortcut all
       the pointers to it on the way back *)
    fun lookup(n as Node{parent=ref NONE, ...}) = n
      | lookup(Node{parent=pRef as ref(SOME p), ...}) =
	let val root = lookup p
	in 
	    (pRef := SOME root; root)
	end

    (* Check if two elements are in the same set in the collection.
       If any of the elements is not in the collection, *)
    fun sameSubset (coll as {hash=hash, v2str=v2str, eq=eq, ...}: 'a Collection) (v1, v2) =
	let val n1 = valOf(findHash(hash, v1)) handle Option => raise UnionFindError
	               ("sameSubset: Value is not in the collection: "
			^(v2str v1))
	    val n2 = valOf(findHash(hash, v2)) handle Option => raise UnionFindError
	               ("sameSubset: Value is not in the collection: "
			^(v2str v2))
	in 
	    nodeEq coll (lookup n1, lookup n2)
	end

    (* Merge two subsets to which the two elements belong *)
    fun mergeSubsets(coll as {hash=hash, v2str=v2str, eq=eq, ...}: 'a Collection) (v1, v2) =
	let val n1 = valOf(findHash(hash, v1)) handle Option => raise UnionFindError
	               ("mergeSubsets: Value is not in the collection: "
			^(v2str v1))
	    val n2 = valOf(findHash(hash, v2)) handle Option => raise UnionFindError
	               ("mergeSubsets: Value is not in the collection: "
			^(v2str v2))
	    val (r1 as Node{parent=pRef, ...}, r2) = (lookup n1, lookup n2)
	in 
	    if nodeEq coll (r1, r2) then () else (pRef := SOME r2)
	end

    (* Return the entire collection as a list of lists of elements, each list
       representing a set. *)

    (* `listHash' is the mapping from each root to the lists of nodes
       under it; nodes are looked up one by one and added to their
       root's list. *)

    fun listCollection(coll as {hash=hash, v2str=v2str, eq=eq, ...}: 'a Collection) =
	let val nodeEq = nodeEq coll
	    val listHash = makeHashDefault(nodeEq, fn Node{value=v,...} => v2str v)
	    val nodes = List.map #2 (hash2any(fn x=>x)(fn x=>x) hash)
	    fun addNode n =
		let val root = lookup n
		in
		    case findHash(listHash, root) of
			NONE => (insertHashDestructive(listHash, root,
						       if nodeEq(n, root) then [n]
						       else [n, root]); ())
		      (* Assume the root is always in its list and don't add it *)
		      | SOME lst => if nodeEq(n, root) then ()
				    else (insertHashDestructive(listHash, root, n::lst); ())
		end
	    val _ = List.app addNode nodes
	in
	    List.map ((List.map node2value) o #2) (hash2any(fn x=>x)(fn x=>x) listHash)
	end

  end

					       
		    