functor TimingStatFun(structure Hash: HASH): TIMING_STAT =
  struct
    structure Hash = Hash
    open Hash
    open SympBug
    open Str

    type CPUtimer = {timer: Timer.cpu_timer,
		     name: string}

    type Entry = {CPUtime: {usr:Time.time, sys:Time.time, gc:Time.time},
		  num: int}

    val funHash = ref(makeHashDefault(op =, fn x=>x): (string, Entry ref) Hash)

    fun startTimer name =
	let val timer = Timer.startCPUTimer()
	in {timer = timer, name = name}: CPUtimer
	end

    (* Number of times substTypeCommon was called *)
    fun newStatEntry(name) =
	let val entry = ref{CPUtime={usr=Time.zeroTime,
				     sys=Time.zeroTime,
				     gc=Time.zeroTime},
			    num=0}
	in (insertHashDestructive(!funHash, name, entry); entry)
	end

    (* If there is no requested entry, create a new one. *)
    fun StatEntry name =
	(case findHash(!funHash, name) of
	     SOME entry => entry
	   | NONE => newStatEntry name)

    fun updateStat{timer=timer, name=name} =
	let val entry = StatEntry name
	    val {CPUtime={usr=usr0,sys=sys0,gc=gc0},num=n} = !entry
	    val {usr=usr,sys=sys,gc=gc}=Timer.checkCPUTimer timer
	in entry:={CPUtime=let open Time in {usr=usr0+usr,sys=sys0+sys,gc=gc0+gc} end,
		   num=n+1}
	end

    (* Print all of the statistics *)
    fun stat2string () =
	let fun pad n str =
		let val len = String.size str
		    val m = if len >= n then 0 else n-len
		    fun makepad 0 acc = String.implode acc
		      | makepad k acc = makepad (k-1) (#" "::acc)
		    val padding=makepad m []
		in str^padding
		end
	    fun statEntry2string(name, entry) =
	        let val {CPUtime={usr=usr0,sys=sys0,gc=gc0},num=n} = !entry
		    val (usr,sys,gc,num,average)=(pad 15 (Time.fmt 6 usr0),
						  pad 15 (Time.fmt 6 sys0),
						  pad 15 (Time.fmt 6 gc0),
						  pad 10 (Int.toString n),
						  pad 10 (Real.toString(((Time.toReal usr0)
									 +(Time.toReal sys0)
									 +(Time.toReal gc0))
									/(Real.fromInt n))))
		in ((pad 25 name)^usr^sys^gc^num^average)
		end
	    val table = hash2any(fn x=>x)(fn x=>x) (!funHash)
	in
	    ("\n\n  ----  Timing Statistics ----\n"
	     ^"   --- Functions ---\n"
	     ^(pad 25 "Function Name")
	     ^(pad 15 "User Time")
	     ^(pad 15 "System Time")
	     ^(pad 15 "GC Time")
	     ^(pad 10 "#calls")
	     ^(pad 10 "Time/call")^"\n"
	     ^(strlist2str "\n" (List.map statEntry2string table)))
	end

    (* Reset all the statistical variables *)
    fun resetStat() = (funHash := wipeHash(!funHash))
  end

