(* load the "load-all.sml". Run `symp'. *)

functor SympFun () : SYMP =
  struct

    structure Version: VERSION =
      struct
	val version = ("symp: version 0.2-beta.  Wed Dec 27 14:24:13 2000")
	val authors = ["Sergey.Berezin@cs.cmu.edu", "Alex.Groce@cs.cmu.edu"]
      end

    structure Interact = InteractFun(structure Version = Version)

    open Str
    open SympBug
    open Interact
    open Interface
    open Options

    fun startup (options: options) commands  =
	let val _ = initSymp options
	    (* Assume that this option never changes *)
	    val {interface=interface, ... } = options
	    (* Printing functions must fetch options every time in case they changed *)
	in
	    (* First, execute any command line commands, and if
	       successful, either terminate (in non-interactive mode),
	       or execute user's commands interactively. *)
	    (case tryExecuteList commands of
		 true => 
		   (case interface of
			commandLine => OS.Process.success
		      | _ => (readExecLoop(); OS.Process.success))
	       | false => OS.Process.failure)
	end
            handle SympExit => OS.Process.success (* Clean exit *)
		 | SympExitError => ((printError options "general" "Unknown fatal error") : unit;
				     OS.Process.failure) (* Exit with error *)
		(* In theory, all "bad" exceptions must have been
		   already taken care of by now, and we only need to
		   terminate with error.  But, you never know... *)
		 | SympError str => ((printError options "general" str): unit;
				     OS.Process.failure)
		 | SympBug str => (printError options "bug" ("\nOops, internal error!  "
					^"Please report this to sergey.berezin@cs.cmu.edu:\n\n"
					^str^"\n\nInclude your program and this output.\n"
					^"The internal SyMP state has been reset.\n\n"
					^(FunStack2string()));
				   OS.Process.failure)
		 | exn => ((printError options "general" 
			     ("\nUnrecognized exception caught:\n"
			      ^(exnMessage exn)^"\n"): unit);
			 OS.Process.failure)
    fun symp(name,args) = 
	let exception Quit of string
	in let val usage =
("Usage: "
 ^"symp [options] <file.symp> [options]\n\n"
 ^"If the file is `-', then the program is read from standard input.\n"
 ^"The options are:\n"
 ^" --spec THEOREM  -- Generate a transition system for the spec (theorem).\n"
 ^"                    Specs are full SyMP theorem expressions.\n"
 ^"                    There can be several --spec instances in one call.\n"
 ^" -mc name\n"
 ^" --modelchecker name\n"
 ^"                 -- the name of the back-end model checker.\n"
 ^"                    Currently, only \"smv\" is supported, and is the default.\n"
 ^" -o file, --output-file file\n"
 ^"                 -- the name of the output file.  A dash `-' means stdout.\n"
 ^"                    default is \"out.smv\".  This is a temporary option\n"
 ^"                    before the call to the back-end model checker is automated.\n"
 ^" --limit N       -- set the threshold on the type size after which the type is\n"
 ^"                    considered too large or infinite for translation purposes.\n"
 ^" --version       -- print a version string and exit\n"
 ^" -emacs          -- run in emacs server mode (don't use on command line)\n"
 ^" -h, -help, --help\n"
 ^"                 -- print this help and exit\n"
 ^" -v (--verbose)  -- verbose mode\n"
 ^" -q (--quiet)    -- quiet mode (default)\n"
 ^" --debug <debug spec>  -- dump (a lot of!) debug output\n"
 ^"     where <debug spec> stands for:\n"
 ^"       all          -- dump all possible debug info\n"
 ^"       f1[,f2,...]  -- print debug info only for the functions f1,f2, etc..\n"
 ^"                       The function names must be separated by commas or spaces,\n"
 ^"                       but in the latter case they must be enclosed in quotes.\n")
	       (* Parse debug spec, return the list of function names *)
	       fun parseDspec str =
		   let val chlst=String.explode str
		       fun loop [] acc flist = rev((String.implode(List.rev acc))::flist)
			 | loop (ch::lst) acc flist =
			     if ch = #"," orelse ch = #" " then
				 loop lst [] ((String.implode(List.rev acc))::flist)
			     else loop lst (ch::acc) flist
		   in loop chlst [] []
		   end
	       fun addTheorem (options as {spec=spec,...}) thm = upd_spec(options, thm::spec)
	       fun processArgs options files [] = (options,files)
		 | processArgs options files ("--spec"::thm::args) =
		     processArgs(addTheorem options thm) files args
		 | processArgs _ _ ["--spec"] =
		     raise Quit("Option --spec requires an argument\n\n"^usage)
		 | processArgs options files ("-mc"::name::args) =
		     processArgs (upd_backendMC(options,name)) files args
		 | processArgs options files (["-mc"]) =
		     raise Quit("Option -mc requires an argument\n\n"^usage)
		 | processArgs options files ("--modelchecker"::name::args) =
		     processArgs (upd_backendMC(options,name)) files args
		 | processArgs options files (["--modelchecker"]) =
		     raise Quit("Option --modelchecker requires an argument\n\n"^usage)
		 | processArgs options files ("-o"::file::args) =
		     processArgs (upd_outputFile(options, file)) files args
		 | processArgs options files (["-o"]) =
		     raise Quit("Option -o requires an argument\n\n"^usage)
		 | processArgs options files ("--limit"::lim::args) =
		     (case Int.fromString lim of
			  SOME n => processArgs (upd_limit(options, n)) files args
			| NONE => raise Quit("Options --limit requires an integer argument."))
		 | processArgs options files (["--limit"]) =
		     raise Quit("Option --limit requires an argument\n\n"^usage)
		 | processArgs options files ("--output-file"::file::args) =
		     processArgs (upd_outputFile(options, file)) files args
		 | processArgs options files (["--output-file"]) =
		     raise Quit("Option --output-file requires an argument\n\n"^usage)
		 | processArgs options files ("-v"::args) =
	             processArgs (upd_verbose(options,true)) files args
		 | processArgs options files ("--verbose"::args) =
	             processArgs (upd_verbose(options,true)) files args
		 | processArgs options files ("-emacs"::args) =
	             processArgs (upd_interface(options,Emacs)) files args
		 | processArgs options files ("--debug"::lst::args) =
		     processArgs (upd_debug(options,SOME(parseDspec lst))) files args
		 | processArgs options files ["--debug"] =
		     raise Quit("Option --debug requires an argument\n\n"^usage)
		 | processArgs options files ("-q"::args) =
		     processArgs (upd_verbose(options,false)) files args
		 | processArgs options files ("--quite"::args) =
		     processArgs (upd_verbose(options,false)) files args
		 | processArgs _ _ ("--version"::_) =
		     raise Quit(Version.version)
		 | processArgs _ _ ("-h"::_) =
		     raise Quit(usage)
		 | processArgs _ _ ("-help"::_) =
		     raise Quit(usage)
		 | processArgs _ _ ("--help"::_) =
		     raise Quit(usage)
		 | processArgs options files (x::args) =
		     processArgs options (x::files) args
	       val _ = (case args of
			    [] => raise Quit(usage)
			  | _ => ())
	       val (options,files) =
		      processArgs defaultOptions [] args
	       val reportError = reportError options
	   in (case List.length(files) of
		   0 => startup options []
		 | 1 => 
		     let val {spec=specs, outputFile=outf, ...} = options 
		     in  startup options 
			  ([UIcommand("typecheck", List.map(fn s=>UIstring s) files),
			    UIcommand("add_theorems", List.map(fn s=>UIstring s) specs),
			    UIcommand("set_current_module", [UIstring "main"]),
			    UIcommand("modelcheck",[UIstring outf])])
		     end
		 | n =>  raise Quit 
		     ("Please provide no more than one file name"
		      ^" (provided "^(Int.toString n)
		      ^" instead)\n\n"^usage))
	   end
            handle Quit str => (print(str^"\n"); OS.Process.success)
	end

  end

structure Symp = SympFun()
(* open Symp *)
