functor StatusFileFun(structure Interface: INTERFACE): STATUS_FILE =
  struct

    structure StatusFileStruct: STATUS_FILE_STRUCT =
      struct
	  (* SFfield(name, value) - the `name' attribute gets `value' string *)
	  datatype StatusFileField = SFfield of string * string
	  (* Section type and its list of assignments *)
	  datatype StatusFileSection = 
	      SFsection of string * (StatusFileField list)
	  (* Representation of the status file *)    
	  type StatusFileType = StatusFileSection list
      end

    open StatusFileStruct
    open Interface
    open Str

    structure ParseError: PARSE_ERROR =
      struct
        exception ParseError of string 
	val wasError = ref false
      end

    structure StatusFileLrVals =
	StatusFileLrValsFun(structure Token = LrParser.Token
			     structure StatusFileStruct = StatusFileStruct
			     structure ParseError = ParseError)

    structure StatusFileLex =
	StatusFileLexFun(structure Tokens = StatusFileLrVals.Tokens
			  structure ParseError = ParseError)

    structure StatusFileParser =
	StatusFileJoin(structure LrParser = LrParser
		 structure StatusFileStruct = StatusFileStruct
		 structure ParserData = StatusFileLrVals.ParserData
		 structure Lex = StatusFileLex)

    type result = StatusFileLrVals.ParserData.result
    open StatusFileLrVals.ParserData

    fun StatusFileParse nameOpt ins =
	let open ParseError
	    val name = (case nameOpt of
			    SOME n => n^":"
			  | NONE => "")
	    val print_error = fn (s,l,c) =>
		(wasError := true;
		 raise SympError("\nStatus file "^name^(Int.toString l) ^"."
				 ^ (Int.toString c) ^ ": " ^ s ^ "\n"))
	    (* Use 15 tokens for error correction *)
	    val invoke = fn lexstream => 
		(StatusFileParser.parse(15,lexstream,print_error,())
		 handle StatusFileParser.ParseError => 
		     raise SympError("Fatal StatusFile parser error")
		      | ParseError str => raise SympError str)
	    val lexer = StatusFileParser.makeLexer 
		           (fn (i: int) => TextIO.inputLine ins) print_error
	    val (result,_) = (StatusFileLex.UserDeclarations.pos := 1;
			      StatusFileLex.UserDeclarations.yypos_bol := 0;
			      wasError := false;
			      invoke lexer)
			      
	in result
	end

    fun readStatusFile filename =
	let val ins = (TextIO.openIn filename)
	             handle _ => raise SympError("Cannot read status file: "^filename)
	in
	    StatusFileParse (SOME filename) ins
	end

    fun StatusFile2str status =
	let fun op *(x,y) = Conc(x,y)
	    fun field2str(SFfield(name,value)) = (Str (name^" = \""^value^"\""))
	    fun section2str(SFsection(title, lst)) =
	         (Str ("["^title^"]\n  "))*(Strlist2Str "\n  " (List.map field2str lst))
	in
	    (* Print all the sections separated by an empty line, and
	       add an empty line at the end. *)
	    (Strlist2Str "\n\n" (List.map section2str status))
	    *(Str "\n")
	end

    fun writeStatusFile filename status =
	let val outs = (TextIO.openOut filename)
	            handle _ => raise SympError("Cannot write status file: "^filename)
	    fun dump outs (Str s) = TextIO.output(outs,s)
	      | dump outs (Conc(x,y)) = ((dump outs x);(dump outs y))
	in
	    (dump outs (StatusFile2str status);
	      TextIO.closeOut outs) handle _ => raise SympError
	           ("Cannot write status file: "^filename)
	end
	     


  end
