#!FullPathToPerl -w

use IO::Handle;

# Parse the command line: 
$usage = 'smvrun.pl [ -c smv-command ] [ input-file | - ] [ --  <smv options> ]';

# Defaults
$command = "smv";
$infile = "-";

sub err {
    print("Error: ",@_,"\n");
    exit 1;
}

if($#ARGV < 0) {
    err("must give at least one argument.\nUsage: $usage");
}

# the arguments are ours when true, otherwise it's for SMV
$localargs = 1;

for($i=0; $i <= $#ARGV; $i++) {
    if($localargs) {
	if($ARGV[$i] eq "-c") {
	    $i++;
	    if($i <= $#ARGV) {
		$command = $ARGV[$i+1];
	    } else {
		err "-c requires an argument";
	    }
	} elsif($ARGV[$i] eq "--") {
	    $localargs = 0;
	# This must be file name
	} else {
	    $infile = $ARGV[$i];
	}
    } else {
	@smvargs = @ARGV[$i..$#ARGV];
	$i = $#ARGV + 1;
    }
}

# Read the header of the input file (upto the first non-empty
# uncommented line) and collect all the SMV variables and values and
# their mapping to SyMP names

# First, read the entire SMV program in
if("$infile" eq "-") {
    @lines = <STDIN>;
} else {
    open(SMV,"$infile") or err("Can't open input file $infile");
    @lines = <SMV>;
    close SMV;
}

# Autoflush STDOUT (otherwise it accumulates in each child and gets duplicated)
STDOUT->autoflush(1);

my $startMapping = 0;
# Parse the comments in the header, fetch the name mapping
for($i = 0, $done = 0; (!$done) && $i <= $#lines; $i++) {
    $str = $lines[$i];
    if($str =~ /^\s*--(.*)$/) {
	$line = $1;
	print "header: $line\n";
	# if we are in the right section of the header, 
        # collect the variable mapping
	if($startMapping && ($line =~ /^\s*(\S(.*\S)?)\s+\-\>\s+(\S+)\s*$/)) {
	    $sympvar = $1; $smvvar = $3;
	    $sympvar =~ s/^\"(.*)\"$/$1/;
	    $varMap{$smvvar} = $sympvar;
	}
	# This starts the var mapping section, turn our flag on
	if($line =~ /^\s*Mapping of all names/) { $startMapping = 1; }
    } elsif($str =~ /^\s*$/) { } # skip empty line
    else {
	# Otherwise it's an SMV line, the header is over, wrap up
	$done = 1;
    }
}

# OK, now we're ready to start SMV.  The war plan is:
# 1. Fork off a child that redirects STDIN/OUT to dedicated pipes,
#    redirect STDERR to STDOUT, and exec smv;
# 2. Fork off another child that would dump all @lines to SMV's STDIN
# 3. In the main thread, read SMV's STDOUT and parse it line by line.

# Zombie killer
sub REAPER {
    wait;
    # $waitedpid = wait;
    # loathe sysV: it makes us not only reinstate
    # the handler, but place it after the wait
    $SIG{CHLD} = \&REAPER;
}
$SIG{CHLD} = \&REAPER;

pipe(PARENT_RDR, CHILD_WTR);
pipe(CHILD_RDR,  PARENT_WTR);
CHILD_WTR->autoflush(1);
PARENT_WTR->autoflush(1);

# When set, the property didn't verify
$result = 1;

# Flush the parent's STDOUT - otherwise the child gets confused somehow
print STDOUT "\n";

# Fork SMV-handling process
$smvpid = fork;
die "cannot fork SMV thread: $!" unless defined $smvpid;
if ($smvpid == 0) {
    # SMV thread: redirect STDIN/OUT/ERR:
    close CHILD_RDR; close CHILD_WTR;
    open(STDIN, "<&PARENT_RDR")
	or die "Can't redirect STDIN in SMV thread";
    open(STDOUT, ">&PARENT_WTR")
	or die "Can't redirect STDOUT in SMV thread";
    open(STDERR, ">&STDOUT")
	or die "Can't redirect STDERR in SMV thread";
    # Start smv process
    print STDOUT ("Running SMV: ","$command ", join(" ",@smvargs), "\n");
    exec("$command", @smvargs) or die "Can't exec smv";
} else {
    # Main thread: continue the forking adventure
    close PARENT_RDR; close PARENT_WTR;
    $dumppid = fork;
    die "cannot fork the dumping thread: $!" unless defined $dumppid;
    if($dumppid == 0) {
	# The dumping thread: dump all the @lines to SMV process
	close CHILD_RDR;
	foreach $str (@lines) {
	    print CHILD_WTR "$str";
	}
	close CHILD_WTR;
	exit 0;
    } else {
	# Main thread: read SMV output and try to make sense of it
	close CHILD_WTR;
	while(defined($str = <CHILD_RDR>)) {
	    # Catch the spec reporting line
	    if("$str" =~ /^-- specification.*is (true|false)/) {
		$result = ("$1" eq "false");
	    }
	    $line = $str;
	    foreach $key (keys(%varMap)) {
		$line =~ s/\b$key\b/$varMap{$key}/g;
	    }
	    print $line;
	}
    }
}

if($result) {
    print "\nSpecification is false\n";
} else {
    print "\nSpecification is true\n";
}

exit $result;
