(* Abstract types of sequents, inference rules, and strategies,
   together with functions for applying them.

   There is no file `sequent.sml', because each implementation is
   specific to the problem domain, and should be placed in a separate
   appropriate module. *)

signature SEQUENT =
  sig
    structure SequentCommon: SEQUENT_COMMON
    structure InputLanguage: INPUT_LANGUAGE
    sharing InputLanguage = SequentCommon.InputLanguage
    sharing SequentCommon.Interface.Options = InputLanguage.Options

    type Sequent = SequentCommon.Sequent
    type InferenceRule

    type ProofSystemCommand
    type ProofSystemCommandResult
    (* An abstract type for instantiating a rule.  It preserves some
       useful information obtained at the time of matching to be
       passed to the applyRule function. *)
    type Substitution

    (* Some hints of what a sequent should look like, so that subgoals
       in a proof tree can be matched better when the theorem changes.
       Currently, the hints are added only by the internal functions
       like rule application, etc. *)

    type SequentHints = SequentCommon.SequentHints

    (* The result of the application of an inference rule.  *)
    type InferenceRuleResult = SequentCommon.InferenceRuleResult
    type SequentPart = SequentCommon.SequentPart
    type InferenceRuleArgumentValue = SequentCommon.InferenceRuleArgumentValue
    type InferenceRuleArgument = SequentCommon.InferenceRuleArgument
    type InferenceRuleArgumentType = SequentCommon.InferenceRuleArgumentType

    type ProgramObject = InputLanguage.ProgramObject
    type ProgramContext = InputLanguage.ProgramContext

    type InferenceRuleArgumentSpec

    type UserInputArgument = SequentCommon.Interface.UserInputArgument

    val IRArgSpecName: InferenceRuleArgumentSpec -> string
    val IRArgSpecType: InferenceRuleArgumentSpec -> InferenceRuleArgumentType

    (* Create a new sequent from a theorem (which may include a model) *)
    val makeSequent: ProgramContext -> ProgramObject -> Sequent
    val sequentEq: Sequent * Sequent -> bool
    (* Check if the first sequent subsumes the second.
       That is, proving the first one also proves the second one.
       This function can be soundly approximated with sequentEq. *)
    val sequentSubsumes: Sequent * Sequent -> bool
(*      val getHints: Sequent -> SequentHints option *)
    val hintsEq: SequentHints * SequentHints -> bool

    val UIArg2FormulaIndex: UserInputArgument -> SequentPart * (int option)
    val FormulaIndex2UIArg: SequentPart * (int option) -> UserInputArgument

    (* Operations on inference rules *)

    (* Each rule has a unique string identifier *)
    val getRuleName: InferenceRule -> string
    val getCommandName: ProofSystemCommand -> string

    (* Each rule has a set of arguments *)
    val getRuleArgs: InferenceRule -> InferenceRuleArgumentSpec list
    val getCommandArgs: ProofSystemCommand -> InferenceRuleArgumentSpec list

    (* Help on the rule *)
    val getRuleHelp: InferenceRule -> string
    val getCommandHelp: ProofSystemCommand -> string

    (* "Typecheck" the rule's arguments and return SOME error message
       if they don't match.  Otherwise return NONE. *)
    val checkRuleArgs: InferenceRuleArgumentSpec list -> InferenceRuleArgument list
	  -> string option

    (* See if the sequent matches the conclusion of the rule, and if
       it does, compute the necessary parameters for the rule. `SOME'
       doesn't guarantee yet that the rule applies, but `NONE'
       guarantees that it doesn't. *)

    val matchRule: (Sequent * ProgramContext)
	  -> (InferenceRule * (InferenceRuleArgument list))
	  -> Substitution option

    (* Try to apply the rule and get a list of new sequents as
       premisses.  NONE means the rule doesn't apply.  May take a
       (partial) substitution obtained by the matchRule funciton.
       The first argument is a printing function to send messages to
       the user while applying the rule. *)
    val applyRule: (string -> unit) -> (Sequent * ProgramContext)
	        -> InferenceRule * (InferenceRuleArgument list) * (Substitution option)
	        -> InferenceRuleResult

    (* Run the proof sytem command on the sequent.  Commands normally
       do not change the sequent or the proof tree, and are only used
       to display some information or test various conditions for the
       sequent. *)

    val applyCommand: (string -> unit) -> (Sequent * ProgramContext)
	        -> ProofSystemCommand * (InferenceRuleArgument list)
                -> ProofSystemCommandResult option

    (* Get the list of rules that match a given sequent, together with
       the lists of rules' arguments *)
    val matchAllRules: Sequent
	  -> (InferenceRule * (InferenceRuleArgument list)) list

    val string2hints: string -> SequentHints

    (* Getting extra information about a sequent.  The type of request
       is specified by a string, the output is also a string which the
       user will see. *)

    val getSequentInfo: Sequent -> string -> string

  end
