(* Some functions that can be useful in implementing a custom
   'Sequent' module. *)

functor SequentCommonFun(structure Interface: INTERFACE
			 structure SequentBase: SEQUENT_BASE
			 sharing Interface.Options = SequentBase.InputLanguage.Options)
    : SEQUENT_COMMON =
  struct

    structure SequentBase = SequentBase
    structure Interface = Interface
    structure InputLanguage = SequentBase.InputLanguage
    open SequentBase
    open Interface
    open InputLanguage
    open Str

    (* The result of the application of an inference rule.  *)
    datatype InferenceRuleResult =
	(* The rule applies and new subgoals are generated *)
	InferenceRuleSuccess of (Sequent * (SequentHints option)) list
	(* The rule proves the sequent false *)
      | InferenceRuleDisproven
	(* The rule doesn't apply *)
      | InferenceRuleFailure

    (* Actual argument to the inference rules.
       Each argument has a name and a value. *)
    datatype InferenceRuleArgumentValue =
	ProverArgString of string
      | ProverArgNumber of int
      | ProverArgSubgoal of string
	(* When formula number is NONE, this means all/any formulas from a given part *)
      | ProverArgFormulaNum of SequentPart * (int option)
	(* An Object in the input language.  Represent it as a string
           and let the rule decide how to parse/typecheck it *)
      | ProverArgObject of string
	(* A special case of Object that must be of type `bool' *)
      | ProverArgFormula of string
      | ProverArgList of InferenceRuleArgumentValue list

    type InferenceRuleArgument = 
	{ name: string,
	  value: InferenceRuleArgumentValue }

    (* Formal argument to an inference rule.  Each argument has a
       name, a "type", and an optional default value, whose presence
       means the argument is optional. *)
    datatype InferenceRuleArgumentType =
	ProverArgUnknownType
      | ProverArgStringType
      | ProverArgNumberType
      | ProverArgSubgoalType
      | ProverArgFormulaNumType
      | ProverArgObjectType
      | ProverArgFormulaType
      | ProverArgListType of InferenceRuleArgumentType

    fun getInfRuleArgType (ProverArgString _) = ProverArgStringType
      | getInfRuleArgType (ProverArgNumber _) = ProverArgNumberType
      | getInfRuleArgType (ProverArgSubgoal _) = ProverArgSubgoalType
      | getInfRuleArgType (ProverArgFormulaNum _) = ProverArgFormulaNumType
      | getInfRuleArgType (ProverArgObject _) = ProverArgObjectType
      | getInfRuleArgType (ProverArgFormula _) = ProverArgFormulaType
      | getInfRuleArgType (ProverArgList []) = ProverArgListType ProverArgUnknownType
      | getInfRuleArgType (ProverArgList(x::_)) = ProverArgListType(getInfRuleArgType x)

    (* Printing of the inference rule argument, for help and debugging *)
    fun InfRuleArgValue2string (ProverArgString s) = s
      | InfRuleArgValue2string (ProverArgNumber n) = Int.toString n
      | InfRuleArgValue2string (ProverArgSubgoal s) = s
      | InfRuleArgValue2string (ProverArgFormulaNum pair) = formulaInd2str pair
      | InfRuleArgValue2string (ProverArgObject s) = s
      | InfRuleArgValue2string (ProverArgFormula s) = s
      | InfRuleArgValue2string (ProverArgList lst) = 
	  "("^(strlist2str " " (List.map InfRuleArgValue2string lst))^")"

    fun InfRuleArg2string{name=name, value=v} =
	  name^" : "^(InfRuleArgValue2string v)

    fun InfRuleArgType2string ProverArgStringType = "string"
      | InfRuleArgType2string ProverArgNumberType = "integer"
      | InfRuleArgType2string ProverArgSubgoalType = "subgoal"
      | InfRuleArgType2string ProverArgFormulaNumType = "formulaNum"
      | InfRuleArgType2string ProverArgObjectType = "expression"
      | InfRuleArgType2string ProverArgFormulaType = "formula"
      | InfRuleArgType2string (ProverArgListType tp) =
	  (InfRuleArgType2string tp)^" list"
      | InfRuleArgType2string ProverArgUnknownType = "unknown"

    fun InfRuleResult2string(InferenceRuleSuccess pairs) =
	let fun pair2str(seq, hintsOpt) =
	        (sequent2string seq)
		^(option2string "" (Option.map (fn h=>"["^(hints2string h)^"]") hintsOpt))
	in
	    "InferenceRuleSuccess[\n"^(strlist2str ", " (List.map pair2str pairs))^"\n]"
	end
      | InfRuleResult2string InferenceRuleFailure = "InferenceRuleFailure"
      | InfRuleResult2string InferenceRuleDisproven = "InferenceRuleDisproven"

    fun IRArgStringValue(ProverArgString str) = SOME str
      | IRArgStringValue _ = NONE
    fun IRArgNumberValue(ProverArgNumber n) = SOME n
      | IRArgNumberValue _ = NONE
    fun IRArgSubgoalValue(ProverArgSubgoal str) = SOME str
      | IRArgSubgoalValue _ = NONE
    fun IRArgFormulaNumValue(ProverArgFormulaNum x) = SOME x
      | IRArgFormulaNumValue _ = NONE
    fun IRArgObjectValue(ProverArgObject str) = SOME str
      | IRArgObjectValue _ = NONE
    fun IRArgFormulaValue(ProverArgFormula str) = SOME str
      | IRArgFormulaValue _ = NONE
    fun IRArgListValue(ProverArgList lst) = SOME lst
      | IRArgListValue _ = NONE

    fun findIRArg args name =
	  Option.map(fn {value=v,...}: InferenceRuleArgument => v)
	            (List.find(fn {name=n,...} => n=name) args)

    fun findIRArgAnyValue ff args name = Option.mapPartial ff (findIRArg args name)

    val findIRArgValue = findIRArgAnyValue(fn x=>SOME x)
    val findIRArgStringValue = findIRArgAnyValue IRArgStringValue
    val findIRArgNumberValue = findIRArgAnyValue IRArgNumberValue
    val findIRArgSubgoalValue = findIRArgAnyValue IRArgSubgoalValue
    val findIRArgFormulaNumValue = findIRArgAnyValue IRArgFormulaNumValue
    val findIRArgObjectValue = findIRArgAnyValue IRArgObjectValue
    val findIRArgFormulaValue = findIRArgAnyValue IRArgFormulaValue
    val findIRArgListValue = findIRArgAnyValue IRArgListValue

  end
