signature SEQUENT_COMMON =
  sig
    structure SequentBase: SEQUENT_BASE
    structure Interface: INTERFACE
    structure InputLanguage: INPUT_LANGUAGE
    sharing Interface.Options = InputLanguage.Options
    sharing InputLanguage = SequentBase.InputLanguage

    type Sequent = SequentBase.Sequent
    type SequentPart = SequentBase.SequentPart

    (* Some hints of what a sequent should look like, so that subgoals
       in a proof tree can be matched better when the theorem changes.
       Currently, the hints are added only by the internal functions
       like rule application, etc. *)

    type SequentHints = SequentBase.SequentHints

    (* The result of the application of an inference rule.  *)
    datatype InferenceRuleResult =
	(* The rule applies and new subgoals are generated *)
	InferenceRuleSuccess of (Sequent * (SequentHints option)) list
	(* The rule proves the sequent false *)
      | InferenceRuleDisproven
	(* The rule doesn't apply *)
      | InferenceRuleFailure

    (* Actual argument to the inference rules.
       Each argument has a name and a value. *)

    datatype InferenceRuleArgumentValue =
	ProverArgString of string
      | ProverArgNumber of int
      | ProverArgSubgoal of string
	(* When formula number is NONE, this means all/any formulas from a given part *)
      | ProverArgFormulaNum of SequentPart * (int option)
	(* An Object in the input language.  Represent it as a string
           and let the rule decide how to parse/typecheck it *)
      | ProverArgObject of string
	(* A special case of Object that must be of type `bool' *)
      | ProverArgFormula of string
      | ProverArgList of InferenceRuleArgumentValue list

    type InferenceRuleArgument = 
	{ name: string,
	  value: InferenceRuleArgumentValue }

    (* Formal argument to an inference rule.  Each argument has a
       name, a "type", and an optional default value, whose presence
       means the argument is optional. *)
    datatype InferenceRuleArgumentType =
	ProverArgUnknownType
      | ProverArgStringType
      | ProverArgNumberType
      | ProverArgSubgoalType
      | ProverArgFormulaNumType
      | ProverArgObjectType
      | ProverArgFormulaType
      | ProverArgListType of InferenceRuleArgumentType

    val getInfRuleArgType: InferenceRuleArgumentValue -> InferenceRuleArgumentType
    val InfRuleArg2string: InferenceRuleArgument -> string
    val InfRuleArgValue2string: InferenceRuleArgumentValue -> string
    val InfRuleArgType2string: InferenceRuleArgumentType -> string
    val InfRuleResult2string: InferenceRuleResult -> string

    (* If the argument is of the right type, extract its value, otherwise return NONE. *)
    val IRArgStringValue: InferenceRuleArgumentValue -> string option
    val IRArgNumberValue: InferenceRuleArgumentValue -> int option
    val IRArgSubgoalValue: InferenceRuleArgumentValue -> string option
    val IRArgFormulaNumValue: InferenceRuleArgumentValue -> (SequentPart * (int option)) option
    val IRArgObjectValue: InferenceRuleArgumentValue -> string option
    val IRArgFormulaValue: InferenceRuleArgumentValue -> string option
    val IRArgListValue: InferenceRuleArgumentValue -> InferenceRuleArgumentValue list option

    (* Find the value of an argument in the list by the argument's name *)
    val findIRArgValue: InferenceRuleArgument list -> string -> InferenceRuleArgumentValue option

    (* Similarly to the above, but search in the argument list for the name *)
    val findIRArgStringValue: InferenceRuleArgument list -> string -> string option
    val findIRArgNumberValue: InferenceRuleArgument list -> string -> int option
    val findIRArgSubgoalValue: InferenceRuleArgument list -> string -> string option
    val findIRArgFormulaNumValue: InferenceRuleArgument list -> string -> (SequentPart * (int option)) option
    val findIRArgObjectValue: InferenceRuleArgument list -> string -> string option
    val findIRArgFormulaValue: InferenceRuleArgument list -> string -> string option
    val findIRArgListValue: InferenceRuleArgument list -> string -> InferenceRuleArgumentValue list option

  end
