(* Join all the prover types under one API *)
functor ProversJoinFun(structure Hash: HASH
		       structure Interface: INTERFACE): PROVERS_JOIN =
  struct
    structure Hash = Hash
    structure Interface = Interface

    structure ProofSystemDefault = 
	ProofSystemDefaultFun(structure Interface = Interface
			      structure Hash = Hash)

    structure PIDefault =
	ProverInterfaceFun(structure ProverInterfaceCommon = ProverInterfaceCommon
			   structure Interface = Interface
			   structure Hash = Hash
			   structure ProofSystem = ProofSystemDefault)

    structure ProofSystemAthena = (*  ProofSystemAthenaFun(structure Interface = Interface) *)
	AthenaProofSystemFun(structure Interface = Interface
			     structure Hash = Hash)

    structure PIAthena =
	ProverInterfaceFun(structure ProverInterfaceCommon = ProverInterfaceCommon
			   structure Interface = Interface
			   structure Hash = Hash
			   structure ProofSystem = ProofSystemAthena)

    open Interface
    open Options
    open ProverInterfaceCommon

    datatype ProverType = DefaultProver | AthenaProver

    type State = { proverType: ProverType option ref }

    (* Initialize all the proof systems and prover interfaces *)
    fun initProvers () =
	(ProofSystemDefault.initProofSystem();
	 ProofSystemAthena.initProofSystem())

    fun str2ProverType "default" = DefaultProver
      | str2ProverType "athena" = AthenaProver
      | str2ProverType x = raise SympError
	  ("Unrecognized prover type: "^x)

    local val state as ({ proverType=pt }: State) = { proverType=ref(NONE) }
    in
	fun getProverType() = !pt

        (* Consider all the cases for all the provers and call the
           appropriate functions *)

	fun typeCheckFile (SOME DefaultProver) (options: options) file =
	      (pt := SOME DefaultProver; 
	       PIDefault.typeCheckFile options file)
	  | typeCheckFile (SOME AthenaProver) options file =
	      (pt := SOME AthenaProver;
	       PIAthena.typeCheckFile options file)
	  | typeCheckFile NONE options file =
	      let fun action choice = typeCheckFile (SOME(str2ProverType choice)) options file
	      in 
		 (userChoice options ("strict", "Select proof system: ", ["default"], action);
		  ())
	      end

	fun hasContext() =
	    (case getProverType() of
		 NONE => false
	       | SOME DefaultProver => isSome(PIDefault.getContext())
	       | SOME AthenaProver => isSome(PIAthena.getContext()))

	fun getActiveProvers() =
	    (case getProverType() of
		 NONE => []
	       | SOME DefaultProver => PIDefault.getActiveProvers()
	       | SOME AthenaProver => PIAthena.getActiveProvers())

	fun getProverStatus id =
	    (case getProverType() of
		 NONE => []
	       | SOME DefaultProver => PIDefault.getProverStatusList id
	       | SOME AthenaProver => PIAthena.getProverStatusList id)

	fun proveTheorem options name theorem =
	    (case getProverType() of
		 NONE => raise ProverError
		     ("No context to prove a theorem, typecheck your program first")
	       | SOME DefaultProver => PIDefault.proveTheorem options name theorem
	       | SOME AthenaProver => PIAthena.proveTheorem options name theorem)

	fun printCurrentSubgoal options id =
	    (case getProverType() of
		 NONE => raise ProverError
		     ("No context to print a subgoal, typecheck your program first")
	       | SOME DefaultProver => PIDefault.printCurrentSubgoal options id
	       | SOME AthenaProver => PIAthena.printCurrentSubgoal options id)

	fun ProverCommand options (id, command, args) =
	    (case getProverType() of
		 NONE => raise ProverError
		     ("No context for prover command, typecheck your program first")
	       | SOME DefaultProver => PIDefault.ProverCommand options (id, command, args)
	       | SOME AthenaProver => PIAthena.ProverCommand options (id, command, args))

	fun exitAllProvers options cont =
	    (case getProverType() of
		 (* No context, therefore, no provers to exit *)
		 NONE => cont()
	       | SOME DefaultProver => PIDefault.exitAllProvers options cont
	       | SOME AthenaProver => PIAthena.exitAllProvers options cont)

    end

  end
