(* Abstract interface for all types of provers.  The implementation
   funnels the commands to the correct prover type. *)

signature PROVERS_JOIN =
  sig
    structure Interface: INTERFACE
    type options = Interface.Options.options

    datatype ProverType = DefaultProver | AthenaProver

    val initProvers: unit -> unit

    val str2ProverType: string -> ProverType
    (* Return the current prover type if the prover interface is
       active, that is, there is a typechecked program currently in
       the memory. *)

    val getProverType: unit -> ProverType option

    (* Return the list of IDs for all active provers *)
    val getActiveProvers: unit -> string list

    (* Return the list of status flags for the prover.  A status flag
       can currently be "active", "proven", and "saved".  If flag is
       not present, its negation is assumed.  So, when the list is
       empty, the prover is not active. *)
    val getProverStatus: string -> string list

    (* Takes a file name, parses/typechecks the program from it for a
       given prover type (which defines the input language), and
       installs it as the currrent context. *)
    val typeCheckFile: ProverType option -> options -> string -> unit
    (* Check if we have typechecked a program before *)
    val hasContext: unit -> bool

    (* Creates a new prover for a theorem.
       proveTheorem options optional_name theorem_string => new prover ID *)
    val proveTheorem: options -> string -> (string option) * string -> string

    (* Print the current subgoal of the prover with the given proverID.
       printCurrentSubgoal options proverID. *)
    val printCurrentSubgoal: options -> string -> unit

    (* All the interactive communication with the provers goes
       through this function.  When proverID is NONE, the command
       is targeted to the global state rather than a particular
       prover.

       ProverCommand options (proverID, command, arguments) *)
    val ProverCommand: options
          -> (string option) * string * Interface.UserInputArgument list
	  -> unit

    (* exitAllProvers options cont
       Exit all active provers and run the continuation `cont' *)
    val exitAllProvers: options -> (unit -> unit) -> unit

  end
