(* Implementation of a generic prover module *)
functor ProverFun(structure Interface: INTERFACE
		  structure ProofSystem: PROOF_SYSTEM
		  structure Hash: HASH
		  sharing Interface = ProofSystem.Sequent.SequentCommon.Interface): PROVER =
  struct
    structure Interface = Interface
    structure ProofSystem = ProofSystem

    structure ProofTree = 
	ProofTreeFun(structure ProofSystem = ProofSystem
		     structure Hash = Hash
		     structure Interface = Interface)

    open SympBug
    open Str
    open ListOps
    open Hash
    open Interface
    open ProofSystem
    open Sequent
    open SequentCommon
    open SequentBase
    open InputLanguage
    open Options
    open ProofTree

    type Prover = { id: string, (* unique ID for each prover *)
		    (* flag whether the proof has changed.
		       It gets reset when the proof is saved. *)
		    changed: bool ref,
		    context: ProgramContext,
		    proofrun: ProofRun }

    type ProverCommand = { name: string,
			   args: UserInputArgument list,
			   subgoal: Subgoal }

    (* Create a new prover object associated with a certain theorem
       and having an optional string ID (name) for the proof.  Don't
       mix it up with the prover ID, which is always unique can only
       be generated automatically when creating a new prover. *)

    local val proverID = ref 0
    in 
	fun makeProver name (obj, context) = 
	    let val seq = makeSequent context obj
		val run = makeProofRun(makeProofTree name)
		val tree = ProofRun2ProofTree run
		val prover = ({ id = Int.toString(!proverID),
			        changed = ref false,
				context = context,
			        proofrun = run }: Prover)
	    in (addSequent run (getRoot tree, seq);
		proverID := (!proverID) + 1;
		prover)
	    end
    end

    fun getProverID({id=id,...}: Prover) = id
    fun getProverChanged({changed=ref x,...}: Prover) = x
    fun setProverChanged({changed=xRef,...}: Prover) x = (xRef := x)
    fun getProverContext({context=c,...}: Prover) = c
    fun getCurrentProofTree({proofrun=pr,...}: Prover) = ProofRun2ProofTree pr
    fun getCurrentProofRun({proofrun=pr,...}: Prover) = pr

    fun SubgoalID2Subgoal prover id = ID2Subgoal(getCurrentProofTree prover) id

    fun isInferenceRule name = isSome(findRuleByName name)
    fun isCommand name = isSome(findCommandByName name)
    fun isTactic name = isSome(findTacticByName name)

    fun printInProver prover str = 
	  printOutCommand (getOptions()) 
	                  ("prover", [UIstring(getProverID prover), UIstring str])


    (* Translate `args' into the `InferenceRuleArgument list' according
       to the `specs' of an inference rule arguments. 

       The way the args are translated: by default, we match the
       argument with the spec in the order they occur in the lists.
       If the current argument is a list, and the first element is a
       string and the second is "=", then it is a *named* argument,
       and the rest of the list is the value.  E.g., ["Gamma", "=", 1,
       2, 3] assigns "Gamma" the value [1,2,3].  If "Gamma" takes only
       scalar values, then only the first element of the list is
       taken. *)

    fun UIargs2IRargs(args: UserInputArgument list,
		      specs: InferenceRuleArgumentSpec list) =

	(* The war plan: take each argument in the list and try to
	   figure out the corresponding argument name.  Then
	   "match" the value, and construct the final IRArgValue.

	   Top level loop: (loop inorder args specs acc)
	   When inorder=true, we haven't seen any named arguments yet;
	   and if the argument is unnamed, pick the first spec and match it.
	   If inorder=false, the next argument MUST be named.
	   Each matched spec is removed from the `specs' list, not to be matched again.
	   All matched args are placed in `acc' list. *)

	let fun findSpecName specs name = 
	          List.find (fn s=> (IRArgSpecName s) = name) specs
	    (* match the argument with the spec and return `InferenceRuleArgument' *)
	    fun match(arg,spec) = 
		let val name = IRArgSpecName spec
		    val tp = IRArgSpecType spec
		    (* The actual translation of UIArg to IRArg *)
		    fun ff (UIstring str, ProverArgStringType) = ProverArgString str
		      | ff (UInumber n, ProverArgStringType) =
			  ProverArgString((if n >= 0 then "" else "-")^(Int.toString(abs n)))
		      | ff (UInumber n, ProverArgNumberType) = ProverArgNumber n
		      | ff (UIstring str, ProverArgSubgoalType) = ProverArgSubgoal str
		      | ff (arg, ProverArgFormulaNumType) = 
			  ProverArgFormulaNum(UIArg2FormulaIndex arg)
		      | ff (UIstring obj, ProverArgObjectType) =
			  ProverArgObject obj
		      | ff (UIstring obj, ProverArgFormulaType) =
			  ProverArgFormula obj
		      | ff (UIlist lst, ProverArgListType tp) =
			  ProverArgList(List.map (fn arg => ff(arg, tp)) lst)
		      | ff (arg, tp) = raise ProverError
			  ("The argument has the wrong type: "^(UIArg2string arg)
			   ^"\n  Expected: "^(InfRuleArgType2string tp))
		in
		    { name = name,
		      value = ff(arg,tp) }: InferenceRuleArgument
		end
	    (* Same as `match', only fish out the spec with the given
	       name.  Return the rest of the specs and the IR argument. *)
	    fun matchNamed(name, arg, specs) = 
		(case findSpecName specs name of
		     NONE => raise ProverError
			 ("Unknown or duplicate named argument: "
			  ^(UIArg2string(UIassoc(UIstring name, arg))))
		   | SOME s =>
		      let val newSpecs = List.filter(fn s=> (IRArgSpecName s) <> name) specs
		      in
			  (newSpecs, match(arg, s))
		      end)
	    fun loop _ [] _ acc = acc
	      | loop inorder (args as _::_) [] _ = raise ProverError
	         ("Unmatched arguments: ["
		  ^(strlist2str ", " (List.map UIArg2string args))
		  ^"]")
	      | loop _ ((UIassoc(UIstring name, arg))::args) specs acc =
		 let val (newSpecs, newArg) = matchNamed(name, arg, specs)
		 in
		     loop false args newSpecs (newArg::acc)
		 end
	      | loop true (arg::args) (spec::specs) acc =
		 loop true args specs ((match(arg,spec))::acc)
	      | loop false (arg::args) specs acc =
		 raise ProverError("All unnamed arguments must preceed named ones: "
				   ^(UIArg2string arg))
	in 
	    loop true args specs []
	end

    (* The reverse translation, needed for inlined tactics that return
       rules' parameters in IRargs format *)

    fun IRargs2UIargs [] = []
      | IRargs2UIargs(({name=name, value=v}: InferenceRuleArgument)::lst) = 
	let fun loop(ProverArgString str) = UIstring str
	      | loop(ProverArgNumber n) = UInumber n
	      | loop(ProverArgSubgoal str) = UIstring str
	      | loop(ProverArgFormulaNum fnum) = FormulaIndex2UIArg fnum
	      | loop(ProverArgObject str) = UIstring str
	      | loop(ProverArgFormula str) = UIstring str
	      | loop(ProverArgList lst) = UIlist(List.map loop lst)
	in
	    (UIassoc(UIstring name, loop v))::(IRargs2UIargs lst)
	end

    (* Installs a successfully applied `rule' with `args' at `subgoal'
       that generated the list of (seq, hints) `pairs' into the proof
       tree with `proofTreeID' *)

    fun installRule (prover, subgoal, rule, args, pairs) =
	let val debug = lazyVerbDebug(getOptions()) "installRule"
	    val proofRun = getCurrentProofRun prover
	    val proofTreeID = getProofTreeID(getCurrentProofTree prover)
	    (* Create a new subgoal for each sequent *)
	    fun newSubgoal(seq, hints) = 
		let val subgoal = makeSubgoal(SOME proofTreeID)
		in 
		    (subgoal, seq, hints)
		end
	    val _ = debug(fn()=>"installRule: generating new subgoals...")
	    val triples = List.map newSubgoal pairs
	    val _ = debug(fn()=>"done\n")
	    val proofRule = (rule, args)
	    fun triple2string(sub, seq, hints) =
		"("^(subgoal2stringDebug sub)
		^", sequent=[\n"^(sequent2string seq)
		^"]"
		^(case hints of
		      SOME h => ", hints=\""^(hints2string h)^"\""
		    | NONE => "")
		^")"
	    val subgoals = List.map(fn (s,_,h)=>(s,h)) triples
	    val _ = debug(fn()=>"\ninstallRule: triples:\n"
			  ^(strlist2str "\n" (List.map triple2string triples))
			  ^"\n")
	in (debug(fn()=>"installRule: installing new subgoals into the proof...");
	    setProverChanged prover true;
	    (* First add the subgoals to the proof tree *)
	    addProofRule(subgoal, proofRule, subgoals);
	    debug(fn()=> "\ninstallRule: new proof tree without new sequents:\n"
		  ^(ProofRun2stringDebug proofRun));
	    (* Then associate them with the sequents *)
	    List.app(fn (sub,seq,_)=>addSequent proofRun (sub, seq)) triples;
	    debug(fn()=> "\ninstallRule: new proof tree:\n"
		  ^(ProofRun2stringDebug proofRun));
	    subgoals)
	end

    (* Applies a rule to a subgoal and returns the InferenceRuleResult which
       contains new sequents upon success.  Doesn't try to generate
       new subgoals or install the rule. *)

    fun applyRuleSequents prover (command: ProverCommand) = 
	let val debug = lazyVerbDebug(getOptions()) "applyRule"
	    val { name=name,
		  args=args,
		  subgoal=subgoal } = command
	    val rule = (case findRuleByName name of
			    SOME r => r
			  | NONE => raise ProverError
				("No such proof rule: "^name
				 ^"\nType `help' for more information about rules, "
				   ^"commands, and tactics."))
	    val specs = getRuleArgs rule
	    val IRargs = UIargs2IRargs(args, specs)
	    val id = getProverID prover
	    val cxt = getProverContext prover
	    val proofRun = getCurrentProofRun prover
	    val _ = debug(fn()=>"\napplyRule: original proof tree:\n"
			  ^(ProofRun2stringDebug proofRun)^"\n")
	    val sequent = (case getSequent proofRun subgoal of
			       SOME s => s
			     | NONE => raise ProverError
				   ("This subgoal doesn't have a sequent"))
	    val printFun = printInProver prover
	in
	    ProofSystem.applyRule printFun (sequent, cxt) (rule, IRargs, NONE)
	end

    (* Tries to apply a rule to the subgoal, and if succeeds, installs
       the rule on the subgoal and returns the list of new generated
       subgoals; otherwise return NONE.  The current subgoal must have
       a sequent in the proof run.  The new subgoals are guaranteed to
       have sequents. *)
    fun applyRule prover (command: ProverCommand) = 
	let val debug = lazyVerbDebug(getOptions()) "applyRule"
	    val { name=name,
		  args=args,
		  subgoal=subgoal } = command
	    val id = getProverID prover
	    val printFun = printInProver prover
	    val hasProof = isSome(getProofRule subgoal)
	in
	    if hasProof then
		(printFun "This subgoal already has a proof.  Erase it first with `eraseproof'\n";
		 NONE)
	    else (case applyRuleSequents prover command of
		      InferenceRuleSuccess l =>
			  SOME(installRule(prover, subgoal, name, args, l))
		    | _ => NONE)
	end

    (* Run the proof system command on the current subgoal *)
    fun applyCommand prover (command: ProverCommand) =
	let val debug = lazyVerbDebug(getOptions()) "applyCommand"
	    val { name=name,
		  args=args,
		  subgoal=subgoal } = command
	    val comm = (case findCommandByName name of
			    SOME c => c
			  | NONE => raise ProverError
				("No such proof system command: "^name
				 ^"\nType `help' for more information about rules, "
				   ^"commands, and tactics."))
	    val specs = getCommandArgs comm
	    val IRargs = UIargs2IRargs(args, specs)
	    val id = getProverID prover
	    val cxt = getProverContext prover
	    val proofRun = getCurrentProofRun prover
	    val sequent = (case getSequent proofRun subgoal of
			       SOME s => s
			     | NONE => raise ProverError
				   ("This subgoal doesn't have a sequent"))
	    val printFun = printInProver prover
	in
	    ProofSystem.applyCommand printFun (sequent, cxt) (comm, IRargs)
	end

    (* Apply proof system specific tactic to the current subgoal and
       either expand the generated proof or collapse it into a single
       mega-step depending on the value of `inline'.
       In the sequentOnly mode (which implies NOT `inline') do not generate
       new subgoals or install the rules, just generate the list of new sequents.  
       Return the list of pairs (subgoalOpt, sequent) or NONE
       if the tactic doesn't apply. *)
    fun applyTacticCommon (sequentsOnly, inline) prover (command: ProverCommand) =
	let val debug = lazyVerbDebug(getOptions()) "applyCommand"
	    (* Make sure we do not inline in the `sequentOnly' mode *)
	    val inline = if sequentsOnly then false else inline
	    val { name=name,
		  args=args,
		  subgoal=subgoal } = command
	    val tactic = (case findTacticByName name of
			      SOME c => c
			    | NONE => raise ProverError
				  ("No such proof system tactic: "^name
				   ^"\nType `help' for more information about rules, "
				   ^"commands, and tactics."))
	    val specs = getTacticArgs tactic
	    val IRargs = UIargs2IRargs(args, specs)
	    val id = getProverID prover
	    val cxt = getProverContext prover
	    val proofRun = getCurrentProofRun prover
	    val sequent = (case getSequent proofRun subgoal of
			       SOME s => s
			     | NONE => raise ProverError
				   ("This subgoal doesn't have a sequent"))
	    val printFun = printInProver prover
	    (* Apply the tactic recursively until it finishes.  Parameters:
	       apply ([(subgoalOpt, seq, hintsOpt, stateOpt),...], changed, accum)
	       Return `((Subgoal option) * Sequent) list option' for the new 
	       unproven nodes generated by the tactic. *)
	    fun apply([], true, acc) = SOME acc  (* Finished with changes *)
	      | apply([], false, _) = NONE       (* Finished without changes *)
	      (* Tactic finishes with this sequent *)
	      | apply((subgoalOpt, seq, _, NONE)::lst, changed, acc) =
		  apply(lst, changed, (subgoalOpt, seq)::acc)
	      (* Tactic generates new seqs and continues *)
	      | apply((subgoalOpt, seq, hintsOpt, SOME state)::lst, changed, acc) =
		let fun inlineAndRecurse(sub, rule, args, pairs, stateOpt) =
		    (* Assume `InstallRule' preserves the order of `pairs' *)
		    let val newSubgoals = installRule(prover, sub, rule, args, pairs)
			fun pairs2tuple((sub, hints),(seq, _)) = (SOME sub, seq, hints, stateOpt)
			val tuples =
			    case zipOpt(newSubgoals, pairs) of
				SOME lst => List.map pairs2tuple lst
			      | NONE => raise SympBug
				    ("Prover/applyTactic/apply: #subgoals <> #seqs... Bizarre...")
		    in
			apply(tuples@lst, true, acc)
		    end
		    fun recurse(pairs, stateOpt) =
			let val tuples = List.map(fn (seq, h)=> (NONE, seq, h, stateOpt)) pairs
			in
			    apply(tuples@lst, changed, acc)
			end
		in
		    case (ProofSystem.applyTactic inline printFun 
			  (tactic, IRargs, cxt, SOME state)
			  (seq, hintsOpt)) of
			NONE => apply(lst, changed, (subgoalOpt, seq)::acc)
		      | SOME(rule, newIRargs, pairs, stateOpt) => 
			 if inline then
			     (case subgoalOpt of
				  NONE => raise SympBug
				      ("Prover/applyTactic/apply: no subgoal with `inline' on")
				| SOME sub =>
				      inlineAndRecurse(sub, rule, IRargs2UIargs newIRargs,
						       pairs, stateOpt))
			 else recurse(pairs, stateOpt)
		end
	in
	    apply([(SOME subgoal, sequent, NONE, NONE)], false, [])
	end

    (* Apply proof system specific tactic to the current subgoal and
       either expand the generated proof or collapse it into a single
       mega-step depending on the value of `inline'. *)
    fun applyTactic inline prover (command: ProverCommand) =
	let val debug = lazyVerbDebug(getOptions()) "applyCommand"
	    val { name=name,
		  args=args,
		  subgoal=subgoal } = command
	    (* Depending on `inline' (the first argument), install the
	       tactic and generate the resulting new subgoals *)
	    fun res2subgoals true lst = 
		let fun pair2sub(SOME sub, _) = (sub, NONE)
		      | pair2sub(NONE, _) = raise SympBug
		         ("Prover/applyTactic/res2subgoals: no subgoal for"
			  ^" a sequent when `inline' is on")
		in 
		    List.map pair2sub lst
		end
	      (* Tactic is not inlined, install it as a single rule *)
	      | res2subgoals false lst =
		let val seqPairs = List.map(fn(_, seq)=>(seq, NONE)) lst
		in 
		    installRule(prover, subgoal, name, args, seqPairs)
		end
	    (* Now, apply the tactic already *)
	    val resOpt = applyTacticCommon (false, inline) prover command
	in
	    Option.map (res2subgoals inline) resOpt
	end

    (* Rerun the proof installed at the current subgoal.  If the first
      argument `complete' is true, rerun the entire proof and rebuild
      the corresponding sequents.  Otherwise, skip parts of the proof
      that seem checked.  That is, if a subgoal has a sequent, a proof
      rule, and all of its children have sequents, then this rule's
      application is not rerun.

      The result is NONE if the proof fails at the first step (that
      is, it didn't change anything except for installing new proof),
      otherwise it's the list of new unproven subgoals.  In
      particular, if the given subgoal is proven, the result is
      SOME[].  *)

    fun runInstalledProof (complete, verbosity) prover subgoal = 
	let val funName = "runInstalledProof"
	    val _ = pushFunStackLazy(funName,
				     fn()=>"complete = "^(if complete then "true" else "false")
				     ^", verbosity = "^(Int.toString verbosity)
				     ^", subgoal = "^(subgoal2string subgoal))
	    val {changed=changed, ...} = prover
	    val proofRun = getCurrentProofRun prover
	    val printFun = printInProver prover
	    val proofSize = ProofTreeSize(getCurrentProofTree prover)
	    (* Count how many subgoals we have processed *)
	    val count = ref 0
	    val hash = makeHashDefault(op =, subgoalUID2string)
	    fun mark s = (insertHashDestructive(hash, getSubgoalUID s, s); ())
	    val options = getOptions()
	    val UI = get_interface options
	    fun isMarked s = isSome(findHash(hash, getSubgoalUID s))
	    fun lst2str lst = "["^(strlist2str ", " (List.map subgoal2string lst))^"]"
	    fun nDots n = String.implode(List.tabulate(n, fn _=> #"." ))

	    (* Do a DFS over the tree, rerun each proof rule/tactic as
	    necessary, then compare the result to what's in the tree.
	    If there is a difference, remove the current subtree and
	    install the new result.  Otherwise leave it alone and move
	    on.  This way we preserve the sharing of common subgoals
	    in the tree when possible. *)

	    fun loop complete subgoal =
		let val funName = funName^"/loop"
		    val _ = pushFunStackLazy(funName, fn()=>"complete="^(bool2str complete)
					     ^", "^(subgoal2string subgoal))
		    fun looplist complete [] = []
		      | looplist complete (s::lst) = 
			let val newS = loop complete s (* This insures the correct execution order *)
			    val rest = looplist complete lst
			in newS::rest
			end
		    (* Apply the rule to the sequent in the subgoal
		       and generate the list of pairs (sequent,hints),
		       or NONE if the rule doesn't apply *)
		    fun runStep(rule as (name, args)) =
			let val funName = funName^"/runStep"
			    val _ = pushFunStackLazy(funName, fn()=>
						     formatProverCommand UI
						      (name, List.map UIArg2string args))
			    val command = {name=name, args=args, subgoal=subgoal}
			    fun appRule() =
			        (case applyRuleSequents prover command of
				     InferenceRuleSuccess l => SOME l
				   | _ => NONE)
			    fun appTactic() = 
				let val seqsOpt = applyTacticCommon(true, false) prover command
				    fun ff(_,seq) = (seq, NONE: SequentHints option)
				in Option.map(List.map ff) seqsOpt
				end
			    fun resolveAndApply [] = raise ProverError
				  ("Can't find rule or tactic:\n  "
				   ^(formatProverCommand UI (name, List.map UIArg2string args)))
			      | resolveAndApply ((isX, applyX)::lst) =
				  if isX name then applyX() else resolveAndApply lst
			    val pairs = [(isInferenceRule, appRule),
					 (isTactic, appTactic)]
			    val seq = (case getSequent proofRun subgoal of
					   SOME seq => seq
					 | NONE => raise SympBug
					   (funName^": subgoal doesn't have a sequent: "
					    ^(subgoal2string subgoal)))
			    val _ = 
				(count := (!count)+1;
				 (case verbosity of
				      0 => ()
				    | 1 => lazyVerb options 
					  (fn()=>"running proof"
					   ^(nDots(((!count) * 65) div proofSize)))
				    | _ => printFun
					  ("\n"^(subgoal2string subgoal)^":\n\n"
					   ^(sequent2string seq)
					   ^"\nApplying rule:  "
					   ^(formatProverCommand UI
					       (name, List.map UIArg2string args))
					   ^"\n")))
			    val res = resolveAndApply pairs
			    val _ = (case res of
					 NONE => printFun 
					     "\nThe rule does not apply to this sequent\n"
				       | SOME lst => 
					   let val n = List.length lst
					   in if n > 1 then
					       printFun ("\nGenerated "^(Int.toString n)
							 ^" new subgoals\n")
					      else ()
					   end)
			    val _ = popFunStackLazy(funName, fn()=>"Done")
			in
			    res
			end

		    val childrenHints = getChildrenHints subgoal
		    val children = List.map #1 childrenHints

		    fun hintsOptEq(SOME h1, SOME h2) = hintsEq(h1, h2)
		      | hintsOptEq(NONE, NONE) = true
		      | hintsOptEq _ = false

		    fun compareOne(hints, pairs) =
			let fun loop([], acc, hints) = NONE
			      | loop((pair as (seq, h))::pairs, acc, hints) =
			        if hintsOptEq(h, hints) then
				    SOME(pair, (List.rev acc)@pairs)
				else loop(pairs, pair::acc, hints)
			in loop(pairs, [], hints)
			end

		    (* Compare the new pairs with what's in the actual
		       proof step, and either install the new proof
		       rule, or leave the original alone.  In either
		       case, return the list of children subgoals. *)
		    fun compareAndInstall((rule, args), pairs) =
			let val funName = funName^"/compareAndInstall"
			    fun pairs2str pairs =
				let fun pair2str(seq, NONE) = sequent2string seq
				      | pair2str(seq, SOME hints) = 
				           (sequent2string seq)^"["^(hints2string hints)^"]"
				in
				    "[\n "^(strlist2str "\n\n " (List.map pair2str pairs))
				    ^"\n]"
				end
			    val _ = pushFunStackLazy(funName, fn()=>
						     (formatProverCommand UI 
						      (rule, List.map UIArg2string args))
						     ^", "^(pairs2str pairs))
			    fun matchKids([], pairs, matched, unmatched) =
			          (List.rev matched, List.rev unmatched, pairs)
			      | matchKids((chPair as (ch,hints))::kids, pairs,
					  matched, unmatched) =
			        (case compareOne(hints, pairs) of
				     SOME(pair, rest) =>
					 matchKids(kids, rest, (chPair, pair)::matched, unmatched)
				   | NONE => matchKids(kids, pairs, matched, chPair::unmatched))
			    (* FIXME: this should be done somewhat smarter than that *)
			    fun guessMatches(l1, l2) = zipOpt(l1, l2)

			    fun installMatch((ch, hints), (seq, _)) =
				((case getSequent proofRun ch of
				      SOME s => if sequentEq(s, seq) then ()
						else raise SympBug 
		         ("runInstalledProof/compareAndinstall: merging proof branches with\n"
			  ^"non-matching sequents: handling of this case is not implemented yet.")
				    | NONE => addSequent proofRun (ch, seq));
				  ch)
			    val (matched, unmatched, pairs) =
				   matchKids(childrenHints, pairs, [], [])
			    val unmatchedNum = List.length unmatched
			    val pairsNum = List.length pairs
			    val _ = (case unmatched of
					 [] => ()
				       | _ => printFun
				    ("\n *** Warning: guessing matches for subgoals\n  "
				     ^(strlist2str "\n  " (List.map (subgoal2string o #1)
							   unmatched))
				     ^"\n"))
			    val _ = if unmatchedNum <> pairsNum then printFun
				("\n *** Warning: the number of new subgoals is "
				 ^(if pairsNum < unmatchedNum then "less" else "greater")
				 ^" than the number of\n"
				 ^"branches in the proof.  NOT recursing into these branches.\n")
				    else ()
			    val matchesOpt = 
				  Option.map(fn x=>matched@x)(guessMatches(unmatched, pairs))
			    val res = (case matchesOpt of
					   SOME matches => List.map installMatch matches
					 | NONE => (deleteSubtree subgoal;
						    changed := true;
						    List.map #1 (installRule(prover, subgoal,
									     rule, args, pairs))))
			    val _ = popFunStackLazy
				(funName, 
				 fn()=>"["^(strlist2str ", " (List.map subgoal2string res))^"]")
			in
			    res
			end
		    fun comment [] = printFun "\nThis completes the proof of the branch\n\n"
		      | comment _ = ()
		    val res = 
			if isMarked subgoal then [] (* Been there, done that *)
			else 
			 (mark subgoal;
			  if complete orelse not(isProofStepComplete proofRun subgoal) then
			      (* rerun the step *)
			      case getProofRule subgoal of
				     NONE => [subgoal]
				   | SOME rule => 
				      (case runStep rule of
					   (* Rule doesn't apply; delete the subproof and stop here *)
					   NONE => (deleteSubtree subgoal;
						    changed := true; [subgoal])
					 | SOME pairs => 
					    (comment pairs;
					     List.foldr(op @) []
					      (looplist true (compareAndInstall(rule, pairs)))))
			  (* Just recurse into the subgoals *)
			  else List.foldr(op @) [] (looplist complete children))
		    val _ = popFunStackLazy(funName, fn()=>lst2str res)
		in res
		end
	    val resList = loop complete subgoal
	    val res = if isProofStepComplete proofRun subgoal then SOME resList else NONE
	in
	    (popFunStackLazy(funName, fn()=>option2string "NONE" (Option.map lst2str res));
	     res)
	end

    (* Run the proof (ProofTree) on the given subgoal.  The current
       prover's proof tree is updated dynamically, step-by-step, so if
       the proof ultimately fails, at least some successful portion of
       it will be installed.  The result is NONE if the proof fails at
       the first step (that is, it didn't change anything except for
       installing new proof), otherwise it's the list of new unproven
       subgoals.  In particular, if the given subgoal is proven, the
       result is SOME[]. *)

    fun runProof verbosity prover (tree, subgoal) = 
	(installProofTree true (subgoal, tree);
	 runInstalledProof (true, verbosity) prover subgoal)

    (* I/O functions. *)

    (* If the subgoal has a sequent, it is printed along with the
       subgoal info *)

    fun subgoal2str prover subgoal =
	let val subInfo = Str((ProofTree.subgoal2string subgoal)^":\n\n")
	    val seq = (case getSequent (getCurrentProofRun prover) subgoal of
			   NONE => Str "No sequent for this subgoal"
			 | SOME s => sequent2str s)
	in Conc(subInfo, seq)
	end

    fun subgoal2string prover subgoal = Str2string(subgoal2str prover subgoal)

  (* Install the proof from the proof file *)
    fun installProofFileAt subgoal fileName =
	  installProofTree true (subgoal, file2ProofTree fileName)

    fun installProofFile prover fileName =
	let val root = getRoot(getCurrentProofTree prover)
	in
	    installProofFileAt root fileName
	end

    fun saveProof prover fileName =
	let val tree = getCurrentProofTree prover
	in
	    ProofTree2file(tree, fileName)
	end

  end
