(* Generic prover API.

   - Accepts a command to apply a rule or a strategy to a particular
     subgoal it the proof tree, checks if the rule applies, and
     modifies the tree

   - Redirect status requests to the proof tree module

   - General proof-search capabilities (in the future) *)

signature PROVER =
  sig
    structure Interface: INTERFACE
    structure ProofSystem: PROOF_SYSTEM
    structure InputLanguage: INPUT_LANGUAGE

    sharing InputLanguage = ProofSystem.Sequent.InputLanguage
    sharing InputLanguage.Options = Interface.Options

    type UserInput = Interface.UserInput
    type UserInputArgument = Interface.UserInputArgument
    type ProgramObject = InputLanguage.ProgramObject
    type ProgramContext = InputLanguage.ProgramContext
    type options = Interface.Options.options
    type InferenceRule = ProofSystem.Sequent.InferenceRule
    type InferenceRuleArgumentValue = ProofSystem.Sequent.InferenceRuleArgumentValue
    type InferenceRuleArgument = ProofSystem.Sequent.InferenceRuleArgument
    type InferenceRuleArgumentType = ProofSystem.Sequent.InferenceRuleArgumentType
    type InferenceRuleArgumentSpec = ProofSystem.Sequent.InferenceRuleArgumentSpec

    type Sequent = ProofSystem.Sequent.Sequent
    type SequentHints = ProofSystem.Sequent.SequentHints

    type Subgoal
    type SubgoalID
    type ProofTree
    type ProofRun
    (* ProofRule is an inference rule or tactics name (string) with
       specific arguments as the user gave them *)
    type ProofRule = string * (UserInputArgument list)

    (* The type of a prover as an object.  Provers are created
       dynamically for each attempt to prove a theorem.  Several
       provers may coexist in memory at the same time, and are
       completely independent. *)
    type Prover

    type ProverCommand = { name: string,
			   args: UserInputArgument list,
			   subgoal: Subgoal }

    (* Functions inherited from the ProofTree module.  The ProofTree
       module itself is not exported to prevent inapropriate changes
       to the proof tree. *)

    val getProofRule: Subgoal -> ProofRule option
    val getRoot: ProofTree -> Subgoal
    (* Return the entire proof tree this subgoal belongs to.
       This is NOT a subtree at this subgoal. *)
    val getProofTree: Subgoal -> ProofTree option
    val getChildren: Subgoal -> Subgoal list
    val getParents: Subgoal -> Subgoal list
    val getSubgoalID: Subgoal -> SubgoalID
    val SubgoalID2Subgoal: Prover -> SubgoalID -> Subgoal option
    (* Proof IDs are strings (e.g. theorem names) *)
    val getProofTreeID: ProofTree -> string

    val ProofRun2ProofTree: ProofRun -> ProofTree
    val getSequent: ProofRun -> Subgoal -> Sequent option

    (* A proof tree is considered "complete" when all leaves have an
       inference rule but no children *)
    val isProofTreeComplete: ProofTree -> bool

    (* A proof run is complete if the corresponding proof tree is
       complete, and every subgoal has an associated sequent *)
    val isProofRunComplete: ProofRun -> bool

    (* Find the next unproven subgoal after the given subgoal, if given *)
    val nextUnprovenSubgoal: ProofRun * (Subgoal option) -> Subgoal option
    (* Find all unproven subgoals and return them in a list *)
    val getLeafSubgoals: ProofTree -> Subgoal list
    val getUnprovenSubgoals: ProofRun -> Subgoal list

    (* Extract a ProofTree rooted at the subgoal, and optionally
       provide a new proof tree id (string).  This will create a clean
       copy of the subtree, so any further modifications to it will
       not affect the original tree. *)
    val extractSubtree: string option -> Subgoal -> ProofTree option

    (* Delete the subtree originating at the subgoal, but leave the subgoal. *)
    val deleteSubtree: Subgoal -> unit

    (* Now the actual prover functions *)

    (* Create a new prover object associated with a certain theorem
       and having an optional string ID (name) *)
    val makeProver: string option -> ProgramObject * ProgramContext -> Prover

    (* Functions for checking status *)

    val getProverID: Prover -> string
    (* Check if the proof has changed since last saved *)
    val getProverChanged: Prover -> bool
    (* Set the prover's `changed' flag *)
    val setProverChanged: Prover -> bool -> unit
    val getCurrentProofTree: Prover -> ProofTree
    val getCurrentProofRun: Prover -> ProofRun

    val isInferenceRule: string -> bool
    val isCommand: string -> bool
    val isTactic: string -> bool
    (* Check if the rule was applied correctly to the given subgoal
       (if any).  Free of side effects.  If the subgoal doesn't have a
       sequent or a rule, return false. *)
    (* Do we need this at all??? *)

(*      val checkProofStep: Prover -> Subgoal -> bool *)

    (* Tries to apply a rule to the subgoal, and if succeeds, installs
       the rule on the subgoal and returns the list of new generated
       subgoals; otherwise return NONE.  The current subgoal must have
       a sequent in the proof run.  The new subgoals are guaranteed to
       have sequents. *)

    val applyRule: Prover -> ProverCommand -> (Subgoal * (SequentHints option)) list option
    val applyCommand: Prover -> ProverCommand -> ProofSystem.ProofSystemCommandResult option
    (* The first argument is `inline'; when true, the found proof of
       the subgoal is inlined in the main proof *)
    val applyTactic: bool -> Prover -> ProverCommand -> (Subgoal * (SequentHints option)) list option

    (* runProof verbosity prover (tree, subgoal)

       Run the proof (ProofTree) on the given subgoal.  The current
       prover's proof tree is updated dynamically, step-by-step, so if
       the proof ultimately fails, at least some successful portion of
       it will be installed.  The result is NONE if the proof fails at
       the first step (that is, it didn't change anything except for
       installing new proof), otherwise it's the list of new unproven
       subgoals.  In particular, if the given subgoal is proven, the
       result is SOME[].

       Verbosity is the same as in `runInstalledProof'. *)

    val runProof: int -> Prover -> ProofTree * Subgoal -> Subgoal list option

    (* runInstalledProof (complete, verbosity) prover subgoal

      Rerun the proof installed at the current subgoal.  If `complete'
      is true, rerun the entire proof and rebuild the corresponding
      sequents.  Otherwise, skip parts of the proof that seem checked.
      That is, if a subgoal has a sequent, a proof rule, and all of
      its children have sequents, then this rule's application is not
      rerun.

      Verbosity: 0 - as silent as you can,
                 1 - print dots as you go,
	       >=2 - print sequents and rules as if the user typed them in. *)

    val runInstalledProof: bool * int -> Prover -> Subgoal -> Subgoal list option

    (* I/O functions. *)

    (* If the subgoal has a sequent, it is printed along with the
       subgoal info *)

    val subgoal2str: Prover -> Subgoal -> Str.Str
    val subgoal2string: Prover -> Subgoal -> string
    (* Print an ID of the subgoal (e.g. `1.4.5' like in PVS) *)
    val subgoalID2string: SubgoalID -> string

    (* Reverse translation from strings *)
    val string2subgoalID: string -> SubgoalID option

    (* Proof tree I/O functions must be one-to-one, so that a proof
       tree could be completely restored from the string or the
       `UserInputArgument' value *)
    val ProofTree2str: ProofTree -> Str.Str
    val ProofTree2string: ProofTree -> string
    (* For more efficient communication with GUI/Emacs/whatever *)
    val ProofTree2UI: ProofTree -> UserInputArgument

    (* And the reverse translation: build a proof tree from the
       `string' or `UserInputArgument'.  This, of course, may raise an
       exception if the input is invalid. *)

    val UI2ProofTree: UserInputArgument -> ProofTree
    val string2ProofTree: string -> ProofTree

    (* Install the ProofTree or its clean copy (if the first arg is
       true) at the Subgoal in the current proof tree.  The subgoal
       replaces the root subgoal of the new ProofTree.  The old
       subtree rooted at the subgoal is deleted.  *)
    val installProofTree: bool -> Subgoal * ProofTree -> unit

    (* Reads a proof file and installs it at the root.  May raise
       exceptions while parsing/openning the file *)
    val installProofFile: Prover -> string -> unit
    (* Installs a proof from file at a given subgoal. *)
    val installProofFileAt: Subgoal -> string -> unit

    val saveProof: Prover -> string -> unit

  end
