functor ProverInterfaceFun
    (structure Hash: HASH
     structure Interface: INTERFACE
     structure ProofSystem: PROOF_SYSTEM
     structure ProverInterfaceCommon: PROVER_INTERFACE_COMMON
     sharing Interface = ProofSystem.Sequent.SequentCommon.Interface)
	 : PROVER_INTERFACE =
  struct

    structure ProverInterfaceCommon = ProverInterfaceCommon

    structure Prover = ProverFun(structure Interface = Interface
				 structure ProofSystem = ProofSystem
				 structure Hash = Hash)

    structure StatusFile = StatusFileFun(structure Interface = Interface)

    open Str
    open SympOSPath
    open ProverInterfaceCommon
    open Prover
    open ProofSystem
    open InputLanguage
    open StatusFile
    open StatusFileStruct
    open Interface
    open Options
    open Hash
    open SympBug

    type ProverState =
	{ prover: Prover,
	  subgoal: Subgoal option ref, (* Current subgoal (optional) *)
	  theoremFile: string option ref, (* File with the proof *)
	  (* The theorem ID, it will be used for saving all the status info *)
	  theoremName: string ref,
	  (* Holds the most recently killed proof (sub)tree *)
	  proofKillRing: ProofTree option ref,
	  theorem: ProgramObject ref }

    type InterfaceState =
	{ provers: (string, ProverState) Hash, (* Set of active provers *)
	  (* Current file names *) 
	  inputFile: string option ref,
	  statusFile: string option ref,
	  status: StatusFileType ref, (* The global status of the program *)
	  context: ProgramContext option ref }
	  
    local 
	fun initState() = { provers=makeHashDefault(op =, fn x=>x),
			    inputFile = ref NONE,
			    statusFile = ref NONE,
			    status = ref [],
			    context = ref NONE }: InterfaceState
	val state as 
	    ref { provers=provers,
		  inputFile = inputFile,
		  statusFile = statusFile,
		  status = status,
		  context=context } = ref(initState())
    in

    (* Resets the entire state of the prover interface, kills all the
       running provers. *)
       fun ResetProverState() = (state := initState())
       fun setContext c = (context := c)
       fun getContext() = !context

       (* Returns the file's "stat" or NONE, if the file doesn't exist. *)
       fun statOpt name = SOME(Posix.FileSys.stat name) handle _ => NONE

       (* Return `true' if the file exists and is a regular file *)
       fun existsFile name = isSome(statOpt name)

       (* Returns `true' if file1 is newer than file2 (acc. to mtime).
	  Takes "stat" of each file *)

       fun fileNewer(file1,file2) = 
	   let open Posix.FileSys.ST
	   in 
	       Time.>(mtime file1, mtime file2)
	   end

       (* Replace the value of the first field matching `field' with the `value'.
	  If no such field found and add=true, then add a new field with this value.
	  If value=NONE, remove the field. *)

       fun changeField true (field, SOME value) [] = [SFfield(field, value)]
	 | changeField _ _ [] = []
	 | changeField add (field, value) ((f as SFfield(name, _))::lst) =
	   if name = field then 
	       (case value of
		    SOME v => (SFfield(name, v))::lst
		  | NONE => lst)
	   else f::(changeField add (field, value) lst)

       (* Same as above, only for the list of field/value pairs *)
       fun changeFields add [] fields = fields
	 | changeFields add (pair::lst) fields =
	     changeField add pair (changeFields add lst fields)	   

       (* Replace certain fields in sections that satisfy `ff'
	  predicate.  fieldVals: (string * (string option)) list - is
	  a list of pairs of (fieldName, newValue).  If the
	  `addSec: string option' is not NONE, add a new section with
	  fieldsVals if the section is not found.  If `addField: bool'
	  is true, add new fields from fieldsVals to the existing
	  section if they are missing.  *)

       fun changeSection ff (NONE, addField) fieldVals [] = []
	 | changeSection ff (SOME secName, _) fieldsVals [] =
	     [SFsection(secName,changeFields true fieldsVals [])]
	 | changeSection ff (addSec, addField) fieldVals ((s as SFsection(name,fields))::lst) =
	   if ff s then
	       (SFsection(name, changeFields addField fieldVals fields))
	       (* One section found, don't add more to the end *)
	       ::(changeSection ff (NONE, addField) fieldVals lst)
	   else s::(changeSection ff (addSec, addField) fieldVals lst)


       (* Find information in the status.  All functions are partial. *)
       fun findField field [] = NONE
	 | findField field ((SFfield(name, v))::lst) =
	     if field = name then SOME v else findField field lst

       fun findSection ff (lst: StatusFileType) = List.find ff lst
       fun findSectionName name = findSection(fn SFsection(n,_) => n = name)
       fun findSectionField ff field lst =
	   let val sec = findSection ff lst
	   in 
	       Option.mapPartial (fn SFsection(_, fields) => findField field fields) sec
	   end

       (* Find a particular field in the THEOREM named `name' *)
       fun findTheoremField (name, field) =
	   let fun theoremName name (SFsection("THEOREM",fields)) =
	            (case Option.map (fn n => n = name) (findField "name" fields) of
			 NONE => false
		       | SOME x => x)
		 | theoremName _ _ = false
	   in 
	       findSectionField (theoremName name) field (!status)
	   end
	   

       (* Same as changeSection, only for a specific section "THEOREM" with a given name.
	  If `add' is true, add a new THEOREM section if there is none for this theorem. *)

       fun changeTheoremSection add name fieldVals lst =
	   let fun theoremName (SFsection("THEOREM",fields)) =
	            (case Option.map (fn n => n = name) (findField "name" fields) of
			 NONE => false
		       | SOME x => x)
		 | theoremName _ = false
	       val addSec = if add then SOME "THEOREM" else NONE
	   in
	       changeSection theoremName (addSec, true) (("name",SOME name)::fieldVals) lst
	   end

       (* Read the `statusFile' and update the result according to the
	  mtime of the `inputFile' and proof files.  If the input file
	  is newer than the status file, mark all proven/unfinished
	  theorems as `unchecked'.  Otherwise for each proof file, if
	  it is newer than the status file, mark that theorem
	  `unchecked'. *)

       fun readStatusUpdate inputFile statusFile =
	   let val funName = "readStatusUpdate"
	       val debug = lazyVerbDebug(getOptions()) funName
	       val _ = pushFunStackLazy(funName, fn()=> statusFile)
	       val st = readStatusFile statusFile
	       val _ = debug(fn()=>"\n"^funName^": status = [\n"
			     ^(Str2string(StatusFile2str st))^"]\n")
	       val statusStat = 
		   case statOpt statusFile of
		       SOME s => s
		     | NONE => raise SympBug
			   ("readStatusUpdate: Can't stat the status file `"^statusFile^"'")
	       val inputStat = statOpt inputFile
	       fun isTheorem(SFsection("THEOREM",_)) = true
		 | isTheorem _ = false
	       (* For each theorem, check the mtime of its proof file
                  and update its status entry *)
	       fun checkProofs [] = []
		 | checkProofs ((sec as SFsection("THEOREM", fields))::lst) =
		    let val proofFile = findField "file" fields
			val proofStat = Option.mapPartial statOpt proofFile
			val _ = debug(fn()=>"proofFile = "
				      ^(case proofFile of
					    NONE => "NONE"
					  | SOME f => f)
				      ^", proofStat = "
				      ^(case proofStat of
					    NONE => "NONE"
					  | SOME _ => "SOME(stat)")
				      ^"\n")
			val newSec = 
			    (case proofStat of
				 (* If the file doesn't exist, remove the `file' field and 
				    update the status *)
				 NONE => SFsection("THEOREM",
						   changeFields true [("file", NONE),
								      ("status", SOME "untried")]
						   fields)
			       | SOME stat => 
				     if fileNewer(stat, statusStat) then
					 SFsection("THEOREM", 
						   changeField true ("status", SOME "unchecked") fields)
				     else sec)
			val _ = debug(fn()=>"sec =\n"^(Str2string (StatusFile2str [sec]))
				      ^"\nnewSec = \n"^(Str2string (StatusFile2str [newSec]))^"\n")
		    in
			newSec::(checkProofs lst)
		    end
		 | checkProofs (sec::lst) = sec::(checkProofs lst)
	   val newSt =
	       (case inputStat of
		    NONE => [] (* The input file doesn't exist, why bother with status? *)
		  | SOME stat => 
			if fileNewer (stat, statusStat) then
			    changeSection isTheorem (NONE, true) [("status", SOME "unchecked")] st
			else
			    checkProofs st)
	   val _ = popFunStackLazy(funName, fn()=>"[\n"^(Str2string(StatusFile2str newSt))^"]")
	   in
	       newSt
	   end

       fun saveStatus() =
	   let val file = (case (!statusFile, !inputFile) of
			       (SOME f, _) => f
			     | (NONE, SOME f) =>
				   let val {base=base,...} = OS.Path.splitBaseExt f
				       val file = OS.Path.joinBaseExt{base=base,ext=SOME "status"}
				   in (statusFile := SOME file; file)
				   end
			     | _ => raise SympError
				   ("Can't determine status file name.  Status is not saved."))
	   in 
	       writeStatusFile file (!status)
	   end

       (* Updates file and status fields in a "THEOREM" section with a
	  given name.  If no such section exists, create one. *)

       fun statusUpdateTheorem name (file, st) =
	   let val filePair = ("file", file)
	       val statusPair = ("status", st)
	   in 
	       status := changeTheoremSection true name [filePair, statusPair] (!status)
	   end

    (* A more powerful function that takes a file name,
       parses/typechecks the program in it, and installs it as the
       currrent context. *)
       fun typeCheckFile options file =
	   let val pt = ParseFile options file
	       (* The parsing was successful, set the file name *)
	       val _ = inputFile := SOME file
	       (* Also check for status file and read it if it exists *)
	       val statusFileName = file^".status"
	       val _ = 
		   (if existsFile statusFileName then
			status := readStatusUpdate file statusFileName
		    else ();
		    (* Set the status file name only if the reading succeeds *)
		    statusFile := SOME(statusFileName))
			handle SympError str => (printError options "parse" str)
			     | _ => (printError options "parse" 
				     ("Can't read status file "^statusFileName))
	       val _ = verb options ("Typechecking "^file^"...")
	       val prog = typeCheckProgram options pt
	       val _ = verb options ("Typechecking "^file^"...done")
	   in setContext(SOME prog)
	   end

    (* Lookup the status of a prover by its ID.  If there is no prover
       with such an ID, return NONE. *)
       fun getProverStatus id =
	   let fun status({ prover=prover,...}: ProverState) =
	           if isProofRunComplete(getCurrentProofRun prover) then Complete
		   else Incomplete
	   in Option.map status (findHash(provers, id))
	   end

       fun getProverStatusList id =
	   (case getProverStatus id of
		NONE => []
	      | SOME Complete => ["active","proven"]
	      | SOME Incomplete => ["active"])

       (* Create a new prover for a (typechecked) theorem with a `name'.
	  Return a new prover ID as a string. *)
       fun newProver name (thm, cxtOpt) =
	   let val cxt = (case (getContext(), cxtOpt) of
			      (NONE, _) => raise ProverError
				  ("No typechecked program to create new prover for")
			    | (SOME cxt, NONE) => cxt
			    | (SOME _, SOME cxt) => cxt)
	       val prover = makeProver (SOME name) (thm, cxt)
	       val id = getProverID prover
	       (* Find the proof file name from the global status *)
	       val proofFile = findTheoremField(name, "file")
	       (* Try to read the proof file; file = proofFile if successful,
		NONE otherwise *)
	       val file = (Option.map (installProofFile prover) proofFile; proofFile)
		   handle ProverError msg => 
		             (printOutCommand (getOptions())
			      ("prover", [UIstring id, UIstring msg]);
			      NONE)
			| SympBug str => raise SympBug str
			| _ => (printOutCommand (getOptions())
				("prover", [UIstring id,
					    UIstring("Can't read proof file: "
						     ^(case proofFile of
							   SOME s => s
							 | NONE => "")
						     ^"\n")]);
				NONE)
	       (* Compute the theorem status: if the proof file is read successfully,
		  then leave the status the same as it was, or set to "unchecked"
		  if status was undefined; otherwise "untried" (no proof file). *)
	       fun thmStatus name =
		   (case file of
			SOME _ =>
			    (case findTheoremField (name, "status") of
				 NONE => "unchecked"
			       | SOME s => s)
		      | NONE => "untried")
	       (* Update the global status, if the theorem has a name *)
	       val _ = statusUpdateTheorem name (file, SOME(thmStatus name))
	       (* Reset the `changed' status of the proof in the prover if the proof
		file has been read.  Otherwise the proof is `changed'. *)
	       val _ = setProverChanged prover (not(isSome file))
	       (* OK, wrap all up into the prover state *)
	       val state = { prover=prover,
			     subgoal=ref(SOME(getRoot(getCurrentProofTree prover))),
			     theoremFile = ref file,
			     theoremName = ref name,
			     proofKillRing = ref NONE,
			     theorem=ref thm }: ProverState
	   in (insertHashDestructive(provers, id, state); id)
	   end

    (* Similar to newProver, but parses the user input as a theorem in
       the current context and creates a new prover for it.

       proveTheorem options optional_name theorem_string => new prover ID *)
       fun proveTheorem options name (moduleOpt, theorem) =
	   let val debug = lazyVerbDebug options "proveTheorem"
	       val context =
	           (case (getContext(), moduleOpt) of
			(NONE, _) => raise ProverError
			    ("No typechecked program to create new prover for")
		      | (SOME cxt, NONE) => cxt
		      | (SOME cxt, SOME module) => 
			    let val moduleStr = contextKeyword^" "^module
				val _ = debug(fn ()=>
					      "\nProverInterface.proveTheorem: moduleStr = "
					      ^moduleStr^"\n")
				val modulePT = ParseInput options (TextIO.openString moduleStr)
			    in typeCheckContext options cxt modulePT
			    end)
	       val pt = ParseInput options (TextIO.openString(theoremKeyword^" "^theorem))
	       val thm = typeCheckTheorem options context pt
	   in newProver name (thm, SOME context)
	   end

       (* Print the current subgoal of the prover with the given proverID. *)
       fun printCurrentSubgoal options proverID =
	   (case findHash(provers,proverID) of
		NONE => raise ProverError("Prover with this ID doesn't exist: \""^proverID^"\"")
	      | SOME {prover=prover, subgoal=ref s,...} =>
		  let val str = (case s of
				     NONE => "No current subgoal\n"
				   | SOME s' => (subgoal2string prover s')^"\n")
		  in printOutCommand options("prover",[UIstring proverID, UIstring str])
		  end)

       fun getActiveProvers() = List.map(fn(x,_)=>x) (hash2any(fn x=>x)(fn x=>x) provers)

       (* List of prover command names with help strings *)
       val commandsHelp =
	   [("eraseproof", [],
	       "Erases the proof (if any) at the current subgoal and places it in the\n"
	       ^"temporary proof kill ring."),
	    ("exit", [],
	       "Exit the prover.  You will be prompted to save the current proof\n"
	       ^"if it has changed"),
	    ("goto", ["SUBGOAL"],
	       "Jump to the SUBGOAL in the current proof.  The subgoal can be\n"
	       ^"specified, e.g., as `theorem.1.3.2', or just `1.3.2'.  That is, the\n"
	       ^"theorem name can be omitted."),
	    ("help", ["&optional NAME1: string", "&optional NAME2: string", "..."],
	       "Print help on the commands and proof rules.\n"
	       ^"Without arguments, prints the list of available commands and rules."),
	    ("postpone", [],
	       "[arguments are ignored]\n"
	       ^"Leaves the current subgoal unproven and goes to the next unproven subgoal."),
	    ("refresh", [],
	       "[arguments are ignored]\n"
	       ^"Prints the current sequent.  Normally, the sequent is only printed\n"
	       ^"when it is changed by a proof rule, and after a few commands or failed\n"
	       ^"rule applications it may scroll too far up.  This command helps to\n"
	       ^"view the current sequent more conveniently."),
	    ("rerun", ["&optional FORCE: boolean"],
	       "Rerun the proof installed at the current sequent.\n" 
	       ^"If there are completed steps already and the FORCE parameter is not\n"
	       ^"present or false, then these steps are not rerun, and already existing\n"
	       ^"sequents are considered correct.  If FORCE is true, all the sequents\n"
	       ^"are recomputed."),
	    ("restoreproof", ["&optional FORCE: boolean"],
	       "Install the proof from the kill ring at the current subgoal.\n"
	       ^"If the subgoal already has a proof, do it only if the positive\n"
	       ^"argument is given (`t', `true', or `yes').  The original proof will then\n"
	       ^"be placed into the kill ring."),
	    ("save", ["&optional FILE: string"], 
	       "Saves the current proof into a file FILE.\n"
	       ^"If the FILE is not given, either take the file name specified\n"
	       ^"in the *.status file for the current theory, or asks the user."),
	    ("showproof", [],
	       "[arguments are ignored]\n\n"
	       ^"Show the entire current proof of the theorem."),
	    ("showsubproof", [],
	       "[arguments are ignored]\n\n"
	       ^"Show the current proof at the current subgoal."),
	    ("status", ["&optional QUIET: boolean"],
	       "Report the current status of the proof and the prover.\n"
	       ^"If QUIET is true, do not print anything in the prover, only update the\n"
	       ^"internal state of the user interface."),
	    ("subgoals", ["&optional PRINT_FLAG: boolean"],
	       "Print all unproven subgoals.  If a `positive' argument is given,\n"
	       ^"also print the sequents associated with each subgoal.\n"
	       ^"The set of `positive' arguments includes `t', `yes', and `print'.\n"
	       ^"Otherwise, only subgoal numbers are printed."),
	    ("why", [],
	       "[arguments are ignored]\n\n"
	       ^"Gives you The True Reason of why your last attempt has failed.\n"
	       ^"Take it with a grain of salt.  And pepper.\n"
	       ^"And go get another coffee or beer...")]

       val whyList = ["I don't know.",
		      "Because it's wrong.",
		      "Why not?",
		      "Hmm... I don't know.  Why are you asking me?"]
       val whyLength = List.length whyList

       fun isProverCommand name = List.exists (fn(n,_,_)=>n=name) commandsHelp

       local val seed = ref(0)
       in
	   fun tellWhy(nOpt) =
	       let fun random n = 
		   ((!seed) before seed := ((!seed)+1) mod n)
	       in (case nOpt of
		       NONE => List.nth(whyList, random whyLength)
		     | SOME n => List.nth(whyList, n mod whyLength))
	       end
       end

	   
       (* Save proof and run the continuation `cont' *)
       fun saveProofCont options (state: ProverState) args cont =
	   let val {theoremFile=fileRef,
		    theoremName=ref thmName,
		    prover=prover,...} = state
	       val thmFile = !fileRef
	       (* Save the file, update all the status info, and run the `cont' *)
	       fun save file = 
		   let val localDir = (case OS.Process.getEnv "PWD" of
					   SOME d => d
					 | NONE => "/")
		       val name =
			   if String.extract(file,0,SOME 2) = "~/" then
			       let val home = (case OS.Process.getEnv "HOME" of
						   SOME h => h
						 (* No home is set up, bail out *)
						 | NONE => raise ProverError
						       ("Saving proof in "^file^":\n  "
							^"Can't determine your home directory, "
							^"please specify the full path"))
			       in home^(String.extract(file,1,NONE))
			       end
			   else file
		       (* The absolute path to the file *)
		       val absPath = makePathAbsolute{path=name,relativeTo=localDir}
		       (* The name relative to the current dir, if it is under the current dir. *)
		       val localName =
			   let val s = makePathRelative{path=name,relativeTo=localDir}
			   in 
			       if String.extract(s,0,SOME 2) = ".." then name else s
			   end
		   in
		       (saveProof prover absPath;
			fileRef := SOME localName;
			status := changeTheoremSection true thmName [("file",SOME localName)] (!status);
			setProverChanged prover false;
			saveStatus();
			cont())
		   end
	       val file = (case (thmFile: string option, args) of
			       (_, [UIstring file]) => SOME file
			     | (SOME file, []) => SOME file
			     | (_, [x]) => raise ProverError ("Non-string argument to save")
			     | (_, _::_) => raise ProverError("save takes at most one argument")
			     | (NONE, _) =>
				   (userChoice options ("file","Save proof in file: ",[],save);
				    NONE))
	   in
	       (* If file = NONE, then `save' is postponed until the user's response. *)
	       case file of
		   SOME f => save f
		 | NONE => ()
	   end

       fun proverStatusList prover =
	   let val saved = not(getProverChanged prover)
	       val completed = isProofRunComplete(getCurrentProofRun prover)
	       val active = isSome(findHash(provers, getProverID prover))
	       fun wrap((c, s)::lst) =
		   if c then (UIstring s)::(wrap lst) else wrap lst
		 | wrap [] = []
	   in wrap[(active, "active"),
		   (saved, "saved"),
		   (completed, "proven")]
	   end
	       
       (* Print a string in the prover buffer *)
       fun printStr proverID str =
	    printOutCommand (getOptions()) ("prover", [UIstring proverID, UIstring str])
       (* Exit the prover and at the end run the continuation `cont' *)
       fun exitProver options (state as {prover=prover,
					 theoremName = ref thm, ...}: ProverState) cont =
	   let val proverID = getProverID prover
	       (* The actual exit from the prover *)
	       fun exit() =
		   (removeHashDestructive(provers, proverID);
		    printOutCommand options ("proverstatus",
					     (UIstring proverID)::(proverStatusList prover));
		    cont())
	       (* Asking the user THE LAST TIME when the proof is unsaved *)
	       fun sureYN "yes" = exit()
		 | sureYN _ = ()
	       (* Asking whether to save the proof *)
	       fun saveYN "yes" = saveProofCont options state [] exit
		 (* The user refused to save the proof.  Give him his last chance. *)
		 | saveYN _ = 
		   (userChoice options ("confirm",
					"You will lose all changes to the proof of "^thm^".  Proceed? ",
					[], sureYN); ())
	   in
	       if getProverChanged prover then
		   (userChoice options ("yesno", "The proof of "^thm^" has changed.  Save it? ",
					[], saveYN); ())
	       else exit()
	   end

       (* Determine if the UI argument means `true' or `yes' *)
       fun isTrueArg(UInumber 0) = false
	 | isTrueArg(UInumber _) = true
	 | isTrueArg(UIstring "t") = true
	 | isTrueArg(UIstring "yes") = true
	 | isTrueArg(UIstring "true") = true
	 | isTrueArg UItrue = true
	 | isTrueArg _ = false

       (* Apply a generic prover command *)
       fun applyProverCommand options (state as {prover=prover,...}: ProverState) (command, args) =
	   let val {interface=UI, ...} = options
	       val debug=lazyVerbDebug options "applyProverCommand"
	       val _ = debug(fn()=>"\napplyProverCommand: UI = "
			     ^(case UI of
				   commandLine => "commandLine"
				 | Emacs => "Emacs")^"\n")
	       val {prover=prover,
		    theoremFile=ref fileOpt,
		    theoremName=ref thm,
		    proofKillRing = proofKillRing,
		    subgoal=subgoalOpt, ...} = state
	       val proverID = getProverID prover
	       		 
	       fun loop("exit", []) = exitProver options state (fn()=>())
		 | loop ("exit", args) = raise ProverError
		     ("`exit' doesn't take any arguments, but was given this:\n  "
		      ^"("^(strlist2str " " (List.map UIArg2string args))^")")
		 | loop ("save",args) =
		     saveProofCont options state args (fn()=>())
		 | loop ("status",args) =
		     let fun verbose [] = true
			   | verbose [arg] = not(isTrueArg arg)
			   | verbose args = raise ProverError
			      ("STATUS command takes at most one argument: QUIET.")
			 fun makeReport() =
			     let val changed = getProverChanged prover
				 val completed = isProofRunComplete(getCurrentProofRun prover)
				 val completeProof = isProofTreeComplete(getCurrentProofTree prover)
				 val completionStr =
				     "The proof is "
				     ^(if completed then "completed"
				       else if completeProof then "unverified"
					    else "incomplete")
				 val savedStr = ("The proof has "^(if changed then "" else "NO")
						 ^" unsaved changes")
			     in
				 (strlist2str ";\n" [completionStr, savedStr])
			     end
			 val print = printStr proverID
		     in
			 (printOutCommand options ("proverstatus",
						   (UIstring proverID)::(proverStatusList prover));
			  if verbose args then print((makeReport())^"\n\n") else ())
		     end		      
		 | loop ("refresh",_) = printCurrentSubgoal options proverID
		 | loop ("why",_) = printOutCommand options
		     ("prover",[UIstring proverID, UIstring ("\n"^(tellWhy NONE)^"\n")])
		 | loop ("goto", args) =
		     let fun str2subID str =
			      (case string2subgoalID str of
				   SOME id => id
				 | NONE => (* Allow the user to omit the theorem name *)
				       (case string2subgoalID (thm^"."^str) of
					    SOME id => id
					  | NONE => raise ProverError
						("bad subgoal argument: "^str)))
			 fun UI2subID(UIstring sub) = str2subID sub
			   (* The user skipped the theorem name *)
			   | UI2subID(UInumber n) = str2subID(thm^"."^(Int.toString n))
			   | UI2subID x = raise ProverError
			     ("The first argument is not a subgoal: "^(UIArg2string x))
			 val subgoalID =
			     (case args of
				  [sub] => UI2subID sub
				| [] => raise ProverError
				      ("`goto' requires one argument - a subgoal")
				| sub::_ => (printStr proverID 
					     ("\nWarning: `goto' was given more than one "
					      ^"argument, only the first one was used\n");
					     UI2subID sub))
		     in
			 case SubgoalID2Subgoal prover subgoalID of
			     NONE => raise ProverError
				 ("no such subgoal in the proof tree: "
				  ^(subgoalID2string subgoalID))
			   | SOME subgoal =>
				 (subgoalOpt := SOME subgoal;
				  printCurrentSubgoal options proverID)
		     end
		 | loop ("postpone", args) =
		     let val subgoal = (case subgoalOpt of
					    ref(SOME sub) => sub
					  | ref NONE => raise ProverError
						("There is no current subgoal"))
		     in 
			 case nextUnprovenSubgoal(getCurrentProofRun prover, SOME subgoal) of
			     x as SOME _ => (subgoalOpt := x;
					     printCurrentSubgoal options proverID)
			   | NONE => printStr proverID "There are no other unproven subgoals"
		     end
		 | loop ("subgoals", args) =
		     let (* Whether to print the sequents with the subgoal numbers *)
			 val doPrint = (case args of
					    (UIstring "print")::_ => true
					  | arg::_ => isTrueArg arg
					  | _ => false)
			 fun sub2str s =
			     if doPrint then subgoal2string prover s
			     else subgoalID2string(getSubgoalID s)
			 fun printSub s = printStr proverID ("\n"^(sub2str s))
			 val subgoals = getUnprovenSubgoals (getCurrentProofRun prover)
		     in
			 case subgoals of
			     [] => printStr proverID "\nThere are no unproven subgoals\n"
			   | _ => (printStr proverID "\nUnproven subgoals:\n\n";
				   List.app printSub subgoals;
				   printStr proverID "\n\n")
		     end
		 | loop ("eraseproof", args) =
		     let val subgoal = (case subgoalOpt of
					    ref(SOME sub) => sub
					  | ref NONE => raise ProverError
						("There is no current subgoal"))
			 val proofOpt = extractSubtree NONE subgoal
		     in
			 case proofOpt of
			     SOME proof => (proofKillRing := proofOpt;
					    deleteSubtree subgoal;
					    setProverChanged prover;
					    printStr proverID 
					    ("\nThe proof has been moved to the kill ring. "
					     ^"Reinstall it with `restoreproof'.\n"))
			   | NONE => printStr proverID
				 ("\nThere is no proof at this subgoal.  "
				  ^"The proof kill ring is unchanged.")
		     end
		 | loop ("restoreproof", args) =
		     let val subgoal = (case subgoalOpt of
					    ref(SOME sub) => sub
					  | ref NONE => raise ProverError
						("There is no current subgoal"))
			 val proofOpt = extractSubtree NONE subgoal
			 val force = (case args of
					  arg::_ => isTrueArg arg
					| _ => false)
			 val UI = get_interface(getOptions())
		     in
			 case (proofOpt, !proofKillRing) of
			     (SOME current, SOME killed) =>
				 if force then (* swap killed and current proofs *)
				     (installProofTree true (subgoal, killed);
				      proofKillRing := proofOpt;
				      setProverChanged prover;
				      printStr proverID
				      ("\nThe proof is restored.  "
				       ^"The old proof is placed in the kill ring.\n"
				       ^"Run `"
				       ^(formatProverCommand UI ("restoreproof", ["true"]))
				       ^"' again if you want to put it back.\n"))
				 else printStr proverID
				  ("\nThis subgoal already has a proof.\n"
				   ^"Run `"
				   ^(formatProverCommand UI ("restoreproof", ["true"]))
				   ^"' to force the proof restoration.\n")
			   | (NONE, SOME killed) =>
				(installProofTree true (subgoal, killed);
				 setProverChanged prover;
				 printStr proverID "\nThe proof is restored.\n")
			   | (_, NONE) => printStr proverID
				  ("\nThere is no proof in the kill ring to restore.\n")
		     end
		 | loop ("showproof", args) =
		     let val proof = getCurrentProofTree prover
		     in 
			 printOutCommand options ("showproof", 
						  [UIstring proverID, ProofTree2UI proof])
		     end
		 | loop ("showsubproof", args) =
		     (case !subgoalOpt of
			  NONE => printOutCommand options 
			            ("prover", [UIstring proverID,
						UIstring "There is no current subgoal\n"])
			| SOME sub => 
			      (case extractSubtree NONE sub of
				   NONE => printOutCommand options
				       ("prover", [UIstring proverID,
						   UIstring "Current subgoal has no proof\n"])
				 | SOME proof => 
				       printOutCommand options 
				         ("showproof", [UIstring proverID, ProofTree2UI proof])))
		 | loop ("rerun", args) =
		    let val force = (case args of
					 [] => false
				       | [UItrue] => true
				       | [UIstring "t"] => true
				       | [UIstring "true"] => true
				       | [UIstring "yes"] => true
				       | [UIstring "force"] => true
				       | [_] => false
				       | _::_ => raise ProverError
					   ("RERUN takes at most one argument"))
			val subgoal = (case subgoalOpt of
					   ref(SOME s) => s
					 | ref NONE => raise ProverError
					   ("No current subgoal"))
			val print = printStr proverID
		    in
			(case runInstalledProof(force, 2) prover subgoal of
			     SOME[] =>
			      (print("\nThe rerun proof proves the original subgoal\n");
			       subgoalOpt := nextUnprovenSubgoal(getCurrentProofRun prover,
								 SOME subgoal);
			       
			       printCurrentSubgoal options proverID)
			   | SOME(s::_) =>
			      (subgoalOpt := SOME s;
			       printCurrentSubgoal options proverID)
			   | NONE => print("\nThe installed proof doesn't apply to the subgoal\n"))
		    end
		 | loop ("help",args) =
		     let val comm = formatProverCommand UI
			 fun getHelp(UIstring name) =
			 let fun getHelpByName name = 
			         let fun formatHelp(name, args, help) =
				          (comm(name, args))^"\n\n"^help
			         in
				     Option.map formatHelp
				        (List.find (fn(n,_,_)=>n=name) commandsHelp)
				 end
			     fun getRuleHelpByName name =
				 Option.map getRuleHelp (findRuleByName name)
			     fun getCommandHelpByName name =
				 Option.map getCommandHelp (findCommandByName name)
			     fun getTacticHelpByName name =
				 Option.map getTacticHelp (findTacticByName name)
			     fun loop [] = "There is no rule or command named "^name
			       | loop ((ff,str)::lst) =
				 (case ff name of
				      SOME help => name^" is "^str^":\n\n"^help
				    | NONE => loop lst)
			 in
			     loop[(getHelpByName, "a general prover command"),
				  (getRuleHelpByName, "an inference rule"),
				  (getCommandHelpByName, "a command specific to the proof system"),
				  (getTacticHelpByName, "a tactic specific to the proof system")]
			 end
			   | getHelp a = "Argument to `help' must be a string: "^(UIArg2string a)
			 fun commandList() =
			     let val commands = List.map #1 commandsHelp
				 val rules = List.map getRuleName (getAllRules())
				 val tactics = List.map getTacticName (getAllTactics())
				 val PScommands = List.map getCommandName (getAllCommands())
			     in
				 "General Prover Commands:\n\n  "
				 ^(strlist2str "\n  " commands)
				 ^"\n\nProof Rules:\n\n  "
				 ^(strlist2str "\n  " rules)
				 ^"\n\nProof Tactics:\n\n  "
				 ^(strlist2str "\n  " tactics)
				 ^"\n\nProof System Commands:\n\n  "
				 ^(strlist2str "\n  " PScommands)
			     end
			 val help = (case args of
					 [] =>
			   ((commandList())^"\n\n"
			    ^"To learn more about particular commands and proof rules, type\n\n"
			    ^(comm ("help", ["command-or-rule1", "command-or-rule2", "..."]))
			    ^"\n\n"
			    ^(comm("help", ["help"]))^" for more details about help.")
				       | _ => strlist2str "\n\n" (List.map getHelp args))
		     in
			 printOutCommand options ("prover",[UIstring proverID,
							    UIstring ("\n"^help^"\n\n")])
		     end
		 | loop (name, args) = raise SympBug
		     ("applyProverCommand: no such command: "^name)
	   in
	       loop(command, args)
	   end

       fun localCommand options state (command, args) =
	     let val {prover=prover,
		      subgoal=subgoalOpt, ...} = state
		 val proverID = getProverID prover
		 fun proverCommand() =
		     let val subgoal = (case !subgoalOpt of
					    NONE => raise ProverError
						("Can't apply rule: no current subgoal")
					  | SOME s => s)
		     in
			 { name=command,
			   args=args,
			   subgoal=subgoal }: ProverCommand
		     end
		 (* Common functionality for rules and tactics *)
		 fun handleNewSubgoals subgoal newSubgoals =
		     (case newSubgoals of
			  (* The rule failed *)
			  NONE => printOutCommand options
			      ("prover",
			       [UIstring proverID,
				UIstring "\nThe rule does not apply to this sequent\n"])
			(* The subgoal is proven *)
			| SOME [] => 
			      (* First, check if we finished the theorem *)
			      if isProofRunComplete(getCurrentProofRun prover) then
				  (* Later we need to save the proof and, perhaps, exit the prover *)
				  (subgoalOpt := NONE;
				   printStr proverID "\nQ.E.D.\n")
			      else
			       (subgoalOpt := nextUnprovenSubgoal(getCurrentProofRun prover,
								  SOME subgoal);
				(printStr proverID "This completes the proof of the branch\n");
				printCurrentSubgoal options proverID)
			| SOME(lst as (s,_)::_) => 
			    let val n = List.length lst
			    in
				(subgoalOpt := SOME s;
				 (if n>1 then 
				      printStr proverID ("\nGenerated "^(Int.toString n)
							 ^" new subgoals\n")
				  else ());
				 printCurrentSubgoal options proverID)
			    end)
		 fun appProverCommand() = applyProverCommand options state (command, args)
		 fun appRule() = 
		     let val command as {subgoal=subgoal,...} = proverCommand()
		     in handleNewSubgoals subgoal (Prover.applyRule prover command)
		     end
		 fun appTactic() =
		     let val command as {subgoal=subgoal,...} = proverCommand()
		     in handleNewSubgoals subgoal (Prover.applyTactic false prover command)
		     end
		 fun appCommand() = (Prover.applyCommand prover (proverCommand()); ())
		 fun tryApply [] = 
		       printStr proverID (command
					  ^": no such inference rule, tactic, or command.\n")
		   | tryApply ((isX, appX)::lst) =
		       if isX command then
			   (printOutCommand options ("proverstatus",
						     [UIstring proverID, UIstring "busy"]
						     @(proverStatusList prover));
			    appX();
			    printOutCommand options ("proverstatus",
						     (UIstring proverID)
						     ::(proverStatusList prover)))
		       else tryApply lst
		 val applyList = [(isProverCommand, appProverCommand),
				  (isInferenceRule, appRule),
				  (isTactic, appTactic),
				  (isCommand, appCommand)]
	     in
		 tryApply applyList
		 handle ProverError str => printStr proverID ("\nError: "^str^"\n")
	     end
       (* Exit all active provers and run the continuation `cont' *)
       fun exitAllProvers options cont =
	   let val proverIDs = getActiveProvers()
	       (* Exit the list of provers one by one *)
	       fun exitList [] = cont()
		 | exitList (id::lst) =
		     (case findHash(provers, id) of
			  (* This shouldn't happen, but for completeness, anyway *)
			  NONE => exitList lst
			| SOME state =>
			      exitProver options state (fn () => exitList lst))
	   in
	       exitList proverIDs
	   end

       (* Interpretation of interactive commands *)
       fun globalCommand options ("reset", []) = 
	     exitAllProvers options ResetProverState
	 | globalCommand _ ("reset", args) = raise ProverError
	     ("`reset' doesn't take any arguments, but was given this:\n  "
	      ^"("^(strlist2str " " (List.map UIArg2string args))^")")
	 | globalCommand options ("typecheck", _) = ()
	 | globalCommand _ (c,_) = raise ProverError
	     ("Unknown global prover command: "^c)

       (* All the interactive communication with the provers goes
	  through this function.  When proverID is NONE, the command
	  is targeted to the global state rather than a particular
	  prover.

	  ProverCommand options (proverID, command, arguments) *)

       fun ProverCommand options (id, command, args) =
	   (case id of
		(* The command is global *)
		NONE => globalCommand options (command, args)
	      | SOME x => 
		   (case findHash(provers,x) of
			NONE => raise ProverError("Prover with this ID doesn't exist: \""^x^"\"")
		      | SOME state => localCommand options state (command, args)))

    end

  end (* struct *)
