signature PROVER_INTERFACE =
  sig
    structure Interface: INTERFACE
    structure ProverInterfaceCommon: PROVER_INTERFACE_COMMON
    structure InputLanguage: INPUT_LANGUAGE

    type ProverStatus = ProverInterfaceCommon.ProverStatus
    type options = Interface.Options.options
    type ProgramContext = InputLanguage.ProgramContext
    type ProgramObject = InputLanguage.ProgramObject

    (* Resets the entire state of the prover interface, kills all the
       running provers. *)

    val ResetProverState: unit -> unit

    (* Set a new proof context (or remove it if it is NONE).  This
       operation will terminate all the running provers and reset the
       state of the prover interface.

       At any given time there is at most one proof context, and it is
       usually the abstract representation of the typechecked program.
       However, in principle, it could be any context within a
       program.  The context is stored in the global state of the
       prover interface and is shared amoung all the running
       provers. *)

    val setContext: ProgramContext option -> unit

    (* A more powerful function that takes a file name,
       parses/typechecks the program in it, and installs it as the
       currrent context. *)

    val typeCheckFile: options -> string -> unit

    (* This basically checks whether we have a working context or not. *)
    val getContext: unit -> ProgramContext option

    (* Lookup the status of a prover by its ID.  If there is no prover
       with such an ID, return NONE. *)
    val getProverStatus: string -> ProverStatus option

    (* Return the list of status flags for the prover.  A status flag
       can currently be "active", "proven", and "saved".  If flag is
       not present, its negation is assumed.  So, when the list is
       empty, the prover is not active. *)
    val getProverStatusList: string -> string list

    (* Create a new prover for a (typechecked) theorem with a name
       `name'.  Return a new prover ID as a string. *)

    val newProver: string -> ProgramObject * (ProgramContext option) -> string

    (* Similar to newProver, but parses the user input as a theorem in
       the current context and creates a new prover for it.

       proveTheorem options name theorem_string => new prover ID *)

    val proveTheorem: options -> string -> (string option) * string -> string

    (* Print the current subgoal of the prover with the given proverID.
       printCurrentSubgoal options proverID. *)
    val printCurrentSubgoal: options -> string -> unit

    (* Return the list of IDs for all active provers *)
    val getActiveProvers: unit -> string list


       (* All the interactive communication with the provers goes
	  through this function.  When proverID is NONE, the command
	  is targeted to the global state rather than a particular
	  prover.

	  ProverCommand options (proverID, command, arguments) *)
    val ProverCommand: options
	  -> (string option) * string * Interface.UserInputArgument list
	  -> unit

    (* exitAllProvers options cont
       Exit all active provers and run the continuation `cont' *)
    val exitAllProvers: options -> (unit -> unit) -> unit

  end
