(* Parser of the proof tree file format for the SyMP prover *)

structure ProofTreeStruct = ProofTreeStruct
structure ParseError = ParseError
open ProofTreeStruct
open ParseError

(* Datatype for collecting elements of a subgoal ID *)
datatype StringOrNumber = StrValue of string | NumValue of int

fun StringOrNumber2str(StrValue s) = s
  | StringOrNumber2str(NumValue n) = Int.toString n

fun symbols2str [] = ""
  | symbols2str [v] = StringOrNumber2str v
  | symbols2str (v::lst) = (StringOrNumber2str v)^"."^(symbols2str lst)

(* Check that the subgoal ID is well-formed *)
fun symbols2subgoalID lst =
    (* loop(symbols, name, intlist) *)
    let fun loop ([], _, []) = raise ParseError 
             ("Syntax Error: subgoalID doesn't have a number: "
              ^(symbols2str lst))
          | loop ([], name, intlist) = (name, List.rev intlist)
          | loop ((NumValue n)::_, "", _) = raise ParseError
             ("Syntax Error: subgoalID doesn't have a name: "
              ^(symbols2str lst))
          | loop ((NumValue n)::lst, name, intlist) = loop(lst, name, n::intlist)
          | loop ((StrValue s)::lst, name, []) = loop(lst, name^"."^s, [])
          | loop ((StrValue _)::_, _, _) = raise ParseError
             ("Syntax Error: subgoalID is malformed: "
              ^(symbols2str lst))
    in loop(lst, "", [])
    end

%%

%name ProofTree


%verbose
%pure
%header (functor ProofTreeLrValsFun
           (structure Token : TOKEN
            structure ParseError: PARSE_ERROR
            structure ProofTreeStruct: PROOF_TREE_STRUCT)
  : sig structure ParserData : PARSER_DATA
            where type result = ProofTreeStruct.ProofTreeType;
        structure Tokens : ProofTree_TOKENS
   end)

%eop EOF
%pos int
%noshift EOF

%term 
  STRING of string
| SYMBOL of string
| NUMBER of int
| DOT
| COMMA
| LP
| RP
| LB
| RB
| EQ
| EOF

%nonassoc STRING SYMBOL NUMBER DOT EQ
%nonassoc COMMA LP RP LB RB EOF

%nonterm begin of ProofTreeType
| subgoals of ProofTreeType list
| subgoal of ProofTreeType
| children of ProofTreeType list
| child of ProofTreeType
| rule of ProofTreeType
| ruleargs of ProofTreeType list
| rulearg of ProofTreeType
| subgoalid of ProofTreeType
| symbol of StringOrNumber
| symbols of StringOrNumber list

%start begin

%%

begin       : subgoals (PTproof subgoals)
subgoals    : ([])
            | subgoal subgoals (subgoal::subgoals)
subgoal     : LP subgoalid RP (PTsubgoal(subgoalid, NONE, NONE))
            | LP subgoalid rule RP (PTsubgoal(subgoalid, SOME(rule), NONE))
            | LP subgoalid LP children RP RP (PTsubgoal(subgoalid, NONE, SOME children))
            | LP subgoalid rule  LP children RP RP (PTsubgoal(subgoalid, SOME rule, SOME children))
rule        : SYMBOL LP ruleargs RP (PTrule(SYMBOL, ruleargs))
ruleargs    : ([])
            | rulearg ([rulearg])
            | rulearg COMMA ruleargs (rulearg::ruleargs)
rulearg     : STRING (PTstring(STRING))
            | SYMBOL (PTsymbol(SYMBOL))
            | NUMBER (PTnumber(NUMBER))
            | LB ruleargs RB (PTlist ruleargs)
            | SYMBOL EQ rulearg (PTassoc(SYMBOL,rulearg))
subgoalid   : symbols (PTsubgoalID(symbols2subgoalID symbols))
symbol      : SYMBOL (StrValue SYMBOL)
            | NUMBER (NumValue NUMBER)
symbols     : symbol ([symbol])
            | symbol DOT symbols (symbol::symbols)
children    : ([])
            | child children (child::children)
            | child COMMA children (child::children)
child       : subgoalid (PTchild(subgoalid, NONE))
            | subgoalid LB STRING RB (PTchild(subgoalid, SOME STRING))
