functor ProofTreeFun(structure ProofSystem: PROOF_SYSTEM
		     structure Hash: HASH
		     structure Interface: INTERFACE
		     sharing ProofSystem.Sequent.InputLanguage.Options = Interface.Options): PROOF_TREE =
  struct
    structure ProofSystem = ProofSystem
    structure Sequent = ProofSystem.Sequent
    structure Interface = Interface
    structure Hash = Hash

    open ProofSystem
    open Sequent
    open SequentCommon
    open SequentBase
    open Hash
    open Interface
    open SympBug
    open Str
    open ListOps

    (* ProofRule is an inference rule or tactic name with specific
       arguments as the user gave them *)
    type ProofRule = string * (UserInputArgument list)

    (* ID is a pair (pid, subgoal #), where pid is a unique ID of the
       proof tree, and the subgoal # is a numbering of subgoals inside
       that proof tree. *)

    type SubgoalID = string * (int list)
    type SubgoalUID = int

    datatype Subgoal = Subgoal of { id: SubgoalID ref,

       (* When true, any child of this subgoal will get an ID by
	  extending it with one more number, even if it's the only child.  When
	  false, the only child will increment the last number in the list. *)
				    extend: bool ref,
				    (* Unique id - everything internally should use it,
				       not the id, which may change with time *)
				    uid: SubgoalUID,
				    proofRule: ProofRule option ref,
				    children: (SubgoalUID * (SequentHints option)) list ref,
				    parents: SubgoalUID list ref,
				    prooftree: ProofTree option ref }

    and ProofTree = ProofTree of { root: Subgoal,
				   hash: (SubgoalUID, Subgoal) Hash }

    type ProofRun = { prooftree: ProofTree,
		      (* Associate subgoal IDs with sequents *)
		      assocHash: (SubgoalUID, Sequent) Hash }

    (* Compare subgoals by UIDs *)
    fun SubgoalEq(Subgoal{uid=uid1,...}, Subgoal{uid=uid2,...}) = (uid1 = uid2)

    (* Return the number of subgoals in the tree *)
    fun ProofTreeSize(ProofTree{hash=hash, ...}) =
	  List.length(hash2any(fn x=>x)(fn x=>x) hash)

    (* If `old' is a prefix of `lst', replace it by `new' and return
       the new list; otherwise return NONE *)
    fun replacePrefix(old, new) lst = 
	let fun loop([], new) lst = SOME(new@lst)
	      | loop(x::old, new) (y::lst) =
	    if x = y then loop(old, new) lst
	    else NONE
	      | loop (x::old, new) [] = NONE
	in
	     loop(old, new) lst
	end

    fun Option_app ff (SOME x) = ff x
      | Option_app ff NONE = ()

    fun subgoalID2string (s,lst) = strlist2str "." (s::(List.map Int.toString lst))
    fun subgoalUID2string uid = Int.toString uid
    fun subgoal2string(Subgoal{id=ref id,...}) = subgoalID2string id
    fun subgoal2stringDebug(Subgoal{id=ref id, uid=uid, extend=ref ex, ...}) =
	  (subgoalID2string id)^"["^(subgoalUID2string uid)^", extend="
	  ^(case ex of true => "true" | false => "false")^"]"

    (* Reverse translation of string to subgoal ID *)
    fun string2subgoalID str =
	let fun isDigit ch = List.exists(fn x=>x = ch)(String.explode "0123456789")
	    (* Collect all elements in lst till the first `.' with a 
	       digit after it, and return (elements, rest-after-dot) *)
	    fun nextChunk [] acc = (String.implode(List.rev acc), [])
	      | nextChunk (#"."::ch::lst) acc =
		if isDigit ch then (String.implode(List.rev acc), ch::lst)
		else nextChunk(ch::lst)(#"."::acc)
	      | nextChunk (ch::lst) acc = nextChunk lst (ch::acc)
	    fun splitString [] acc = List.rev acc
	      | splitString lst acc =
		let val (str, rest) = nextChunk lst []
		in splitString rest (str::acc)
		end
	    fun strings2nums [] acc = SOME(List.rev acc)
	      | strings2nums (str::lst) acc = 
		  Option.mapPartial(fn n=>strings2nums lst (n::acc)) (Int.fromString str)
	    fun isName str =
		let val lst = String.explode str
		in
		    case lst of
			ch::_ => not(isDigit ch)
		      | _ => false
		end
	in
	    case splitString(String.explode str) [] of
		[] => NONE
	      | name::nums =>
		    if isName name then
			Option.map(fn nums=> (name, nums): SubgoalID)(strings2nums nums [])
		    else NONE
	end

    fun getProofRule(Subgoal{proofRule=rule,...}) = !rule
    fun getRoot(ProofTree{root=root,...}) = root
    fun getProofTree(s as Subgoal{prooftree=tree,...}) = !tree
    fun getChildren(Subgoal{children=ch, prooftree=ref(SOME(ProofTree{hash=hash, ...})), ...}) =
	  List.mapPartial (fn (x,_) => findHash(hash,x)) (!ch)
      | getChildren x = raise ProverError
	  ("Can't list children of a subgoal without a proof tree: "
	   ^(subgoal2string x))
    fun getChildrenHints(Subgoal{children=ch,
				 prooftree=ref(SOME(ProofTree{hash=hash, ...})), ...}) =
	  List.mapPartial (fn (id,hints) => Option.map (fn s=>(s,hints))(findHash(hash,id))) (!ch)
      | getChildrenHints x = raise ProverError
	  ("Can't list children with hints of a subgoal without a proof tree: "
	   ^(subgoal2string x))
    fun getParents(Subgoal{parents=ref parents,
			   prooftree=ref(SOME(ProofTree{hash=hash, ...})),...}) =
	  List.mapPartial (fn x=>findHash(hash,x)) parents
      | getParents x = raise ProverError
	  ("Can't list parents of a subgoal without a proof tree: "
	   ^(subgoal2string x))
    fun getSubgoalID(Subgoal{id=ref id,...}) = id
    fun getSubgoalUID(Subgoal{uid=uid,...}) = uid
    fun getSubgoalExtend(Subgoal{extend=ref x, ...}) = x
    fun getProofTreeID p = #1 (getSubgoalID(getRoot p))
    (* Proof tree is uniquely identified by the UID of its root subgoal *)
    fun getProofTreeUID p = getSubgoalUID(getRoot p)

    fun ProofRun2ProofTree({prooftree=tree,...}: ProofRun) = tree

    (* List all the subgoals in the tree in a DFS order, starting from the root. *)
    fun listSubgoals tree =
	let val root = getRoot tree
	    val markHash = makeHashDefault(op =, subgoalUID2string)
	    fun mark id = insertHashDestructive(markHash, id, ())
	    fun loop sub acc =
		let val (uid, ch) = (getSubgoalUID sub, getChildren sub)
		in (case findHash(markHash, uid) of
			SOME _ => acc
		      | NONE => (mark uid; looplist (List.rev ch) (sub::acc)))
		end
	    and looplist [] acc = acc
	      | looplist (sub::lst) acc =
		  loop sub (looplist lst acc)
	in 
	    List.rev(loop root [])
	end

    fun ProofTree2strCommon (verbose, runOpt) tree = 
	let fun op *(x,y) = Conc(x,y)
	    val subgoals = listSubgoals tree
	    fun showSeq NONE uid = NONE
	      | showSeq(SOME run) uid =
		let val {assocHash=hash, ...}: ProofRun = run
		in
		    SOME(Str("\n"
			     ^(case findHash(hash, uid) of
				   NONE => "subgoal has no sequent"
				 | SOME seq => sequent2string seq)
			     ^"\n"))
		end

	    fun doSubgoal sub = 
		 let val id = getSubgoalID sub
		     val uid = getSubgoalUID sub
		     fun idStr(id, uid) = 
			 Str((if verbose then
				  (subgoalID2string id)^"["^(subgoalUID2string uid)^"]"
			      else (subgoalID2string id))
			     ^" ")
		     fun subStr sub = 
			 Str((if verbose then subgoal2stringDebug sub
			      else (subgoal2string sub))
			     ^" ")
		     fun rule2str (r, args) =
			  (Str (r^"("))
			  *(Strlist2Str ", " (List.map (fn x=>Str (UIArg2string x)) args))
			  *(Str ")")
		     val ruleStr = Option.map rule2str (getProofRule sub)
		     fun child2str (ch,hints) =
			 (subStr ch)
			  *(Str (case hints of
				     NONE => ""
				   | SOME h => "[\""^(hints2string h)^"\"]"))
		     val children = List.map child2str (getChildrenHints sub)
		     val childrenStr =
			  Option.map (fn _=> (Str "(" )*(Strlist2Str ", " children)*(Str ")"))
			             ruleStr
		 in Strlist2Str " " (List.mapPartial (fn x=>x) 
				     [SOME(Str "("),SOME(idStr(id,uid)),
				      showSeq runOpt uid,
				      ruleStr,childrenStr,SOME(Str ")\n"),
				      Option.map(fn _ => Str "\n") runOpt])
		 end
	in 
	    Strlist2Str "" (List.map doSubgoal subgoals)
	end

    val ProofTree2str = ProofTree2strCommon(false,NONE)
    fun ProofTree2string tree = Str2string(ProofTree2str tree)

    val ProofTree2strDebug = ProofTree2strCommon(true,NONE)
    fun ProofTree2stringDebug tree = Str2string(ProofTree2strDebug tree)

    fun ProofRun2str run = ProofTree2strCommon(false, SOME run) (ProofRun2ProofTree run)
    fun ProofRun2strDebug run = ProofTree2strCommon(true, SOME run) (ProofRun2ProofTree run)
    fun ProofRun2string run = Str2string(ProofRun2str run)
    fun ProofRun2stringDebug run = Str2string(ProofRun2strDebug run)

    (* Save the proof tree to a file. *)
    fun ProofTree2file (tree, file) =
	let val outs = TextIO.openOut file
	    fun dump (Str s) = TextIO.output(outs,s)
	      | dump (Conc(x,y)) = ((dump x);(dump y))
	in 
	    (dump(ProofTree2str tree);
	     TextIO.closeOut outs)
	end handle _ => raise ProverError ("Can't save proof to file "^file)

    (* For now, we do not check that the subgoal is from the same
       proof as the run, just use the subgoal's ID *)
    fun getSequent ({assocHash=hash,...}: ProofRun) (Subgoal{uid=uid,...}) = findHash(hash,uid)

    fun getSequentByUid ({assocHash=hash,...}: ProofRun) uid = findHash(hash,uid)

    (* A proof tree is considered "complete" when all leaves have an
       inference rule but no children *)
    fun isProofTreeComplete tree = 
	let val visited = makeHashDefault(op =, subgoalUID2string)
	    fun notVisited s = not(isSome(findHash(visited, getSubgoalUID s)))
	    fun mark s = (insertHashDestructive(visited, getSubgoalUID s, ()); ())
	    fun loop (Subgoal{children=ref [], proofRule=ref(SOME _), ...}) = true
	      | loop (Subgoal{children=ref [], proofRule=ref NONE, ...}) = false
	      | loop s = (mark s; List.all loop (List.filter notVisited (getChildren s)))
	in loop(getRoot tree)
	end

    (* A proof run is complete if the corresponding proof tree is
       complete, and every subgoal has an associated sequent *)
    fun isProofRunComplete({prooftree=tree, assocHash=hash}: ProofRun) = 
	let val ProofTree{ hash=treeHash, ...} = tree
	    fun checkSequents() = 
		let val ids = List.map #1 (hash2any(fn x=>x)(fn x=>x) treeHash)
		in List.all (fn x => isSome(findHash(hash, x))) ids
		end
	in isProofTreeComplete tree andalso checkSequents()
	end

    (* Check if the proof step at this subgoal is complete.  That is,
       the subgoal has a sequent and a proof rule, and all of its
       children have a sequent. *)

    fun isProofStepComplete proofRun s =
	let val Subgoal{uid=uid, proofRule=ref ruleOpt, children=ref children, ...} = s
	    fun isComplete(uid, _) = isSome(getSequentByUid proofRun uid)
	in
	    isSome ruleOpt andalso List.all isComplete ((uid,NONE)::children)
	end

    (* Find all subgoals without a proof and return them in a list.  A
       subgoal is a leaf if it doesn't have a proof rule. *)
    fun getLeafSubgoals tree =
	let val visited = makeHashDefault(op =, subgoalUID2string)
	    fun notVisited s = not(isSome(findHash(visited, getSubgoalUID s)))
	    fun mark s = (insertHashDestructive(visited, getSubgoalUID s, ()); ())
	    fun loop (s as Subgoal{children=ref [], proofRule=ref(SOME _), ...}) = []
	      | loop (s as Subgoal{children=ref [], proofRule=ref NONE, ...}) = [s]
	      | loop s =
		let val ch = getChildren s
		    val newCh = List.filter notVisited ch
		in (mark s; List.foldr (op @) [] (List.map loop newCh))
		end
	in loop(getRoot tree)
	end

    fun getUnprovenSubgoals proofRun =
	let val _ = pushFunStack("getUnprovenSubgoals", "")
	    val visited = makeHashDefault(op =, subgoalUID2string)
	    fun notVisited s = not(isSome(findHash(visited, getSubgoalUID s)))
	    fun mark s = (insertHashDestructive(visited, getSubgoalUID s, ()); ())
	    fun loop s =
		if isProofStepComplete proofRun s then
		    let val ch = List.filter notVisited (getChildren s)
		    in (mark s; List.foldr (op @) [] (List.map loop ch))
		    end
		else [s]
	    val res = loop(getRoot(ProofRun2ProofTree proofRun))
	    val _ = popFunStackLazy("getUnprovenSubgoals", fn()=>
				    "["^(strlist2str ", " (List.map subgoal2string res))^"]")
	in res
	end

    (* Handling the subgoal's IDs *)
    local val pid = ref 0
    in 
	
	(* Generate a new unique proof ID based on the old one *)
	fun newProofTreeID s = 
	    ((option2string "" s)^"<"^(Int.toString(!pid))^">") before (pid := (!pid) + 1)
	(* Generate an ID for a new subgoal *)
	fun initID s = (s, [1])
    end

    (* Find the next unproven subgoal after the given subgoal, if
       given.  Currently, we get the entire list of the unproven
       subgoals, and if the given subgoal is also unproven (and thus,
       occur in the list), get the next subgoal after it in the list
       order (or the first one, if the current subgoal is the last in
       the list).  In any other case, just give the first subgoal in
       the list if it's not empty. *)
    fun nextUnprovenSubgoal(proofRun, sub: Subgoal option) = 
	let val funName = "nextUnprovenSubgoal"
	    val _ = pushFunStackLazy(funName, fn()=> 
				     option2string "NONE" (Option.map subgoal2string sub))
	    fun nextSub hd [] = hd
	      | nextSub _ (hd::_) = hd
	    fun loop (hd,s) lst =
		let val _ = pushFunStackLazy
		    (funName^"/loop",
		     fn()=> "hd="^(subgoal2string hd)
		     ^", s="^(subgoal2string s)
		     ^", lst=["^(strlist2str "," (List.map subgoal2string lst))^"]")
		    val res = loop' (hd, s) lst
		    val _ = popFunStackLazy(funName^"/loop", fn()=> subgoal2string res)
		in res
		end
	    and loop' (hd,s) [] = hd
	      | loop' (hd,s) (s'::lst) = 
	          if SubgoalEq(s, s') then nextSub hd lst
		  else loop (hd,s) lst
	    val res = (case (getUnprovenSubgoals proofRun, sub) of
			   ([],_) => NONE
			 | (hd::lst, NONE) => SOME hd
			 | (hd::lst, SOME s) =>
			       if SubgoalEq(s, hd) then SOME(nextSub hd lst)
			       else SOME(loop (hd,s) lst))
	    val _ = popFunStackLazy(funName, fn()=> 
				    option2string "NONE" (Option.map subgoal2string res))
	in res
	end

    (* Generate new subgoal *)
    local 
	val nextUID = ref 0
	fun newUID() = (!nextUID) before (nextUID := (!nextUID) + 1)
    in
	fun makeSubgoal(proofTreeIDopt) =
	    Subgoal{ id=ref(initID(case proofTreeIDopt of
				       NONE => newProofTreeID NONE
				     | SOME s => s)),
		     extend = ref false,
		     uid=newUID(),
		     proofRule = ref(NONE),
		     children = ref([]),
		     parents=ref [],
		     prooftree=ref(NONE) }
    end

    fun copySubgoal pid (Subgoal{id=ref (_,id),
				 extend=ref ex,
				 proofRule=proofRule,
				 children=ch,
				 parents=parents,
				 prooftree=tree, ... }) =
	let val s as Subgoal{ id=newID,
			      extend=newExtend,
			      proofRule=newProofRule,
			      children=newCh,
			      parents=newP,
			      prooftree=newProofTree, ...} = makeSubgoal(SOME pid)
	in (newID :=(pid, id);
	    newExtend := ex;
	    newCh := !ch;
	    newP := !parents;
	    newProofRule := !proofRule;
	    newProofTree := !tree;
	    s)
	end

    fun makeProofTree pid =
	let val ptref = ref(NONE)
	    val newPid = 
		 (case pid of
		      NONE => newProofTreeID NONE
		    | SOME s => s)
	    val s as Subgoal{ id=id, prooftree=ptref, ...} = makeSubgoal(SOME newPid)
	    val hash = makeHashDefault(op =, subgoalUID2string)
	    val pt = ProofTree{root=s, hash=hash}
	in (id := initID newPid;
	    ptref := SOME(pt);
	    insertHashDestructive(hash,getSubgoalUID s, s);
	    pt)
	end

    (* Find subgoal in the proof tree by its UID *)
    fun UID2Subgoal (ProofTree{hash=hash,...}) uid = findHash(hash,uid)

    (* For now, just search the entire tree for the subgoal ID;
       inefficient, but simple and correct.  Besides, we do not expect
       to have behemoth proof trees here. *)

    fun ID2Subgoal (ProofTree{hash=hash,...}) id =
	let val subgoals = List.map #2 (hash2any(fn x=>x)(fn x=>x) hash)
	in 
	    List.find (fn s=> id = (getSubgoalID s)) subgoals
	end

    (* Build a copy of the proof subtree rooted at `s' (if there is one),
        and give it an optional `pid' *)
    fun extractSubtree pid (s as Subgoal{proofRule=ref(SOME rule), ...}) =
	let val funName = "extractSubtree"
	    val _ = pushFunStackLazy(funName,
				     fn()=>"pid="^(case pid of SOME s => s | NONE => "NONE")
				     ^", "^(subgoal2string s))
	    val ch = getChildrenHints s
	    val hash = makeHashDefault(op =, subgoalUID2string)
	    val oldPid = Option.map getProofTreeID (getProofTree s)
	    val newPid = 
		 (case pid of
		      NONE => newProofTreeID oldPid
		    | SOME s => s)
	    val newS = copySubgoal newPid s
	    val Subgoal{id=ref id, uid=uid, children=newCh, proofRule=newRule,
			parents=newP, prooftree=newTree, ...} = newS
	    val tree = ProofTree{root=newS, hash=hash}
	    fun loop parent (s, hints: SequentHints option) =
		let fun hOpt2str NONE = "NONE"
		      | hOpt2str (SOME h) = hints2string h
		    val _ = pushFunStackLazy(funName^"/loop", 
					     fn()=>(subgoal2string s)
					     ^", "^(hOpt2str hints))
		    val res as (uid, hints) = loop' parent (s, hints)
		    val _ = popFunStackLazy(funName^"/loop",
					    fn()=>"("^(subgoalUID2string uid)
					    ^", hints="^(hOpt2str hints)^")")
		in res
		end
	    and loop' parent (s, hints: SequentHints option) =
		let val (uid, ch) = (getSubgoalUID s, getChildrenHints s)
		    val Subgoal{children=parentCh, ...} = parent
		    val newSub as Subgoal{uid=newUID, ...} =
			(case findHash(hash,uid) of
			     SOME newS =>
				 let val Subgoal{parents=p, ...} = newS
				 in (p := ((getSubgoalUID parent)::(!p));
				     newS)
				 end
			   | NONE => 
				 let val newS = copySubgoal newPid s
				     val Subgoal{children=newCh,
						 uid=uid,
						 parents=newP,
						 prooftree=newTree, ...} = newS
				 in (newP := [getSubgoalUID parent]; newTree := SOME tree;
				     insertHashDestructive(hash,uid,newS);
				     newCh := List.map(loop newS) ch;
				     newS)
				 end)
		in (parentCh := (!parentCh)@[(newUID, hints)];
		    (newUID, hints))
		end
	in (newP := []; newTree := SOME tree;
	    insertHashDestructive(hash,uid,newS);
	    newCh := List.map(loop newS) ch;
	    popFunStackLazy(funName, fn()=> ProofTree2string tree);
	    SOME tree)
	end
      (* There is no proof tree at this subgoal *)
      | extractSubtree _ (Subgoal{proofRule=ref NONE, ...}) = NONE

    (* Copy the entire proof tree and give it an optional `pid' *)
    fun copyProofTree pid tree = 
	let val _ = pushFunStackLazy("copyProofTree", fn()=>ProofTree2string tree)
	    val res = (case extractSubtree pid (getRoot tree) of
			   SOME t => t
			 | NONE => makeProofTree pid)
	    val _ = popFunStackLazy("copyProofTree", fn()=>ProofTree2string res)
	in res
	end

    fun makeProofRun pt =
	{ prooftree=copyProofTree (SOME(getProofTreeID pt)) pt,
	  assocHash=makeHashDefault(op =, subgoalUID2string) } : ProofRun

    (* Check if the first id is the desendant of the second *)
    fun isDescendant _ [] = true
      | isDescendant [] _ = false
      | isDescendant (x1::lst1) [x2] = (x1 >= x2)
      | isDescendant (x1::lst1) (x2::lst2) = x1 = x2 andalso isDescendant lst1 lst2

    (* Same, only the probable ancestor is given as a UID *)
    fun isDescendantUID proofTree sid parentUID =
	let val (_,parentSid) = getSubgoalID
	    (case UID2Subgoal proofTree parentUID of
		 SOME s => s
	       | NONE => raise SympBug
		     ("isDescendantUID: parent ["
		      ^(subgoalUID2string parentUID)^"] not found"))
	in isDescendant sid parentSid
	end
    (* Change the true parent of the subgoal s to `newParent'.
       Destructively change the IDs of the subgoal `s' and its
       descendants, and update the proof tree of `s' (not the
       newParent's tree). *)

    fun changeParent (newParentUID: SubgoalUID) s =
	let val Subgoal{prooftree=ref prooftreeOpt, ...} = s
	    val proofTree as ProofTree{hash=hash, ...} =
		  (case prooftreeOpt of
		       SOME pt => pt
		     | NONE => raise SympBug
			   ("changeParent: the subgoal "^(subgoal2string s)
			    ^" doesn't have a proof tree"))
	    val visited = makeHashDefault(op =, subgoalUID2string)
	    val newParent = (case UID2Subgoal proofTree newParentUID of
				 SOME p => p
			       | NONE => raise SympBug
				   ("changeParent: the new parent ["
				    ^(subgoalUID2string newParentUID)
				    ^"] not found"))
	    val (newPid, _) = getSubgoalID newParent
	    val (oldPid, _) = getSubgoalID s
	    val updateAll = not(oldPid = newPid)

	    (* Look at the list of children of `parent' and figure out
	       the new name of the subgoal when `parent' becomes its
	       true parent *)

	    fun computeNewID parent (id,uid) =
		let val Subgoal{id=ref parID, 
				extend=ref extend,
				children=ref ch,...} = parent
		    val (pid, sid) = parID
		    fun loop [] _ = raise SympBug
			  ("changeParent/loop: new parent "^(subgoal2string parent)
			   ^" doesn't have the subgoal "^(subgoalID2string id)
			   ^"["^(subgoalUID2string uid)^"]"
			   ^" in its list of children")
		      | loop ((uid',_)::lst) n =
			  if uid = uid' then (pid, sid@[n]) else loop lst (n+1)
		    (* Increment the last integer in the list and return the new list *)
		    fun incLast [n] = [n+1]
		      | incLast (x::lst) = x::(incLast lst)
		      | incLast [] = raise SympBug
			  ("changeParent/incLast: the SID list is empty")
		in
		    case ch of
			lst as [(uid',_)] => 
			    if uid = uid' then
				if extend then (loop lst 1, false)
				else ((pid, incLast sid), false)
			    else raise SympBug
				("changeParent: new parent "^(subgoal2string parent)
				 ^" doesn't have the subgoal "^(subgoalID2string id)
				 ^"["^(subgoalUID2string uid)^"]"
				 ^" in its list of children (the only child ["
				 ^(subgoalUID2string uid')^"] differs)")
		      | lst => (loop lst 1, true)
		end

	    (* updateID(oldID, newParentUID) id
	       If `id' matches the `oldID' in SID, rename it to be a child of `newParent',
	       and update PID if needed. Return NONE otherwise. *)
	    fun updateID((_,oldSid),newParentUID) (id as (_, sid), uid) = 
		let val parent = (case findHash(hash,newParentUID) of
				      SOME p => p
				    | NONE => raise SympBug
					  ("changeParent/updateID: parent not found: "
					   ^(subgoalUID2string newParentUID)))
		    val ((newPid,newSid),newExtend) = computeNewID parent (id,uid)
		in 
		    if oldSid = sid then SOME((newPid, newSid), newExtend)
		    else if updateAll then SOME((newPid, sid), newExtend)
		    else NONE
		end

	    fun updateSubgoal (oldID, parentUID) (s as Subgoal{id=id, uid=uid,
							       extend=exRef, ...}) =
	        Option.map(fn (newID, newExtend) =>
			   (id := newID;
			    exRef := newExtend;
			    newID)) (updateID (oldID, parentUID) (!id, uid))
	    (* The main loop; p is the current new parent for s *)
	    fun loop p s =
		(case findHash(visited, getSubgoalUID s) of
		     SOME _ => ()
		   | NONE => 
			 let val Subgoal{id=id, uid=uid,
					 children=chRef,
					 parents=parentsRef, ...} = s
			     val myOldID = !id
			     val newIdOpt = updateSubgoal (myOldID, p) s
			     (* Insert the *new* ID to the set of visited IDs *)
			     val _ = insertHashDestructive(visited, uid, s)
			 in
			     case newIdOpt of
				 (* If it didn't change, do nothing further *)
				 NONE => ()
			       (* Recurse on children with `s' as a new parent *)
			       | SOME newID => List.app (loop uid) (getChildren s)
			 end)
	in loop newParentUID s
	end

    (* Delete the parent from the subgoal's list of parents.
       The parent must be present. *)
    fun delParent parent (s as Subgoal{parents=plRef as ref pUIDlist,...}) =
	let val funName = "delParent"
	    val pl = getParents s
	    val _ = pushFunStackLazy(funName, fn()=>
				     (subgoal2stringDebug parent)^", "
				     ^(subgoal2stringDebug s)^", parents=["
				     ^(strlist2str ", " (List.map subgoalUID2string pUIDlist))
				     ^"], pl=["^(strlist2str ", " 
						 (List.map subgoal2stringDebug pl))^"]")
	    fun loop [] = raise SympBug
	           ("ProofTree/garbageCollect: parent subgoal is not "
		    ^"in the parents list:\n  "
		    ^"Parent: "^(subgoal2string parent)
		    ^"\n  Subgoal: "^(subgoal2string s))
	      | loop (s1::tl) =
		   (if SubgoalEq(parent,s1) then tl
		    else s1::(loop tl))
	in 
	    (plRef := List.map getSubgoalUID (loop pl);
	     popFunStack(funName, ""); ())
	end

    (* Garbage-collect the subgoal and all its children that apply.
       When removing the true parent of a child, renumber the child's
       subtree to be a descendant of some other remaining parent. *)

    fun garbageCollect s =
	let val funName = "garbageCollect"
	    val Subgoal{prooftree=ref proofTreeOpt, ...} = s
	    fun ptOpt2str ptOpt =
		(option2string "NONE"
		 (Option.map (fn p=>"[\n"^(ProofTree2stringDebug p)^"]") ptOpt))
	    fun argsFn(Subgoal{uid=uid, parents=ref plist, prooftree=ref ptOpt, ...}) () =
		("Subgoal{uid="^(subgoalUID2string uid)
		 ^",\n   parents=["^(strlist2str ", " (List.map subgoalUID2string plist))
		 ^"],\n   prooftree="^(ptOpt2str ptOpt)^"}")
	    val _ = pushFunStackLazy(funName, argsFn s)
	    fun loop s =
		let val funName = "garbageCollect/loop"
		    val _ = pushFunStackLazy(funName, argsFn s)
		    val res = loop' s
		    val _ = popFunStackLazy(funName, fn()=>ptOpt2str proofTreeOpt)
		in res
		end
	    (* If the parent list is empty, remove the subgoal. *)
	    and loop' (s as Subgoal{uid=uid, parents=ref [],
						 prooftree=ref(SOME pt), ...}) =
		let val ProofTree{hash=hash,...} = pt
		    val ch = getChildren s
		in (List.app (delParent s) ch;
		    removeHashDestructive(hash,uid);
		    List.app loop ch)
		end
	      (* If the parent list is non-empty, renumber the subgoal and its
	         subtree, if necessary, to remove all the traces of the old subgoal. *)
	      | loop' (s as Subgoal{id=idRef as ref id,
				   uid=uid,
				   parents=ref(plist as newParentUID::_),
				   prooftree=ref(SOME pt), ...}) = 
		let val (pid, sid) = id
		    fun getID uid = 
			  (case UID2Subgoal pt uid of
			       SOME s => getSubgoalID s
			     | NONE => raise SympBug
				   ("garbageCollect/getID: subgoal ["
				    ^(subgoalUID2string uid)^"] not found"))
		in
		    (* If the true parent is present, do nothing further *)
		    if List.exists(isDescendantUID pt sid) plist then ()
		    (* Otherwise change the parent *)
		    else changeParent newParentUID s
		end
	      (* If there is no proof tree, don't garbage collect *)
	      | loop' _ = ()
	in 
	    (loop s;
	     popFunStackLazy(funName, fn()=>ptOpt2str proofTreeOpt); ())
	end

    (* Delete the subtree originating at the subgoal, but leave the subgoal. *)
    fun deleteSubtree(s as Subgoal{children=chRef, proofRule=rule, ...}) =
	let val ch = getChildren s
	in (List.app (delParent s) ch;
	    List.app garbageCollect ch;
	    rule := NONE; chRef := [])
	end

    (* Check if the parent has the subgoal as its predecessor *)
    fun checkCycle parent subgoal =
	let val hash = makeHashDefault(op =, subgoalUID2string)
	    fun loop s =
		  if isSome(findHash(hash,getSubgoalUID s)) then false
		  else if SubgoalEq(subgoal,s) then true
		       else (insertHashDestructive(hash,getSubgoalUID s, s);
			     List.exists loop (getParents s))
	in loop parent
	end
		  
    (* Verify that the subgoal is from the same proof tree, and only
       then associate the sequent with it. *)

    fun addSequent (proofrun as {prooftree=pt, assocHash=hash}: ProofRun) (s, seq) =
	((case getProofTree s of
	      NONE => raise ProofTreeError
		  ("Subgoal doesn't have a proof tree: "^(subgoal2string s))
	    | SOME p => if (getProofTreeUID p) = (getProofTreeUID pt) then ()
			else raise ProofTreeError
			    ("Subgoal is not from the current proof tree: "
			     ^(subgoal2string s)));
	 lazyVerbDebug (getOptions()) "addSequent"
	   (fn()=>"addSequent: subgoal "
	    ^(subgoal2stringDebug s)
	    ^" is assigned sequent:\n"
	    ^(sequent2string seq)^"\n");
	 insertHashDestructive(hash, getSubgoalUID s, seq); ())

    (* Install a clean COPY of the ProofTree (if `copy' is true) at
       the Subgoal in the current proof tree.  The subgoal replaces
       the root subgoal of the new ProofTree.  The old subtree rooted
       at the subgoal is deleted.  It is an error for the subgoal to
       not have a proof tree. *)

    fun installProofTree copy (s, tree) =
	let val funName = "installProofTree"
	    val debug = lazyVerbDebug (getOptions()) funName
	    val Subgoal{id=ref id,
			parents=parents,
			proofRule=oldProofRuleRef,
			children = children, ...} = s
	    val oldTree as ProofTree{hash=hash,...} = 
		  (case getProofTree s of
		       SOME t => t
		     | NONE => raise SympBug
			   ("\ninstallProofTree: the subgoal doesn't have a proof tree: "
			    ^(subgoal2string s)))
	    val _ = pushFunStackLazy(funName,
				     fn()=>(subgoal2string s)
				     ^"\n Old tree:\n"
				     ^(ProofTree2stringDebug oldTree)
				     ^"\n New tree:\n"
				     ^(ProofTree2stringDebug tree))
	    val oldCh = getChildren s
			
	    (* Proof tree ID and Subgoal ID of `s' *)
	    val (pid, sid) = id
	    (* The true parent of `s', if any *)
	    fun findTrueParent s = 
		let val Subgoal{parents=ref parents,
				id=ref(pid,sid),
				prooftree=tree, ...} = s
		in (Option.map(fn uid=> 
			       (case UID2Subgoal oldTree uid of
				    SOME p => p
				  | NONE => raise SympBug
					("installProofTree: parent ["
					 ^(subgoalUID2string uid)^"] of s = "
					 ^(subgoal2stringDebug s)^" not found in the proof tree")))
		    (List.find(isDescendantUID oldTree sid) parents))
		end
	    (* The new subgoal replacing `s' contents *)
	    val rootProofTree = (if copy then copyProofTree (SOME pid) tree 
				 else tree)
	    val root = getRoot rootProofTree
	    val Subgoal{children=ref rootCh,
			proofRule= ref newProofRule, ...} = root
	    (* The list of children subgoals (not IDs) *)
	    val rootChildren = getChildren root
	    (* Add all the new subgoals to the proof tree hash *)
	    fun addToHash s =
		if isSome(findHash(hash,getSubgoalUID s)) then ()
		else let val ch = getChildren s
			 val Subgoal{prooftree=pt,...} = s
		     in (insertHashDestructive(hash,getSubgoalUID s, s);
			 pt := SOME oldTree;
			 List.app addToHash ch)
		     end
	    (* Replace parent with oldUID by newUID *)
	    fun replaceParent(oldUID, newUID) (Subgoal{parents=p,...}) =
		let fun loop [] = raise SympBug
		          ("insatllProofTree/replaceParent: parent ["
			   ^(subgoalUID2string oldUID)^"] is not in the list:\n ["
			   ^(strlist2str "," (List.map subgoalUID2string (!p)))^"]")
		      | loop (uid::lst) =
			  if uid = oldUID then newUID::lst
			  else uid::(loop lst)
		in p := loop(!p)
		end
	    fun changeChildsParent newParent child =
		let val parentUID = getSubgoalUID newParent
		in
		    case findTrueParent child of
			SOME _ => () (* It's not our child, leave it alone *)
		      | NONE => changeParent parentUID child
		end
	in ((* First, fix the numbering in the new subtree.
	       If `s' is a root, then no renumbering is needed. *)
	    (* Option_app changeRootParent trueParentOpt; *)
	    debug(fn()=>" New tree renumbered:\n"
		  ^(ProofTree2stringDebug tree)^"\n");
	    (* Detach children of the old subgoal *)
	    List.app (delParent s) oldCh;
	    (* And garbage collect.  This should also renumber
	       children to detach them from this subtree completely *)
	    List.app garbageCollect oldCh;
	    debug(fn()=>" Old tree after removing the old subtree and GC'ing:\n"
		  ^(ProofTree2stringDebug oldTree)^"\n");
	    (* Copy the new root configuration to the old root (the
	       old root remains - we can't always replace it) *)
	    children := rootCh;
	    oldProofRuleRef := newProofRule;
	    List.app(replaceParent(getSubgoalUID root, getSubgoalUID s)) rootChildren;
	    (* And add its children to the tree hash recursively *)
	    List.app addToHash rootChildren;
	    (* Now renumber the new descendants *)
	    List.app (changeChildsParent s) rootChildren;
	    popFunStackLazy(funName,fn()=> ProofTree2stringDebug oldTree))
	end

    (* Add a ProofRule to the Subgoal that produces a Subgoal list.
       The existing subtree rooted at the Subgoal is deleted. Subgoals
       are NOT COPIED, but rather MODIFIED IN PLACE, so you may later
       safely associate them with sequents.

       This is the only EXPORTED function that builds a proof tree
       structure.  You cannot change the inference rule or the subgoal
       without changing the entire subtree, nor you can change any
       information at the subgoal.

       The new subgoals must either belong to the current proof tree,
       or be completely new. *)

    fun addProofRule(s,proofRule,lst) =
	let val funName = "addProofRule"
	    val debug = lazyVerbDebug (getOptions()) funName
	    val _ = pushFunStackLazy(funName,
				     fn()=>(subgoal2stringDebug s)
				     ^", "^(#1 proofRule)^", ["
				     ^(strlist2str "," (List.map (subgoal2stringDebug o #1) lst))
				     ^"]")
	    val tree as ProofTree{ hash=hash,...} = 
		(case getProofTree s of
		     SOME x => x
		   | NONE => raise ProofTreeError
			 ("Base subgoal "^(subgoal2string s)^"doesn't have a proof tree"))
	    val pid = getProofTreeID tree
	    val pUid = getProofTreeUID tree
	    val (_,sid) = getSubgoalID s
	    val extend = getSubgoalExtend s
	    fun wrongPid (ss,_) = 
		 (case Option.map(fn pt => pUid = (getProofTreeUID pt)) (getProofTree ss) of
		      SOME false => true
		    | _ => false)
	    val _ = (case List.find wrongPid lst of
			 SOME (ss,_) => raise ProofTreeError
			     ("Can't add subgoal "^(subgoal2string ss)
			      ^" from a different proof tree")
		       | _ => ())
	    (* Increment the last integer in the list and return the new list *)
	    fun incLast [n] = [n+1]
	      | incLast (x::lst) = x::(incLast lst)
	      | incLast [] = raise SympBug
		("addProofRule/incLast: the SID list is empty")
	    (* Renumber the new subgoals (without a proof tree) to be proper children of `s',
	       and add them to the tree hash.
	       Assume that `lst' are the only children of the parent. *)
	    fun renumber(hash,parentUID, parentID, proofTree) lst =
		let fun renameSub(s, sid, ex) =
		         let val Subgoal{id=id, uid=uid,
					 extend=extendRef,
					 parents=parents,
					 prooftree=pt, ...} = s
			 in  (* Only rename if the subgoal is brand new *)
			     case getProofTree s of
				 NONE => (id := (pid,sid);
					  extendRef := ex;
					  debug(fn()=>"addProofRule/renumber: renaming subgoal to "
						^(subgoalID2string(!id))^"\n");
					  parents := [parentUID];
					  pt := SOME proofTree;
					  insertHashDestructive(hash, uid, s); ())
			       | SOME _ => ()
			 end
		    fun loop _ [] _ = ()
		      | loop ex ((ch,_)::lst) n =
			  (renameSub(ch, sid@[n], ex);
			   loop ex lst (n+1))
		in
		    case lst of 
			lst as [(ch,_)] => 
			    if extend then loop false lst 1
			    else renameSub(ch, incLast sid, false)
		      | _ => loop true lst 1
		end
	in 
	    ((case List.find (fn (x,_)=>checkCycle s x) lst of
		  SOME (ss,_) => raise ProofTreeError
		      ("Can't add the subgoal "^(subgoal2string ss)
		       ^" because it creates a cycle in the proof")
		| NONE =>
		      (* Build a new proof tree and install it *)
		      let val pt as ProofTree{hash=hash,...} = makeProofTree(SOME pid)
			  val Subgoal{id=ref id, uid=uid,
				      children=ch,
				      proofRule=r, ...} = getRoot pt
		      in (r := SOME proofRule;
			  renumber(hash, uid, id, pt) lst;
			  ch := List.map (fn(s,h)=>(getSubgoalUID s,h)) lst;
			  installProofTree false (s,pt))
		      end);
	     popFunStackLazy(funName,fn()=>"new proof tree:\n  "
			     ^(ProofTree2stringDebug tree)))
	end

    (* Proof tree I/O functions must be one-to-one, so that a proof
       tree could be completely restored from the string or the
       `UserInputArgument' value *)

    (* For more efficient communication with GUI/Emacs/whatever *)
    fun ProofTree2UI tree = 
	let val subgoals = listSubgoals tree
	    fun doSubgoal sub = 
		let val id = getSubgoalID sub
		    fun id2UI id = UIstring(subgoalID2string id)
		    fun rule2UI (r, args) = UIlist((UIstring r)::args)
		    val ruleUI = Option.map rule2UI (getProofRule sub)
		    fun child2UI (ch,hints) =
			let val id = id2UI(getSubgoalID ch)
			in
			    case hints of
				NONE => id
			      | SOME h => UIlist[id, UIstring(hints2string h)]
			end
		    val children = UIlist(List.map child2UI (getChildrenHints sub))
		    val ruleChildren = Option.map(fn r => [r, children]) ruleUI
		in
		    UIlist(case ruleUI of
			       NONE => [id2UI id]
			     | SOME r => [id2UI id, r, children])
		end
	in
	    UIlist(List.map doSubgoal subgoals)
	end

    (* And the reverse translation: build a proof tree from the
       `string' or `UserInputArgument'.  This, of course, may raise an
       exception if the input is invalid. *)

    structure ProofTreeStruct: PROOF_TREE_STRUCT =
      struct
	datatype ProofTreeType =
	    PTsymbol of string
	  | PTstring of string
	  | PTnumber of int
	  | PTlist of ProofTreeType list
	  (* For named rule's arguments *)
	  | PTassoc of string * ProofTreeType
	  (* The name of the rule and the list of arguments *)
	  | PTrule of string * (ProofTreeType list)
	  | PTsubgoalID of string * (int list)
	  (* PTchild(subgoalID, optional-hints) *)
	  | PTchild of ProofTreeType * (string option)
	  (* The entire subgoal entry with rule and children; the last two are optional *)
	  | PTsubgoal of ProofTreeType * (ProofTreeType option) * (ProofTreeType list option)
	  (* The entire proof *)
	  | PTproof of ProofTreeType list
      end

    open ProofTreeStruct

    fun ptt2str(PTsymbol s) = s
      | ptt2str(PTstring s) = "\""^s^"\""
      | ptt2str(PTnumber n) = Int.toString n
      | ptt2str(PTlist lst) = "["^(strlist2str ", " (List.map ptt2str lst))^"]"
      | ptt2str(PTassoc(n, t)) = "("^n^" = "^(ptt2str t)^")"
      | ptt2str(PTrule(n, lst)) = "rule("^n^", ["^(strlist2str "," (List.map ptt2str lst))^"])"
      | ptt2str(PTsubgoalID sub) = "subgoalID("^(subgoalID2string sub)^")"
      | ptt2str(PTchild(t, sOpt)) =
	 ("child("^(ptt2str t)
	  ^(option2string "" (Option.map(fn s=>", "^s) sOpt))
	  ^")")
      | ptt2str(PTsubgoal(t,tOpt,tlstOpt)) =
	 ("subgoal("
	  ^(ptt2str t)
	  ^(option2string "" (Option.map(fn t => ", "^(ptt2str t)) tOpt))
	  ^(option2string "" 
	    (Option.map(fn lst => ", ["^(strlist2str "," (List.map ptt2str lst))^"]")
	               tlstOpt))
	  ^")")
      | ptt2str(PTproof lst) = "proof[\n"^(strlist2str "\n" (List.map ptt2str lst))^"]\n"

    structure ParseError: PARSE_ERROR =
      struct
        exception ParseError of string 
	val wasError = ref false
      end

    structure ProofTreeLrVals =
	ProofTreeLrValsFun(structure Token = LrParser.Token
			     structure ProofTreeStruct = ProofTreeStruct
			     structure ParseError = ParseError)

    structure ProofTreeLex =
	ProofTreeLexFun(structure Tokens = ProofTreeLrVals.Tokens
			  structure ParseError = ParseError)

    structure ProofTreeParser =
	ProofTreeJoin(structure LrParser = LrParser
		 structure ProofTreeStruct = ProofTreeStruct
		 structure ParserData = ProofTreeLrVals.ParserData
		 structure Lex = ProofTreeLex)

    type result = ProofTreeLrVals.ParserData.result
    open ProofTreeLrVals.ParserData

    fun ProofTreeParse nameOpt ins =
	let open ParseError
	    val name = (case nameOpt of
			    SOME n => n^":"
			  | NONE => "")
	    val print_error = fn (s,l,c) =>
		(wasError := true;
		 raise ProofTreeError("\n"^name^(Int.toString l) ^"."
					^ (Int.toString c) ^ ": " ^ s ^ "\n"))
		  (* Use 15 tokens for error correction *)
	    val invoke = fn lexstream => 
		(ProofTreeParser.parse(15,lexstream,print_error,())
		 handle ProofTreeParser.ParseError => 
		     raise ProofTreeError("Fatal ProofTree parser error")
		      | ParseError str => raise ProofTreeError str)
	    val lexer = ProofTreeParser.makeLexer 
		           (fn (i: int) => TextIO.inputLine ins) print_error
	    val (result,_) = (ProofTreeLex.UserDeclarations.pos := 1;
			      ProofTreeLex.UserDeclarations.yypos_bol := 0;
			      wasError := false;
			      invoke lexer)
			      
	in result
	end

    fun ProofTreeType2ProofTree (ptt as PTproof(root::lst)) =
	let val funName = "ProofTreeType2ProofTree"
	    val _ = pushFunStackLazy(funName, fn()=>ptt2str ptt)
	    val ProofTree{hash=hash, root=oldRoot } = makeProofTree NONE
	    fun ptt2arg (PTsymbol s) = UIstring s
	      | ptt2arg (PTstring s) = UIstring s
	      | ptt2arg (PTnumber n) = UInumber n
	      | ptt2arg (PTlist lst) = UIlist(List.map ptt2arg lst)
	      | ptt2arg (PTassoc(s, x)) = UIassoc(UIstring s, ptt2arg x)
	      | ptt2arg x = raise SympBug
		   ("ProofTreeType2ProofTree/ptt2arg: bad rule argument:\n "
		    ^(ptt2str x))
	    fun ptt2rule (PTrule(name,args)) = (name, List.map ptt2arg args)
	      | ptt2rule x = raise SympBug
		  ("ProofTreeType2ProofTree/ptt2ptt2rule: bad rule spec:\n "
		   ^(ptt2str x))
	    (* Recreate the tree with the same hash but different root *)
	    fun newRoot sub = 
		let val tree = ProofTree{hash=hash, root=sub}
		    val Subgoal{prooftree=treeRef,...} = sub
		in (removeHashDestructive(hash, getSubgoalUID oldRoot);
		    insertHashDestructive(hash, getSubgoalUID sub, sub);
		    treeRef := SOME tree;
		    tree)
		end
	    fun ptt2subgoal tree (PTsubgoal(PTsubgoalID id, rule, children)) =
		let val s as Subgoal{id=idRef,
				     proofRule=ruleRef,
				     prooftree=ptRef, ... } = makeSubgoal(NONE)
		in (idRef := id;
		    ruleRef := Option.map ptt2rule rule;
		    ptRef := tree;
		    s)
		end
	      | ptt2subgoal _ x = raise ProverError("Error in the proof file: bad rule")

	    val tree = newRoot(ptt2subgoal NONE root)

	    fun insertSubgoal sub =
		let val subgoal = ptt2subgoal(SOME tree) sub
		in 
		    (insertHashDestructive(hash, getSubgoalUID subgoal, subgoal);
		     subgoal)
		end

	    fun ID2Sub id = (case ID2Subgoal tree id of
				 SOME s => s
			       | NONE => raise SympBug
				   ("ProofTreeType2ProofTree/ID2Sub: "
				    ^"subgoal is not in hash: "^(subgoalID2string id)))

	    fun ptt2children NONE = []
	      | ptt2children (SOME lst) =
		let fun ptt2child(PTchild(PTsubgoalID id, hints)) =
		          (getSubgoalUID(ID2Sub id), Option.map string2hints hints)
		      | ptt2child x = raise SympBug
			  ("ProofTreeType2ProofTree/ptt2child: bad child spec:\n "
			   ^(ptt2str x))
		in List.map ptt2child lst
		end

	    (* After all subgoals are in the hash, fix the children/parent links *)
	    fun fixChildren(PTsubgoal(PTsubgoalID id, rule, children)) =
		let val s as Subgoal{children=chRef,...} = ID2Sub id
		in 
		    chRef := ptt2children children
		end
	      | fixChildren x = raise SympBug
		  ("ProofTreeType2ProofTree/fixChildren: wrong subgoal spec:\n "
		   ^(ptt2str x))

	    fun fixParents sub =
		let val ch = getChildren sub
		    val uid = getSubgoalUID sub
		    (* Add `id' to the list of parents of the given subgoal *)
		    fun addParent(Subgoal{parents=p,...}) = (p := uid::(!p))
		in List.app addParent ch
		end
	    (* Check for cycles in the proof tree (which should actually be a DAG).
	       Returns a cycle if found, or NONE. *)
	    fun findCycle tree =
		let val root = getRoot tree
		    val hash = makeHashDefault(op =, subgoalUID2string)
		    fun mark uid = insertHashDestructive(hash, uid, ())
		    fun inStack uid [] acc = NONE
		      | inStack uid (uid'::lst) acc =
			 if uid = uid' then SOME(uid::acc) else inStack uid lst (uid::acc)
		    fun loop stack sub =
			let val (uid, ch) = (getSubgoalUID sub, getChildren sub)
			in 
			   case inStack uid stack [] of
			       SOME lst => SOME lst
			     | NONE =>
				   (case findHash(hash, uid) of
					SOME _ => NONE
				      | NONE => (mark uid; looplist (uid::stack) ch))
			end
		    and looplist stack [] = NONE
		      | looplist stack (s::lst) =
			  (case loop stack s of
			       SOME x => SOME x
			     | NONE => looplist stack lst)
		in 
		    loop [] root
		end
	    fun getID tree uid = 
		(case UID2Subgoal tree uid of
		     SOME s => getSubgoalID s
		   | NONE => raise SympBug
			 ("ProofTreeType2ProofTree/getID: subgoal ["
			  ^(subgoalUID2string uid)^"] not found"))

	    val subgoals = List.map insertSubgoal lst
	    val _ = List.app fixChildren (root::lst)
	    val _ = List.app fixParents subgoals
	    val res = 
		(case findCycle tree of
		     NONE => tree
		   | SOME lst =>  raise ProverError
			 ("The proof tree contains a cycle:\n  "
			  ^(strlist2str " -> " (List.map (subgoalID2string o (getID tree)) lst))))
	    val _ = popFunStackLazy(funName, fn()=>ProofTree2string res)	    
	in res
	end
      | ProofTreeType2ProofTree x = raise SympBug
	  ("ProofTreeType2ProofTree: not PTproof:\n "
	   ^(ptt2str x))

    fun stream2ProofTree nameOpt ins =
	let val ptt = (ProofTreeParse nameOpt ins)
	              handle ProofTreeError msg => raise ProverError msg
	in 
	    ProofTreeType2ProofTree ptt
	end

    fun string2ProofTree str = stream2ProofTree NONE (TextIO.openString str)

    fun file2ProofTree filename =
	let val funName = "file2ProofTree"
	    val _ = pushFunStack(funName, filename)
	    val ins = (TextIO.openIn filename)
	                handle _ => raise SympError("Cannot open input file: "^filename)
	    val res = stream2ProofTree (SOME filename) ins
	    val _ = popFunStackLazy(funName, fn()=> ProofTree2stringDebug res)
	in 
	    res
	end

    fun UI2ProofTree arg = raise SympBug("UI2ProofTree: not implemented")

  end
