(* Proof Tree module.  The ProofTree type and its componentns are
   fully abstracted, so no accidental tempering with proof tree could
   happen *)

signature PROOF_TREE =
  sig
    structure ProofSystem: PROOF_SYSTEM
    structure Interface: INTERFACE
    sharing ProofSystem.Sequent.InputLanguage.Options = Interface.Options
    (* Inherited types *)
    type UserInputArgument = Interface.UserInputArgument
    type InferenceRule = ProofSystem.Sequent.InferenceRule
    type InferenceRuleArgument = ProofSystem.Sequent.InferenceRuleArgument
    type Sequent = ProofSystem.Sequent.Sequent
    type SequentHints = ProofSystem.Sequent.SequentHints

    (* ProofRule is an inference rule or tactics name (string) with
       specific arguments as the user gave them *)
    type ProofRule = string * (UserInputArgument list)

    (* The ProofTree type is (generally) a proof DAG with
       bi-directional pointers to children and a parent in each node,
       if there are any.

       A proof tree is a "box" that holds a proof.  There are no
       notions of a "subtree" in the representation, however, subtrees
       rooted at a particular subgoal can be extracted. *)

    type ProofTree

    (* A Subgoal is a handler for each node in a proof tree.  It
       doesn't have to contain the actual sequent (since one and the
       same proof may be applied to diff. theorems), but may contain
       some hints to what a sequent could be. *)

    type Subgoal
    (* A unique ID for each subgoal in a proof tree, computed automatically *)
    eqtype SubgoalID
    eqtype SubgoalUID

    (* A structure that binds a ProofTree to particular sequents
       resulting from applying the proof to a theorem.  *)
    type ProofRun

    (* Information gathering functions *)

    val getProofRule: Subgoal -> ProofRule option
    val getRoot: ProofTree -> Subgoal
    (* Return the entire proof tree this subgoal belongs to.
       This is NOT a subtree at this subgoal. *)
    val getProofTree: Subgoal -> ProofTree option
    val getChildren: Subgoal -> Subgoal list
    val getChildrenHints: Subgoal -> (Subgoal * SequentHints option) list
    val getParents: Subgoal -> Subgoal list
    val getSubgoalID: Subgoal -> SubgoalID
    val getSubgoalUID: Subgoal -> SubgoalUID
    (* Proof IDs are strings (e.g. theorem names) *)
    val getProofTreeID: ProofTree -> string

    val ProofRun2ProofTree: ProofRun -> ProofTree
    val getSequent: ProofRun -> Subgoal -> Sequent option

    (* A proof tree is considered "complete" when all leaves have an
       inference rule but no children *)
    val isProofTreeComplete: ProofTree -> bool

    (* A proof run is complete if the corresponding proof tree is
       complete, and every subgoal has an associated sequent *)
    val isProofRunComplete: ProofRun -> bool

    (* Check if the proof step at this subgoal is complete.  That is,
       the subgoal has a sequent and a proof rule, and all of its
       children have a sequent. *)
    val isProofStepComplete: ProofRun -> Subgoal -> bool

    (* Find the next unproven subgoal after the given subgoal, if given *)
    val nextUnprovenSubgoal: ProofRun * (Subgoal option) -> Subgoal option
    (* Find all unproven subgoals and return them in a list *)
    val getLeafSubgoals: ProofTree -> Subgoal list
    val getUnprovenSubgoals: ProofRun -> Subgoal list

    (* Extract a ProofTree rooted at the subgoal, and optionally
       provide a new proof tree id (string).  This will create a clean
       copy of the subtree, so any further modifications to it will
       not affect the original tree. *)
    val extractSubtree: string option -> Subgoal -> ProofTree option

    (* Delete the subtree originating at the subgoal, but leave the subgoal. *)
    val deleteSubtree: Subgoal -> unit

    (* Modifying the proof tree *)

    (* Create a new proof tree consisting of a single (empty) root
       subgoal with no inference rule, with an optional proof ID. *)
    val makeProofTree: string option -> ProofTree

    (* Make a clean copy of the proof tree.  The first argument is the
       new proof tree ID.  If it is omitted, a new unique name is
       generated automatically.  Note, that if the string is provided,
       it is not checked for uniqueness. *)
    val copyProofTree: string option -> ProofTree -> ProofTree

    (* Return the number of subgoals in the tree *)
    val ProofTreeSize: ProofTree -> int

    (* Create a new proof run for a clean COPY of the ProofTree.  Any
       modification to the proof tree results in the proof run being
       incomplete (usually being cut off at the modified subtree). *)

    val makeProofRun: ProofTree -> ProofRun

    (* Create a new Subgoal, optionally belonging to a proof tree with
       the given ID. *)
    val makeSubgoal: string option -> Subgoal

    (* Install the ProofTree or its clean copy (if the first arg is
       true) at the Subgoal in the current proof tree.  The subgoal
       replaces the root subgoal of the new ProofTree.  The old
       subtree rooted at the subgoal is deleted.  *)

    val installProofTree: bool -> Subgoal * ProofTree -> unit

    (* Add a ProofRule to the Subgoal that produces a Subgoal
       list.  The existing subtree rooted at the Subgoal is deleted. 

       This is the only function that builds a proof tree structure.
       You cannot change the inference rule or the subgoal without
       changing the entire subtree, nor you can change any information
       at the subgoal.

       The new subgoals must either belong to the current proof tree,
       or be completely new. *)

    val addProofRule: Subgoal * ProofRule
	    * ((Subgoal * (SequentHints option)) list) -> unit

    (* Associate a Sequent with a Subgoal in the ProofRun.  This
       overrides an existing sequent for that subgoal, if any. *)

    val addSequent: ProofRun -> Subgoal * Sequent -> unit

    (* Printing and some UI functions *)

    (* Print all the info about subgoal *)
    val subgoal2string: Subgoal -> string
    val subgoal2stringDebug: Subgoal -> string
    (* Print an ID of the subgoal (e.g. `1.4.5' like in PVS) *)
    val subgoalID2string: SubgoalID -> string
    val subgoalUID2string: SubgoalUID -> string

    (* Reverse translation from strings *)
    val string2subgoalID: string -> SubgoalID option

    (* Extract the Subgoal from the ProofTree by its string ID *)
    val ID2Subgoal: ProofTree -> SubgoalID -> Subgoal option

    (* Proof tree I/O functions must be one-to-one, so that a proof
       tree could be completely restored from the string or the
       `UserInputArgument' value *)
    val ProofTree2str: ProofTree -> Str.Str
    val ProofTree2string: ProofTree -> string
    val ProofTree2strDebug: ProofTree -> Str.Str
    val ProofTree2stringDebug: ProofTree -> string
    val ProofTree2file: ProofTree * string -> unit

    val ProofRun2str: ProofRun -> Str.Str
    val ProofRun2string: ProofRun -> string
    val ProofRun2strDebug: ProofRun -> Str.Str
    val ProofRun2stringDebug: ProofRun -> string

    (* For more efficient communication with GUI/Emacs/whatever *)
    val ProofTree2UI: ProofTree -> UserInputArgument

    (* And the reverse translation: build a proof tree from the
       `string' or `UserInputArgument'.  This, of course, may raise an
       exception if the input is invalid. *)

    val UI2ProofTree: UserInputArgument -> ProofTree
    val string2ProofTree: string -> ProofTree
    val file2ProofTree: string -> ProofTree

  end
