(* This is an abstract interface to a generic proof system.  Each
   implementation is specific to the input laguage, the sequent
   representation, the proof system, etc..  The implementation is also
   a bridge from a custom representation of each language and proof
   system to the uniform API used by the rest of the prover. *)

signature PROOF_SYSTEM =
  sig
    structure Sequent: SEQUENT
    type Sequent = Sequent.Sequent
    type SequentHints = Sequent.SequentHints
    type InferenceRule = Sequent.InferenceRule
    type InferenceRuleResult = Sequent.InferenceRuleResult
    type InferenceRuleArgument = Sequent.InferenceRuleArgument
    type InferenceRuleArgumentSpec = Sequent.InferenceRuleArgumentSpec

    type ProofSystemCommand = Sequent.ProofSystemCommand
    type ProofSystemCommandResult = Sequent.ProofSystemCommandResult

    type Substitution = Sequent.Substitution
    type ProgramContext = Sequent.ProgramContext

    type ProofSystemTactic

    (* Internal state of a tactic to be passed back to `applyTactic'
       when in the middle of the tactic execution *)
    type ProofSystemTacticState

    (* Must be run once at the program startup to initialize the proof system *)
    val initProofSystem: unit -> unit

    val findRuleByName: string -> InferenceRule option
    val findCommandByName: string -> ProofSystemCommand option
    val findTacticByName: string -> ProofSystemTactic option

    (* Try to apply the rule and get a list of new sequents as
       premisses.  NONE means the rule doesn't apply.  May take a
       (partial) substitution obtained by the matchRule funciton.
       The first argument is a printing function to send messages to
       the user while applying the rule. *)
    val applyRule: (string -> unit) -> (Sequent * ProgramContext)
	        -> InferenceRule * (InferenceRuleArgument list) * (Substitution option)
	        -> InferenceRuleResult

    (* Run the proof sytem command on the sequent.  Commands normally
       do not change the sequent or the proof tree, and are only used to
       display some extra information about the sequent. *)
    val applyCommand: (string -> unit) -> (Sequent * ProgramContext)
	        -> ProofSystemCommand * (InferenceRuleArgument list)
	        -> ProofSystemCommandResult option

    (* applyTactic inline printFun (tactic, args, context, stateOpt) (seq, hintsOpt)

       Applies `tactic' to the sequent `seq'.  If `inline' is true,
       include each atomic step in the current proof, otherwise keep
       the tactic application as a single step.  This can be an
       intermediate step within the tactic from an optional state
       `stateOpt'.  Actions may also depend on the optional sequent
       hints.  If any rule has been applied to the sequent, its name,
       args, and newly generated sequents with hints will be returned.
       If also a new state is returned, this means the tactic has not
       finished, and it has to be called again with this state on each
       of the new sequents. *)

    val applyTactic: bool -> (string -> unit)
	        -> (ProofSystemTactic * (InferenceRuleArgument list) * ProgramContext
		    * (ProofSystemTacticState option))
                -> (Sequent * (SequentHints option))
	        (* Result *)
	        -> (string * (InferenceRuleArgument list)
		    * ((Sequent * (SequentHints option)) list)
		    * (ProofSystemTacticState option)) option

    (* Each rule has a unique string identifier *)
    val getRuleName: InferenceRule -> string
    val getCommandName: ProofSystemCommand -> string
    val getTacticName: ProofSystemTactic -> string
    (* Help on the rule *)
    val getRuleHelp: InferenceRule -> string
    val getCommandHelp: ProofSystemCommand -> string
    val getTacticHelp: ProofSystemTactic -> string
    (* Extracting argument "types" from a tactic *)
    val getTacticArgs: ProofSystemTactic -> InferenceRuleArgumentSpec list
    (* Add new inference rule, possibly overriding an existing one with the same name *)
    val addNewRule: InferenceRule -> unit
    val addNewCommand: ProofSystemCommand -> unit
    val addNewTactic: ProofSystemTactic -> unit
    (* Get the list of all the rules in the system *)
    val getAllRules: unit -> InferenceRule list
    val getAllCommands: unit -> ProofSystemCommand list
    val getAllTactics: unit -> ProofSystemTactic list
  end
