functor InterfaceFun (structure Options: OPTIONS
		      structure Hash: HASH) : INTERFACE =
  struct

    exception SympError of string
    exception ParseError of string
    exception TypeError of string
    exception SequentError of string
    exception ProverError of string
    exception ProofTreeError of string
    exception TransError of string



    (* Clean exit on user's request *)
    exception SympExit
    (* Exit with error (some fatal error, or killing a run-away thread) *)
    exception SympExitError

    open SympBug
    open Str

    structure Options = Options
    structure Hash = Hash
    structure TimingStat = TimingStatFun(structure Hash = Hash)

    open Options

    (* The unified type of user inputs.  The input from any interface
       is translated to this datatype before becoming visible to the
       rest of the system. *)
    datatype UserInputArgument =
	UIstring of string
      | UInumber of int
      (* Associate two args, e.g. for named arguments with values *)
      | UIassoc of UserInputArgument * UserInputArgument
      | UIlist of UserInputArgument list
      | UItrue
      | UIfalse

    datatype UserInput =
	UIyes | UIno (* yes/no answers from the user *)
      | UIexit (* Terminate the entire program, exit from ML *)
      (* No input.  E.g. the user wants to use default values and just
         hits <Enter>; or we are in commandLine (batch) mode. *)
      | UInull
      (* Any other command in the form ("command name", [<arg list>]) *)
      | UIcommand of string * (UserInputArgument list)

    structure EmacsServerInterface = EmacsServerInterfaceFun()
    open EmacsServerInterface
    open EmacsServerStruct

    local 
	val options = ref defaultOptions
    in
	fun setOptions newOptions = (options := newOptions)
	fun getOptions() = !options
    end

    val debugList =
	["all",  (* Print all the debugging info available *)
	 "ReadUserCommand",
	 "applyProverCommand",
	 "common",
	 (* General interface *)
	 "lazyFun",
	 (* Common functions for different proof systems *)
	 "ParseFile",
	 (* Prover *)
	 "readStatusUpdate",
	 "applyRule",
	 "proveTheorem",
	 "matchTerm",
	 "matchTerm/loop",
	 "matchTerms",
	 "matchTerms/loop",
	 "matchTerms/matchFilter",
	 "matchTerms/matchFirst",
	 "matchSeq",
	 "skolemMatch",
	 "skolemApply",
	 "updateSubstitution",
	 "substTermID",
	 "SequentDefault.subst",
	 "SequentDefault.subst/loop",
	 "SequentDefault.subst/constrValue",
	 "substSequent",
	 "updateSubstitutionRaw",
	 (* Proof Tree module in the prover *)
	 "extractSubtree",
	 "extractSubtree/loop",
	 "copyProofTree",
	 "getUnprovenSubgoals",
	 "nextUnprovenSubgoal",
	 "nextUnprovenSubgoal/loop",
	 "delParent",
	 "garbageCollect",
	 "garbageCollect/loop",
	 "installProofTree",
	 "addProofRule",
	 "addSequent",
	 "term2termID",
	 "runInstalledProof",
	 "runInstalledProof/loop",
	 "runInstalledProof/loop/runStep",
	 "runInstalledProof/loop/compareAndInstall",
	 "ProofTreeType2ProofTree",
	 "file2ProofTree",
	 (* Default Proof System *)
	 "SequentDefault.applyRule",
	 "completeRuleArgs",
	 "compileArgs",
	 "replaceApply",
	 "ProofRules.split.apply",
	 "ProofRules.split.apply/splitPair",
	 "ProofRules.abstractSplit.apply",
	 (* Typechecking in the default proof system *)
	 "stat",
	 "findinContextCommon",
	 "findinContextCommonDebug",
	 "finalizeExpr",
	 "finalizeExpr/loop",
	 "expandType",
	 "instantiateModule",
	 "instantiateModule/resolveParam",
	 "instantiateModule/loop",
	 "unifyTypes",
	 "coerce",
	 "tcName",
	 "tcObject",
	 "tcNameDebug",
	 "tc",
	 "tcDebug",
	 "typeCheckPattern",
	 "tcPattern/loop",
	 "typeCheckModuleExpr",
	 "typeCheckModuleExprDebug",
	 "tcModuleName",
	 "tcModuleInst",
	 "tcModuleExpr",
	 "typeCheckStatParam",
	 "typeCheckType",
	 "typeCheckType/loop",
	 "evaluateExpr",
	 "evaluateExpr/recurCall",
	 "Evaluate.subs",
	 "Evaluate.applyFun",
	 (* Abstraction *)
	 "getTypeValues",
	 "getTypeValues/loop",
	 (* Translation: common functions *)
	 "addToCone",
	 "updateConeDestructive",
	 "inConeOfCommon",
	 "getVarCone",
	 "getVarConeHashCommon",
	 "getVarConeHashCommon/checkCycle",
	 "getVarConeHashCommon/loop",
	 "getVarCone1",
	 "getVarCone1Delayed",
	 "getVarCone1Delayed/split",
	 (* Translation: preparation phase (translating to AsstVarsTree) *)
	 "splitEType",
	 "varsFromDelayed",
	 "varsFromDelayed/loop",
	 "collectAsstVars",
	 "splitType",
	 "splitType/loop",
	 "extractAsstTree",
	 "rebuildAsstVars",
	 "rebuildAsstVars/collect",
	 (* Specialization phase (specializing trans. rel. to the property, cone of influence) *)
	 "customizeAsstVars",
	 "customizeAsstVars0",
	 "coneFromVar",
	 "coneFromVar/loop",
	 (* Generation phase (translation to boolean formulas) *)
	 "substVarValuesPairs",
	 "TransGen.matches",
	 "TransGen.matchesDebug",
	 "walkTree",
	 "eqFun",
	 "eqFun/loop",
	 "eqPrim",
	 "wrapPrim",
	 "asstGen",
	 "asstGen1",
	 "asstGen1Debug",
	 "asstGen2",
	 "asstGen3",
	 "asstGen4",
	 "eqBuiltin",
	 (* Generating SMV code *)
	 "SMVcode",
	 "formulaStr",
	 "formulaStr/formStr",
	 "TransSMV.getMap",
	 "TransSMV.getTVMap",
	 (* Athena *)
	 "ParserAthenaFun.Parse",
	 "Athena:typeCheckProgram"]
    

    (* Run ff on every atomic substring (ff is usually `print' or a similar function) *)
    fun walkStr ff (Conc(s1,s2)) = (walkStr ff s1; walkStr ff s2)
      | walkStr ff (Str s) = ff s

    fun printerr str = 
        (TextIO.output(TextIO.stdErr,str);
	 TextIO.flushOut(TextIO.stdErr))

    fun printerrStr str =
	let fun ff str = TextIO.output(TextIO.stdErr,str)
	in (walkStr ff str;
	    TextIO.flushOut(TextIO.stdErr))
	end

    fun printEmacs key str =
	(print(EmacsExpr2stringServer(EmacsList[EmacsSymbol key, EmacsString str]));
	 TextIO.flushOut(TextIO.stdOut))

    (* Convert `UserInputArgument' value to `EmacsExpr' *)
    fun UIarg2emacs (UIstring s) = EmacsString s
      | UIarg2emacs (UInumber n) = EmacsNumber n
      (* Association is translated into a list (a1 = a2), where 
         `=' is a lisp symbol. *)
      | UIarg2emacs (UIassoc(a1,a2)) =
	  EmacsList[UIarg2emacs a1, EmacsSymbol "=", UIarg2emacs a2]
      | UIarg2emacs (UIlist lst) = EmacsList(List.map UIarg2emacs lst)
      | UIarg2emacs UItrue = EmacsTrue
      | UIarg2emacs UIfalse = EmacsNil

    (* Convert `UserUnput' value to `EmacsExpr' *)
    fun UI2emacs UIyes = EmacsTrue
      | UI2emacs UIno = EmacsNil
      | UI2emacs UIexit = EmacsSymbol "exit"
      | UI2emacs UInull = EmacsNil
      | UI2emacs (UIcommand(c, args)) = EmacsList((EmacsSymbol c)::(List.map UIarg2emacs args))

    fun printEmacsCommand command =
	(print(EmacsExpr2stringServer(UI2emacs command));
	 TextIO.flushOut(TextIO.stdOut))

    (* This needs to be rewritten more efficiently *)
    fun printEmacsStr key str = printEmacs key (Str2string str)
(*     fun printEmacsStr key str =
	(print("\000("^key^" \"");
	 walkStr (print o str2emacs) str;
	 print("\")\001\n");
	 TextIO.flushOut(TextIO.stdOut)) *)

    fun UIArg2string UItrue = "True"
      | UIArg2string UIfalse = "False"
      | UIArg2string (UIstring s) = "\""^s^"\""
      | UIArg2string (UInumber n) = Int.toString n
      | UIArg2string (UIassoc(a1,a2)) =
	  (UIArg2string a1)^" = "^(UIArg2string a2)
      | UIArg2string (UIlist lst) = "["^(strlist2str ", " (List.map UIArg2string lst))^"]"

    fun UI2string UIyes = "Yes"
      | UI2string UIno = "No"
      | UI2string UInull = "<No input>"
      | UI2string UIexit = "Exit"
      | UI2string (UIcommand(c,args)) =
	  c^"("^(strlist2str ", " (List.map UIArg2string args))^")"

    (* Format a prover command for a particular interface *)
    fun formatProverCommand commandLine (name, args) = (name^"("^(strlist2str ", " args)^")")
      | formatProverCommand Emacs (name, args) =
	let val sep = (case args of
			   [] => ""
			 | [_] => " "
			 | _ => "\n   ")
	in
	    ("("^name^sep^(strlist2str "\n   " args)^")")
	end

    fun printOut ({interface=Emacs, ...}: options) key str = printEmacs key str
      | printOut {interface=commandLine,...} _ str = printerr(str^"\n")

    fun printOutStr ({interface=Emacs, ...}: options) key str = printEmacsStr key str
      | printOutStr {interface=commandLine,...} _ str = printerrStr(Conc(str, Str "\n"))

    fun printOutInput ({interface=Emacs, ...}: options) command =
	  printEmacsCommand command
      | printOutInput {interface=commandLine,...} command = 
	  printerr (UI2string command)

    fun printOutCommand ({interface=Emacs, ...}: options) (command, args) =
	  printEmacsCommand(UIcommand(command, args))
      | printOutCommand {interface=commandLine,...} (command, args) = 
	  printerr (UI2string(UIcommand(command, args)))

    fun printError (options: options) tp str =
	  printOutCommand options ("error",[UIstring tp, UIstring str])
    fun printErrorStr (options: options) tp str =
	  printOutCommand options ("error",[UIstring tp, UIstring(Str2string str)])
(*  	  printOutStr options "error" str *)

    local
	(* ID for `userchoice' prompts *)
	val promptId = ref(0)
	val prompts = ref([]: (int * (string -> unit)) list)
    in
	fun insertPrompt(id, action) = 
	      prompts := (id, action)::(!prompts)
	fun removePrompt id =
	      prompts := List.filter(fn(x,_) => x<>id) (!prompts)
	fun findPrompt id = List.find(fn(x,_) => x=id) (!prompts)

	(* Prompt the user with a question, a possibly strict choice,
	   and an action (function taking the choice and returning
	   unit), and return the prompt ID *)

	fun userChoice options (choiceType, message, choices, action) =
	    let val id = !promptId before (promptId := (!promptId + 1))
	    in (insertPrompt(id,action);
		printOutCommand options ("userchoice",
					 [UInumber id, UIstring choiceType, UIstring message]
					 @(List.map (fn x=>UIstring x) choices));
		id)
	    end

	(* The user returns his choice, execute the corresponding
	   action and remove it from the hash *)
	fun acceptUserChoice(id, choice) =
	    let val (_, action) = (case findPrompt id of
				  NONE => raise SympError
				    ("Unexpected user response, id = "^(Int.toString id))
				| SOME a => a)
	    in (removePrompt id;
		action choice)
	    end

	fun cancelUserChoice id = removePrompt id
    end

    fun verb (options: options) str = 
	let val { verbose=verbose, ... } = options
	in 
	    if verbose then printOut options "verbose" str
	    else ()
	end

    fun verbStr (options: options) str = 
	let val { verbose=verbose, ... } = options
	in 
	    if verbose then printOutStr options "verbose" str
	    else ()
	end

    fun lazyVerb (options: options) str = 
	let val { verbose=verbose, ... } = options
	in 
	    if verbose then printOut options "verbose" (str())
	    else ()
	end

    fun lazyVerbStr (options: options) str = 
	let val { verbose=verbose, ... } = options
	in 
	    if verbose then printOutStr options "verbose" (str())
	    else ()
	end

    fun verbDebug (options: options) name str = 
	let val { debug=debug, ... } = options
	    fun doit lst = (name="") 
		           orelse List.exists(fn x=>x="all") lst
			   orelse List.exists(fn x=>x=name) lst
	in (case debug of
		NONE => ()
	      | SOME(lst) => (if doit lst then printOut options "debug" str else ()))
	end

    fun verbDebugStr (options: options) name str = 
	let val { debug=debug, ... } = options
	    fun doit lst = (name="") 
		           orelse List.exists(fn x=>x="all") lst
			   orelse List.exists(fn x=>x=name) lst
	in (case debug of
		NONE => ()
	      | SOME(lst) => (if doit lst then printOutStr options "debug" str else ();
				  TextIO.flushOut(TextIO.stdErr)))
	end

    fun lazyVerbDebug (options: options) name strFun = 
	let val { debug=debug, ... } = options
	    fun doit lst = (name="") 
		           orelse List.exists(fn x=>x="all") lst
			   orelse List.exists(fn x=>x=name) lst
	in (case debug of
		NONE => ()
	      | SOME(lst) => (if doit lst then printOut options "debug" (strFun()) else ()))
	end

    fun lazyVerbDebugStr (options: options) name strFun = 
	let val { debug=debug, ... } = options
	    fun doit lst = (name="") 
		           orelse List.exists(fn x=>x="all") lst
			   orelse List.exists(fn x=>x=name) lst
	in (case debug of
		NONE => ()
	      | SOME(lst) => (if doit lst then printOutStr options "debug" (strFun()) else ()))
	end

    fun printCommonUsage (options: options) = 
	"  Nothing to report for the interface...\n"
	
    fun catastrophe (options: options) f str =
	(printOut options "bug" 
	 ("\n\n*** internal error ***\n"^str
	  ^"\n\nPlease report this error to sergey.berezin@cs.cmu.edu\n"
	 ^"Send a copy of this output and your input.\n");
	 if isSome(f) then (valOf f)() else ();
	 raise SympError("*** internal error ***\n"^str))

    fun reportError (options: options) str = raise SympError str

    fun reportWarning (options: options) str =
	   printOut options "warning" ("\nWarning: "^str^"\n")

    (* Reads the user's command through the appropriate user interface
       (currently stdin), and convert it to `UserInput' type *)

    fun ReadUserCommand (options: options) () =
	let val debug = lazyVerbDebug options "ReadUserCommand"
	    val _ = debug(fn()=>"\nReadUserCommand: reading...")
	    val {interface=UI,...} = options
	    fun EmacsList2list EmacsNil = []
	      | EmacsList2list (EmacsCons(x,y)) = x::(EmacsList2list y)
	      | EmacsList2list x = raise SympBug
		  ("ReadUserCommand/EmacsList2list: not a list element:\n  "
		   ^(EmacsExpr2string x))
	    (* Convert emacs expression to an argument *)
	    fun emacs2arg (EmacsString s) = UIstring s
	      | emacs2arg (EmacsSymbol s) = UIstring s
	      | emacs2arg (EmacsNumber n) = UInumber n
	      | emacs2arg EmacsNil = UIfalse
	      | emacs2arg EmacsTrue = UItrue
	      (* First, catch all associations - lists with `=' as a
	         second element *)
	      | emacs2arg (e as EmacsCons _) = 
		let val lst = EmacsList2list e
		in 
		    case lst of
			x::(EmacsSymbol "=")::rest =>
			    UIassoc(emacs2arg x, 
				    case rest of
					[y] => emacs2arg y
				      | _ => UIlist(emacs2args rest))
		      | _ => UIlist(emacs2args lst)
		end
	    and emacs2args lst = List.map emacs2arg lst
	    fun EmacsList2args lst = emacs2args(EmacsList2list lst)
	    val res = 
		(case UI of
		     (* Command line interface is non-interactive *)
		     commandLine => UInull
		   | Emacs => 
			 let val expr =(EmacsExprParse TextIO.stdIn 
					handle EmacsServerError str => 
					    raise SympError("Emacs Server Error: "^str))
			     val _ = debug(fn()=>"\nReadUserCommand: parsed input = "
					   ^(EmacsExpr2string expr)^"\n")
			 in (case expr of
				 EmacsNil => UIno
			       | EmacsTrue => UIyes
			       | EmacsSymbol "exit" => UIexit
			       | EmacsSymbol "quit" => UIexit
			       | EmacsSymbol s => UIcommand(s,[])
			       | EmacsCons(c,args) => 
				     UIcommand(EmacsExpr2string c, EmacsList2args args)
			       | _ => UInull)
			 end)
	    val _ = debug(fn()=>"\nReadUserCommand => "^(UI2string res)^"\n")
	in res
	end

    (* Debugging engine: function call stack *)
    local 
	val stack = ref[]
	fun lazyFun ff =
	    let val res = ref NONE
		val debug = lazyVerbDebug(getOptions()) "lazyFun"
	    in
		fn() => (case !res of
			     NONE => (res := SOME(ff());
				      debug(fn()=>("\nlazyFun: Evaluating crap\n"
						   ^(valOf(!res))));
				      valOf(!res))
			   | SOME x => x)
	    end
    in 
	fun pushFunStackLazy(name, commentsFun) = 
	    let val options as { debug=debug, ...} = getOptions()
		val lazyComments = lazyFun commentsFun
		val _ = lazyVerbDebug options name (fn()=>"\n"^name^"("^(lazyComments())^")[\n")
	    in
		case debug of
		    NONE => ()
		  | SOME _ => stack := (name, lazyComments)::(!stack)
	    end
	(* Pop all the elements up to and including the function `name' *)
	fun popFunStackLazy(name, commentsFun) =
	    let val options as { debug=debug, ...} = getOptions()
		val verb = lazyVerbDebug options
	    in
		case debug of
		    NONE => ()
		  | SOME _ => 
			(case !stack of
			     [] => ()
			   | (ff, ffCom)::lst => 
				 (stack := lst;
				  if ff = name then 
				      verb name (fn()=>"\n"^name^" => "^(commentsFun())^"]\n")
				  else (verb ff (fn()=>"\n"^ff^" exited abnormally]\n");
					popFunStackLazy(name, commentsFun))))
	    end
	fun pushFunStack(name, comments) = pushFunStackLazy(name, fn()=>comments)
	fun popFunStack(name, comments) = popFunStackLazy(name, fn()=>comments)
	(* Return the current stack *)
	fun getFunStack() = !stack
	fun resetFunStack() = stack := []
	fun FunStack2string() = 
	    let fun pair2str(name, commentsFun) = name^"("^(commentsFun())^")"
	    in
		"The function call stack:\n\n"
		^(strlist2str "\n\n" (List.map pair2str (!stack)))
	    end
    end
  end
