signature INTERFACE =
  sig
    structure Options: OPTIONS
    structure TimingStat: TIMING_STAT

    exception SympError of string
    exception ParseError of string
    exception TypeError of string
    exception SequentError of string
    exception ProverError of string
    exception ProofTreeError of string
    exception TransError of string

    (* Clean exit on user's request *)
    exception SympExit
    (* Exit with error (some fatal error, or killing a run-away thread) *)
    exception SympExitError

    (* The unified type of user inputs.  The input from any interface
       is translated to this datatype before becoming visible to the
       rest of the system. *)
    datatype UserInputArgument =
	UIstring of string
      | UInumber of int
      (* Associate two args, e.g. for named arguments with values *)
      | UIassoc of UserInputArgument * UserInputArgument
      | UIlist of UserInputArgument list
      | UItrue
      | UIfalse

    datatype UserInput =
	UIyes | UIno (* yes/no answers from the user *)
      | UIexit (* Terminate the entire program, exit from ML *)
      (* No input.  E.g. the user wants to use default values and just
         hits <Enter>. *)
      | UInull
      (* Any other command in the form ("command name", [<arg list>]) *)
      | UIcommand of string * (UserInputArgument list)

    (* Set and access global options *)
    val setOptions: Options.options -> unit
    val getOptions: unit -> Options.options
    (* The list of names available for debugging.  These names are the
       ones that can be added to the `debug' option. *)
    val debugList: string list

    val ReadUserCommand: Options.options -> unit -> UserInput
    val UIArg2string: UserInputArgument -> string
    val UI2string: UserInput -> string

    (* Format a prover command for a particular interface *)
    val formatProverCommand: Options.UserInterface -> string * (string list) -> string

    (* Printing functions *)

    (* printOut[Str] options tag message.  `tag' is a message type,
       like "error", "verbose", etc. *)
    val printOut: Options.options -> string -> string -> unit
    val printOutStr: Options.options -> string -> Str.Str -> unit
    (* Send an arbitrary command to the abstract user interface *)
    val printOutInput: Options.options -> UserInput -> unit
    val printOutCommand: Options.options -> string * (UserInputArgument list) -> unit

    (* Print a message tagged as "error" of a certain type:
       printError options error_type message *)
    val printError: Options.options -> string -> string -> unit
    val printErrorStr: Options.options -> string -> Str.Str -> unit

    (* Interactive multi-choice dialog with the user *)

    (* Prompt the user with a question, a possibly strict choice, and
       an action (function taking the choice and returning unit), and
       return the prompt ID:
       userChoice options (type, message, choices, action),
       type is one of { "any", "strict", "file", "yesno", "confirm" }. *)
    val userChoice: Options.options -> string * string * (string list) * (string -> unit) -> int
    (* The user returns his choice, execute the corresponding
       action and remove it from the hash *)
    val acceptUserChoice: int * string -> unit
    (* Cancel a choice and its action *)
    val cancelUserChoice: int -> unit
    (* Print short one-line messages *)
    val verb: Options.options -> string -> unit
    val verbStr: Options.options -> Str.Str -> unit
    (* Same, but doesn't compute the string if not needed *)
    val lazyVerb: Options.options -> (unit -> string) -> unit
    val lazyVerbStr: Options.options -> (unit -> Str.Str) -> unit
    (* (verbDebug options funName debug-message)
       Value funName="" matches any value in the debug list. *)
    val verbDebug: Options.options -> string -> string -> unit
    val verbDebugStr: Options.options -> string -> Str.Str -> unit
    val lazyVerbDebug: Options.options -> string -> (unit -> string) -> unit
    val lazyVerbDebugStr: Options.options -> string -> (unit -> Str.Str) -> unit
    (* printUsage options (trans,invar,reachableStates,typeMask,NumBddVars)
       prints all kinds of statistics *)
    val printCommonUsage: Options.options -> string
    (* Takes an optional function to execute before exiting *)
    val catastrophe: Options.options -> (unit -> unit) option -> string -> 'a
    val reportError: Options.options -> string -> 'a
    val reportWarning: Options.options -> string -> unit


    (* Debugging engine: function call stack.  The push/pop functions
       take effect only if debugging is enabled.  They also print the
       trace for those functions mentioned in the `debug' option.

       The second argument is a comment to print.  In the lazy
       version, the comment is computed by a function which will only
       be evaluated ONCE, and only if the comment is actually
       needed. *)

    val pushFunStackLazy: string * (unit -> string) -> unit
    val popFunStackLazy: string * (unit -> string) -> unit
    val pushFunStack: string * string -> unit
    val popFunStack: string * string -> unit

    (* Returns the current stack contents.  Each element in the list is a
       pair (functionName: string, commentFun: unit -> string). *)

    val getFunStack: unit -> (string * (unit -> string)) list
    val resetFunStack: unit -> unit
    val FunStack2string: unit -> string
  end 

