functor InteractFun(structure Version: VERSION): INTERACT =
  struct
    structure Hash: HASH = HashFun()

    (* Command line options *)
    structure Options = OptionsFun()

    (* Parsing, reporting errors, etc. *)
    structure Interface
	= InterfaceFun(structure Options = Options
		       structure Hash = Hash)

    structure ProversJoin = 
	ProversJoinFun(structure Hash = Hash
		       structure Interface = Interface)

    open SympBug
    open Str
    open Options
    open Interface
    open Hash
    open ProversJoin
    open Version

    type SympState = unit

    local 
	fun initState() = ()
	val state = ref(initState())
    in
	fun conv ff args = ff(getOptions()) args
	val printError = conv printError 
	val printOut = conv printOut
	val printOutCommand = conv printOutCommand
	val verb = conv verb
	val verbStr = conv verbStr
	val lazyVerbDebug = conv lazyVerbDebug
	val verbDebug = conv verbDebug
	val lazyVerbDebugStr = conv lazyVerbDebugStr

	fun initSymp options =
	    (setOptions options;
	     initProvers())
	     

	(* Resets the entire tool.  Not implemented yet. *)
	fun ResetSympState() =
	    (resetFunStack())

	fun executeCommandOptions(display, args) =
	    let val command = if display then "displayoptions" else "options"
		val allOptions = ["verbose", "interface", "debug", "limit", "backendmc",
				  "outputfile"]
		fun checkArgs([], change, display) = (change, display)
		  | checkArgs((UIstring field)::args, change, display) =
		     checkArgs(args, change, field::display)
		  | checkArgs((UIassoc(UIstring field, v))::args, change, display) =
		     checkArgs(args, (field, v)::change, display)
		  | checkArgs(x::_, _, _) = raise SympBug
		     ("Wrong parameter from the external user interface in the `"
		      ^command^"' command:\n  "
		      ^(UIArg2string x))
		val (changed, displayed) = checkArgs(args, [], [])
		(* Construct UIassoc pair for the value of one option *)
		fun displayArg "verbose" = 
		      UIassoc(UIstring "verbose",
			      if get_verbose(getOptions()) then UItrue else UIfalse)
		  | displayArg "interface" =
		      UIassoc(UIstring "interface",
			      case get_interface(getOptions()) of
				  commandLine => UIstring "commandLine"
				| Emacs => UIstring "Emacs")
		  | displayArg "debug" =
		      UIassoc(UIstring "debug",
			      case get_debug(getOptions()) of
				  NONE => UIfalse
				| SOME lst => UIlist(List.map(fn x=>UIstring x) lst))
		  (* Skip "spec" - it doesn't make much sense anymore *)
		  | displayArg "limit" =
		      UIassoc(UIstring "limit", UInumber(get_limit(getOptions())))
		  | displayArg "backendmc" =
		      UIassoc(UIstring "backendMC", UIstring(get_backendMC(getOptions())))
		  | displayArg "outputfile" =
		      UIassoc(UIstring "outputFile", UIstring(get_outputFile(getOptions())))
		  | displayArg x = raise SympError("No such option: "^x)
		fun updateOption("verbose", v) =
		    let val newV = (case v of
					UItrue => true
				      | UIstring "t" => true
				      | UIstring "true" => true
				      | UIstring "yes" => true
				      | _ => false)
		    in setOptions(upd_verbose(getOptions(), newV))
		    end
		  | updateOption("debug", v) =
		    let fun convertOne(UIstring s) = s
			  | convertOne x = raise SympError
			      ("not a string element in `debug' option: "^(UIArg2string x))
			val oldV = get_debug(getOptions())
			val newV = (case v of
					UIfalse => NONE
				      | UIstring "no" => NONE
				      | UItrue => SOME[]
				      | UIstring "yes" => SOME[]
				      | UIstring "true" => SOME[]
				      | UIstring "t" => SOME[]
				      | UIlist lst => SOME(List.map convertOne lst)
				      (* Can't parse the value, leave it unchanged *)
				      | _ => oldV)
		    in
			setOptions(upd_debug(getOptions(), newV))
		    end
		  | updateOption("limit", v) =
		    let val newV = (case v of
					UInumber n => n
				      | _ => raise SympError
					("Option `limit' requires a number, but was given this: "
					 ^(UIArg2string v)))
		    in setOptions(upd_limit(getOptions(), newV))
		    end
		  | updateOption("backendmc", v) =
		    let val newV = (case v of
					UIstring n => n
				      | _ => raise SympError
				      ("Option `backendMC' requires a string, but was given this: "
				       ^(UIArg2string v)))
		    in setOptions(upd_backendMC(getOptions(), newV))
		    end
		  | updateOption (x, _) = raise SympError
		      ("Can't modify option "^x
		       ^"\nIt either doesn't exist, or cannot be changed dynamically.")
	    in
		(List.map updateOption changed;
		 (case (changed, displayed) of
		      ([], []) => (printOutCommand(command,
					   List.map displayArg allOptions))
		    | (_, []) => ()
		    | _ => printOutCommand(command,
					   List.map displayArg displayed)))
	    end

	fun executeCommand UInull = verb("Null command")
	  | executeCommand UIexit = 
	    let val options = getOptions()
		fun confirm "yes" = (verb "Exiting...done";raise SympExit)
		  | confirm _ = verb "Exiting...cancelled"
		fun userLastConfirm() = 
		    (userChoice options ("yesno",
					 "Are you absolutely sure you want to quit SyMP? ",
					 [], confirm); ())
		fun sureYN "yes" = exitAllProvers (getOptions()) userLastConfirm
		  | sureYN _ = verb "Exiting...cancelled"
		fun userConfirm() = 
		    (userChoice options ("confirm", "You want to quit SyMP? ", [], sureYN); ())
	    in (verb("Exiting..."); userConfirm())
	    end
	  | executeCommand UIyes = verb("Yes!..")
	  | executeCommand UIno = verb("No!..")
	  | executeCommand (c as UIcommand("typecheck", args)) =
	      let fun parseArgs[UIstring file, UIstring tp] = (file, SOME(str2ProverType tp))
		    | parseArgs[UIstring file] = (file, NONE)
		    | parseArgs _ = raise SympError
		       ("executeCommand: wrong arguments to typecheck:\n  "
			^(UI2string c)
			^"\n format: (typecheck \"fileName\"  [ \"proverType\" ])")
		  val (file, tp) = parseArgs args
		  fun cont() = typeCheckFile tp (getOptions()) file
	      in
		  (verb("Preparing for typechecking...");
		   exitAllProvers (getOptions()) cont)
	      end
	  | executeCommand (UIcommand("version",_)) = printOut "version" version

	  (* `prover' commands always come from a particular prover
	     interface (e.g. prover emacs buffer), and must have the
	     prover ID and the command in the list of arguments *)

	  | executeCommand (UIcommand("prover",(UIstring id)::(UIstring command)::lst)) =
	      ProverCommand (getOptions()) (SOME id, command, lst)
	  | executeCommand (c as UIcommand("prover",lst)) = raise SympError
	      ("executeCommand: wrong arguments to `prover' command:\n  "
	       ^(UI2string c)
	       ^"\n format: (prover \"proverID\" \"command\" [ args ])")
	  | executeCommand (UIcommand("proverstatus", [UIstring id])) =
	      (* Call the prover's status command in a silent mode *)
	      ProverCommand (getOptions()) (SOME id, "status", [UIstring "true"])
	  | executeCommand (c as UIcommand("proverstatus", args)) =
	      raise SympError
		  ("executeCommand: wrong arguments to `proverstatus' command:\n  "
		   ^(UI2string c)
		   ^"\n format: (proverstatus \"proverID\")")
	  | executeCommand (c as UIcommand("provetheorem",args)) =
	      let val options = getOptions()
		  fun getThm(UIstring thm) = ((NONE, thm), thm)
		    | getThm(UIlist[UIstring thm, UIstring module]) =
		        ((SOME module, thm), module^"."^thm)
		    | getThm x = raise SympError
		      ("executeCommand: wrong theorem spec argument to `provetheorem' command:\n  "
		       ^(UI2string c)
		       ^"\n format: <thmName> or  [<thmName>, <module>]")
		  val (thmSpec, name) =
		       (case args of
			    [UIstring thm] => ((NONE, thm), thm)
			  | [thm] => getThm thm
			  | [thm, UIstring str] => (#1 (getThm thm), str)
			  | _ => raise SympError
			     ("executeCommand: wrong arguments to `provetheorem' command:\n  "
			      ^(UI2string c)
			      ^"\n format: (provetheorem \"thm_spec\" [ \"new_name\" ])"))
		  val proverID = proveTheorem options name thmSpec
	      in 
		  (printOutCommand ("newprover",
				    [UIstring proverID, UIstring name]);
		   printOutCommand ("proverstatus",
				    (UIstring proverID)::(List.map(fn x=>UIstring x)
							    (getProverStatus proverID)));
		   printCurrentSubgoal options proverID)
	      end
	  | executeCommand (UIcommand("options", args)) = executeCommandOptions(false, args)
	  | executeCommand (UIcommand("displayoptions", args)) = executeCommandOptions(true, args)
	  | executeCommand (UIcommand("debugnames", _)) =
	      printOutCommand("debugnames", List.map(fn x=>UIstring x) debugList)
	  | executeCommand (UIcommand("userchoice",[UInumber id, UIstring choice])) =
	      acceptUserChoice(id, choice)
	  | executeCommand (c as UIcommand("userchoice",_)) = raise SympError
	      ("executeCommand: wrong arguments to `userchoice' command:\n  "
	       ^(UI2string c))
	  | executeCommand (UIcommand("cancelchoice",[UInumber id])) =
	      cancelUserChoice id
	  | executeCommand (c as UIcommand("cancelchoice",_)) = raise SympError
	      ("executeCommand: wrong arguments to `cancelchoice' command:\n  "
	       ^(UI2string c))
	  | executeCommand (C as UIcommand(c,args)) = raise SympError
	      ("Unknown command: "^(UI2string C))


	(* Wrapper for `executeCommand' to catch all the exceptions
	   and return the exit status `true' (success) or `false'
	   (exception caught).  The only exceptions we pass through is
	   `SympExit' (clean exit) and `SympError' (fatal error),
	   which are caught on the top level. *)

	fun tryExecute f = (f(); true)
	    handle SympError str => (printError "general" str; false)
		 | SympBug str =>
		     (printError "bug" ("\nOops, internal error!  "
					^"Please report this to sergey.berezin@cs.cmu.edu:\n\n"
					^str^"\n\nInclude your program and this output.\n"
					^"The internal SyMP state has been reset.\n\n"
					^(FunStack2string()));
		      (* lazyVerbDebug "" (typeCheckStat2string); *)
		      (* Since this is a nasty event, reset all the state and lose
		         all the typechecked/proved status info.  Play safe. *)
		      ResetSympState();
		      false)
		 | ParseError str => (printError "parse" str; false)
		 | TypeError str => (printError "type" str; false)
		 | SequentError str => (printError "prover" str; false)
		 | TransError str => (printError "trans" str; false)
		 | ProverError str => (printError "prover" str; false)
		 | ProofTreeError str => (printError "prover" str; false)
		 | SympExit => raise SympExit
		 (*  | _ => raise SympError("\nUnknow fatal error. Exiting.\n") *)

	fun tryExecuteCommand command = tryExecute(fn()=>executeCommand command)

	(* Same for a list of commands *)
	fun tryExecuteList [] = true
	  | tryExecuteList (c::lst) =
	    if tryExecuteCommand c then tryExecuteList lst
	    else false

	(* Read the user's command and execute it in an endless loop *)
	fun readExecLoop () = 
	    let fun loop() =
		    let val command = ReadUserCommand (getOptions()) ()
		    in 
			(tryExecuteCommand command; loop())
		    end
	    in (loop())
		handle SympError str => (printError "general" ("\nError: "^str^"\n"))
		     | SympExit => (verb "Symp is exiting";
				    raise SympExit)
		     | SympExitError => (printError "general" ("\nAbnormal exit\n");
					 raise SympExitError)
		     | SympBug str => (printError "bug" ("\nOops, internal error!  "
					^"Please report this to sergey.berezin@cs.cmu.edu:\n\n"
					^str^"\n\nInclude your program and this output.\n"
					^"The internal SyMP state has been reset.\n\n"
					^(FunStack2string())); loop())
		     | exn => 
		       let val str = ("\nreadExecLoop: Unrecognized exception caught:\n"
				      ^(exnMessage exn)^"\n")
		       in 
			   raise SympBug str
		       end
	    end
	


    end
	

  end (* struct *)
