signature INPUT_LANGUAGE =
  sig
    structure Options: OPTIONS
    type options = Options.options

    exception ParseError of string
    exception TypeError of string

    type ParseTree

    (* Typechecked program, expression, declaration, whatever, which
       is the result of typechecking a ParseTree. *)
    type ProgramObject
    (* Type or an expression*)
    type ExpressionType
    (* Context for typechecking *)
    type ProgramContext

    (* String values for keywords starting a context ("module",
     "protocol", etc.), a theorem (usually "theorem", an expression
     (e.g. "val"), and a type (e.g. "type") *)

    val contextKeyword: string
    val theoremKeyword: string 
    val expressionKeyword: string
    val typeKeyword: string

    val Parse: Options.options -> TextIO.instream -> ParseTree
    val ParseInput: Options.options -> TextIO.instream -> ParseTree
    val ParseFile: Options.options -> string -> ParseTree
    val wasError: bool ref

    (* Typechecking *)

    (* Typecheck the entire program, return the top-level context *)
    val typeCheckProgram: options -> ParseTree -> ProgramContext

    (* Typecheck a construct that represents some context (e.g. a
       module).  Take the top context and the parse tree of the
       construct and return the new (local) context. *)
    val typeCheckContext: options -> ProgramContext -> ParseTree
	  -> ProgramContext

    (* Typecheckng an expression and return its new typechecked
       representation together with its type *)
    val typeCheckExpr: options -> ProgramContext -> ParseTree * ParseTree
	      -> ProgramObject * ExpressionType

    (* Same for a theorem *)
    val typeCheckTheorem: options -> ProgramContext -> ParseTree
	  -> ProgramObject


    (* Equality on parse trees and objects *)
    val ptEq: ParseTree * ParseTree -> bool
    val poEq: ProgramObject * ProgramObject -> bool

  end
