(* New rewrite of the hash functions.  Now hashes can change size
   dynamically as they grow or shrink. *)

functor HashFun () :> HASH =
  struct
    type ('a, 'b) Hash = { hash: ('a * 'b) list ref vector ref,
			   size: int ref,
			   (* If false, the hash cannot be resized *)
			   resizable: bool,
			   (* Count the number of elements in the hash *)
			   count: int ref,
			   F: 'a -> int,
			   eq: 'a * 'a -> bool}

    (* Somewhat arbitrary.  Makes it easy to implement F. *)
    val defaultSize = 11

    fun hash2any (pkey: 'a -> 'c) (pval: 'b -> 'd)
	         ({hash=ref hash,...}: ('a,'b) Hash) = 
	let fun loop x = List.map(fn(key,el)=>(pkey key, pval el))(!x)
	    val tmp = Vector.map loop hash
	in Vector.foldr(op @) [] tmp
	end

    fun findHash({hash=ref hash,size=ref size,F=F,eq=eq,...}: ('a, 'b) Hash, key) = 
	let fun strip (SOME(key,el)) = SOME el
	      | strip NONE = NONE
	in strip(List.find (fn (key1,_) => eq(key,key1))
		           (!(Vector.sub(hash, (abs(F key)) mod size))))
	end

    (* Generate a word with n-1 1's, or with all 1's if n is NONE *)
    fun nOnes NONE = Word.notb 0wx0
      | nOnes (SOME n) =
	let fun loop 2 = 0wx1
	      | loop n =
	        if n <= 1 then 0wx0
		else Word.orb(Word.<<(loop(n-1), 0wx1), 0wx1)
	in loop n
	end

    val IntMask = nOnes(Int.precision)

    (* The "almost perfect" hash function.  Converts a string into an
       integer from 0 to 255. *)
    fun F x =
	let val w = map(Word.fromInt o ord)(explode x)
	    val bytes = (Word.wordSize + 1) div 8
	    val size = Word.fromInt(List.length w)
	    fun fold _ acc [] = acc
	      | fold 1 acc lst = Word.xorb(acc, fold bytes 0wx0 lst)
	      | fold n acc (x::tl) = fold(n-1)(Word.orb(Word.<<(acc,0wx8),x)) tl
	    val z = Word.andb(IntMask, fold bytes 0wx0 (size::w))
	in 
	    Word.toInt z
	    (* abs(Word.toInt(List.foldr(fn(x,y) => Word.xorb(x,y)) 0wx0 z)) *)
	end

    fun makeHashCustom(eq,size,F,resizable) =
	{ hash=ref(Vector.tabulate(size,(fn x => ref []))),
	   eq=eq,
	   F = F,
	   size=ref size,
	   count=ref 0,
	   resizable=resizable}: ('a, 'b) Hash

    fun makeHashDefault(eq,toString) = 
	  makeHashCustom(eq, defaultSize, F o toString, true)

    (* Inefficient hashes for objects nonrepresentable by strings *)
    fun makeHash eq = makeHashCustom(eq, 1, fn _ => 0, false)

    (* Make a clean copy of the hash, so any destructive operations wouldn't
     * change the original one *)
    fun copyHash({hash=ref hash,size=ref size,count=ref count,
		  F=F,eq=eq,resizable=resizable}: ('a, 'b) Hash) =
	({hash=ref(Vector.map(fn x => ref(!x)) hash),
	  count=ref count,
	  size=ref size,F=F,eq=eq,
	  resizable=resizable}: ('a, 'b) Hash)

    fun insertHashNoResize(theHash: ('a, 'b) Hash,key,el) = 
	let val {hash=ref hash, eq=eq, F=F, size=ref size,
		 count=countRef, ... } = theHash
	    val listref = Vector.sub(hash, (abs(F key)) mod size)
	in  (if isSome(List.find(fn (key1,_) => eq(key,key1))(!listref)) then
		 listref := (key,el)::(List.filter(fn(key1,_)=>not(eq(key1,key)))(!listref))
	     else
		 (countRef := (!countRef) + 1;
		  listref := (key,el)::(!listref));
	     theHash)
	end

    fun resizeHash(theHash as {hash=hash,
			       size=size,
			       count=count,
			       F=F,eq=eq,
			       resizable=resizable}: ('a, 'b) Hash,
		   newSize) =
	(if resizable then
	    let val newHash = makeHashCustom(eq, newSize, F, true)
		fun insertPair(key, el) = insertHashNoResize(newHash, key, el)
	    in (List.map insertPair (hash2any(fn x=>x)(fn x=>x) theHash);
		newHash)
	    end
	else theHash)

    fun resizeHashDestructive(theHash as {hash=hash,
					  size=size,
					  count=count,
					  F=F,eq=eq,
					  resizable=resizable,...}: ('a, 'b) Hash,
			      newSize) =
	(if resizable then
	    let val {hash=ref newVector, size=ref newSize, ...} = resizeHash(theHash, newSize)
	    in (hash := newVector;
		size := newSize;
		theHash)
	    end
	else theHash)

    (* it is supposed to find a near-prime number close to the requested size *)
    fun findCloseSize n = n

    (* Resize the hash when #elements > 4*size, or 4*#elements < size *)
(*    fun balance hash = hash *)
     fun balance(theHash as {size=ref size, count = ref count, ...}) =
	  if count > size * 2 then
	      resizeHashDestructive(theHash, findCloseSize(size*4+1))
	  else if count*2 < size then
	      resizeHashDestructive(theHash, if count < defaultSize then defaultSize
					     else findCloseSize count)
	  else theHash

    fun insertHashDestructive(theHash,key,el) = 
	  balance(insertHashNoResize(theHash, key, el))

    fun removeHashDestructive(theHash,key) = 
	let val {hash=ref hash, eq=eq, F=F, size=ref size,
		 count=count, ... } = theHash
	    val listref = Vector.sub(hash, (abs(F key)) mod size)
	in  ((if isSome(List.find(fn (key1,_) => eq(key,key1))(!listref)) then
		  (listref := List.filter(fn(key1,_)=>not(eq(key1,key))) (!listref);
		   count := (!count) - 1)
	      else ()); 
	     (balance theHash))
	end

    fun insertHash(theHash,key,el) = 
	  insertHashDestructive(copyHash(theHash),key,el)

    fun removeHash(theHash,key) = 
	  removeHashDestructive(copyHash(theHash),key)

    fun wipeHash({size=ref size,F=F,eq=eq,resizable=resizable,...}: ('a, 'b) Hash) =
	  makeHashCustom(eq, if resizable then defaultSize else size, F, resizable)

    fun wipeHashDestructive(h as {hash=hash,size=sizeRef,count=count,
				  resizable=resizable, ...}: ('a,'b) Hash) =
	let val newSize = if resizable then defaultSize else (!sizeRef)
	in 
	    (hash := Vector.tabulate(newSize,(fn x => ref []));
	     sizeRef := newSize;
	     count := 0;
	     h)
	end

    fun stats({hash=ref hash, ...}: ('a,'b) Hash) =
	Vector.foldr(fn(x,y) => (List.length(!x))::y) [] hash
  end
