functor EmacsServerInterfaceFun(): EMACS_SERVER_INTERFACE =
  struct

    open Str
    open SympBug

    exception EmacsServerError of string

    structure ParseError: PARSE_ERROR =
      struct
        exception ParseError of string 
	val wasError = ref false
      end

    structure EmacsServerStruct: EMACS_SERVER =
      struct
	datatype EmacsExpr =
	    EmacsString of string
	  | EmacsSymbol of string 
	  | EmacsNumber of int
	  | EmacsNil
	  | EmacsTrue
	  | EmacsCons of EmacsExpr * EmacsExpr
      end

    open ParseError
    open EmacsServerStruct

    fun EmacsList [] = EmacsNil
      | EmacsList (x::lst) = EmacsCons(x,EmacsList lst)

    (* Convert a string to the sequence of characters readable by
       emacs as the original string.  E.g. replace `"' by `\"', etc. *)
    fun str2emacs str =
	let val lst = String.explode str
	    fun loop ( #"\"" :: lst) = ( #"\\" :: #"\"" :: (loop lst))
	      | loop ( #"\\" :: lst) = #"\\" :: #"\\" :: (loop lst)
	      | loop (c::lst) = c::(loop lst)
	      | loop [] = []
	in String.implode(loop(String.explode str))
	end
    local fun op *(x,y) = Conc(x,y)
    in
	fun EmacsExpr2str (EmacsString s) = Str("\""^(str2emacs s)^"\"")
	  | EmacsExpr2str (EmacsSymbol s) = Str s
	  | EmacsExpr2str (EmacsNumber n) = Str(Int.toString n)
	  | EmacsExpr2str EmacsNil = Str "nil"
	  | EmacsExpr2str EmacsTrue = Str "t"
	  | EmacsExpr2str (EmacsCons(x,y)) = 
	    let fun loop (x, EmacsCons(x',y')) = 
		      (EmacsExpr2str x)*(Str " ")*(loop(x',y'))
		  | loop (x,EmacsNil) = (EmacsExpr2str x)
		  | loop (x,y) = (EmacsExpr2str x)*(Str " . ")*(EmacsExpr2str y)
	    in (Str "(")*(loop(x,y))*(Str ")")
	    end

	and EmacsExpr2string e = Str2string(EmacsExpr2str e)

    (* Same as above, only wrap the entire expression into special delimiters *)
	fun EmacsExpr2strServer e = (Str "\n\000")*(EmacsExpr2str e)*(Str "\001\n")
	fun EmacsExpr2stringServer e = "\n\000"^(EmacsExpr2string e)^"\001\n"
    end

    structure EmacsServerLrVals =
	EmacsServerLrValsFun(structure Token = LrParser.Token
			     structure EmacsServerStruct = EmacsServerStruct
			     structure ParseError = ParseError)

    structure EmacsServerLex =
	EmacsServerLexFun(structure Tokens = EmacsServerLrVals.Tokens
			  structure ParseError = ParseError)

    structure EmacsServerParser =
	EmacsServerJoin(structure LrParser = LrParser
		 structure EmacsServerStruct = EmacsServerStruct
		 structure ParserData = EmacsServerLrVals.ParserData
		 structure Lex = EmacsServerLex)

    type result = EmacsServerLrVals.ParserData.result
    open EmacsServerLrVals.ParserData

    val EmacsExprParse = fn ins => 
	let val print_error = fn (s,l,c) =>
		(wasError := true;
		 raise EmacsServerError("\nError: " ^ (Int.toString l) ^"."
					^ (Int.toString c) ^ ", " ^ s ^ "\n"))
		  (* Use 0 tokens for error correction *)
	    val invoke = fn lexstream => 
		(EmacsServerParser.parse(0,lexstream,print_error,())
		 handle EmacsServerParser.ParseError => 
		     raise EmacsServerError("Fatal emacs lisp parser error")
		      | ParseError str => raise EmacsServerError str)
	    val lexer = EmacsServerParser.makeLexer 
		           (fn (i: int) => TextIO.inputLine ins) print_error
	    val (result,_) = (EmacsServerLex.UserDeclarations.pos := 1;
			      EmacsServerLex.UserDeclarations.yypos_bol := 0;
			      wasError := false;
			      invoke lexer)
			      
	in result
	end


  end
