;;;; Running SYMP 

(require 'symp-common)
(require 'symp-error-mode)
(require 'symp-prover-mode)
(require 'symp-proof-mode)
(require 'symp-debug-mode)
(require 'symp-message-mode)
(require 'symp-options-mode)

(defvar symp-server-command "symp" 
  "The command name to run SYMP. The defaul is usually \"symp\"")

(defvar symp-server-args 
  (list "-emacs" 
	"--debug" 
	(mapconcat 'identity
		   '(
		     "ReadUserCommand"
		     "applyProverCommand"
		     "common"
		     ;; General interface
;		     "lazyFun"
		     ;; Typechecking in the default proof system
; 		     "stat"
; 		     "instantiateModule"
; 		     "instantiateModule/resolveParam"
; 		     "instantiateModule/loop"
; 		     "tcModuleExpr"
; 		     "tcModuleInst"
; 		     "tcModuleName"
; 		     "findinContextCommon"
; 		     "findinContextCommonDebug"
; 		     "tcName"
; 		     "tcNameDebug"
; 		     "tc"
; 		     "typeCheckModuleExpr"
; 		     "applyRule"
; 		     "proveTheorem"
; 		     "matchTerm"
; 		     "matchTerm/loop"
; 		     "matchTerms"
; 		     "matchTerms/loop"
; 		     "matchSeq"
; 		     "skolemMatch"
; 		     "skolemApply"
; 		     "updateSubstitution"
; 		     "substTermID"
; 		     "SequentDefault.subst"
; 		     "SequentDefault.subst/loop"
; 		     "SequentDefault.subst/constrValue"
; 		     "substSequent"
; 		     "updateSubstitutionRaw"
; 		     "installProofTree"
; 		     "addProofRule"
; 		     "addSequent"
; 		     "term2termID"
; 		     ;; Translation: preparation phase (translating to AsstVarsTree)
; 		     "varsFrom"
; 		     "collectAsstVars"
; 		     "splitType"
; 		     ;; Generation phase (translation to boolean formulas)
; 		     "transmatches" ; "matches" in default/trans_gen.sml
; 		     "walkTree"
; 		     "eqFun"
; 		     "eqPrim"
; 		     "asstGen"
; 		     "customizeAsstVars"
; 		     "asstGen1"
; 		     "asstGen2"
; 		     "asstGen3"
; 		     "asstGen4"
; 		     "eqBuiltin"
; 		     ;; Generating SMV code
; 		     "SMVcode"
; 		     "formulaStr"
		     )
		   ","))
  "Thes list of default command line args.  Normally, set to
'(\"-emacs\") to run SyMP in emacs server mode.")

(defvar symp-server-verbose t
  "If not nil, run SyMP in verbose mode (with -v options).")

;;;; Internal variables.  You shouldn't need to modify those.

(defvar symp-server-process nil
  "The SyMP process handle.  This variable is local in each SyMP
 buffer.")
(make-variable-buffer-local 'symp-server-process)

(defvar symp-server-process-buffer nil
  "The buffer associated with inferior SYMP process.  This variable is
 updated automatically each time SYMP process is started.  

 This variable is local in each SyMP buffer.")
(make-variable-buffer-local 'symp-server-process-buffer)

(defvar symp-server-buffer-list nil
  "The list of buffers relevant to the current symp process (provers,
input sources, various message buffers, etc.).  Each such buffer has a
set of local variables updated automatically on every process event.
Also, each buffer for that process must register in this list.")
(make-variable-buffer-local 'symp-server-buffer-list)

(defvar symp-server-status-hook nil
  "This buffer-local hook is run in each buffer from
`symp-server-buffer-list' on each status change after the
`symp-server-status' variable is updated.")
;;(make-variable-buffer-local 'symp-server-status-hook)

(defvar symp-server-start-regexp "\000"
  "Regexp matching the start delimiter in the server's output.")

(defvar symp-server-end-regexp "\001"
  "Regexp matching the end delimiter in the server's output.")

(defvar symp-server-max-buffer-size 100000
  "The maximum size of the symp I/O buffer in bytes.  If nil, no limit.")

(defvar symp-server-status nil
  "Status of the SyMP server displayed in the mode line of SyMP buffers.")
(make-variable-buffer-local 'symp-server-status)

(defvar symp-server-process-marker nil
  "Marker in the SyMP I/O buffer that remembers the current point.

 Normally it is before the next output block to be processed.  This
 variable is local in each SyMP buffer.")
(make-variable-buffer-local 'symp-server-process-marker)

(defun symp-server-update-args (&optional debug-list other-args)
  "Update the `symp-server-args' variable with the optional DEBUG-LIST
and OTHER-ARGS.  When DEBUG-LIST is nil or not present, do not set the
--debug flag at all."
  (setq symp-server-args
	(cons "-emacs"
	      (if debug-list
		  (cons "--debug"
			(cons (mapconcat 'identity debug-list ",") other-args))
		nil))))

(defun symp-server-start (&optional buffers)
  "Start a new SyMP server.  Optional argument BUFFERS is the initial
buffer list for this server (see `symp-server-buffer-list')."
  (interactive)
  (if (and symp-server-process 
	   (eq (process-status symp-server-process) 'run))
      (error "SyMP server already running")
    (let ((buffer (current-buffer))
	  (symp-buffer nil)
	  (start-point nil)
	  (args symp-server-args)
	  (tmp nil))
      ;; If we had symp buffers already for this file, reuse them,
      ;; unless forced otherwise (parameter BUFFERS overrides it)
      (if (and (null buffers)
	       (boundp 'symp-server-process-buffer)
	       (bufferp symp-server-process-buffer)
	       (buffer-live-p symp-server-process-buffer))
	  (progn
	    (set-buffer symp-server-process-buffer)
	    (setq buffers symp-server-buffer-list)
	    (set-buffer buffer)))
      (if symp-server-verbose 
	  (setq args (cons "-v" args)))
      (setq symp-buffer 
	    (if (and (boundp 'symp-server-process-buffer)
		     (bufferp symp-server-process-buffer)
		     (buffer-live-p symp-server-process-buffer))
		symp-server-process-buffer
	      (get-buffer-create
	       (concat "*symp-server:" (buffer-name buffer) "*"))))
      (buffer-disable-undo symp-buffer)
      (set-buffer symp-buffer)
      (setq symp-server-process-buffer symp-buffer)
      ;; Initialize the buffer list
      (setq symp-server-buffer-list (list symp-buffer))
      (setq tmp (cons buffer buffers))
      (while tmp
	(add-to-list 'symp-server-buffer-list (car tmp))
	(setq tmp (cdr tmp)))
      ;; Set the markers
      (setq old-point (make-marker))
      (set-marker old-point (point))
      (goto-char (point-max))
      ;; print the command line for the server
      (insert symp-server-command)
      (setq tmp args)
      (while tmp
	(insert (concat " " (car tmp)))
	(setq tmp (cdr tmp)))
      (insert "\n")
      (goto-char (point-max))
      (setq symp-server-process-marker (make-marker))
      (set-marker symp-server-process-marker (point))
      ;; Now actually start the server
      (message "Starting SyMP...")
      (setq symp-server-process nil)
      (if symp-server-process-buffer
	  (setq symp-server-process 
		(apply 'start-process 
		       (append (list "symp"
				     symp-server-process-buffer
				     symp-server-command)
			       args)))
	(error "Process buffer is nil!"))	
      (if (null symp-server-process)
	  (error "Can't start the SyMP server %S" symp-server-command))
      (set-process-sentinel symp-server-process 'symp-server-process-sentinel)
      (setq symp-server-status " Run")
      ;; Set the variables in all the relevant buffers
      (setq tmp symp-server-buffer-list)
      (let ((process symp-server-process)
	    (buffer (current-buffer))
	    (status symp-server-status))
	(while tmp
	  (if (buffer-live-p (car tmp))
	      ;; The buffer exists, update the status
	      (progn
		(set-buffer (car tmp))
		(setq symp-server-process process)
		(setq symp-server-process-buffer buffer)
		(setq symp-server-status status))
	    ;; The buffer was killed, remove it from the list
	    (progn
	      (set-buffer buffer)
	      (setq symp-server-buffer-list 
		    (delete (car tmp) symp-server-buffer-list))))
	  (setq tmp (cdr tmp)))
	(set-buffer buffer))
      ;; Set the filter *after* all buffers are set, to avoid
      ;; surprizes.  We may lose some really fast messages from the
      ;; server this way, but why should we care...
      (set-process-filter symp-server-process 'symp-server-process-filter)
      (symp-server-process-sentinel symp-server-process "server started")
      ;; Request current options for our own cache
      (symp-server-send-expr '(options))
      (symp-server-send-expr '(debugnames))
      ;; Restore the state
      (goto-char old-point)
      (set-buffer buffer))))

(defun symp-server-process-sentinel (process event)
  "Sentinel function for SyMP process"
  (let* ((status (process-status process))
	 (status-str (cond ((eq status 'run) " Run")
			   ((eq status 'stop) " Stopped")
			   ((eq status 'exit) " Exited")
			   ((eq status 'signal) " Error")
			   (t nil)))
	 (symp-buffer (process-buffer process))
	 (lst nil)
	 (buffer (current-buffer)))
    (if symp-buffer
	(progn
	  (set-buffer symp-buffer)
	  (setq lst symp-server-buffer-list)
	  (while lst
	    (if (buffer-live-p (car lst))
		;; The buffer exists, update the status
		(progn
		  (set-buffer (car lst))
		  (setq symp-server-status status-str)
		  (run-hooks 'symp-server-status-hook))
	      ;; The buffer was killed, remove it from the list
	      (progn
		(set-buffer buffer)
		(setq symp-server-buffer-list 
		      (delete (car lst) symp-server-buffer-list))))
	    (setq lst (cdr lst)))
	  (set-buffer buffer))))
  (if (string-match "\C-j$" event)
      (setq event (substring event 0 (- (length event) 1))))
  ;;(beep)
  (message "SyMP: %s" event))

(defun symp-server-search-output ()
  "Search for the first complete output block in the buffer and return
its start and end as a pair '(start . end). If no complete block
found, return nil."
  (let ((current-point (point))
	(start-point nil)
	(end-point nil))
    ;; Search for the start of the first output block.
    (setq start-point (search-forward-regexp symp-server-start-regexp nil t))
    (if start-point
	(progn
	  (goto-char start-point)
	  ;; now look for the end of 
	  (setq end-point (search-forward-regexp symp-server-end-regexp nil t))
	  (if end-point
	      (setq end-point (match-beginning 0)))))
    ;; Restore the point
    (goto-char current-point)
    ;; and return the result
    (if (and start-point end-point) 
	(progn
	  ;; (message "found output!")
	  (cons start-point end-point))
      nil)))

(defun symp-server-process-filter (process string)
  "Reads messages from the SyMP server and makes sense out of them."
  (if (equal (process-status process) 'run)
	(let ((old-buffer (current-buffer))
	      (pair nil)
	      (old-point nil)
	      (start-point nil)
	      (end-point nil)
	      (input nil)
	      (symp-buffer (process-buffer process)))
	  (if symp-buffer
	      (progn
		(set-buffer symp-buffer)
		(setq old-point (make-marker))
		(set-marker old-point (point))
		;; Insert the server's output into the I/O buffer
		(goto-char (point-max))
		(insert string)
		(goto-char symp-server-process-marker)
		(setq pair (symp-server-search-output))
		;; Process all server output chunks
		;; Loop invariant: we are in the symp-buffer
		(while pair
		  ;; First, move to the next input, in case it comes
		  ;; before we are done with the current one
		  (setq input (buffer-substring (car pair) (cdr pair)))
		  (goto-char (cdr pair))
		  (set-marker symp-server-process-marker (point))
		  ;; Digest the input.  We always start from the symp
		  ;; server buffer, and then see where it takes us
		  (symp-server-process-input input)
		  ;; We may end up in a diff. buffer here.  Save the
		  ;; current buffer and go back to the symp-buffer for
		  ;; the next chunk of output
		  (if (eq (current-buffer) symp-buffer) nil
		    (setq old-buffer (current-buffer)))
		  (set-buffer symp-buffer)
		  (setq pair (symp-server-search-output)))
		(symp-server-trim-buffer symp-server-process-marker
					 (current-buffer))
		(goto-char old-point)
		(set-buffer old-buffer))
	    (error "Process has no buffer!")))))

(defun symp-server-trim-buffer (&optional current-marker buffer)
  "Truncates the BUFFER from top to the size given by
 `symp-server-max-buffer-size', but no further than CURRENT_MARKER.
 (symp-server-trim-buffer &optional current-marker buffer)"

  (let ((old-buffer nil)
	(end-point nil))
    (if buffer
	(progn
	  (setq old-buffer (current-buffer))
	  (set-buffer buffer)))
    (if (null current-marker) (setq current-marker (point-max)))
    (if (and symp-server-max-buffer-size
	     (> (buffer-size) (+ symp-server-max-buffer-size 1000)))
	(progn
	  ;; We need to trim.  Find the end-point of the region to cut.
	  (setq end-point (- (buffer-size) symp-server-max-buffer-size))
	  (if (>= end-point current-marker)
	      (setq end-point (- current-marker 1)))
	  (delete-region (point-min) end-point)))))

(defun symp-server-process-input (string)
  "Takes a string from the SyMP server and interprets it."
  ;; First, parse it as an S-exp
  (let ((expr (read string))
	(command nil)
	(args nil))
    (if (consp expr)
	(progn
	  (setq command (car expr))
	  (setq args (cdr expr))
	  (cond
	   ;; Print verbose messages in the minibuffer
	   ((or (eq command 'verbose) (eq command 'version))
	    (if (consp args) (symp-message "%s" (car args))
	      (symp-message "%S" args)))
	   ;; The debug messages go to a special debug buffer
	   ((eq command 'debug)
	    (if (consp args) (symp-debug-display (car args))
	      (symp-debug-display (format "%S" args))))
	   ;; Error messages must end up in yet another special popup
	   ;; buffer
	   ((eq command 'error)
	    (if (and (listp args) (= 2 (length args)))
		(symp-server-report-error (car args) (car (cdr args)))
	      (symp-server-report-error "general" (format "%S" args))))
	   ((eq command 'newprover)
	    (if (and (listp args) (= 2 (length args)))
		(symp-server-new-prover (car args) (car (cdr args)))
	      (symp-server-report-error "emacs" 
	       (format "%s:\n  %S"
		       "`newprover' command from the server has wrong arguments:"
		       args))))
	   ((eq command 'oldprover)
	    (if (and (consp args) (consp (cdr args)))
		(symp-server-old-prover (car args) (car (cdr args)))
	      (symp-server-report-error "emacs"
	       (format "%s:\n  %S"
		       "`oldprover' command from the server has wrong arguments:"
		       args))))
	   ((eq command 'proverstatus)
	    (if (consp args)
		(symp-prover-update-status (car args) (cdr args))
	      (symp-server-report-error "emacs"
	       (format "%s:\n  %S"
		       "`proverstatus' command from the server has wrong arguments:"
		       args))))
	   ;; Print prover output
	   ((eq command 'prover)
	    (if (and (listp args) (= 2 (length args)))
		(symp-prover-display (nth 0 args) (nth 1 args))
	      (symp-server-report-error
	       "emacs"
	       (format "%s:\n  %S"
		       "`prover' command from the server has wrong arguments:"
		       args))))
	   ;; Display a proof
	   ((eq command 'showproof)
	    (if (and (listp args) (= 2 (length args)))
		(symp-proof-display (nth 0 args) (nth 1 args))
	      (symp-server-report-error
	       "emacs"
	       (format "%s:\n  %S"
		       "`showproof' command from the server has wrong arguments:"
		       args))))
	   ;; The server asks the user a question
	   ((eq command 'userchoice)
	    (if (and (listp args) 
		     (>= (length args) 3))
		(let* ((id (nth 0 args))
		       (type (nth 1 args))
		       (strict (string= type "strict"))
		       (msg (nth 2 args))
		       (choices (cdr (cdr (cdr args))))
		       (choice nil))
		  ;; Ask the user and read his input, and if something
		  ;; goes wrong, cancel the request
		  (condition-case nil
		      (progn
			(cond ((string= type "file")
			       (setq choice
				     (read-file-name
				      msg nil nil nil
				      (if choices (car choices) nil))))
			      ((string= type "yesno")
			       (setq choice (if (y-or-n-p msg) "yes" "no")))
			      ((string= type "confirm")
			       (setq choice (if (yes-or-no-p msg) "yes" "no")))
			      (t ;;(member type '("any" "strict"))
			       (setq choice 
				     (completing-read 
				      msg
				      (symp-list-to-alist choices)
				      nil strict
				      (if choices (car choices) nil)))))
			;; send the response
			(symp-server-send-expr
			 (list 'userchoice id choice)))
		    ;; If an error happens, or the user quits, trap it
		    ;; and send a cancel msg to the server
		    ((error quit)
		     (beep)
		     (symp-server-send-expr (list 'cancelchoice id)))))
	      (symp-server-report-error "emacs"
	       (format "Bad `userchoice' args: %S" args))))
	   ;; Ring the bell
	   ((eq command 'bell) (beep))
	   ;; Update and optionally display the values of the current options
	   ((or (eq command 'options)
		(eq command 'displayoptions))
	    (let ((options nil))
	      (while (and (listp args) (not (null args)))
		(let ((arg (car args)))
		  (if (and (listp arg)
			   (= (length arg) 3))
		      (setq options (cons (cons (nth 0 arg) (nth 2 arg)) options))
		    (symp-message 
		     "Warning: command `options' from the server has a bad argument: %S"
		     arg))
		  (setq args (cdr args))))
	      (symp-options-update options)
	      (symp-options-display options (eq command 'options))))
	   ;; Update the available names for the `debug' option
	   ((eq command 'debugnames)
	    (setq symp-options-debug args))
	   ;; Unknown command??? Bug the user.
	   (t (beep) (message "??? %S" expr))))
      ;; Hmm... the expression is in a strange format.  Scream about
      ;; it and ignore it otherwise.
      (progn
	;;(beep)
	;; (message "Unrecognized message from the server.")
	(message "%S" expr)
	))))

(defun symp-server-report-error (type str)
  "Display an error message STR received from the server."
  (let ((type-string
	 (cond ((string= type "bug")
		"Internal SyMP Error (this shouldn't happen):")
	       ((string= type "emacs")
		"Internal SyMP Emacs Error (this shouldn't happen):")
	       ((string= type "general") "Error:")
	       ((string= type "type") "Type Error:")
	       ((string= type "prover") "Prover Error:")
	       ((string= type "trans") "Transition Relation Generator Error:")
	       (t (format "%s Error:" type)))))
    (symp-error-display-message
     (format "%s %s" type-string str)
     ;; Erase the error buffer before displaying these types of error
     ;; messages
     (member type '("bug" "type" "prover")))))

(defun symp-server-new-prover (prover-id name)
  "Create a new buffer for prover interaction for a prover PROVER-ID
and a theorem NAME."
  (let ((prover-buffer (symp-prover-buffer prover-id name)))
    (switch-to-buffer-other-window prover-buffer)
;    (symp-prover-update-status prover-id '(active))
    (setq symp-prover-id prover-id)
    (setq symp-prover-name name)
    ;; ask the server for the prover status
    (symp-server-send-expr (list 'proverstatus prover-id))
    (goto-char (point-max))))

(defun symp-server-old-prover (prover-id name)
  "Find the prover interaction buffer for a prover PROVER-ID and a
theorem NAME, or create a new one if it doesn't exist."
  (symp-server-new-prover prover-id name))

(defun symp-server-send-expr (expr)
  "Send EXPR to the SyMP server.  This is the most primitive function
to communicate to the server, and all the other functions that want to
talk to the server must use it."

  (if (and symp-server-process
	   (eq (process-status symp-server-process) 'run))
      (let ((old-buffer (current-buffer))
	    (str (format "%S\n" expr)))
	(if symp-debug
	    (symp-debug-display (format "Sending \"%s\"\n" str)))
	;; (set-buffer (process-buffer symp-server-process))
	(process-send-string symp-server-process str)
	;; It'd be good to see what we're sending, but don't mess the
	;; I/O buffer, let the server echo the string back.
	)
    (error "SyMP is not running")))

(defun symp-server-exit ()
  "Ask the server to exit cleanly."
  (interactive)
  (symp-server-send-expr 'exit))

(defun symp-server-interrupt ()
  "Interrupt the SyMP server.  This usually kills the server."
  (interactive)
  (if (and symp-server-process
	   (equal (process-status symp-server-process) 'run))
      (interrupt-process symp-server-process)))

(defun symp-server-register-buffer (&optional buffer)
  "Register (optional) BUFFER or the current buffer with the current
SyMP process.  This adds the current buffer to the master
`symp-server-buffer-list' in the main SyMP buffer for this session.
The SyMP buffer is determined by `symp-server-process-buffer'."
  (if symp-server-process-buffer
      (let ((buf (or buffer (current-buffer)))
	    (old-buffer (current-buffer)))
	(set-buffer symp-server-process-buffer)
	;;(message "symp-server-buffer-list = %S" symp-server-buffer-list)
	(add-to-list 'symp-server-buffer-list buf)
	;;(message "Added prover buffer, symp-server-buffer-list = %S"
	;;	 symp-server-buffer-list)
	(set-buffer old-buffer))
    (message "Warning: no SyMP process buffer")))

;(defun symp-interrupt ()
;  "Kills current SYMP process."
;  (interactive)
;  (quit-process (get-buffer-process symp-compile-buffer) t))

; (defun symp-send-signal (sig)
;  "Sends signal SIG to the SyMP process. SIG must be an integer."
;  (if (get-buffer-process symp-compile-buffer)
;      (if (file-exists-p ".symp-pid")
; 	(save-excursion
; 	  (let ((buf (get-buffer-create ".symp-pid")))
; 	    (set-buffer buf)
; 	    (erase-buffer)
; 	    (insert-file-contents ".symp-pid")
; 	    (let ((pid (read buf)))
; 	      (if (integerp pid)
; 		  (signal-process pid sig)
; 		(error "The file .symp-pid is screwed up: %s" pid)))
; 	    (kill-buffer buf)))
; 	(error "Your SYMP version does not support signal handling"))
;    (error "SYMP is not running")))

;(defun symp-send-usr1 () 
;  "Sends SIGUSR1 to the current SYMP process. I have a version of SYMP
;that uses it to toggle dynamic variable ordering."
;  (interactive)
;  (symp-send-signal 10))

;(defun symp-send-usr2 () 
;  "Sends SIGUSR2 to the current SYMP process. I have a version of SYMP
;that uses it to force garbage collection."
;  (interactive)
;  (symp-send-signal 12))


(provide 'symp-server)
