;;;; Major mode for interactive SyMP prover

(require 'ring)
(require 'comint)
(require 'symp-common)
(require 'symp-debug-mode)

(defvar symp-prover-prompt "Rule? "
  "The prover prompt.")

(defvar symp-prover-prompt-regexp "^Rule\\? "
  "Regexp to match the prover prompt.")

(defvar symp-prover-history nil
  "Command history, local in every prover buffer.")

(defvar symp-prover-history-position 0
  "Position in the history, 0 = the last element.")

(defvar symp-prover-escrow-command nil
  "Temporarily saved command")

(make-variable-buffer-local 'symp-prover-history)
(make-variable-buffer-local 'symp-prover-history-position)
(make-variable-buffer-local 'symp-prover-escrow-command)

(defun symp-prover-buffer (prover-id &optional name dont-create)
  "Find a prover buffer for the PROVER-ID and, optionally, NAME, or
create a new one if optional DONT-CREATE is nil, and select it.  The
buffer name is constructed by appending NAME value to \"prover:\" and
making it unique if necessary."
  (let ((old-buffer (current-buffer))
	(process symp-server-process)
	(symp-buffer symp-server-process-buffer)
	(buffer-list nil)
	(prover-buffer nil))
    (if symp-buffer
	(progn
	  (set-buffer symp-buffer)
	  ;; Search the relevant buffers for the prover-id 
	  (setq buffer-list symp-server-buffer-list)
	  (while (and (not prover-buffer) buffer-list)
	    (if (buffer-live-p (car buffer-list))
		(progn
		  (set-buffer (car buffer-list))
		  ;; First, check if the prover ID is the same
		  (if (or (and (boundp 'symp-prover-id)
			       (equal symp-prover-id prover-id))
			;; then compare the name, and if the prover
			;; became inactive, reuse the buffer
			  (and name 
			       (boundp 'symp-prover-name)
			       (equal symp-prover-name name)
			       (not (symp-prover-active-p))))
		      (setq prover-buffer (car buffer-list)))))
	    (setq buffer-list (cdr buffer-list)))
	  (if (and (not prover-buffer) (not dont-create))
	    ;; Didn't find prover buffer, create one
	      (progn
		(setq prover-buffer
		      (generate-new-buffer (concat "Prover:" name)))
		(set-buffer prover-buffer)
		(setq symp-server-process process)
		(setq symp-server-process-buffer symp-buffer)
		(symp-prover-mode)))
	  (if prover-buffer
	      (progn
		;; Update the prover's status
		;; We must be in the prover buffer here
		(if prover-id (setq symp-prover-id prover-id))
		(if name (setq symp-prover-name name))))))
    (set-buffer old-buffer)
    prover-buffer))

(defun symp-prover-active-p ()
  "Returns non-nil if the prover is active.  Currently fetches the
result from a buffer-local variable."
  (and (boundp 'symp-prover-status)
       (member "active" symp-prover-status)))

(defun symp-prover-status-hook ()
  "Hook to run on SyMP server status change"
  (cond
   ;; Server is running, keep the status of the prover unchanged
   ((equal symp-server-status " Run") nil)
   ;; otherwise set the prover status string to the server status
   (t (setq symp-prover-status-string symp-server-status)
      ;; and nullify the actual prover status
      (setq symp-prover-status nil))))

(defvar symp-prover-mode-map nil
  "Keymap for SyMP prover interactive buffer.")

;;; Use comint-mode-map as the basis and customize it
(if symp-prover-mode-map nil
  (setq symp-prover-mode-map (copy-keymap comint-mode-map))
  (define-key symp-prover-mode-map "\C-m" 'symp-prover-maybe-send-input)
  (define-key symp-prover-mode-map "\M-p" 'symp-prover-previous-command)
  (define-key symp-prover-mode-map "\M-n" 'symp-prover-next-command))

(defun symp-prover-maybe-send-input ()
  "If the point after the prover's active prompt \(determined by the
`symp-prover-insert-marker'\), determine whether the current command
is complete and either send it to the SyMP server, or just insert a
new line."
  (interactive)
  (let ((insert-marker (and (boundp 'symp-prover-insert-marker)
			    symp-prover-insert-marker))
	(old-point (point)))
    (if (and insert-marker (>= (point) insert-marker))
	(let* ((command (buffer-substring insert-marker (point-max)))
	       (prompt-end nil))
	  ;; Remove the prompt from the command string
	  (if (string-match symp-prover-prompt-regexp command)
	      (setq prompt-end (match-end 0)))
	  (if symp-debug (symp-debug-display
			  (format "\nprompt-end=%S\n" prompt-end)))
	  (if prompt-end
	      (progn
		(setq command (substring command prompt-end))
		(set-marker insert-marker (+ insert-marker prompt-end))))
	  ;; Now decide if the command is complete and send it
	  (if (symp-prover-command-complete-p insert-marker (point-max))
	      (progn
		(goto-char (point-max))
		;; If there is no end of line, add it
		(backward-char 1)
		(if (not (looking-at "\n"))
		    (progn
		      (goto-char (point-max))
		      (insert "\n")))
		;; move the insert marker to the end
		(goto-char (point-max))
		(set-marker symp-prover-insert-marker (point))
		;; parse and send the command
		(symp-prover-send-input command)
		;; and draw the next prompt
		(goto-char (point-max))
		(insert symp-prover-prompt)
		(goto-char (point-max)))
	    ;; Command is incomplete, insert new line
	    (insert "\n")))
      ;; We are not at the prover prompt.  Just insert a new line.
      (insert "\n"))))

(defun symp-prover-send-input (command)
  "Send the command to the SyMP server."
  (if (and (boundp 'symp-prover-id)
	   (symp-prover-active-p))
      (let ((expr (symp-prover-parse-command command)))
	;; Send the command if it's not empty
	(if symp-debug
	    (symp-debug-display
	     (format "prover expr = %S\n" expr)))
	(if expr 
	    (symp-server-send-expr
	     (cons 'prover (cons symp-prover-id expr))))
	;; Insert the command into the input history
	(if (and expr
		 (or (null symp-prover-history)
		     (not (equal command (car symp-prover-history)))))
	    (setq symp-prover-history
		  (cons command symp-prover-history)))
	;; and reset the history position
	(setq symp-prover-history-position 0))
    (error "No active prover in this buffer")))

(defun symp-prover-previous-command ()
  "If the cursor is on the command line, inserts the
`symp-prover-history-position's command from the end of the history
`symp-prover-history'.  The current position is then incremented.  If
the current position is 0, and the current command on the command line
is non-empty, it is saved in `symp-prover-escrow-command'."
  (interactive)
;   (if symp-debug
;       (symp-debug-display
;        (format "\nsymp-prover-previous-command: escrow = %S, position = %S, history = %S\n"
; 	       symp-prover-escrow-command
; 	       symp-prover-history-position
; 	       symp-prover-history)))
  (let ((insert-marker (and (boundp 'symp-prover-insert-marker)
			    symp-prover-insert-marker))
	(old-point (point)))
    (if (and insert-marker (>= (point) insert-marker))
	(let ((command (buffer-substring insert-marker (point-max)))
	      (prompt-end nil)
	      (new-command (nth symp-prover-history-position
				symp-prover-history)))
	  ;; Remove the prompt from the command string
	  (if (string-match symp-prover-prompt-regexp command)
	      (setq prompt-end (match-end 0)))
	  (if prompt-end 
	      (progn
		(setq command (substring command prompt-end))
		(set-marker insert-marker (+ insert-marker prompt-end))))
	  ;; Now save the command, if necessary
	  (if (and (= symp-prover-history-position 0)
		   new-command
		   ;; (not (string-match "^\\s-*$" command))
		   )
	      (setq symp-prover-escrow-command command))
	  ;; And replace the command with whatever's in the history
	  (if new-command
	      (progn
		(delete-region insert-marker (point-max))
		(insert new-command)
		(setq symp-prover-history-position
		      (1+ symp-prover-history-position)))
	    (error "No more commands in history")))
      (error "The cursor is not at a command prompt"))))

(defun symp-prover-next-command ()
  "If the cursor is on the command line, inserts the
`symp-prover-history-position-1'-st command from the end of the
history `symp-prover-history'.  The current position is then
decremented.  If the current position is 0, the command from
`symp-prover-escrow-command' is inserted, if any, and the command is
deleted from the escrow."
  (interactive)
;   (if symp-debug
;       (symp-debug-display
;        (format "\nsymp-prover-next-command: escrow = %S, position = %S, history = %S\n"
; 	       symp-prover-escrow-command
; 	       symp-prover-history-position
; 	       symp-prover-history)))
  (let ((insert-marker (and (boundp 'symp-prover-insert-marker)
			    symp-prover-insert-marker))
	(old-point (point)))
    (if (and insert-marker (>= (point) insert-marker))
	(let ((prompt-end nil)
	      (command (buffer-substring insert-marker (point-max)))
	      (new-command nil))
	  ;; First, figure out if we have a next command
	  (if (> symp-prover-history-position 0)
	      (setq symp-prover-history-position
		    (- symp-prover-history-position 1)))
	  (if (= symp-prover-history-position 0)
	      (if symp-prover-escrow-command
		  (progn
		    (setq new-command symp-prover-escrow-command)
		    (setq symp-prover-escrow-command nil)))
	    (progn
	      ;; Yes, we need to subtract 1 again from the position.
	      ;; Draw a picture if you don't believe it.
	      (setq new-command 
		    (nth (- symp-prover-history-position 1)
			 symp-prover-history))))
	  (if new-command
	      (progn
		;; Remove the prompt from the command string
		(if (string-match symp-prover-prompt-regexp command)
		    (setq prompt-end (match-end 0)))
		(if prompt-end 
		    (set-marker insert-marker (+ insert-marker prompt-end)))
		(delete-region insert-marker (point-max))
		(insert new-command))
	    (error "No more commands in history")))
      (error "The cursor is not at a command prompt"))))  

(defun symp-prover-command-complete-p (start end)
  "Test if the prover command in the current buffer from START to END
is complete or not."
  (= 0 (nth 0 (parse-partial-sexp start end))))

(defun symp-prover-parse-command (command)
  "Parse the prover command and convert it into a list s-expression."
  (let ((raw-expr (if (string-match "^\\s-*$" command) nil (read command))))
    (cond ((and raw-expr (symbolp raw-expr)); nil is also a symbol...
	   (list raw-expr))
	  ((and (consp raw-expr)
		(symbolp (car raw-expr)))
	   raw-expr)
	  ((null raw-expr) nil)
	  (t (error "Badly formed prover command")))))

(defun symp-prover-update-status (prover-id status-list)
  "Update the prover PROVER-ID status with values from STATUS-LIST."
  ;; First, find the prover's  buffer, if any
  (let ((old-buffer (current-buffer))
	(buffer (symp-prover-buffer prover-id nil t))
	(was-open nil))
    (if symp-debug
	(symp-debug-display 
	 (format "Updating status for prover %S: %S\nBuffer = %S\n"
		 prover-id status-list buffer)))
    (if buffer
	(progn
	  (set-buffer buffer)
	  (if (and (boundp 'symp-prover-status)
		   (not (symp-prover-closedp symp-prover-status)))
	      (setq was-open t))
	  (setq symp-prover-status status-list)
	  (setq symp-prover-status-string
		(symp-prover-compute-status-string symp-prover-status))
	  (if (and was-open (symp-prover-closedp))
	      (symp-prover-display prover-id "\n  The prover exited.\n\n"))
	  (set-buffer old-buffer)))))

(defun symp-prover-closedp (&optional prover-status)
  "Check if the prover with the optional PROVER-STATUS is closed.  If
PROVER-STATUS is not provided or NIL, check the prover in the current
buffer."
  (if (and (not prover-status)
	   (boundp 'symp-prover-status))
      (setq prover-status symp-prover-status))
  (not (member "active" prover-status)))

(defun symp-prover-display (prover-id string)
  "Display STRING in the prover buffer for PROVER-ID.

 (symp-prover-display prover-id string)"
  (let ((old-buffer (current-buffer))
	(prover-buffer (symp-prover-buffer prover-id)))
    (if prover-buffer
	(let ((old-point (make-marker)))
	  (set-buffer prover-buffer)
	  (set-marker old-point (point))
	  (goto-char symp-prover-insert-marker)
	  (insert string)
	  (set-marker symp-prover-insert-marker (point))
	  (goto-char old-point)
	  (set-buffer old-buffer))
      ;; No prover buffer found (which is a bug)
      (symp-debug-display 
       (format "Can't find prover buffer for prover-id = %s\n%s\n"
	       prover-id string)))))

(defun symp-prover-compute-status-string (status)
  "Compute status string to display in the mode line."
  (let ((str " Closed")
	(active nil)
	(busy nil)
	(init nil)
	(proven nil)
	(saved nil))
    (while (consp status)
      (cond ((string= (car status) "active") (setq active t))
	    ((string= (car status) "init") (setq init t))
	    ((string= (car status) "busy") (setq busy t))
	    ((string= (car status) "proven") (setq proven t))
	    ((string= (car status) "saved") (setq saved t)))
      (setq status (cdr status)))
    (if active 
	(if busy
	    (setq str " Running")
	  (setq str " Ready")))
    (if proven (setq str (concat str " Proven")))
    (if init (setq str  (concat str " Initializing")))
    (if saved (setq str (concat str " Saved")))
    str))

(defun symp-prover-mode ()
  "Major mode for SyMP prover interaction."
  (interactive)
  ;; Save local values that we really need
  (let ((symp-buffer symp-server-process-buffer)
	(process symp-server-process))
    (comint-mode)
    ;; Define the major mode
    (setq major-mode 'symp-prover-mode)
    (setq mode-name "SyMP Prover")
    (use-local-map symp-prover-mode-map)
    ;; Restore local vars
    (setq symp-server-process-buffer symp-buffer)
    (setq symp-server-process process))
  ;; Set up new local vars
  (make-local-variable 'symp-prover-id)
  (make-local-variable 'symp-prover-name)
  (make-local-variable 'symp-prover-status)
  (make-local-variable 'symp-prover-status-string)
  (make-local-variable 'symp-prover-insert-marker)
  (setq symp-prover-status '("init"))
  (setq symp-prover-status-string
	(symp-prover-compute-status-string symp-prover-status))
  (setq symp-prover-id nil)
  (setq symp-prover-name nil)
  (setq symp-prover-insert-marker (make-marker))
  ;; Display the status of the prover in the mode line
  (setq mode-line-process 'symp-prover-status-string)
  ;; Change the prover status on the server status change
  (make-local-hook 'symp-server-status-hook)
  (add-hook 'symp-server-status-hook 'symp-prover-status-hook)
  ;; Insert the first prompt
  (goto-char (point-max))
  (insert "\n")
  (set-marker symp-prover-insert-marker (point))
  (setq comint-accum-marker symp-prover-insert-marker)
  (insert symp-prover-prompt)
  ;; Register the buffer with the server
  (symp-server-register-buffer))

(provide 'symp-prover-mode)
