;;;; Major mode for SyMP error messages.

(require 'symp-common)

(defun symp-proof-buffer nil
  "Buffer for displaying a proof.  This variable is buffer local, and
is set in each prover buffer.")
(make-variable-buffer-local 'symp-proof-buffer)

(defun symp-proof-find-buffer (prover-id)
  "Find the proof buffer for the PROVER-ID, or create one if not
found."
  (let ((old-buffer (current-buffer))
	(status symp-server-status)
	(process symp-server-process)
	(prover-buffer (symp-prover-buffer prover-id nil t))
	(buffer nil))
    (if prover-buffer
	;; there is a prover buffer, go there
	(let ((name nil))
	  (set-buffer prover-buffer)
	  (if (and (boundp 'symp-prover-name)
		   (stringp symp-prover-name))
	      (setq name symp-prover-name)
	    setq name "<unnamed>")
	  (if (and symp-proof-buffer
		   (bufferp symp-proof-buffer)
		   (buffer-live-p symp-proof-buffer))
	      ;; the error buffer already exists, just take it
	      (setq buffer symp-proof-buffer)
	    ;; no proof buffer, create a new one
	    (progn
	      (setq buffer (generate-new-buffer (concat "Proof:" name)))
	      ;; register it in the prover buffer
	      (setq symp-proof-buffer buffer)
	      ;; and set up the proof buffer properly
	      (set-buffer buffer)
	      (setq symp-server-process process)
	      (setq symp-server-status status)
	      (setq symp-server-process-buffer symp-buffer)
	      (symp-proof-mode)))
	  (set-buffer old-buffer))
      ;; if there is no prover buffer, give up, leave `buffer' nil.
      )
    buffer))

(defun symp-proof-display (prover-id proof)
  "Pop up the proof buffer for PROVER-ID and display the PROOF in it."
  (let ((old-buffer (current-buffer))
	(proof-buffer (symp-proof-find-buffer prover-id)))
    (if proof-buffer
	(progn
	  ;; put the message into the error buffer
	  (set-buffer proof-buffer)
	  (erase-buffer)
	  (if (listp proof)
	      (progn
		(insert "(\n")
		(while (consp proof)
		  (insert (format " %S\n" (car proof)))
		  (setq proof (cdr proof)))
		(insert ")"))
	    (format "%S" proof))
	  ;; go back and display the error buffer in another window
	  (set-buffer old-buffer)
	  (display-buffer proof-buffer t))
      ;; Ouch, no proof buffer!
      (symp-server-report-error 
       "emacs"
       (format "Can't find proof buffer for prover-id = %d.\n\nProof:\n%S"
	       prover-id proof)))))

(defun symp-proof-mode ()
  "SyMP major mode for displaying proofs."
  (interactive)
  ;;; Define the major mode
  (setq major-mode 'symp-proof-mode)
  ;; Watch the status of the inferior SYMP process
  (setq mode-line-process 'symp-server-status)  
  (setq mode-name "SyMP Proof")
  ;; Register the buffer with the server
  (symp-server-register-buffer))

(provide 'symp-proof-mode)
